/*************************************************************************//**
 * @file LFAST.c
 * @copyright Freescale 2014 All Rights Reserved                             
 * @version 1.0                                                              
 * @brief This file contains the LFAST clock configuration
 * @date 15-Dec-14
 * @author MSG-Austin                                                       */
/*===========================================================================*
 * UPDATE HISTORY                                                            *
 * REV      AUTHOR      DATE       	DESCRIPTION OF CHANGE                    *
 * ---   -----------  ---------    	---------------------                    * 
 * 1.0     MSG-Austin 15-Dec-14     Original source from MSG-Austin          *
 *                                                                           *
 *===========================================================================*
 * COPYRIGHT:                                                                *
 *  Freescale Semiconductor, INC. All Rights Reserved. You are hereby        *
 *  granted a copyright license to use, modify, and distribute the           *
 *  SOFTWARE so long as this entire notice is retained without alteration    *
 *  in any modified and/or redistributed versions, and that such modified    *
 *  versions are clearly identified as such. No licenses are granted by      *
 *  implication, estoppel or otherwise under any patentsor trademarks        *
 *  of Freescale Semiconductor, Inc. This software is provided on an         *
 *  "AS IS" basis and without warranty.                                      *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, Freescale             *
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,       *
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A         *
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD     *
 *  TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY        *
 *  ACCOMPANYING WRITTEN MATERIALS.                                          *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, IN NO EVENT           *
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER       *
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,     *
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER            *
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.     *
 *                                                                           *
 *  Freescale Semiconductor assumes no responsibility for the                *
 *  maintenance and support of this software                                 *
 *                                                                           *
 ****************************************************************************/

/***************Files to be included******************************************/
#include "LFAST.h"
#include "Zipwire.h"

/*****************************************************************************/
/* Prototypes */
uint8_t LFAST_Configure(unsigned char master);

/*****************************************************************************/

/** \fn LFAST_Configure(unsigned char master)
\brief Configures the LFAST as either a Master or a Slave, sets the clocks and bus speeds, and configures PADs to set the LFAST link.
\param[in] master defines whether to configure the LFAST node as master or slave.
*/
uint8_t LFAST_Configure(unsigned char master)
{ 
  
  uint32_t LFAST_timeout;

  /* Steps are according to MPC5777M Reference Manual Rev. 3.1 */
  
  /* Step 0: Configure SIUL for SIPI operation */
  //SIPI_TXN - PD[6]  ALT 1 SIUL 54, SSS 0000_0001
  //SIPI_TXP - PA[14] ALT 1 SIUL 14, SSS 0000_0001
  //SIPI_RXN - PF[13] ALTX  SIUL 93,
  //SIPI_RXP - PD[7]  ALTX  SIUL 55, 
  SIUL2.MSCR_IO[54].R = 0x05000000; //TX_N PINS
  SIUL2.MSCR_IO[14].R = 0x05000001; //TX_P PINS
  SIUL2.MSCR_IO[55].R = 0x00200001; //RX_P PINS
  SIUL2.MSCR_IO[93].R = 0x00200000; //RX_N PINS

  /* Step 1: SLCR and RCDCR are programmed according to the LVDS parameters in the device data sheet. */
  
  LFAST_0.RCDCR.R = 0x000F0000;
  LFAST_0.SLCR.R = 0x12015402;
  
  /* Step 2: The PLLCR is programmed with configuration parameters for the PLL. */
  //LFAST_PLLCR[LPCFG] = 3, 2 x IBASE current
  //LFAST_PLLCR[FBDIV] = 15, Divide by 16
  //LFAST_PLLCR[PREDIV] = 0, Direct clock passed, PLL ref. divide by 1
  LFAST_0.PLLCR.R = 0x0000603C;

  
  
  if(master)	/* Configure as Master */
  { 
    /* Step 3: Initialize the Zipwire clock pin as an input to receive the clock from Slave */
    SIUL2.MSCR_IO[94].R = 0x00380001;
    
    /* Step 4: LCR is Programmed with configuration parameters for the LVDS */
    LFAST_0.LCR.R = LCR_LVTXOE|LCR_LVRFEN|LCR_LVRXOP_4|LCR_LVTXOP;
  
    /* Step 5: Write MCR[MSEN]=1 Then select LFAST modes by configuring MCR[CTSEN/DATAEN]. */
    /* Step 6: Select the fraction of sysclk in Low Speed Select mode */     // As we have a 20MHz reference, LSSEL must be enabled
    /* Step 7: Enable LFAST as Transmitter and receiver */
    /* Step 8: Write MCR[RXEN] = 1 and MCR[TXEN] = 1 to Negate LD Power Down, LR Disable and LR power down signals */
    LFAST_0.MCR.R |= MCR_MSEN|MCR_DATAEN|MCR_LSSEL|MCR_DRFEN|MCR_RXEN|MCR_TXEN;

    do
    {
      /* Step 9: Write ICR[ICLCPLD] = 0x31 to enable Slaves Tx Interface */
      /* Step 10: Write ICR[SNDICLC] = 1 and ICR[ICLCSEQ] = 1 */
      LFAST_0.ICR.R = ICR_ICLCSEQ|ICR_SNDICLC|ICR_ICLCPLD_31;

      /* Step 11: Write MCR[TXARBD]=0, Enable Tx arbiter and framer. When enabled, all frame requests and services are based on priority */
      LFAST_0.MCR.R &= ~MCR_TXARBD;

      /* Step 12: Verify that the ICLC transmission is confirmed */
      LFAST_timeout = 0;
      while((LFAST_0.ICR.R&ICR_SNDICLC) || !(LFAST_0.TISR.R&TISR_TXICLCF))
      {
        LFAST_timeout++;
        if(LFAST_timeout>2000) return (4);
      }

      /* Step 13: Clear TXICLCF Status Flag */
      LFAST_0.TISR.R = TISR_TXICLCF;

      /* Step 14: Write ICR[ICLCPLD] = 0x00 to check LFAST slaves status */
      LFAST_0.ICR.R &= ~ICR_ICLCPLD;//

      /* Step 15: Write ICR[SNDICLC] = 1 */
      LFAST_0.ICR.R |= ICR_SNDICLC;
    } /* Step 16: LFAST Slave status is confirmed by the occurrence of one of the following */
    while(!(LFAST_0.RIISR.R&RIISR_ICPSF)&&(LFAST_0.RIISR.R&RIISR_ICPFF));

    /* Step 17: Clear ICPSF */
    LFAST_0.RIISR.R = RIISR_ICPSF;  

    /* SPEED MODE CHANGE */
    /* Step 18: Write PLLCR[SWPON] = 1 to enable the LFAST masters PLL */
    LFAST_0.PLLCR.R |= PLLCR_SWPON;
    
    /* Step 19: Wait for PLL to relock */
    while(LFAST_0.PLLLSR.R&PLLLSR_PLLDIS);     /* Wait for PLL disable signal to be negated. */
    while(!(LFAST_0.PLLLSR.R&PLLLSR_PLDCR));   /* Wait for PLL to lock. */

    /* Step 20: Write ICLC start PLL frame, ICR[ICLCPLD] = 0x02 */
    LFAST_0.ICR.R &= ~ICR_ICLCPLD;
    LFAST_0.ICR.R |= ICR_ICLCPLD_02;
    
    /* Step 21: Write ICR[SNDICLC] = 1 */
    LFAST_0.ICR.R |= ICR_SNDICLC;

    /* Step 22: Verify that the ICLC transmission is confirmed */
    while((LFAST_0.ICR.R&ICR_SNDICLC) || !(LFAST_0.TISR.R&TISR_TXICLCF)){}

    /* Step 23: Clear TXICLCF Status Flag */
    LFAST_0.TISR.R |= TISR_TXICLCF;
    
    /* Step 24: Change LFAST slave TX interface */
    LFAST_0.ICR.R &= ~ICR_ICLCPLD;
    LFAST_0.ICR.R |= ICR_ICLCPLD_80;
    LFAST_0.ICR.R |= ICR_SNDICLC;
    
    /* Step 25: Verify that the ICLC transmission is confirmed */
    while((LFAST_0.ICR.R&ICR_SNDICLC) || !(LFAST_0.TISR.R&TISR_TXICLCF)){}

    /* Step 26: Clear TXICLCF Status Flag */
    LFAST_0.TISR.R |= TISR_TXICLCF;

    /* Step 27: Change LFAST slave RX interface */
    LFAST_0.ICR.R &= ~ICR_ICLCPLD;
    LFAST_0.ICR.R |= ICR_ICLCPLD_10;
    LFAST_0.ICR.R |= ICR_SNDICLC;
   
    /* Step 28: Verify that the ICLC transmission is confirmed */
    while((LFAST_0.ICR.R&ICR_SNDICLC) || !(LFAST_0.TISR.R&TISR_TXICLCF)){}

    /* Step 29: Clear TXICLCF Status Flag */
    LFAST_0.TISR.R |= TISR_TXICLCF;
    
    /* Step 30: Clear ICLCSEQ to be able to set the TDR bits */
    LFAST_0.ICR.R &= ~ICR_ICLCSEQ;
    
    /* Step 31 and 32: Write SCR[TDR] = 1 and SCR[RDR] = 1 to change LFAST Master Interface's speed */
    LFAST_0.SCR.R |= SCR_RDR|SCR_TDR;

    /* Step 33: Wait until GSR reflects speed change */
    while(!(LFAST_0.GSR.R&GSR_LDSM) || !(LFAST_0.GSR.R&GSR_DRSM));
    
    /* Step 34: Enable ICLCSEQ to send final command */
    LFAST_0.ICR.R |= ICR_ICLCSEQ;

    /* Step 35 and 36.- Write ICR[ICLCPLD] = 0x00 to confirm speed change in the LFAST SLAVE. */
    LFAST_0.ICR.R &= ~ICR_ICLCPLD;
    LFAST_0.ICR.R |= ICR_SNDICLC;

    /* Step 37: Verify that the ICLC transmission is confirmed */
    while((LFAST_0.ICR.R&ICR_SNDICLC) || !(LFAST_0.TISR.R&TISR_TXICLCF)){}

    /* Step 38: Clear TXICLCF Status Flag */
    LFAST_0.TISR.R = TISR_TXICLCF;
  
    /* Step 39: Write MCR[TXARBD] = 1 */
    LFAST_0.MCR.R |= MCR_TXARBD;

    /* Step 40: The LFAST slaves speed mode is confirmed by RIISR[ICPSF] = 1 */
    // if RIISR[ICPFF] = 1, LFAST is in slow speed mode.
    if(LFAST_0.RIISR.R&RIISR_ICPSF)
    {
      /* Step 41: Frame Arbitration enabled. */
      LFAST_0.ICR.R &= ~ICR_ICLCSEQ;
      LFAST_0.MCR.R &= ~MCR_TXARBD;
      return(0);
    }
    else
    {
      //Failed to confirm speed from slave
      return(1);
    }

  }else /* Configure as Slave */
  {
    /* Step 3: Initialize the Zipwire clock as an output */
    //Matterhorn uses PF[14] to input/output the clock to another SIPI interface
    SIUL2.MSCR_IO[94].R = 0x22800001;         /* Enable LFAST CLK pin Output.  0x02000001*/

    /* Step 4: LCR is Programmed with configuration parameters for the LVDS */
    LFAST_0.LCR.R = LCR_LVTXOE|LCR_LVRFEN|LCR_LVRXOP_4|LCR_LVTXOP;
    
    /* Step 5: Write MCR[MSEN] = 0 Then select LFAST modes by configuring MCR[CTSEN/DATAEN]. */
    /* Step 6: Select the fraction of sysclk in Low Speed Select mode */     // As we have a 20MHz reference, LSSEL must be enabled
    LFAST_0.MCR.R &= ~MCR_MSEN;
    LFAST_0.MCR.R |= MCR_DATAEN|MCR_LSSEL;
    
    /* Step 7: Enable LFAST as Transmitter and receiver */
    LFAST_0.MCR.R |= MCR_DRFEN;
    
    /* Step 8: The LR is enabled by writing MCR[RXEN] = 1 This negates LR Disable and LR power down signals */
    LFAST_0.MCR.R |= MCR_RXEN; //Enable interface to receive   
    
    /* Step 9: Configure bits to send ping automatically and be able to configure the clock remotely */
    LFAST_0.PICR.R |= PICR_PNGAUTO;
    LFAST_0.SCR.R |= SCR_DRMD;

    /* Step 10: LD enable signal will be received by the master sending an ICLC of 0x31 */
    while(!(LFAST_0.RIISR.R&RIISR_ICTEF) || !(LFAST_0.MCR.R&MCR_TXEN));
    LFAST_0.RIISR.R = 0;

    /* Step 11: After a write to MCR[TXARBD] = 0, a ping frame will be sent on one of the
       following conditions: PICR[PNGAUTO]=1 and 0x00 is received in ICLC */
    LFAST_0.MCR.R &= ~MCR_TXARBD;
    
    //SPEED MODE CHANGE
    /* Step 12: */
    while(!(LFAST_0.RIISR.R&RIISR_ICPRF));

    /* Step 13:  PLL will start when ICLC frame with 0x02 as payload is received, HW will write RIISR[IPONF] = 1 */
    while(!(LFAST_0.RIISR.R&RIISR_ICPONF));

    /* Step 14: Wait for PLL to lock */
    while(LFAST_0.PLLLSR.R&PLLLSR_PLLDIS);        /* Wait for PLL disable signal to be negated. */
    while(!(LFAST_0.PLLLSR.R&PLLLSR_PLDCR));      /* Wait for PLL to lock. */

    /* Step 15: Speed of Tx interface is changed by receiving a ICLC frame with 0x80 Payload, HW will write RIISR[ICTFF] and SCR[TDR] = 1 */
    while(!(LFAST_0.RIISR.R&RIISR_ICTFF) || !(LFAST_0.SCR.R&SCR_TDR));

    /* Step 16: Speed of Rx interface is changed by receiving a ICLC frame with 0x10 Payload, HW will write RIISR[ICRFF] and SCR[RDR] = 1 */
    while(!(LFAST_0.RIISR.R&RIISR_ICRFF) || !(LFAST_0.SCR.R&SCR_RDR));

    /* Step 17: The interface will send a ping automatically */
    LFAST_0.RIISR.R = 0xFFFFFFFF;
    return(0);
  }
}
