/*************************************************************************//**
 * @file SIPI_HSSL_API.c                                                    
 * @copyright Freescale 2014 All Rights Reserved                             
 * @version 1.1                                                              
 * @brief Provides functions for SIPI_API for Matterhorn
 * @date 24-Nov-14
 * @author S.Becerra                                                        */
/*===========================================================================*
 * UPDATE HISTORY                                                            *
 * REV      AUTHOR      DATE       	DESCRIPTION OF CHANGE                    *
 * ---   -----------  ---------    	---------------------                    *
 * 1.0                              Initial version                          *
 *                                                                           *
 * 1.1   S.Becerra    24-Nov-14     Clarified comments, rid code of bit-     *
 *                                  fields in critical registers             *
 *                                                                           *
 *===========================================================================*
 * COPYRIGHT:                                                                *
 *  Freescale Semiconductor, INC. All Rights Reserved. You are hereby        *
 *  granted a copyright license to use, modify, and distribute the           *
 *  SOFTWARE so long as this entire notice is retained without alteration    *
 *  in any modified and/or redistributed versions, and that such modified    *
 *  versions are clearly identified as such. No licenses are granted by      *
 *  implication, estoppel or otherwise under any patentsor trademarks        *
 *  of Freescale Semiconductor, Inc. This software is provided on an         *
 *  "AS IS" basis and without warranty.                                      *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, Freescale             *
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,       *
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A         *
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD     *
 *  TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY        *
 *  ACCOMPANYING WRITTEN MATERIALS.                                          *
 *                                                                           *
 *  To the maximum extent permitted by applicable law, IN NO EVENT           *
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER       *
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,     *
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER            *
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.     *
 *                                                                           *
 *  Freescale Semiconductor assumes no responsibility for the                *
 *  maintenance and support of this software                                 *
 *                                                                           *
 ****************************************************************************/

#include "SIPI_HSSL_Header_v4.h"

#define MASK_LSB 0xFFFFFFFC
#define ADD_INC_MIN 0
#define ADD_INC_MAX 3
#define SIPI_TIMEOUT_VALUE 0x00000400

uint8_t word_length;

void configureTDC(uint32_t CDR, uint32_t * dma_array);

/*****************************************************************************/
/*                          SIPI Read Function                               */		
/*****************************************************************************/
uint8_t SIPI_read(DATA_TEMPLATE_t data_address, CHANNEL_t channel, uint8_t injected_error)
{

	/* Calculate the data size to be read */
	/* 0 -> 32 bit reading, 1-> 16 bit reading, 2 -> 8 bit reading. Check if the command is consistent with the address */	
  switch(data_address.Size)
  {
    case(32):
      word_length = 0x02;
      break;
    case(16):
      /* if the LSB bit in the address are different to zero do nothing, otherwise specify a 16 bit reading */
      if (data_address.Address^0x00000001)
        return(10);
      else 
        word_length = 0x01;
      break;
    case(8):
      /* if the two LSB bits in the address are different to zero do nothing, otherwise specify an 8 bit reading */
      if ((data_address.Address^0x00000011) == 0x03)
        return(10);
      else 
        word_length = 0x03;
      break;
    default:
      return(1); //Invalid width		
  }

  /* Set up channel configuration register and start transfer */
  switch(channel.Number)
  {
    case(0):
      if (!SIPI_0.CSR0.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        //Configure CCR0 and enable the channel and request
        SIPI_0.CCR0.B.WL = word_length;
        SIPI_0.CCR0.R &= ~(CCR0_TC|CCR0_IDT|CCR0_WRT|CCR0_ST);
        SIPI_0.CCR0.R |= (CCR0_RRT|CCR0_CHEN);
        SIPI_0.MCR.R &= ~MCR_INIT;
        SIPI_0.CAR0.R = data_address.Address;

        while(!SIPI_0.CSR0.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE0)
          {
            return(4);  //Time-out error
          }
        }
        //Read OK, clear flags.
        SIPI_0.CSR0.R = CSR0_RAR;
        SIPI_0.ERR.R = ERR_TOE0;
        
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(1):
      if (!SIPI_0.CSR1.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        //Configure CCR1 and enable the channel and request
        SIPI_0.CCR1.B.WL = word_length;
        SIPI_0.CCR1.R &= ~(CCR1_TC|CCR1_IDT|CCR1_WRT|CCR1_ST);
        SIPI_0.CCR1.R |= (CCR1_RRT|CCR1_CHEN);
        SIPI_0.MCR.R &= ~MCR_INIT;
        SIPI_0.CAR1.R = data_address.Address;
        while(!SIPI_0.CSR1.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE1)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(2):
      if (!SIPI_0.CSR2.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        //Configure CCR2 and enable the channel and request
        SIPI_0.CCR2.B.WL = word_length;
        SIPI_0.CCR2.R &= ~(CCR2_TC|CCR2_IDT|CCR2_WRT|CCR2_ST);
        SIPI_0.CCR2.R |= (CCR2_RRT|CCR2_CHEN);
        SIPI_0.MCR.R &= ~MCR_INIT;
        SIPI_0.CAR2.R = data_address.Address;
        
        while(!SIPI_0.CSR2.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE2)
          {
            return(4); //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
    case(3):
      if (!SIPI_0.CSR3.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        //Configure CCR3 and enable the channel and request
        SIPI_0.CCR3.B.WL = word_length;
        SIPI_0.CCR3.R &= ~(CCR3_TC|CCR3_IDT|CCR3_WRT|CCR3_ST);
        SIPI_0.CCR3.R |= (CCR3_RRT|CCR3_CHEN);
        SIPI_0.MCR.R &= ~MCR_INIT;
        SIPI_0.CAR3.R = data_address.Address;
        
        while(!SIPI_0.CSR3.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE3)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
    default:
      return(3); //Invalid Channel Number
  }

  return(0); //Command sent successfully
}




/*****************************************************************************/
/*               SIPI Read Channel Data Register Function                    */		
/*****************************************************************************/

uint32_t SIPI_read_channel_data(CHANNEL_t channel) 
{
  switch(channel.Number)
  {
    case(0):
      return(SIPI_0.CDR0.R);  //Return the Channel Data Register
    case(1):
      return(SIPI_0.CDR1.R);  //Return the Channel Data Register
    case(2):
      return(SIPI_0.CDR2[0].R); //Return the Channel Data Register
    case(3):
      return(SIPI_0.CDR3.R);  //Return the Channel Data Register
    default:
      return(0);            //Invalid Channel Number
  }
};



/*****************************************************************************/
/*                     SIPI Read Multiple Function                           */		
/*****************************************************************************/
uint8_t SIPI_multiple_read(DATA_TEMPLATE_t * read_array, uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint32_t * read_temp)
{

  switch(channel.Number)
  {

    uint32_t i;
    uint8_t Read_Setup_Result;

    case(0):
      //Check if Channel is free
      if (!SIPI_0.CSR0.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

          //First Read OK, clear flags.
          SIPI_0.CSR0.R = CSR0_RAR;
          SIPI_0.ERR.R = ERR_TOE0;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(1):
      //Check if Channel is free
      if (!SIPI_0.CSR1.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly
          
          //First Read OK, clear flags.
          SIPI_0.CSR1.R = CSR1_RAR;
          SIPI_0.ERR.R = ERR_TOE1;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(2):
      //Check if Channel is free
      if (!SIPI_0.CSR2.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

          //First Read OK, clear flags.
          SIPI_0.CSR2.R = CSR2_RAR;
          SIPI_0.ERR.R = ERR_TOE2;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(3):
      //Check if Channel is free
      if (!SIPI_0.CSR3.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

        //First Read OK, clear flags.
        SIPI_0.CSR3.R = CSR3_RAR;
        SIPI_0.ERR.R = ERR_TOE3;
        //assign read data
        *read_temp = SIPI_read_channel_data(channel);
        read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

      default:
        return(3);	//Invalid channel

  }
};



/*****************************************************************************/
/*                          SIPI Write Function                              */		
/*****************************************************************************/

uint8_t SIPI_write(DATA_TEMPLATE_t write_data, CHANNEL_t channel, uint8_t injected_error)
{

  /* Calculate the data size to be read */
  /* 0 -> 32 bit reading, 1-> 16 bit reading, 2 -> 8 bit reading. Check if the command is consistent with the address */	
  switch(write_data.Size)
  {
    case(32):
      word_length = 0x02;
      break;
    case(16):
      /* if the LSB bit in the address are different to zero do nothing, otherwise specify a 16 bit reading */
      if (write_data.Address^0x00000001)
        return(10);
      else 
        word_length = 0x01;
      break;
    case(8):
      /* if the two LSB bits in the address are different to zero do nothing, otherwise specify an 8 bit reading */
      if ((write_data.Address^0x00000011) == 0x03)
        return(10);
      else 
        word_length = 0x03;
      break;
    default:
      return(1); //Invalid width		
  }


  /* Set up channel config register and start transfer */
  switch(channel.Number)
  {
    case(0):
      if (!SIPI_0.CSR0.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR0.B.WL = word_length;
        
        //Clear other commands
        SIPI_0.CCR0.R &= ~(CCR0_TC|CCR0_IDT|CCR0_RRT|CCR0_ST);
        
        //Enable the channel and request
        SIPI_0.CCR0.R |= CCR0_WRT|CCR0_CHEN;
        SIPI_0.MCR.R &= ~MCR_INIT;
        SIPI_0.CDR0.R = write_data.Data; 
        SIPI_0.CAR0.R = write_data.Address;
        
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI_0.CSR0.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE0||SIPI_0.ERR.B.ACKE0)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(1):
      if (!SIPI_0.CSR1.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR1.B.WL = word_length;
        
        //Clear other commands
        SIPI_0.CCR1.R &= ~(CCR1_TC|CCR1_IDT|CCR1_RRT|CCR1_ST);
        
        //Enable the channel and request
        SIPI_0.CCR1.R |= CCR1_WRT|CCR1_CHEN;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        SIPI_0.CDR1.R = write_data.Data; 
        SIPI_0.CAR1.R = write_data.Address;
        
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI_0.CSR1.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE1||SIPI_0.ERR.B.ACKE1)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(2):
      if (!SIPI_0.CSR2.B.CB) //If channel not busy
        {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR2.B.WL = word_length;
        
        //Clear other commands
        SIPI_0.CCR2.R &= ~(CCR2_TC|CCR2_IDT|CCR2_RRT|CCR2_ST);
        
        //Enable the channel and request
        SIPI_0.CCR2.R |= CCR2_WRT|CCR2_CHEN;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        SIPI_0.CDR2[0].R = write_data.Data; 
        SIPI_0.CAR2.R = write_data.Address;
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI_0.CSR2.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE2||SIPI_0.ERR.B.ACKE2)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(3):
      if (!SIPI_0.CSR3.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR3.B.WL = word_length;
        
        //Clear other commands
        SIPI_0.CCR3.R &= ~(CCR3_TC|CCR3_IDT|CCR3_RRT|CCR3_ST);
        
        //Enable the channel and request
        SIPI_0.CCR3.R |= CCR3_WRT|CCR3_CHEN;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        SIPI_0.CDR3.R = write_data.Data; 
        SIPI_0.CAR3.R = write_data.Address;
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI_0.CSR3.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE3||SIPI_0.ERR.B.ACKE3)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

    return(0); //If you get to here, write request successfully submitted.
  }
  return(0);
}


/*****************************************************************************/
/*                    SIPI Multiple Write Function                           */		
/*****************************************************************************/

uint8_t SIPI_multiple_write(DATA_TEMPLATE_t write_array[], uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint8_t DMA_Enable, uint32_t * dma_array)
{

  uint32_t i = 0;
  uint8_t Write_Setup_Result = 0;

  switch(channel.Number)
  {
    case(0):
      //Check if Channel is free
      if (!SIPI_0.CSR0.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {
          SIPI_0.MCR.B.MOEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x01;
          SIPI_0.CCR0.B.WL = 0x02;
          SIPI_0.MCR.B.TEN = 1;
          SIPI_0.CCR0.B.RRT = 0x00;
          SIPI_0.CCR0.B.WRT = 0x01;
          SIPI_0.CCR0.B.CHEN = 0x01;
          
          //Increase TimerPrescaler to MAX
          SIPI_0.MCR.B.PRSCLR = 0x400;        
          
          //Exit INIT mode and send the frame
          SIPI_0.CCR0.B.DEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x00;

          DMAMUX_1.CHCFG[0].R = 0;

          eDMA_0.ERQL.R = 0xFFFFFFFF;
          eDMA_0.CR.B.EMLM = 1;
          
          //Configure TCD
          configureTDC((uint32_t)&SIPI_0.CDR0.R, dma_array);
          
          //Select source to be SIPI CH 0  
          DMAMUX_1.CHCFG[0].R = 0x8B;

          while(!SIPI_0.CSR0.B.ACKR)
          {
            if(SIPI_0.ERR.B.TOE0||SIPI_0.ERR.B.ACKE0)
            {
              //NO ACKNOWLEDGE
              return(4);
            }
          }
        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            if(Write_Setup_Result){
              // Returns Write_Setup_results if error occurred (Write_Setup_Result non-zero)
              // 1 - Invalid Data Size
              // 2 - Channel Busy
              // 3 - Invalid Channel Number
              // 4 - Unknown Error
              if(Write_Setup_Result < 4)
                return Write_Setup_Result;
              else
                return 4;
            }

            //Verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI_0.CSR0.B.ACKR)
            {
              if(SIPI_0.ERR.B.TOE0||SIPI_0.ERR.B.ACKE0)
              {
                return(4);  //Write Failed
              }
            } 
            
          //Data Written, clear flags and start over
          SIPI_0.CSR0.R = CSR0_ACKR;
          SIPI_0.ERR.R = ERR_TOE0|ERR_ACKE0;
          }
        }
        
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(1):
      //Check if Channel is free
      if (!SIPI_0.CSR1.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {
          SIPI_0.MCR.B.MOEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x01;
          SIPI_0.CCR1.B.WL = 0x02;
          SIPI_0.MCR.B.TEN = 1;
          SIPI_0.CCR1.B.RRT = 0x00;
          SIPI_0.CCR1.B.WRT = 0x01;
          SIPI_0.CCR1.B.CHEN = 0x01;
          
          //Increase TimerPrescaler to MAX
          SIPI_0.MCR.B.PRSCLR = 0x400;        
          
          //Exit INIT mode and send the frame
          SIPI_0.CCR1.B.DEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x00;

          DMAMUX_1.CHCFG[0].R = 0;

          eDMA_0.ERQL.R = 0xFFFFFFFF;
          eDMA_0.CR.B.EMLM = 1;
          
          //Configure TCD
          configureTDC((uint32_t)&SIPI_0.CDR1.R, dma_array);

          //Select source to be SIPI CH 1  
          DMAMUX_1.CHCFG[0].R = 0x8C;

          while(!SIPI_0.CSR1.B.ACKR)
          {
            if(SIPI_0.ERR.B.TOE1||SIPI_0.ERR.B.ACKE1)
            {
              return(4);  //NO ACKNOWLEDGE
            }
          }

        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            if(Write_Setup_Result){
              // Returns Write_Setup_results if error occurred (Write_Setup_Result non-zero)
              // 1 - Invalid Data Size
              // 2 - Channel Busy
              // 3 - Invalid Channel Number
              // 4 - Unknown Error
              if(Write_Setup_Result < 4)
                return Write_Setup_Result;
              else
                return 4;
            }

            //Verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI_0.CSR1.B.ACKR)
            {
              if(SIPI_0.ERR.B.TOE1||SIPI_0.ERR.B.ACKE1)
              {
                return(4); //Write Failed
              }
            } 
            
            //Data Written, clear flags and start over
            SIPI_0.CSR1.R = CSR1_ACKR;
            SIPI_0.ERR.R = ERR_TOE1|ERR_ACKE1;
          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(2):
      //Check if Channel is free
      if (!SIPI_0.CSR2.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {
          SIPI_0.MCR.B.MOEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x01;
          SIPI_0.CCR2.B.WL = 0x02;
          SIPI_0.MCR.B.TEN = 1;
          SIPI_0.CCR2.B.RRT = 0x00;
          SIPI_0.CCR2.B.WRT = 0x01;
          SIPI_0.CCR2.B.CHEN = 0x01;
          
          //Increase TimerPrescaler to MAX
          SIPI_0.MCR.B.PRSCLR = 0x400;        
          
          //Exit INIT mode and send the frame
          SIPI_0.CCR2.B.DEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x00;

          DMAMUX_1.CHCFG[0].R = 0;

          eDMA_0.ERQL.R = 0xFFFFFFFF;
          eDMA_0.CR.B.EMLM = 1;
          
          //Configure TCD
          configureTDC((uint32_t)&SIPI_0.CDR2[0].R, dma_array);

          //Select source to be SIPI CH 2  
          DMAMUX_1.CHCFG[0].R = 0x8D;

          while(!SIPI_0.CSR2.B.ACKR)
          {
            if(SIPI_0.ERR.B.TOE2||SIPI_0.ERR.B.ACKE2)
            {
              return(4);  //NO ACKNOWLEDGE
            }
          }
        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            if(Write_Setup_Result){
              // Returns Write_Setup_results if error occurred (Write_Setup_Result non-zero)
              // 1 - Invalid Data Size
              // 2 - Channel Busy
              // 3 - Invalid Channel Number
              // 4 - Unknown Error
              if(Write_Setup_Result < 4)
                return Write_Setup_Result;
              else
                return 4;
            }

            //If you got here then read set up correctly, now look for result
            //Now verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI_0.CSR2.B.ACKR)
            {
              if(SIPI_0.ERR.B.TOE2||SIPI_0.ERR.B.ACKE2)
              {
                return(4);  //Write Failed
              }
            } 
            
            //Data Written, clear flags and start over
            SIPI_0.CSR2.R = CSR2_ACKR;
            SIPI_0.ERR.R = ERR_TOE2|ERR_ACKE2;
          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(3):
      //Check if Channel is free
      if (!SIPI_0.CSR3.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {
          SIPI_0.MCR.B.MOEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x01;
          SIPI_0.CCR3.B.WL = 0x02;
          SIPI_0.MCR.B.TEN = 1;
          SIPI_0.CCR3.B.RRT = 0x00;
          SIPI_0.CCR3.B.WRT = 0x01;
          SIPI_0.CCR3.B.CHEN = 0x01;
          
          //Increase TimerPrescaler to MAX
          SIPI_0.MCR.B.PRSCLR = 0x400;        
          
          //Exit INIT mode and send the frame
          SIPI_0.CCR3.B.DEN = 0x01;
          SIPI_0.MCR.B.INIT = 0x00;

          DMAMUX_1.CHCFG[0].R = 0;

          eDMA_0.ERQL.R = 0xFFFFFFFF;
          eDMA_0.CR.B.EMLM = 1;
          
          //Configure TCD
          configureTDC((uint32_t)&SIPI_0.CDR3.R, dma_array);

          //Select source to be SIPI CH 3  
          DMAMUX_1.CHCFG[0].R = 0x8E;

          while(!SIPI_0.CSR3.B.ACKR)
          {
            if(SIPI_0.ERR.B.TOE3||SIPI_0.ERR.B.ACKE3)
            {
              return(4);  //NO ACKNOWLEDGE
            }
          }

        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command
            
            if(Write_Setup_Result){
              // Returns Write_Setup_results if error occurred (Write_Setup_Result non-zero)
              // 1 - Invalid Data Size
              // 2 - Channel Busy
              // 3 - Invalid Channel Number
              // 4 - Unknown Error
              if(Write_Setup_Result < 4)
                return Write_Setup_Result;
              else
                return 4;
            }

            //Verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI_0.CSR3.B.ACKR)
            {
              if(SIPI_0.ERR.B.TOE3||SIPI_0.ERR.B.ACKE3)
              {
                return(4);  //Write Failed
              }
            }
            
            //Data Written, clear flags and start over
            SIPI_0.CSR3.R = CSR3_ACKR;
            SIPI_0.ERR.R = ERR_TOE3|ERR_ACKE3;
          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    default:
      return(9);	//Unknown Error
  }


};


/*****************************************************************************/
/*                          SIPI ID Function                                 */		
/*****************************************************************************/

uint8_t SIPI_ID(uint32_t * id_array, CHANNEL_t channel)
{
  switch(channel.Number)
  {
    case(0):
      if (!SIPI_0.CSR0.B.CB) 
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR0.R &= ~(CCR0_WL|CCR0_RRT|CCR0_WRT|CCR0_ST);
        SIPI_0.CCR0.R |= CCR0_WL_2|CCR0_CHEN|CCR0_IDT;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        //Initiate transfer by writing in CAR.
        SIPI_0.CAR0.R = (uint32_t)id_array;

        while(!SIPI_0.CSR0.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE0)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(1):
      if (!SIPI_0.CSR1.B.CB)  //Check channel busy status
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR1.R &= ~(CCR1_WL|CCR1_RRT|CCR1_WRT|CCR1_ST);
        SIPI_0.CCR1.R |= CCR1_WL_2|CCR1_CHEN|CCR1_IDT;
        SIPI_0.MCR.R &= ~MCR_INIT;

        //Initiate transfer by writing in CAR.
        SIPI_0.CAR1.R = (uint32_t)id_array;

        while(!SIPI_0.CSR1.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE1)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(2):
      if (!SIPI_0.CSR2.B.CB)  //Check channel busy status
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR2.R &= ~(CCR2_WL|CCR2_RRT|CCR2_WRT|CCR2_ST);
        SIPI_0.CCR2.R |= CCR2_WL_2|CCR2_CHEN|CCR2_IDT;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        //Initiate transfer by writing in CAR.
        SIPI_0.CAR2.R = (uint32_t)id_array;

        while(!SIPI_0.CSR2.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE2)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
    
    case(3):
      if (!SIPI_0.CSR3.B.CB)  //Check channel busy status
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR3.R &= ~(CCR3_WL|CCR3_RRT|CCR3_WRT|CCR3_ST);
        SIPI_0.CCR3.R |= CCR3_WL_2|CCR3_CHEN|CCR3_IDT;
        SIPI_0.MCR.R &= ~MCR_INIT;
        
        //Initiate transfer by writing in CAR.
        SIPI_0.CAR3.R = (uint32_t)id_array;

        while(!SIPI_0.CSR3.B.RAR)
        {
          //Poll for the time-out flag
          if(SIPI_0.ERR.B.TOE3)
          {
            return(4);  //Time-out error
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    default:
      return(3); //Invalid Channel Number

  }

  return(0); //Command sent successfully.
};




/*****************************************************************************/
/*                  SIPI Initialize Initiator Function                       */		
/*****************************************************************************/

uint8_t SIPI_init_INITIATOR(uint16_t Clock_Prescale)
{

  SIPI_0.MCR.B.MOEN = 1;			//First Enable Module
  SIPI_0.MCR.B.INIT = 1;			//Place module into INIT mode

  switch(Clock_Prescale)
  {
    case(64):
      SIPI_0.MCR.B.PRSCLR = 0x040;
      break;
    case(128):
      SIPI_0.MCR.B.PRSCLR = 0x080;
      break;
    case(256):
      SIPI_0.MCR.B.PRSCLR = 0x100;
      break;
    case(512):
      SIPI_0.MCR.B.PRSCLR = 0x200;
      break;
    case(1024):
      SIPI_0.MCR.B.PRSCLR = 0x400;
      break;
    default:
      SIPI_0.MCR.B.PRSCLR = 0x040;
      return(1);  //Incorrect Clock Prescale Selected, default of 64 set.
  }

  SIPI_0.MCR.B.INIT = 0;
  return(0);

}


/*****************************************************************************/
/*                   SIPI Initialize Target Function                         */		
/*****************************************************************************/

uint8_t SIPI_init_TARGET(uint32_t max_count, uint32_t reload_address, uint8_t Add_Inc)
{
  SIPI_0.MCR.R |= MCR_MOEN;  //Enable module
  SIPI_0.MCR.R |= MCR_INIT;  //Place module into INIT mode

  SIPI_0.MAXCR.R = (max_count & MASK_LSB); 		//2 LSB are reserved.
  SIPI_0.ARR.R = (reload_address & MASK_LSB);	//2 LSB are reserved.

  if (Add_Inc <= ADD_INC_MAX)
  {
    SIPI_0.MCR.B.AID = Add_Inc;
  }
  else
  {
    return(1);  //Incorrect Address Increment Selected.
  }
  
  SIPI_0.MCR.R |= MCR_TEN; //Enable target
  SIPI_0.CTOR0.R = SIPI_TIMEOUT_VALUE;
  //ENABLE ALL CHANNELS IN TARGET
  SIPI_0.CCR0.R = CCR0_CHEN;
  SIPI_0.CCR1.R = CCR1_CHEN;
  SIPI_0.CCR2.R = CCR2_CHEN;
  SIPI_0.CCR3.R = CCR3_CHEN;

  SIPI_0.MCR.R &= ~MCR_INIT;  //Bring module back up

  if (!SIPI_0.SR.B.MCR)
  {
    return(0);
  }
  else
  {
    return(2);  //Error with Max Count Address and SIPI Address Count Register
  }
}

/*****************************************************************************/
/*                 SIPI Initialize Channel Function                          */		
/*****************************************************************************/

uint8_t SIPI_init_channel(CHANNEL_t channel, uint8_t mode, uint8_t error_int_enable, uint8_t data_int_enable)
{
  SIPI_0.MCR.R |= MCR_MOEN;  //Enable module
  SIPI_0.MCR.R |= MCR_INIT;
  switch(channel.Number)
  {
    case(0):	
      //Check requested mode
      switch(mode)
      {
        case(0):                    //Run Mode
          SIPI_0.CCR0.B.CHEN = 1;
          break;
        case(1):                    //Disable
          SIPI_0.CCR0.B.CHEN = 0;
          break;
        case(2):                    //Clear Error
          SIPI_0.CCR0.B.CHEN = 0;
          SIPI_0.ERR.R = ERR_ACKE0|ERR_TIDE0|ERR_TOE0;
          SIPI_0.CCR0.B.CHEN = 1;
          break;
        case(3):                    //Stop
          while (SIPI_0.CSR0.B.CB){}
          SIPI_0.CCR0.B.CHEN = 0;
          break;
        default:                    //Incorrect Channel Mode Selection
          SIPI_0.MCR.R &= ~MCR_INIT;
          return(1);
      }
      //Setup Error Interrupts
      SIPI_0.CIR0.B.TOIE = error_int_enable;
      SIPI_0.CIR0.B.TIDIE = error_int_enable;
      SIPI_0.CIR0.B.ACKIE = error_int_enable;

      //Setup Data Interrupts
      SIPI_0.CIR0.B.WAIE = data_int_enable;
      SIPI_0.CIR0.B.RAIE = data_int_enable;
      SIPI_0.CIR0.B.TCIE = data_int_enable;

      //Setup Channel Timeout Value
      SIPI_0.CTOR0.B.TOR = channel.Timeout;
      SIPI_0.MCR.R &= ~MCR_INIT;
      return(0);

    case(1):
      //Check requested mode
      switch(mode)
      {
        case(0):                    //Run Mode
          SIPI_0.CCR1.B.CHEN = 1;
          break;
        case(1):                    //Disable
          SIPI_0.CCR1.B.CHEN = 0;
          break;
        case(2):                   //Clear Error
          SIPI_0.CCR1.B.CHEN = 0;
          SIPI_0.ERR.R = ERR_ACKE1|ERR_TIDE1|ERR_TOE1;
          SIPI_0.CCR1.B.CHEN = 1;
          break;
        case(3):                    //Stop
          while (SIPI_0.CSR1.B.CB){}
          SIPI_0.CCR1.B.CHEN = 0;
          break;
        default:                    //Incorrect Channel Mode Selection
          SIPI_0.MCR.R &= ~MCR_INIT;
          return(1);
      }
      //Setup Error Interrupts
      SIPI_0.CIR1.B.TOIE = error_int_enable;
      SIPI_0.CIR1.B.TIDIE = error_int_enable;
      SIPI_0.CIR1.B.ACKIE = error_int_enable;

      //Setup Data Interrupts
      SIPI_0.CIR1.B.WAIE = data_int_enable;
      SIPI_0.CIR1.B.RAIE = data_int_enable;
      SIPI_0.CIR1.B.TCIE = data_int_enable;

      //Setup Channel Timeout Value
      SIPI_0.CTOR1.B.TOR = channel.Timeout;
      SIPI_0.MCR.R &= ~MCR_INIT;
      return(0);

    case(2):
    //Check requested mode
      switch(mode)
      {
        case(0):                    //Run Mode
          SIPI_0.CCR2.B.CHEN = 1;
          break;
        case(1):                    //Disable
          SIPI_0.CCR2.B.CHEN = 0;
          break;
        case(2):                    //Clear Error
          SIPI_0.CCR2.B.CHEN = 0;
          SIPI_0.ERR.R = ERR_ACKE2|ERR_TIDE2|ERR_TOE2;
          SIPI_0.CCR2.B.CHEN = 1;
          break;
        case(3):                    //Stop
          while (SIPI_0.CSR2.B.CB){}
          SIPI_0.CCR2.B.CHEN = 0;
          break;
        default:                    //Incorrect Channel Mode Selection
          SIPI_0.MCR.R &= ~MCR_INIT;
          return(1);
      }
      //Setup Error Interrupts
      SIPI_0.CIR2.B.TOIE = error_int_enable;
      SIPI_0.CIR2.B.TIDIE = error_int_enable;
      SIPI_0.CIR2.B.ACKIE = error_int_enable;

      //Setup Data Interrupts
      SIPI_0.CIR2.B.WAIE = data_int_enable;
      SIPI_0.CIR2.B.RAIE = data_int_enable;
      SIPI_0.CIR2.B.TCIE = data_int_enable;

      //Setup Channel Timeout Value
      SIPI_0.CTOR2.B.TOR = channel.Timeout;
      SIPI_0.MCR.R &= ~MCR_INIT;
      return(0);

    case(3):
      //Check requested mode
      switch(mode)
      {
        case(0):                    //Run Mode
          SIPI_0.CCR3.B.CHEN = 1;
          break;
        case(1):                    //Disable
          SIPI_0.CCR3.B.CHEN = 0;
          break;
        case(2):                    //Clear Error
          SIPI_0.CCR3.B.CHEN = 0;
          SIPI_0.ERR.R = ERR_ACKE3|ERR_TIDE3|ERR_TOE3;
          SIPI_0.CCR3.B.CHEN = 1;
          break;
        case(3):                    //Stop
          while (SIPI_0.CSR3.B.CB){}
          SIPI_0.CCR3.B.CHEN = 0;
          break;
        default:                    //Incorrect Channel Mode Selection
          SIPI_0.MCR.R &= ~MCR_INIT;
          return(1);
      }
      //Setup Error Interrupts
      SIPI_0.CIR3.B.TOIE = error_int_enable;
      SIPI_0.CIR3.B.TIDIE = error_int_enable;
      SIPI_0.CIR3.B.ACKIE = error_int_enable;

      //Setup Data Interrupts
      SIPI_0.CIR3.B.WAIE = data_int_enable;
      SIPI_0.CIR3.B.RAIE = data_int_enable;
      SIPI_0.CIR3.B.TCIE = data_int_enable;

      //Setup Channel Timeout Value
      SIPI_0.CTOR3.B.TOR = channel.Timeout;
      SIPI_0.MCR.R &= ~MCR_INIT;
      return(0);

    default:
        return(2);  //Incorrect Channel
  }

}




/*****************************************************************************/
/*                          SIPI Trigger Function                            */		
/*****************************************************************************/

uint8_t SIPI_Trigger(CHANNEL_t channel)
{

  switch(channel.Number)
  {
    case(0):
      if (!SIPI_0.CSR0.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR0.R &= ~(CCR0_WL|CCR0_IDT|CCR0_RRT|CCR0_WRT|CCR0_ST);
        SIPI_0.CCR0.R |= CCR0_WL_2|CCR0_CHEN|CCR0_TC;
        SIPI_0.MCR.R &= ~MCR_INIT;

        //Verify that the trigger acknowledge is received
        while(!SIPI_0.CSR0.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE0||SIPI_0.ERR.B.ACKE0)
          {
            dummyPrintf("TRIGGER TRANSFER FAILED");	
            return(4);
          }
        }
        break;
      }
      else
      {
        return(2);	// Channel Busy
      }
      
    case(1):
      if (!SIPI_0.CSR1.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR1.R &= ~(CCR1_WL|CCR1_IDT|CCR1_RRT|CCR1_WRT|CCR1_ST);
        SIPI_0.CCR1.R |= CCR1_WL_2|CCR1_CHEN;
        SIPI_0.CCR1.R |= CCR1_TC;
        SIPI_0.MCR.R &= ~MCR_INIT;

        //Verify that the trigger acknowledge is received
        while(!SIPI_0.CSR1.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE1||SIPI_0.ERR.B.ACKE1)
          {
            dummyPrintf("TRIGGER TRANSFER FAILED");	
            return(4);
          }
        }
        break;
      }
      else
      {
        return(2);	// Channel Busy
      }
      
    case(2):
      if (!SIPI_0.CSR2.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR2.R &= ~(CCR2_WL|CCR2_IDT|CCR2_RRT|CCR2_WRT|CCR2_ST);
        SIPI_0.CCR2.R |= CCR2_WL_3|CCR2_CHEN;
        SIPI_0.CCR2.R |= CCR2_TC;
        SIPI_0.MCR.R &= ~MCR_INIT;

        //Verify that the trigger acknowledge is received
        while(!SIPI_0.CSR2.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE2||SIPI_0.ERR.B.ACKE2)
          {
            dummyPrintf("TRIGGER TRANSFER FAILED");	
            return(4);
          }
        }
        break;
      }
      else
      {
        return(2);	// Channel Busy
      }
      
    case(3):
      if (!SIPI_0.CSR3.B.CB) //If channel not busy
      {
        SIPI_0.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI_0.CCR3.R &= ~(CCR3_WL|CCR3_IDT|CCR3_RRT|CCR3_WRT|CCR3_ST);
        SIPI_0.CCR3.R |= CCR3_WL_3|CCR3_CHEN;
        SIPI_0.CCR3.R |= CCR3_TC;
        SIPI_0.MCR.R &= ~MCR_INIT;

        //Verify that the trigger acknowledge is received
        while(!SIPI_0.CSR3.B.ACKR)
        {
          if(SIPI_0.ERR.B.TOE3||SIPI_0.ERR.B.ACKE3)
          {
            dummyPrintf("TRIGGER TRANSFER FAILED");	
            return(4);
          }
        }
        break;
      }
      else
      {
        return(2);	// Channel Busy
      }
      
    default:
      return(1);		// Incorrect Channel Number	
  }

  /* Successful */
  return(0);
}





/*****************************************************************************/
/*                          SIPI Reset Function                              */		
/*****************************************************************************/

uint8_t SIPI_reset(void)
{
  /* Reset the module */
  SIPI_0.MCR.R = MCR_SR;

  /* Disable the module */
  SIPI_0.MCR.R |= MCR_MOEN|MCR_INIT;
  SIPI_0.MCR.R &= ~(MCR_TEN|MCR_INIT|MCR_MOEN);

  return(0);
}

/*****************************************************************************/
/*                     SIPI Module Mode Function                             */		
/*****************************************************************************/
 
uint8_t SIPI_module_mode(uint8_t Mode)
{
  switch(Mode)
  {
    case(0):
      //Disable the module
      SIPI_0.MCR.R |= MCR_INIT;
      SIPI_0.MCR.R &= ~(MCR_MOEN|MCR_TEN|MCR_INIT);
      return(0);
    case(1):
      //Enable the module and put into INIT mode
      SIPI_0.MCR.R |= MCR_INIT|MCR_MOEN;
      return(0);
    case(2):
      //Enable the module
      SIPI_0.MCR.R |= MCR_INIT|MCR_MOEN;
      SIPI_0.MCR.R &= ~MCR_INIT;
      return(0);
    default:
      //Disable the module
      SIPI_0.MCR.R |= MCR_INIT;
      SIPI_0.MCR.R &= ~(MCR_MOEN|MCR_TEN|MCR_INIT);
      return(1);  //Invalid Mode Selected
  }
}

/*****************************************************************************/
/*                     SIPI Stream Transfer Function                         */		
/*****************************************************************************/
 
uint8_t SIPI_stream_transfer(uint32_t * temp_data_stream, uint8_t initiator, uint8_t  length)
{
  unsigned char send_index;
  if(length>8)
  {
    dummyPrintf("INVALID LENGTH");
    return(1);
  }

  //CONFIGURE SIPI Module for Streaming Transfer
  if(initiator)
  {
    SIPI_0.MCR.B.MOEN = 0x01;
    SIPI_0.MCR.B.INIT = 0x01;
    SIPI_0.MCR.B.TEN = 0x01;
    SIPI_0.CCR2.B.WL = 0x02;
    SIPI_0.CCR2.B.RRT = 0x00;
    SIPI_0.CCR2.B.WRT = 0x01;
    SIPI_0.CCR2.B.CHEN = 0x01;
    SIPI_0.CCR2.B.ST = 0x01;
    SIPI_0.MCR.B.AID = 0x01; //Increment 4 addresses
    //Increase TimerPrescaler to MAX
    SIPI_0.MCR.B.PRSCLR = 0x400;        
    //Get out of INIT mode and send the frame
    SIPI_0.MCR.B.INIT = 0x00;
    //Configure the address and count in the Initiator
    SIPI_0.MAXCR.R = (uint32_t)(temp_data_stream+(length-1)); 
    SIPI_0.ARR.R = (uint32_t)temp_data_stream;
    SIPI_0.ACR.R = (uint32_t)temp_data_stream; // Not needed in the initiator node

    //Prepare the register contents
    for(send_index=0;send_index<length;send_index++)
    {
      switch(send_index)
      {
        case 0:
          SIPI_0.CDR2[0].R = *temp_data_stream;	
          break;
        case 1:
          SIPI_0.CDR2[1].R = *temp_data_stream;	
          break;
        case 2:
          SIPI_0.CDR2[2].R = *temp_data_stream;	
          break;
        case 3:
          SIPI_0.CDR2[3].R = *temp_data_stream;	
          break;
        case 4:
          SIPI_0.CDR2[4].R = *temp_data_stream;	
          break;
        case 5:
          SIPI_0.CDR2[5].R = *temp_data_stream;	
          break;
        case 6:
          SIPI_0.CDR2[6].R = *temp_data_stream;	
          break;
        case 7:
          SIPI_0.CDR2[7].R = *temp_data_stream;	
          break;
      }
      temp_data_stream++;
    }
    //Send the data
    temp_data_stream -= (length);
    SIPI_0.CAR2.R = (uint32_t)temp_data_stream;

    //Wait for the acknowledge
    while(!SIPI_0.CSR2.B.ACKR)
    {
      if(SIPI_0.ERR.B.TOE2||SIPI_0.ERR.B.ACKE2)
      {
        dummyPrintf("Acknowledge error");
        return(2);
      }
      dummyPrintf("Data written");
      //Now verify that the data was correctly written, verify in the target that the data was indeed written.
      return(0);
    }
  }
  else
  {
    SIPI_0.MCR.B.MOEN = 0x01;
    SIPI_0.MCR.B.INIT = 0x01;
    SIPI_0.CCR2.B.CHEN = 0x01;
    SIPI_0.MCR.B.TEN = 0x01;
    SIPI_0.CCR2.B.WL = 0x02;
    SIPI_0.CCR2.B.WRT = 0x01;
    SIPI_0.CCR2.B.ST = 0x01;
    SIPI_0.MCR.B.AID = 0x01; //Increment 4 addresses
    SIPI_0.MCR.B.INIT = 0x00;
    //Configure the address and count in the target
    SIPI_0.MAXCR.R = (uint32_t)(temp_data_stream+(length-1)); 
    SIPI_0.ARR.R = (uint32_t)temp_data_stream;
    SIPI_0.ACR.R = (uint32_t)temp_data_stream; // Not needed in the initiator node
  }
  return(0);
}

void configureTDC(uint32_t CDR, uint32_t * dma_array)
{
  eDMA_0.TCD[8].SADDR.R = (uint32_t)dma_array; //SEND the Source address
  eDMA_0.TCD[8].ATTR.B.SMOD = 0;
  eDMA_0.TCD[8].ATTR.B.SSIZE = 2;           //32Bit size transfers
  eDMA_0.TCD[8].ATTR.B.DMOD = 0;
  eDMA_0.TCD[8].ATTR.B.DSIZE = 2;           //32Bit size transfers
  eDMA_0.TCD[8].SOFF.R = 4;            //32Bit (4byte) source offset
  eDMA_0.TCD[8].NBYTES.MLOFFYES.R = 0x40002008; //0x7FFFF008;//Transfer 8 Bytes per minor loop (2 unsigned longs per minor loop)
  eDMA_0.TCD[8].SLAST.R = 0;           //No Source Offset correction upon major final count
  eDMA_0.TCD[8].DADDR.R = CDR;         //CDR as destination address
  eDMA_0.TCD[8].CITER.ELINKYES.B.ELINK  = 0;    //No channel to channel linking enabled
  eDMA_0.TCD[8].CITER.ELINKYES.B.LINKCH = 0;    //No channel to channel linking
  eDMA_0.TCD[8].CITER.ELINKYES.B.CITER = 4;           //Do 1 Major LOOP per request
  eDMA_0.TCD[8].DOFF.R = (vuint16_t)-4;           //When a major loop finishes, correct Destination address by 8 (2 minor loops reduced it by 8)
  eDMA_0.TCD[8].DLASTSGA.R = 0;       //Correct Last destination address
  eDMA_0.TCD[8].BITER.ELINKYES.B.ELINK = 0;     // Channel to channel linking disabled
  eDMA_0.TCD[8].BITER.ELINKYES.B.LINKCH = 0;
  eDMA_0.TCD[8].BITER.ELINKYES.B.BITER = 4;           // Starting major loop counter
  eDMA_0.TCD[8].CSR.B.BWC = 0x0;           /* No DMA Bandwidth Control / Throttling */
  eDMA_0.TCD[8].CSR.B.MAJORLINKCH = 0x0;   /* No Ch-Ch linking      */
  eDMA_0.TCD[8].CSR.B.ESG = 0x0;           /* Disable scatter gather on major loop      */
  eDMA_0.TCD[8].CSR.B.DREQ = 0x1;          /* Enable hardware request on citer=0 */
  eDMA_0.TCD[8].CSR.B.INTHALF = 0x0;       /* Disable half point interrupt               */
  eDMA_0.TCD[8].CSR.B.INTMAJOR = 0x0;        /* Disable interrupt on major loop complete   */  
  eDMA_0.TCD[8].CSR.B.DONE = 0;				
  eDMA_0.TCD[8].CSR.B.ACTIVE = 0;
  eDMA_0.TCD[8].CSR.B.INTMAJOR = 0;
  eDMA_0.TCD[8].CSR.B.START = 0;
}


