/*****************************************************************************/
/* FILE NAME: SIPI_HSSL_API.c                   COPYRIGHT (c) Freescale 2014 */
/*                                                       All Rights Reserved */
/* DESCRIPTION:                                                              */		
/*****************************************************************************/

#include "SIPI_HSSL_Header_v4.h"

#define MAX_COUNT 0xFFFFFFFF

uint8_t word_length;

/*****************************************************************************/
/*                          SIPI Read Function                               */		
/*****************************************************************************/
uint8_t SIPI_read(DATA_TEMPLATE_t data_address, CHANNEL_t channel, uint8_t injected_error)
{

	/* Calculate the data size to be read */
	/* 0 -> 32 bit reading, 1-> 16 bit reading, 2 -> 8 bit reading. Check if the command is consistent with the address */	
  switch(data_address.Size)
  {
    case(32):
      word_length = 0x02;
      break;
    case(16):
      /* if the LSB bit in the address are different to zero do nothing, otherwise specify a 16 bit reading */
      if (data_address.Address^0x00000001)
        return(10);
      else 
        word_length = 0x01;
      break;
    case(8):
      /* if the two LSB bits in the address are different to zero do nothing, otherwise specify an 8 bit reading */
      if ((data_address.Address^0x00000011) == 0x03)
        return(10);
      else 
        word_length = 0x03;
      break;
    default:
      return(1); //Invalid width		
  }

  /* Set up channel configuration register and start transfer */
  switch(channel.Number)
  {
    case(0):
      if (!SIPI.CSR0.B.CB) //If channel not busy
      {
        SIPI.MCR.B.INIT = 1; 
        SIPI.CCR0.B.WL = word_length;
        SIPI.MCR.B.TEN = 1;
        //Need to make sure other commands are not set
        SIPI.CCR0.B.TC = 0;
        SIPI.CCR0.B.IDT = 0;
        SIPI.CCR0.B.WRT = 0;
        SIPI.CCR0.B.ST = 0;
        //Enable the channel and request
        SIPI.CCR0.B.RRT = 1;
        SIPI.CCR0.B.CHEN = 1; //Enable the Channel to be used
        SIPI.MCR.B.INIT = 0;
        SIPI.CAR0.R = data_address.Address;

        while(!SIPI.CSR0.B.RAR)
        {
          //Read answer still not received
          //Poll for the timeout flag
          if(SIPI.ERR.B.TOE0)
          {
            //Timeout error!!!
            return(4);
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(1):
      if (!SIPI.CSR1.B.CB) //If channel not busy
      {
        SIPI.MCR.B.INIT = 1;
        SIPI.CCR1.B.WL = word_length;
        SIPI.MCR.B.TEN = 1;
        //Need to make sure other commands are not set
        SIPI.CCR1.B.TC = 0;
        SIPI.CCR1.B.IDT = 0;
        SIPI.CCR1.B.WRT = 0;
        SIPI.CCR1.B.ST = 0;
        //Enable the channel and request
        SIPI.CCR1.B.RRT = 1;
        SIPI.CCR1.B.CHEN = 1; //Enable the Channel to be used
        SIPI.MCR.B.INIT = 0;
        SIPI.CAR1.R = data_address.Address;
        while(!SIPI.CSR1.B.RAR)
        {
          //Read answer still not received
          //Poll for the timeout flag
          if(SIPI.ERR.B.TOE1)
          {
          //Timeout error!!!
          return(4);
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
      
    case(2):
      if (!SIPI.CSR2.B.CB) //If channel not busy
      {
        SIPI.MCR.B.INIT = 1; 
        SIPI.CCR2.B.WL = word_length;
        SIPI.MCR.B.TEN = 1;
        //Need to make sure other commands are not set
        SIPI.CCR2.B.TC = 0;
        SIPI.CCR2.B.IDT = 0;
        SIPI.CCR2.B.WRT = 0;
        SIPI.CCR2.B.ST = 0;
        //Enable the channel and request
        SIPI.CCR2.B.RRT = 1;
        SIPI.CCR2.B.CHEN = 1;
        SIPI.MCR.B.INIT = 0; 
        SIPI.CAR2.R = data_address.Address;
        while(!SIPI.CSR2.B.RAR)
        {
          //Read answer still not received
          //Poll for the timeout flag
          if(SIPI.ERR.B.TOE2)
          {
            //Timeout error!!!
            return(4);
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
    case(3):
      if (!SIPI.CSR3.B.CB) //If channel not busy
      {
        SIPI.MCR.B.INIT = 1; 
        SIPI.CCR3.B.WL = word_length;
        SIPI.MCR.B.TEN = 1;
        //Need to make sure other commands are not set
        SIPI.CCR3.B.TC = 0;
        SIPI.CCR3.B.IDT = 0;
        SIPI.CCR3.B.WRT = 0;
        SIPI.CCR3.B.ST = 0;
        //Enable the channel and request
        SIPI.CCR3.B.RRT = 1;
        SIPI.CCR3.B.CHEN = 1; //Enable the Channel to be used
        SIPI.MCR.B.INIT = 0; 
        SIPI.CAR3.R = data_address.Address;
        while(!SIPI.CSR3.B.RAR)
        {
          //Read answer still not received
          //Poll for the time-out flag
          if(SIPI.ERR.B.TOE3)
          {
            //Time-out error
            return(4);
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }
      break; 
    default:
      return(3); //Invalid Channel Number
      //break;
  }

  return(0); //If you reach here then Command sent successfully
}




/*****************************************************************************/
/*               SIPI Read Channel Data Register Function                    */		
/*****************************************************************************/

uint32_t SIPI_read_channel_data(CHANNEL_t channel) 
{
  switch(channel.Number)
  {
    case(0):
      return(SIPI.CDR0.R);  //Return the Channel Data Register
    case(1):
      return(SIPI.CDR1.R);  //Return the Channel Data Register
    case(2):
      return(SIPI.CDR20.R); //Return the Channel Data Register
    case(3):
      return(SIPI.CDR3.R);  //Return the Channel Data Register
    default:
      return(0);            //Invalid Channel Number
  }
};



/*****************************************************************************/
/*                     SIPI Read Multiple Function                           */		
/*****************************************************************************/
uint8_t SIPI_multiple_read(DATA_TEMPLATE_t * read_array, uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint32_t * read_temp)
{

  switch(channel.Number)
  {

    uint32_t i;
    uint8_t Read_Setup_Result;

    case(0):
      //Check if Channel is free
      if (!SIPI.CSR0.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

          //First Read OK, clear flags.
          SIPI.CSR0.R = CSR0_RAR;
          SIPI.ERR.R = ERR_TOE0;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(1):
      //Check if Channel is free
      if (!SIPI.CSR1.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly
          
          //First Read OK, clear flags.
          SIPI.CSR1.R = CSR1_RAR;
          SIPI.ERR.R = ERR_TOE1;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(2):
      //Check if Channel is free
      if (!SIPI.CSR2.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

          //First Read OK, clear flags.
          SIPI.CSR2.R = CSR2_RAR;
          SIPI.ERR.R = ERR_TOE2;
          //assign read data
          *read_temp = SIPI_read_channel_data(channel);
          read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(3):
      //Check if Channel is free
      if (!SIPI.CSR3.B.CB)
      {
        for (i=0; i<array_length; i++)
        {
          Read_Setup_Result = SIPI_read(read_array[i], channel, 0); //Attempt to set up Read command
          if(Read_Setup_Result)
            return(Read_Setup_Result); //Read not setup correctly

        //First Read OK, clear flags.
        SIPI.CSR3.R = CSR3_RAR;
        SIPI.ERR.R = ERR_TOE3;
        //assign read data
        *read_temp = SIPI_read_channel_data(channel);
        read_temp++;
        }
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

      default:
        return(3);	//Invalid channel

  }
};



/*****************************************************************************/
/*                          SIPI Write Function                              */		
/*****************************************************************************/

uint8_t SIPI_write(DATA_TEMPLATE_t write_data, CHANNEL_t channel, uint8_t injected_error)
{

  /* Calculate the data size to be read */
  /* 0 -> 32 bit reading, 1-> 16 bit reading, 2 -> 8 bit reading. Check if the command is consistent with the address */	
  switch(write_data.Size)
  {
    case(32):
      word_length = 0x02;
      break;
    case(16):
      /* if the LSB bit in the address are different to zero do nothing, otherwise specify a 16 bit reading */
      if (write_data.Address^0x00000001)
        return(10);
      else 
        word_length = 0x01;
      break;
    case(8):
      /* if the two LSB bits in the address are different to zero do nothing, otherwise specify an 8 bit reading */
      if ((write_data.Address^0x00000011) == 0x03)
        return(10);
      else 
        word_length = 0x03;
      break;
    default:
      return(1); //Invalid width		
  }


  /* Set up channel config register and start transfer */
  switch(channel.Number)
  {
    case(0):
      if (!SIPI.CSR0.B.CB) //If channel not busy
      {
        SIPI.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI.CCR0.B.WL = word_length;
        
        //Clear other commands
        SIPI.CCR0.R &= ~(CCR0_TC|CCR0_IDT|CCR0_RRT|CCR0_ST);
        
        //Enable the channel and request
        SIPI.CCR0.R |= CCR0_WRT|CCR0_CHEN;
        SIPI.MCR.B.INIT = 0;
        SIPI.CDR0.R = write_data.Data; 
        SIPI.CAR0.R = write_data.Address;
        
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI.CSR0.B.ACKR)
        {
          if(SIPI.ERR.B.TOE0||SIPI.ERR.B.ACKE0)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(1):
      if (!SIPI.CSR1.B.CB) //If channel not busy
      {
        SIPI.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI.CCR1.B.WL = word_length;
        
        //Clear other commands
        SIPI.CCR1.R &= ~(CCR1_TC|CCR1_IDT|CCR1_RRT|CCR1_ST);
        
        //Enable the channel and request
        SIPI.CCR1.R |= CCR1_WRT|CCR1_CHEN;
        SIPI.MCR.B.INIT = 0;
        
        SIPI.CDR1.R = write_data.Data; 
        SIPI.CAR1.R = write_data.Address;
        
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI.CSR1.B.ACKR)
        {
          if(SIPI.ERR.B.TOE1||SIPI.ERR.B.ACKE1)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(2):
      if (!SIPI.CSR2.B.CB) //If channel not busy
        {
        SIPI.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI.CCR2.B.WL = word_length;
        
        //Clear other commands
        SIPI.CCR2.R &= ~(CCR2_TC|CCR2_IDT|CCR2_RRT|CCR2_ST);
        
        //Enable the channel and request
        SIPI.CCR2.R |= CCR2_WRT|CCR2_CHEN;
        SIPI.MCR.B.INIT = 0;
        
        SIPI.CDR20.R = write_data.Data; 
        SIPI.CAR2.R = write_data.Address;
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI.CSR2.B.ACKR)
        {
          if(SIPI.ERR.B.TOE2||SIPI.ERR.B.ACKE2)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

      return(0); //If you get to here, write request successfully submitted.

    case(3):
      if (!SIPI.CSR3.B.CB) //If channel not busy
      {
        SIPI.MCR.R |= MCR_INIT|MCR_TEN;
        SIPI.CCR3.B.WL = word_length;
        
        //Clear other commands
        SIPI.CCR3.R &= ~(CCR3_TC|CCR3_IDT|CCR3_RRT|CCR3_ST);
        
        //Enable the channel and request
        SIPI.CCR3.R |= CCR3_WRT|CCR3_CHEN;
        SIPI.MCR.B.INIT = 0;
        
        SIPI.CDR3.R = write_data.Data; 
        SIPI.CAR3.R = write_data.Address;
        //Verify that a write acknowledge is received
        //Wait for the acknowledge
        while(!SIPI.CSR3.B.ACKR)
        {
          if(SIPI.ERR.B.TOE3||SIPI.ERR.B.ACKE3)
          {
            return(4); //Write Failed
          }
        }
      }
      else
      {
        return(2); //Channel Busy
      }

    return(0); //If you get to here, write request successfully submitted.
  }
  return(0);
}


/*****************************************************************************/
/*                    SIPI Multiple Write Function                           */		
/*****************************************************************************/

uint8_t SIPI_multiple_write(DATA_TEMPLATE_t write_array[], uint16_t array_length, CHANNEL_t channel, uint8_t injected_error, uint8_t DMA_Enable, uint32_t * dma_array)
{

  uint32_t i = 0;
  uint8_t Write_Setup_Result = 0;

  switch(channel.Number)
  {
    case(0):
      //Check if Channel is free
      if (!SIPI.CSR0.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {

          SIPI.MCR.B.MOEN = 0x01;
          SIPI.MCR.B.INIT = 0x01;
          SIPI.CCR0.B.WL = 0x02;
          SIPI.MCR.B.TEN = 1;
          SIPI.CCR0.B.RRT = 0x00;
          SIPI.CCR0.B.WRT = 0x01;
          SIPI.CCR0.B.CHEN = 0x01;
          //Increase TimerPrescaler to MAX
          SIPI.MCR.B.PRSCLR = 0x400;        
          //Get outta INIT mode and send the frame
          SIPI.CCR0.B.DEN = 0x01;
          SIPI.MCR.B.INIT = 0x00;

          DMACHMUX_1.CHCONFIG[0].R = 0;

          DMA_0.ERQL.R = 0xFFFFFFFF;
          DMA_0.CR.B.EMLM = 1;
          //Configure TCD
          DMA_0.TCD[8].SADDR = (uint32_t)dma_array; //SEND the Source address
          DMA_0.TCD[8].SMOD = 0;
          DMA_0.TCD[8].SSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].DMOD = 0;
          DMA_0.TCD[8].DSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].SOFF = 4; //32Bit (4byte) source offset
          DMA_0.TCD[8].NBYTES = 0x40002008; //0x7FFFF008;//Transfer 8 Bytes per minor loop (2 unsigned longs per minor loop)
          DMA_0.TCD[8].SLAST = 0; //No Source Offset correction upon major final count
          DMA_0.TCD[8].DADDR = (uint32_t)&SIPI.CDR0.R; //CDR as destination address
          DMA_0.TCD[8].CITER_ELINK  = 0; //No channel to channel linking enabled
          DMA_0.TCD[8].CITER_LINKCH = 0; //No channel to channel linking
          DMA_0.TCD[8].CITER = 4; //Do 1 Major LOOP per request
          DMA_0.TCD[8].DOFF = -4; //When a major loop finishes, correct Destination address by 8 (2 minor loops reduced it by 8)
          DMA_0.TCD[8].DLAST_SGA = 0; //Correct Last destination address
          DMA_0.TCD[8].BITER_ELINK = 0; // Channel to channel linking disabled
          DMA_0.TCD[8].BITER_LINKCH = 0;
          DMA_0.TCD[8].BITER = 4; // Starting major loop counter
          DMA_0.TCD[8].BWC = 0x0;         /* No DMA Bandwidth Control / Throttling */
          DMA_0.TCD[8].MAJORLINKCH = 0x0;         /* No Ch-Ch linking      */
          DMA_0.TCD[8].ESG = 0x0;         /* Disable scatter gather on major loop      */
          DMA_0.TCD[8].DREQ = 0x1;         /* Enable hardware request on citer=0 */
          DMA_0.TCD[8].INTHALF = 0x0;         /* Disable half point interrupt               */
          DMA_0.TCD[8].INTMAJ = 0x0;         /* Disable interrupt on major loop complete   */  
          DMA_0.TCD[8].DONE = 0;				
          DMA_0.TCD[8].ACTIVE = 0;
          DMA_0.TCD[8].INTMAJ = 0;
          DMA_0.TCD[8].START = 0;

          //Select source to be SIPI CH 0  
          DMACHMUX_1.CHCONFIG[0].R = 0x8B;

          while(!SIPI.CSR0.B.ACKR)
          {
            if(SIPI.ERR.B.TOE0||SIPI.ERR.B.ACKE0)
            {
              //NO ACKNOWLEDGE
              return(4);
            }
          }

        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            switch(Write_Setup_Result)
            {
              case(0):		//Read Setup correctly
                break;
              case(1):
                return(1);	//Invalid Data Size
              case(2):
                return(2);	//Channel Busy
              case(3):
                return(3);	//Invalid Channel Number
              default:
                return(4);	//Unknown Error Occurred
            }

            //If you got here then read set up correctly, now look for result
            //Now verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI.CSR0.B.ACKR)
            {
              if(SIPI.ERR.B.TOE0||SIPI.ERR.B.ACKE0)
              {
                //Write Failed
                return(4);
              }
            } 
            
          //Data Written, clear flags and start over
          SIPI.CSR0.R = CSR0_ACKR;
          /*SIPI.ERR.B.TOE0 = 1;
          SIPI.ERR.B.ACKE0 = 1;*/
          SIPI.ERR.R = ERR_TOE0|ERR_ACKE0;
          }
        }
        
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(1):
      //Check if Channel is free
      if (!SIPI.CSR1.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {

          SIPI.MCR.B.MOEN = 0x01;
          SIPI.MCR.B.INIT = 0x01;
          SIPI.CCR1.B.WL = 0x02;
          SIPI.MCR.B.TEN = 1;
          SIPI.CCR1.B.RRT = 0x00;
          SIPI.CCR1.B.WRT = 0x01;
          SIPI.CCR1.B.CHEN = 0x01;
          //Increase TimerPrescaler to MAX
          SIPI.MCR.B.PRSCLR = 0x400;        
          //Get outta INIT mode and send the frame
          SIPI.CCR1.B.DEN = 0x01;
          SIPI.MCR.B.INIT = 0x00;

          DMACHMUX_1.CHCONFIG[0].R = 0;

          DMA_0.ERQL.R = 0xFFFFFFFF;
          DMA_0.CR.B.EMLM = 1;
          //Configure TCD
          DMA_0.TCD[8].SADDR = (uint32_t)dma_array; //SEND the Source address
          DMA_0.TCD[8].SMOD = 0;
          DMA_0.TCD[8].SSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].DMOD = 0;
          DMA_0.TCD[8].DSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].SOFF = 4; //32Bit (4byte) source offset
          DMA_0.TCD[8].NBYTES = 0x40002008; //0x7FFFF008;//Transfer 8 Bytes per minor loop (2 unsigned longs per minor loop)
          DMA_0.TCD[8].SLAST = 0; //No Source Offset correction upon major final count
          DMA_0.TCD[8].DADDR = (uint32_t)&SIPI.CDR1.R; //CDR as destination address
          DMA_0.TCD[8].CITER_ELINK  = 0; //No channel to channel linking enabled
          DMA_0.TCD[8].CITER_LINKCH = 0; //No channel to channel linking
          DMA_0.TCD[8].CITER = 4; //Do 1 Major LOOP per request
          DMA_0.TCD[8].DOFF = -4; //When a major loop finishes, correct Destination address by 8 (2 minor loops reduced it by 8)
          DMA_0.TCD[8].DLAST_SGA = 0; //Correct Last destination address
          DMA_0.TCD[8].BITER_ELINK = 0; // Channel to channel linking disabled
          DMA_0.TCD[8].BITER_LINKCH = 0;
          DMA_0.TCD[8].BITER = 4; // Starting major loop counter
          DMA_0.TCD[8].BWC = 0x0;         /* No DMA Bandwidth Control / Throttling */
          DMA_0.TCD[8].MAJORLINKCH = 0x0;         /* No Ch-Ch linking      */
          DMA_0.TCD[8].ESG = 0x0;         /* Disable scatter gather on major loop      */
          DMA_0.TCD[8].DREQ = 0x1;         /* Enable hardware request on citer=0 */
          DMA_0.TCD[8].INTHALF = 0x0;         /* Disable half point interrupt               */
          DMA_0.TCD[8].INTMAJ = 0x0;         /* Disable interrupt on major loop complete   */  
          DMA_0.TCD[8].DONE = 0;				
          DMA_0.TCD[8].ACTIVE = 0;
          DMA_0.TCD[8].INTMAJ = 0;
          DMA_0.TCD[8].START = 0;

          //Select source to be SIPI CH 1  
          DMACHMUX_1.CHCONFIG[0].R = 0x8C;

          while(!SIPI.CSR1.B.ACKR)
          {
            if(SIPI.ERR.B.TOE1||SIPI.ERR.B.ACKE1)
            {
              //NO ACKNOWLEDGE
              return(4);
            }
          }

        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            switch(Write_Setup_Result)
            {
              case(0):		//Read Setup correctly
                break;
              case(1):
                return(1);	//Invalid Data Size
              case(2):
                return(2);	//Channel Busy
              case(3):
                return(3);	//Invalid Channel Number
              default:
                return(4);	//Unknown Error Occured
            }

            //If you got here then read set up correctly, now look for result
            //Now verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI.CSR1.B.ACKR)
            {
              if(SIPI.ERR.B.TOE1||SIPI.ERR.B.ACKE1)
              {
                //Write Failed
                return(4);
              }
            } 
            //Data Written, clear flags and start over
            //SIPI.CSR1.B.ACKR = 1;
            SIPI.CSR1.R = CSR1_ACKR;
            /*SIPI.ERR.B.TOE1 = 1;
            SIPI.ERR.B.ACKE1 = 1;*/
            SIPI.ERR.R = ERR_TOE1|ERR_ACKE1;
            
          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    case(2):
      //Check if Channel is free
      if (!SIPI.CSR2.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {

          SIPI.MCR.B.MOEN = 0x01;
          SIPI.MCR.B.INIT = 0x01;
          SIPI.CCR2.B.WL = 0x02;
          SIPI.MCR.B.TEN = 1;
          SIPI.CCR2.B.RRT = 0x00;
          SIPI.CCR2.B.WRT = 0x01;
          SIPI.CCR2.B.CHEN = 0x01;
          //Increase TimerPrescaler to MAX
          SIPI.MCR.B.PRSCLR = 0x400;        
          //Get outta INIT mode and send the frame
          SIPI.CCR2.B.DEN = 0x01;
          SIPI.MCR.B.INIT = 0x00;

          DMACHMUX_1.CHCONFIG[0].R = 0;

          DMA_0.ERQL.R = 0xFFFFFFFF;
          DMA_0.CR.B.EMLM = 1;
          //Configure TCD
          DMA_0.TCD[8].SADDR = (uint32_t)dma_array; //SEND the Source address
          DMA_0.TCD[8].SMOD = 0;
          DMA_0.TCD[8].SSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].DMOD = 0;
          DMA_0.TCD[8].DSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].SOFF = 4; //32Bit (4byte) source offset
          DMA_0.TCD[8].NBYTES = 0x40002008; //0x7FFFF008;//Transfer 8 Bytes per minor loop (2 unsigned longs per minor loop)
          DMA_0.TCD[8].SLAST = 0; //No Source Offset correction upon major final count
          DMA_0.TCD[8].DADDR = (uint32_t)&SIPI.CDR20.R; //CDR as destination address
          DMA_0.TCD[8].CITER_ELINK  = 0; //No channel to channel linking enabled
          DMA_0.TCD[8].CITER_LINKCH = 0; //No channel to channel linking
          DMA_0.TCD[8].CITER = 4; //Do 1 Major LOOP per request
          DMA_0.TCD[8].DOFF = -4; //When a major loop finishes, correct Destination address by 8 (2 minor loops reduced it by 8)
          DMA_0.TCD[8].DLAST_SGA = 0; //Correct Last destination address
          DMA_0.TCD[8].BITER_ELINK = 0; // Channel to channel linking disabled
          DMA_0.TCD[8].BITER_LINKCH = 0;
          DMA_0.TCD[8].BITER = 4; // Starting major loop counter
          DMA_0.TCD[8].BWC = 0x0;         /* No DMA Bandwidth Control / Throttling */
          DMA_0.TCD[8].MAJORLINKCH = 0x0;         /* No Ch-Ch linking      */
          DMA_0.TCD[8].ESG = 0x0;         /* Disable scatter gather on major loop      */
          DMA_0.TCD[8].DREQ = 0x1;         /* Enable hardware request on citer=0 */
          DMA_0.TCD[8].INTHALF = 0x0;         /* Disable half point interrupt               */
          DMA_0.TCD[8].INTMAJ = 0x0;         /* Disable interrupt on major loop complete   */  
          DMA_0.TCD[8].DONE = 0;				
          DMA_0.TCD[8].ACTIVE = 0;
          DMA_0.TCD[8].INTMAJ = 0;
          DMA_0.TCD[8].START = 0;

          //Select source to be SIPI CH 2  
          DMACHMUX_1.CHCONFIG[0].R = 0x8D;

          while(!SIPI.CSR2.B.ACKR)
          {
            if(SIPI.ERR.B.TOE2||SIPI.ERR.B.ACKE2)
            {
              //NO ACKNOWLEDGE
              return(4);
            }
          }
        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            switch(Write_Setup_Result)
            {
              case(0):		//Read Setup correctly
                break;
              case(1):
                return(1);	//Invalid Data Size
              case(2):
                return(2);	//Channel Busy
              case(3):
                return(3);	//Invalid Channel Number
              default:
                return(4);	//Unknown Error Occured
            }

            //If you got here then read set up correctly, now look for result
            //Now verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI.CSR2.B.ACKR)
            {
              if(SIPI.ERR.B.TOE2||SIPI.ERR.B.ACKE2)
              {
                //Write Failed
                return(4);
              }
            } 
            //Data Written, clear flags and start over
            //SIPI.CSR2.B.ACKR = 1;
            SIPI.CSR2.R = CSR2_ACKR;
            //SIPI.ERR.B.TOE2 = 1;
            //SIPI.ERR.B.ACKE2 = 1;
            SIPI.ERR.R = ERR_TOE2|ERR_ACKE2;

          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }
      //break;

    case(3):
      //Check if Channel is free
      if (!SIPI.CSR3.B.CB)
      {
        //Check Whether to Use DMA or software method 
        if (DMA_Enable)
        {

          SIPI.MCR.B.MOEN = 0x01;
          SIPI.MCR.B.INIT = 0x01;
          SIPI.CCR3.B.WL = 0x02;
          SIPI.MCR.B.TEN = 1;
          SIPI.CCR3.B.RRT = 0x00;
          SIPI.CCR3.B.WRT = 0x01;
          SIPI.CCR3.B.CHEN = 0x01;
          //Increase TimerPrescaler to MAX
          SIPI.MCR.B.PRSCLR = 0x400;        
          //Get outta INIT mode and send the frame
          SIPI.CCR3.B.DEN = 0x01;
          SIPI.MCR.B.INIT = 0x00;

          DMACHMUX_1.CHCONFIG[0].R = 0;

          DMA_0.ERQL.R = 0xFFFFFFFF;
          DMA_0.CR.B.EMLM = 1;
          //Configure TCD
          DMA_0.TCD[8].SADDR = (uint32_t)dma_array; //SEND the Source address
          DMA_0.TCD[8].SMOD = 0;
          DMA_0.TCD[8].SSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].DMOD = 0;
          DMA_0.TCD[8].DSIZE = 2; //32Bit size transfers
          DMA_0.TCD[8].SOFF = 4; //32Bit (4byte) source offset
          DMA_0.TCD[8].NBYTES = 0x40002008; //0x7FFFF008;//Transfer 8 Bytes per minor loop (2 unsigned longs per minor loop)
          DMA_0.TCD[8].SLAST = 0; //No Source Offset correction upon major final count
          DMA_0.TCD[8].DADDR = (uint32_t)&SIPI.CDR3.R; //CDR as destination address
          DMA_0.TCD[8].CITER_ELINK  = 0; //No channel to channel linking enabled
          DMA_0.TCD[8].CITER_LINKCH = 0; //No channel to channel linking
          DMA_0.TCD[8].CITER = 4; //Do 1 Major LOOP per request
          DMA_0.TCD[8].DOFF = -4; //When a major loop finishes, correct Destination address by 8 (2 minor loops reduced it by 8)
          DMA_0.TCD[8].DLAST_SGA = 0; //Correct Last destination address
          DMA_0.TCD[8].BITER_ELINK = 0; // Channel to channel linking disabled
          DMA_0.TCD[8].BITER_LINKCH = 0;
          DMA_0.TCD[8].BITER = 4; // Starting major loop counter
          DMA_0.TCD[8].BWC = 0x0;         /* No DMA Bandwidth Control / Throttling */
          DMA_0.TCD[8].MAJORLINKCH = 0x0;         /* No Ch-Ch linking      */
          DMA_0.TCD[8].ESG = 0x0;         /* Disable scatter gather on major loop      */
          DMA_0.TCD[8].DREQ = 0x1;         /* Enable hardware request on citer=0 */
          DMA_0.TCD[8].INTHALF = 0x0;         /* Disable half point interrupt               */
          DMA_0.TCD[8].INTMAJ = 0x0;         /* Disable interrupt on major loop complete   */  
          DMA_0.TCD[8].DONE = 0;				
          DMA_0.TCD[8].ACTIVE = 0;
          DMA_0.TCD[8].INTMAJ = 0;
          DMA_0.TCD[8].START = 0;

          //Select source to be SIPI CH 3  
          DMACHMUX_1.CHCONFIG[0].R = 0x8E;

          while(!SIPI.CSR3.B.ACKR)
          {
            if(SIPI.ERR.B.TOE3||SIPI.ERR.B.ACKE3)
            {
              //NO ACKNOWLEDGE
              return(4);
            }
          }

        }
        else
        {
          //Loop through the array to be sent, sending each individual message and waiting for the return
          for (i=0; i<array_length; i++)
          {
            Write_Setup_Result = SIPI_write(write_array[i], channel, 0); //Attempt to set up Read command

            switch(Write_Setup_Result)
            {
              case(0):		//Read Setup correctly
                break;
              case(1):
                return(1);	//Invalid Data Size
              case(2):
                return(2);	//Channel Busy
              case(3):
                return(3);	//Invalid Channel Number
              default:
                return(4);	//Unknown Error Occured
            }

            //If you got here then read set up correctly, now look for result
            //Now verify that the write command has been sent
            //Wait for the acknowledge
            while(!SIPI.CSR3.B.ACKR)
            {
              if(SIPI.ERR.B.TOE3||SIPI.ERR.B.ACKE3)
              {
                //Write Failed
                return(4);
              }
            } 
            //Data Written, clear flags and start over
            SIPI.CSR3.R = CSR3_ACKR;
            SIPI.ERR.R = ERR_TOE3|ERR_ACKE3;
          }
        }       
        return(0);
      }
      else
      {
        return(2);		//Channel Busy
      }

    default:
      return(9);	//Unknown Error
  }


};


/*****************************************************************************/
/*                          SIPI ID Function                                 */		
/*****************************************************************************/

uint8_t SIPI_ID(uint32_t * id_array, CHANNEL_t channel)
{

	 
	switch(channel.Number)
	{
		case(0):
			if (!SIPI.CSR0.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1; 
				SIPI.CCR0.B.WL = 0x02;   //Not used
				SIPI.MCR.B.TEN = 1;
				SIPI.CCR0.B.RRT = 0;
				SIPI.CCR0.B.WRT = 0;
				SIPI.CCR0.B.ST = 0;
				SIPI.CCR0.B.CHEN = 1;
				SIPI.CCR0.B.IDT = 1;
				SIPI.MCR.B.INIT = 0;
                 //You have to initiate transfer by writing in CAR.
				SIPI.CAR0.R = (uint32_t)id_array;

				while(!SIPI.CSR0.B.RAR)
				{
				    //Read answer still not received
				    //Poll for the timeout flag
				    if(SIPI.ERR.B.TOE0)
				    {
					//Timeout error
					return(4);
				    }
				}
			}
			else
			{
				return(2); //Channel Busy
			}
			break; 
		case(1):
			if (!SIPI.CSR1.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1; 
				SIPI.CCR1.B.WL = 0x02;   //Not used
				SIPI.MCR.B.TEN = 1;
				SIPI.CCR1.B.RRT = 0;
				SIPI.CCR1.B.WRT = 0;
				SIPI.CCR1.B.ST = 0;
				SIPI.CCR1.B.CHEN = 1;
				SIPI.CCR1.B.IDT = 1;
				SIPI.MCR.B.INIT = 0;
        //Initiate transfer by writing in CAR.
				SIPI.CAR1.R = (uint32_t)id_array;

				while(!SIPI.CSR1.B.RAR)
				{
				    //Read answer still not received
				    //Poll for the timeout flag
				    if(SIPI.ERR.B.TOE1)
				    {
					//Timeout error!!!
					return(4);
				    }
				}
			}
			else
			{
				return(2); //Channel Busy
			}
			break; 
		case(2):
			if (!SIPI.CSR2.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1; 
				SIPI.CCR2.B.WL = 0x02;   //Not used
				SIPI.MCR.B.TEN = 1;
				SIPI.CCR2.B.RRT = 0;
				SIPI.CCR2.B.WRT = 0;
				SIPI.CCR2.B.ST = 0;
				SIPI.CCR2.B.CHEN = 1;
				SIPI.CCR2.B.IDT = 1;
				SIPI.MCR.B.INIT = 0;
                                //You have to initiate transfer by writting in CAR.
				SIPI.CAR2.R = (uint32_t)id_array;

				while(!SIPI.CSR2.B.RAR)
				{
				    //Read answer still not received
				    //Poll for the timeout flag
				    if(SIPI.ERR.B.TOE2)
				    {
					//Timeout error!!!
					return(4);
				    }
				}
			}
			else
			{
				return(2); //Channel Busy
			}
			break; 
		case(3):
			if (!SIPI.CSR3.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1; 
				SIPI.CCR3.B.WL = 0x02;   //Not used
				SIPI.MCR.B.TEN = 1;
				SIPI.CCR3.B.RRT = 0;
				SIPI.CCR3.B.WRT = 0;
				SIPI.CCR3.B.ST = 0;
				SIPI.CCR3.B.CHEN = 1;
				SIPI.CCR3.B.IDT = 1;
				SIPI.MCR.B.INIT = 0;
                                //You have to initiate transfer by writting in CAR.
				SIPI.CAR3.R = (uint32_t)id_array;

				while(!SIPI.CSR3.B.RAR)
				{
				    //Read answer still not received
				    //Poll for the timeout flag
				    if(SIPI.ERR.B.TOE3)
				    {
					//Timeout error!!!
					return(4);
				    }
				}
			}
			else
			{
				return(2); //Channel Busy
			}
			break; 
		default:
			return(3); //Invalid Channel Number
 
	}
	
	return(0); //If you reached here then command sent successfully.
};




/*****************************************************************************/
/*                  SIPI Initialize Initiator Function                       */		
/*****************************************************************************/

uint8_t SIPI_init_INITIATOR(uint16_t Clock_Prescale)
{

	SIPI.MCR.B.MOEN = 1;			//First Enable Module
	SIPI.MCR.B.INIT = 1;			//Place module into INIT mode
	
	/* Need to now check what Clock_Prescale to set up */
	
	switch(Clock_Prescale)
	{
		case(64):
			SIPI.MCR.B.PRSCLR = 0x040;
			break;
		case(128):
			SIPI.MCR.B.PRSCLR = 0x080;
			break;
		case(256):
			SIPI.MCR.B.PRSCLR = 0x100;
			break;
		case(512):
			SIPI.MCR.B.PRSCLR = 0x200;
			break;
		case(1024):
			SIPI.MCR.B.PRSCLR = 0x400;
			break;
		default:
			SIPI.MCR.B.PRSCLR = 0x040;
			return(1);						//Incorrect Clock Prescale Selected, default of 64 set.
	}
	
	
	SIPI.MCR.B.INIT = 0;
	return(0);
	
}


/*****************************************************************************/
/*                   SIPI Initialize Target Function                         */		
/*****************************************************************************/
uint8_t SIPI_init_TARGET(uint32_t max_count, uint32_t reload_address, uint8_t Add_Inc)
{
	
	SIPI.MCR.B.MOEN = 1;			//First ensure module is enabled
	SIPI.MCR.B.INIT = 1;			//Place module into INIT mode
	
	SIPI.MAXCR.R = (max_count & 0xFFFFFFFC); 		//2 LSB are reserved.
	SIPI.ARR.R = (reload_address & 0xFFFFFFFC);	//2 LSB are reserved.
	
	if ((Add_Inc == 0) || (Add_Inc == 1) || (Add_Inc == 2) || (Add_Inc == 3))
	{
		SIPI.MCR.B.AID = Add_Inc;
	}
	else
	{
		return(1);					//Incorrect Address Increment Selected.
	}

	
	SIPI.MCR.B.TEN = 1;			//Enable target
	SIPI.CTOR0.R = 0x0400;
        //ENABLE ALL CHANNELS IN TARGET
	SIPI.CCR0.B.CHEN = 0x01;
	SIPI.CCR1.B.CHEN = 0x01;
	SIPI.CCR2.B.CHEN = 0x01;
	SIPI.CCR3.B.CHEN = 0x01;
	
	SIPI.MCR.B.INIT = 0;			//Bring module back up
	
	if (!SIPI.SR.B.MCR)
	{
		return(0);
	}
	else
	{
		return(2);					//Error with Max Count Address and SIPI Address Count Register
	}
}



/*****************************************************************************/
/*                 SIPI Initialize Channel Function                          */		
/*****************************************************************************/

uint8_t SIPI_init_channel(CHANNEL_t channel, uint8_t mode, uint8_t error_int_enable, uint8_t data_int_enable)
{


switch(channel.Number)
{

	case(0):	
		//Check what mode is being requested
		switch(mode)
		{
			case(0):	//Run Mode
				SIPI.CCR0.B.CHEN = 1;
				break;
			case(1):	//Disable
				SIPI.CCR0.B.CHEN = 0;
				break;
			case(2):	//Clear Error
				SIPI.CCR0.B.CHEN = 0;
				/*SIPI.ERR.B.ACKE0 = 1;
				SIPI.ERR.B.TIDE0 = 1;
				SIPI.ERR.B.TOE0  = 1;*/
        SIPI.ERR.R = ERR_ACKE0|ERR_TIDE0|ERR_TOE0;
				SIPI.CCR0.B.CHEN = 1;
				break;
			case(3):	//Stop
				while (SIPI.CSR0.B.CB){}
				SIPI.CCR0.B.CHEN = 0;
				break;
			default:
				return(1);		//Incorrect Channel Mode Selection
		}
		//Setup Error Interrupts
		SIPI.CIR0.B.TOIE = error_int_enable;
		SIPI.CIR0.B.TIDIE = error_int_enable;
		SIPI.CIR0.B.ACKIE = error_int_enable;
		
		//Setup Data Interrupts
		SIPI.CIR0.B.WAIE = data_int_enable;
		SIPI.CIR0.B.RAIE = data_int_enable;
		SIPI.CIR0.B.TCIE = data_int_enable;
		
		//Setup Channel Timeout Value
		SIPI.MCR.B.INIT = 1;
		SIPI.CTOR0.B.TOR = channel.Timeout;
		SIPI.MCR.B.INIT = 0;
		return(0);
		
		//Do we need to check status here?
	case(1):
		//Check what mode is being requested
		switch(mode)
		{
			case(0):	//Run Mode
				SIPI.CCR1.B.CHEN = 1;
				break;
			case(1):	//Disable
				SIPI.CCR1.B.CHEN = 0;
				break;
			case(2):	//Clear Error
				SIPI.CCR1.B.CHEN = 0;
				/*SIPI.ERR.B.ACKE1 = 1;
				SIPI.ERR.B.TIDE1 = 1;
				SIPI.ERR.B.TOE1  = 1;*/
        SIPI.ERR.R = ERR_ACKE1|ERR_TIDE1|ERR_TOE1;
				SIPI.CCR1.B.CHEN = 1;
				break;
			case(3):	//Stop
				while (SIPI.CSR1.B.CB){}
				SIPI.CCR1.B.CHEN = 0;
				break;
			default:
				return(1);		//Incorrect Channel Mode Selection
		}
		//Setup Error Interrupts
		SIPI.CIR1.B.TOIE = error_int_enable;
		SIPI.CIR1.B.TIDIE = error_int_enable;
		SIPI.CIR1.B.ACKIE = error_int_enable;
		
		//Setup Data Interrupts
		SIPI.CIR1.B.WAIE = data_int_enable;
		SIPI.CIR1.B.RAIE = data_int_enable;
		SIPI.CIR1.B.TCIE = data_int_enable;
		
		//Setup Channel Timeout Value
		SIPI.MCR.B.INIT = 1;
		SIPI.CTOR1.B.TOR = channel.Timeout;
		SIPI.MCR.B.INIT = 0;
		return(0);
		
		//Do we need to check status here?
	case(2):
		//Check what mode is being requested
		switch(mode)
		{
			case(0):	//Run Mode
				SIPI.CCR2.B.CHEN = 1;
				break;
			case(1):	//Disable
				SIPI.CCR2.B.CHEN = 0;
				break;
			case(2):	//Clear Error
				SIPI.CCR2.B.CHEN = 0;
				/*SIPI.ERR.B.ACKE2 = 1;
				SIPI.ERR.B.TIDE2 = 1;
				SIPI.ERR.B.TOE2  = 1;*/
        SIPI.ERR.R = ERR_ACKE2|ERR_TIDE2|ERR_TOE2;
				SIPI.CCR2.B.CHEN = 1;
				break;
			case(3):	//Stop
				while (SIPI.CSR2.B.CB){}
				SIPI.CCR2.B.CHEN = 0;
				break;
			default:
				return(1);		//Incorrect Channel Mode Selection
		}
		//Setup Error Interrupts
		SIPI.CIR2.B.TOIE = error_int_enable;
		SIPI.CIR2.B.TIDIE = error_int_enable;
		SIPI.CIR2.B.ACKIE = error_int_enable;
		
		//Setup Data Interrupts
		SIPI.CIR2.B.WAIE = data_int_enable;
		SIPI.CIR2.B.RAIE = data_int_enable;
		SIPI.CIR2.B.TCIE = data_int_enable;
		
		//Setup Channel Timeout Value
		SIPI.MCR.B.INIT = 1;
		SIPI.CTOR2.B.TOR = channel.Timeout;
		SIPI.MCR.B.INIT = 0;
		return(0);
		
		//Do we need to check status here?
	case(3):
		//Check what mode is being requested
		switch(mode)
		{
			case(0):	//Run Mode
				SIPI.CCR3.B.CHEN = 1;
				break;
			case(1):	//Disable
				SIPI.CCR3.B.CHEN = 0;
				break;
			case(2):	//Clear Error
				SIPI.CCR3.B.CHEN = 0;
				/*SIPI.ERR.B.ACKE3 = 1;
				SIPI.ERR.B.TIDE3 = 1;
				SIPI.ERR.B.TOE3  = 1;*/
        SIPI.ERR.R = ERR_ACKE3|ERR_TIDE3|ERR_TOE3;
				SIPI.CCR3.B.CHEN = 1;
				break;
			case(3):	//Stop
				while (SIPI.CSR3.B.CB){}
				SIPI.CCR3.B.CHEN = 0;
				break;
			default:
				return(1);		//Incorrect Channel Mode Selection
		}
		//Setup Error Interrupts
		SIPI.CIR3.B.TOIE = error_int_enable;
		SIPI.CIR3.B.TIDIE = error_int_enable;
		SIPI.CIR3.B.ACKIE = error_int_enable;
		
		//Setup Data Interrupts
		SIPI.CIR3.B.WAIE = data_int_enable;
		SIPI.CIR3.B.RAIE = data_int_enable;
		SIPI.CIR3.B.TCIE = data_int_enable;
		
		//Setup Channel Timeout Value
		SIPI.MCR.B.INIT = 1;
		SIPI.CTOR3.B.TOR = channel.Timeout;
		SIPI.MCR.B.INIT = 0;
		return(0);
		
		//Do we need to check status here?
	default:
		return(2);  //Incorrect Channel
}

}




/*****************************************************************************/
/*                          SIPI Trigger Function                            */		
/*****************************************************************************/

uint8_t SIPI_Trigger(CHANNEL_t channel)
{
	
	switch(channel.Number)
	{
		case(0):
			if (!SIPI.CSR0.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1;
				SIPI.CCR0.B.WL = 0x02;   //Not used
				SIPI.MCR.B.TEN = 1;
				SIPI.CCR0.B.CHEN = 1;
				SIPI.CCR0.B.IDT = 0;
				SIPI.CCR0.B.ST = 0;
				SIPI.CCR0.B.RRT = 0;
				SIPI.CCR0.B.WRT = 0;
				SIPI.CCR0.B.TC = 1;
				SIPI.MCR.B.INIT = 0;

				//Verify that the trigger acknowledge is received
				while(!SIPI.CSR0.B.ACKR)
				{
				    if(SIPI.ERR.B.TOE0||SIPI.ERR.B.ACKE0)
				    {
					dummyPrintf("TRIGGER TRANSFER FAILED");	
					return(4);
				    }
				}
				break;
			}
			else
			{
				return(2);	// Channel Busy
			}
		case(1):
			if (!SIPI.CSR1.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1;
				SIPI.CCR1.B.WL = 0x02;   //Not used
				SIPI.CCR1.B.CHEN = 1;
				SIPI.CCR1.B.IDT = 0;
				SIPI.CCR1.B.ST = 0;
				SIPI.CCR1.B.RRT = 0;
				SIPI.CCR1.B.WRT = 0;
				SIPI.CCR1.B.TC = 1;
				SIPI.MCR.B.INIT = 0;
				//Verify that the trigger acknowledge is received
				while(!SIPI.CSR1.B.ACKR)
				{
				    if(SIPI.ERR.B.TOE1||SIPI.ERR.B.ACKE1)
				    {
					dummyPrintf("TRIGGER TRANSFER FAILED");	
					return(4);
				    }
				}
				break;
			}
			else
			{
				return(2);	// Channel Busy
			}
		case(2):
			if (!SIPI.CSR2.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1;
				SIPI.CCR2.B.WL = 3;   //Not used
				SIPI.CCR2.B.CHEN = 1;
				SIPI.CCR2.B.IDT = 0;
				SIPI.CCR2.B.ST = 0;
				SIPI.CCR2.B.RRT = 0;
				SIPI.CCR2.B.WRT = 0;
				SIPI.CCR2.B.TC = 1;
				SIPI.MCR.B.INIT = 0;

				//Verify that the trigger acknowledge is received
				while(!SIPI.CSR2.B.ACKR)
				{
				    if(SIPI.ERR.B.TOE2||SIPI.ERR.B.ACKE2)
				    {
					dummyPrintf("TRIGGER TRANSFER FAILED");	
					return(4);
				    }
				}
				break;
			}
			else
			{
				return(2);	// Channel Busy
			}
		case(3):
			if (!SIPI.CSR3.B.CB) //If channel not busy
			{
				SIPI.MCR.B.INIT = 1;
				SIPI.CCR3.B.WL = 3;   //Not used
				SIPI.CCR3.B.CHEN = 1;
				SIPI.CCR3.B.IDT = 0;
				SIPI.CCR3.B.ST = 0;
				SIPI.CCR3.B.RRT = 0;
				SIPI.CCR3.B.WRT = 0;
				SIPI.CCR3.B.TC = 1;
				SIPI.MCR.B.INIT = 0;
				//Verify that the trigger acknowledge is received
				while(!SIPI.CSR3.B.ACKR)
				{
				    if(SIPI.ERR.B.TOE3||SIPI.ERR.B.ACKE3)
				    {
					dummyPrintf("TRIGGER TRANSFER FAILED");	
					return(4);
				    }
				}
				break;
			}
			else
			{
				return(2);	// Channel Busy
			}
		default:
			return(1);		// Incorrect Channel Number	
			
	}
	
	/* If you got to here then successfully completed! */
	
	return(0);
}





/*****************************************************************************/
/*                          SIPI Reset Function                              */		
/*****************************************************************************/

uint8_t SIPI_reset(void)
{
	/* Reset the module */
	SIPI.MCR.B.SR = 1;
	
	/* Disable the module */
	SIPI.MCR.B.MOEN = 1;
	SIPI.MCR.B.INIT = 1;
	SIPI.MCR.B.TEN = 0;
	SIPI.MCR.B.INIT = 0;
	SIPI.MCR.B.MOEN = 0;
	
	return(0);
}

/*****************************************************************************/
/*                     SIPI Module Mode Function                             */		
/*****************************************************************************/
 
uint8_t SIPI_module_mode(uint8_t Mode)
{
	switch(Mode)
	{
		case(0):
			//Disable the module
			SIPI.MCR.B.INIT = 1;
			SIPI.MCR.B.MOEN = 0;
			SIPI.MCR.B.TEN = 0;
			SIPI.MCR.B.INIT = 0;
			return(0);
		case(1):
			//Enable the module and put into INIT mode
			SIPI.MCR.B.INIT = 1;
			SIPI.MCR.B.MOEN = 1;
			return(0);
		case(2):
			//Enable the module
			SIPI.MCR.B.INIT = 1;
			SIPI.MCR.B.MOEN = 1;
			SIPI.MCR.B.INIT = 0;
			return(0);
		default:
			//Disable the module
			SIPI.MCR.B.INIT = 1;
			SIPI.MCR.B.MOEN = 0;
			SIPI.MCR.B.TEN = 0;
			SIPI.MCR.B.INIT = 0;
			return(1);				//Invalid Mode Selected
	}
}

/*****************************************************************************/
/*                     SIPI Stream Transfer Function                         */		
/*****************************************************************************/
 
uint8_t SIPI_stream_transfer(uint32_t * temp_data_stream, uint8_t initiator, uint8_t  length)
{
  unsigned char send_index;
  if(length>8)
  {
    dummyPrintf("INVALID LENGTH");
    return(1);
  }

  //CONFIGURE SIPI Module for Streaming Transfer
  if(initiator)
  {
    SIPI.MCR.B.MOEN = 0x01;
    SIPI.MCR.B.INIT = 0x01;
    SIPI.MCR.B.TEN = 0x01;
    SIPI.CCR2.B.WL = 0x02;
    SIPI.CCR2.B.RRT = 0x00;
    SIPI.CCR2.B.WRT = 0x01;
    SIPI.CCR2.B.CHEN = 0x01;
    SIPI.CCR2.B.ST = 0x01;
    SIPI.MCR.B.AID = 0x01; //Increment 4 addresses
    //Increase TimerPrescaler to MAX
    SIPI.MCR.B.PRSCLR = 0x400;        
    //Get outta INIT mode and send the frame
    SIPI.MCR.B.INIT = 0x00;
    //Configure the address and count in the Initiator
    SIPI.MAXCR.R = (uint32_t)(temp_data_stream+(length-1)); 
    SIPI.ARR.R = (uint32_t)temp_data_stream;
    SIPI.ACR.R = (uint32_t)temp_data_stream; // Not needed in the initiator node

    //Preare the register contents
    for(send_index=0;send_index<length;send_index++)
    {
      switch(send_index)
      {
        case 0:
        SIPI.CDR20.R = *temp_data_stream;	
	break;

	case 1:
        SIPI.CDR21.R = *temp_data_stream;	
	break;

	case 2:
        SIPI.CDR22.R = *temp_data_stream;	
	break;

	case 3:
        SIPI.CDR23.R = *temp_data_stream;	
	break;

	case 4:
        SIPI.CDR24.R = *temp_data_stream;	
	break;

	case 5:
        SIPI.CDR25.R = *temp_data_stream;	
	break;

	case 6:
        SIPI.CDR26.R = *temp_data_stream;	
	break;

	case 7:
        SIPI.CDR27.R = *temp_data_stream;	
	break;
      }
      temp_data_stream++;
    }
    //Send the data
    temp_data_stream -= (length);
    SIPI.CAR2.R = (uint32_t)temp_data_stream;

    //Wait for the acknowledge
    while(!SIPI.CSR2.B.ACKR)
    {
      if(SIPI.ERR.B.TOE2||SIPI.ERR.B.ACKE2)
      {
        dummyPrintf("Acknowledge error");
	    return(2);
      }
      dummyPrintf("Data written");
      //Now verify that the data was correctly written, verify in the target that the data was indeed written.
      return(0);
    }
  }
  else
  {
    SIPI.MCR.B.MOEN = 0x01;
    SIPI.MCR.B.INIT = 0x01;
    SIPI.CCR2.B.CHEN = 0x01;
    SIPI.MCR.B.TEN = 0x01;
    SIPI.CCR2.B.WL = 0x02;
    SIPI.CCR2.B.WRT = 0x01;
    SIPI.CCR2.B.ST = 0x01;
    SIPI.MCR.B.AID = 0x01; //Increment 4 addresses
    SIPI.MCR.B.INIT = 0x00;
    //Configure the address and count in the target
    SIPI.MAXCR.R = (uint32_t)(temp_data_stream+(length-1)); 
    SIPI.ARR.R = (uint32_t)temp_data_stream;
    SIPI.ACR.R = (uint32_t)temp_data_stream; // Not needed in the initiator node
  }
  return(0);
}
