//codewarrior header files
#include <hidef.h>
#include "derivative.h"

//driver header file
#include "CAN.h"
#include "CAN_cfg.h"

//global variables
void (*rx_callback)(unsigned int, unsigned char[8], unsigned char);

//timing values
#if(CAN_BAUDRATE==CAN_125_KBPS)
	#define VALUE_TSEG10 0x0A
	#define VALUE_TSEG20 0x03
	#define VALUE_PRESCA 0x01
#elif(CAN_BAUDRATE==CAN_250_KBPS)
	#define VALUE_TSEG10 0x0A
	#define VALUE_TSEG20 0x03
	#define VALUE_PRESCA 0x00
#elif(CAN_BAUDRATE==CAN_500_KBPS)
	#define VALUE_TSEG10 0x04
	#define VALUE_TSEG20 0x02
	#define VALUE_PRESCA 0x00
#endif

//initialises CAN communications
void CAN_init(void){

	rx_callback = NULL;

	//initialise MSCAN
	CAN0CTL0 = 0x01;
	while (!(CAN0CTL1_INITAK)){}
	CAN0CTL1_CANE = 1;      /* Enable MSCAN module */
	CAN0CTL1_CLKSRC = 0;    /* Clock source is OSCCLK = 4 MHz */    
	CAN0CTL1_LOOPB = 0;     /* Set to 1 for LoopBack Mode, 0 otherwise */
	CAN0CTL1_LISTEN = 0;    /* Not listen only mode */  
	CAN0CTL0_WUPE = 1;		/* Enable WAKEUP */
	
	//Baud rate = CANCLK/(Pre-scaler * time quanta)                                                                           
	CAN0BTR1_TSEG_10 = VALUE_TSEG10;
	CAN0BTR1_TSEG_20 = VALUE_TSEG20;
	CAN0BTR0_BRP = VALUE_PRESCA;
	CAN0BTR0_SJW = 0x03;
	CAN0BTR1_SAMP = 0;
	
	//accept all messages
	CAN0IDAC_IDAM = 0; 		/* Two 32 bit filters */
	CAN0IDMR0 = 0xFF;		/* Accept all incoming ID's */
	CAN0IDMR1 = 0xFF;
	CAN0IDMR2 = 0xFF;
	CAN0IDMR3 = 0xFF;
	CAN0IDMR4 = 0xFF;
	CAN0IDMR5 = 0xFF;
	CAN0IDMR6 = 0xFF;
	CAN0IDMR7 = 0xFF;
	
	//exit initialisation mode
	CAN0CTL0_INITRQ = 0;
	while (CAN0CTL1_INITAK){};
	while(!(CAN0CTL0_SYNCH)){};
	
	//clear flags
	CAN0RFLG_RXF = 1;                                   
	CAN0RIER_RXFIE = 1;
	
	//enable physical layer if available
	#ifdef CP0CR
	CP0CR_CPE = 1;
	CP0CR_SPE = 1;
	#endif
	
	//enable RX interrupt
	CAN0RIER_RXFIE = 1;
}

//sends a CAN message
void CAN_send(unsigned int id, unsigned char * ptr, unsigned char cnt){
	
	unsigned char index, bufnum;
	
	//prepare frame
	CAN0TIER = 0x00;		/* disable TX interrupts */
	CAN0TBSEL = CAN0TFLG;	/* select lowest empty buffer */
	bufnum = CAN0TBSEL;
	//load ID
	CAN0TXIDR0 = (id&0b11111111000)>>3;
	CAN0TXIDR1 = (id&0b00000000111)<<5;
	
	//load data
	for(index=0; index<cnt; index++){
		*(&CAN0TXDSR0 + index) = ptr[index];
	}
	
	//send frame
	CAN0TXDLR = cnt;	/* set data length */
	CAN0TXTBPR = 0;		/* set priority */
	CAN0TFLG = bufnum;	/* start transmission */
}

//assigns a function to be called when a message is received
void CAN_set_rx_callback(void (*ptr)(unsigned int, unsigned char[8], unsigned char)){
	rx_callback = ptr;
}

//CAN message reception interrupt
void interrupt VectorNumber_Vcan0rx CAN_ISR() {
	//extract CAN frame information
	unsigned int id;
	unsigned char data[8];
	unsigned char length, i;
	//copy ID and DLC
	id = CAN0RXIDR0<<3;
	id |= CAN0RXIDR1>>5;
	length = CAN0RXDLR_DLC;
	//copy DATA
	for(i=0; i<length; i++){
		data[i] = CAN0RXDSR_ARR[i];
	}
	//execute callback
	if(rx_callback!=NULL){
		rx_callback(id, data, length);
	}
	//clear interrupt flag
	CAN0RFLG_RXF = 1;
}
