#*******************************************************************************
#
#  FREESCALE SEMICONDUCTOR INC. 2014
#  ALL RIGHTS RESERVED
#  COPYRIGHT (c)
#
#*******************************************************************************
#
#  FILE NAME:  engine_control_demo.tcl
#
#*******************************************************************************

#*******************************************************************************
# Helper scripts 
#*******************************************************************************
etpusource trans.tcl
etpusource common.tcl

#*******************************************************************************
# Included scripts 
#*******************************************************************************
source [get_relative_loc]constants.tcl
source [get_relative_loc]crank_init.tcl
source [get_relative_loc]cam_init.tcl
source [get_relative_loc]spark_init.tcl
source [get_relative_loc]fuel_init.tcl
source [get_relative_loc]knock_init.tcl
source [get_relative_loc]inj_init.tcl
source [get_relative_loc]tg_init.tcl

#*******************************************************************************
# Definitions.
#*******************************************************************************
set ::CRANK_CHAN                0
set ::CAM_CHAN                  1
set ::SPARK_1_CHAN              2
set ::SPARK_2_CHAN              3
set ::SPARK_3_CHAN              4
set ::SPARK_4_CHAN              5
set ::FUEL_1_CHAN               6
set ::FUEL_2_CHAN               7
set ::FUEL_3_CHAN               8
set ::FUEL_4_CHAN               9
set ::KNOCK_1_CHAN             10
set ::KNOCK_2_CHAN             11
set ::INJ_BANK_1_CHAN          12
set ::INJ_BANK_2_CHAN          13
set ::INJ_1_CHAN               14
set ::INJ_2_CHAN               15
set ::INJ_3_CHAN               16
set ::INJ_4_CHAN               17
set ::TG_CRANK_CHAN            31
set ::TG_CAM_CHAN              30

set ::CRANK_BASE_ADDR       0x100
set ::CAM_BASE_ADDR         0x280
set ::SPARK_1_BASE_ADDR     0x300
set ::SPARK_2_BASE_ADDR     0x400
set ::SPARK_3_BASE_ADDR     0x500
set ::SPARK_4_BASE_ADDR     0x600
set ::FUEL_1_BASE_ADDR      0x700
set ::FUEL_2_BASE_ADDR      0x800
set ::FUEL_3_BASE_ADDR      0x900
set ::FUEL_4_BASE_ADDR      0xA00
set ::KNOCK_1_BASE_ADDR     0xB00
set ::KNOCK_2_BASE_ADDR     0xC00
set ::INJ_1_BASE_ADDR       0xD00
set ::INJ_2_BASE_ADDR       0xD40
set ::INJ_3_BASE_ADDR       0xD80
set ::INJ_4_BASE_ADDR       0xDC0
set ::INJ_TAB_BASE_ADDR     0xE00
set ::TG_BASE_ADDR          0xF00

set ::TDC_1_DEG                90
set ::TDC_2_DEG               270
set ::TDC_3_DEG               450
set ::TDC_4_DEG               630

#*******************************************************************************
# ISRs 
#*******************************************************************************
proc crank_isr { CRANK_CHAN } {
  #*****************************************************************************
  # Import automatically generated interface header file
  #*****************************************************************************
  include_generated_file "etpu_crank_auto.h"

  # Verify and clear channel interrupt
  verify_chan_intr $CRANK_CHAN  1
  clear_chan_intr  $CRANK_CHAN 
  
  # Read eng_pos_state
  config MemWidth 8
  set eng_pos_state [mem 0:[format 0x%.2x $::FS_ETPU_OFFSET_ENG_POS_STATE]]
  config MemWidth 32
  
  puts "CRANK ISR at EngPosState: $eng_pos_state"
  
  # Write tcr2_adjustment and HSR_SET_SENC
  if {$eng_pos_state == $::FS_ETPU_ENG_POS_PRE_FULL_SYNC} {
    write_chan_data24 $CRANK_CHAN $::FS_ETPU_CRANK_OFFSET_TCR2_ADJUSTMENT  [deg2tcr2 360]
    write_chan_hsrr   $CRANK_CHAN $::FS_ETPU_CRANK_HSR_SET_SYNC 
    puts "CRANK ISR: CRANK_HSR_SET_SYNC set."
    return 1
  }
  return 0
}


#*******************************************************************************
# Main 
#*******************************************************************************
proc func_main { } {
  #*****************************************************************************
  # eTPU Engine Configuration
  #*****************************************************************************
  write_entry_table_base_addr     0x0000
  write_engine_relative_base_addr 0x1000  

  #*****************************************************************************
  # eTPU Engine Clock and Time Base Configuration 
  #*****************************************************************************
  #clk_period is configured in (Run menu->)Debug configuration->Debugger->ETPU Simulator
  #set_clk_period 5000000;    # 200MHz
  write_tcr1_control    2;    # TCR1: system_clock/ 2 = 100MHz
  write_tcr1_prescaler  1;    # TCR1 =      100MHz/ 1 = 100MHz
  write_tcr2_control    2;    # TCR2: falling external
  write_tcr2_prescaler  1;    # 
  write_angle_mode      1;    # Angle Mode: Channel 0, TCRCLK input

  #*****************************************************************************
  # eTPU Module Configuration
  #*****************************************************************************
  write_global_time_base_enable 1; 
  
  #*****************************************************************************
  # Channel Initializations
  #*****************************************************************************
  # initialize Tooth Generator
  tg_init    $::TG_CRANK_CHAN  $::TG_CAM_CHAN  $::TG_BASE_ADDR;
  # connect TG outputs to TCRCLK and CAM input
  place_buffer   ($::TG_CRANK_CHAN+32) 64
  place_buffer   ($::TG_CAM_CHAN+32)   $::CAM_CHAN
  
  # or, use a vector file to generate input signals
  #parse_vector [get_relative_loc]CrankCam36-1.Vector
  
  # initalize CRANK and CAM
  crank_init $::CRANK_CHAN     $::CRANK_BASE_ADDR
  cam_init   $::CAM_CHAN       $::CAM_BASE_ADDR
  
  # initalize SPARKs
  spark_init $::SPARK_1_CHAN   $::SPARK_1_BASE_ADDR  $::TDC_1_DEG
  spark_init $::SPARK_2_CHAN   $::SPARK_2_BASE_ADDR  $::TDC_2_DEG
  spark_init $::SPARK_3_CHAN   $::SPARK_3_BASE_ADDR  $::TDC_3_DEG
  spark_init $::SPARK_4_CHAN   $::SPARK_4_BASE_ADDR  $::TDC_4_DEG
  
  # initalize FUELs
  fuel_init  $::FUEL_1_CHAN    $::FUEL_1_BASE_ADDR   $::TDC_1_DEG
  fuel_init  $::FUEL_2_CHAN    $::FUEL_2_BASE_ADDR   $::TDC_2_DEG
  fuel_init  $::FUEL_3_CHAN    $::FUEL_3_BASE_ADDR   $::TDC_3_DEG
  fuel_init  $::FUEL_4_CHAN    $::FUEL_4_BASE_ADDR   $::TDC_4_DEG
  
  # initalize KNOCKs
  knock_init  $::KNOCK_1_CHAN  $::KNOCK_1_BASE_ADDR   $::TDC_1_DEG
  knock_init  $::KNOCK_2_CHAN  $::KNOCK_2_BASE_ADDR   $::TDC_2_DEG

  # initalize INJs
  inj_bank_init  $::INJ_BANK_1_CHAN
  inj_bank_init  $::INJ_BANK_2_CHAN
  inj_table_init  $::INJ_TAB_BASE_ADDR
  inj_init   $::INJ_1_CHAN     $::INJ_1_BASE_ADDR    $::TDC_1_DEG  $::INJ_TAB_BASE_ADDR  [list $::INJ_BANK_1_CHAN $::INJ_BANK_2_CHAN]
  inj_init   $::INJ_2_CHAN     $::INJ_2_BASE_ADDR    $::TDC_2_DEG  $::INJ_TAB_BASE_ADDR  [list $::INJ_BANK_1_CHAN $::INJ_BANK_2_CHAN]
  inj_init   $::INJ_3_CHAN     $::INJ_3_BASE_ADDR    $::TDC_3_DEG  $::INJ_TAB_BASE_ADDR  [list $::INJ_BANK_1_CHAN $::INJ_BANK_2_CHAN]
  inj_init   $::INJ_4_CHAN     $::INJ_4_BASE_ADDR    $::TDC_4_DEG  $::INJ_TAB_BASE_ADDR  [list $::INJ_BANK_1_CHAN $::INJ_BANK_2_CHAN]
  
  # loop and poll interrupts
  while {1} {
    if { [get_bit [chanAddrScr 0 $::CRANK_CHAN] $::ETPUCxSCR_CIS 31] } {
      if { [crank_isr $::CRANK_CHAN] } {
        return
      }
    }
  }
}

func_main
puts "Script execution finished."
