/************************************************************************/
/*    Phase-shift PWM with external current limit control               */
/************************************************************************/
/* Desc: Output 2 complementary pairs of PWM signals at 100kHz			*/
/*       PWM0_H and PWM0_L on FlexPWM0 submodule 0						*/
/*       PWM1_H and PWM1_L on FlexPWM0 submodule 1						*/
/*       A "fake" Ilimit pulse is generated with FlexPWM0 submodule 2	*/
/************************************************************************/
/* Author: Yves Briant 													*/
/* Date:   14/03/2010													*/			
/************************************************************************/
#include "MPC5643L.h"
#include "dma.h"

// Table defined in main.c
extern volatile uint16_t RawVoltagesTable[4];

//Const used for DMA transfer must be aligned for use with modulo features
uint64_t forceOut_conf[4] 			__attribute__ ((aligned (32)));

//Constants for DMA config
//							 Normal Mode,        One Mode,		     Normal Mode,        Zero Mode
uint64_t forceOut_conf[] = 	{0x0000000080400800, 0x0008008080300800, 0x0000000080400800, 0x0000008080300800};
uint8_t dma_gpio_toggle[2] = 		{0x00, 0x01};

/************************************************************************/
/* void ConfigureDMA_CTU_FIFO(void)										*/
/************************************************************************/
/* Desc: Configure DMA channel 5 and 4                                  */
/*       ch5: - is triggered by CTU FIFO0 threshold						*/
/*			  - toggles GPIO											*/
/*			  - launches channel 4 on major loop						*/
/*       ch4: - transfer CTU FIFO0 to RawVoltagesTable                  */
/************************************************************************/
void ConfigureDMA_CTU_FIFO(void)
{
   /* Channel 4 config */
   EDMA.SERQR.R = 4; //Enable request to channel 4
     
   EDMA.CHANNEL[4].TCDWORD0_.B.SADDR = (uint32_t) ((uint8_t*)(&(CTU.FRA[0].R)) + 2);   // Source Address 	
   
   EDMA.CHANNEL[4].TCDWORD4_.B.SMOD  = 0;	    //  Source address modulo : no modulo features
   EDMA.CHANNEL[4].TCDWORD4_.B.SSIZE = 1;	    //  Source transfer size : 16 Bits 
   EDMA.CHANNEL[4].TCDWORD4_.B.DMOD  = 3;	    //  Destination address modulo	
   EDMA.CHANNEL[4].TCDWORD4_.B.DSIZE = 1;	    //  Destination transfer size : 16 Bits 
   EDMA.CHANNEL[4].TCDWORD4_.B.SOFF  = 0;	    //  Signed source address offset
   
   EDMA.CHANNEL[4].TCDWORD8_.B.SMLOE = 0;	    //  disable source minor loop offset
   EDMA.CHANNEL[4].TCDWORD8_.B.DMLOE = 0;	    //  disable desti minor loop offset
   EDMA.CHANNEL[4].TCDWORD8_.B.MLOFF = 0;	    //  minor loop offset
   EDMA.CHANNEL[4].TCDWORD8_.B.NBYTES = 0x8;	//  Inner "minor" byte count 
   
   EDMA.CHANNEL[4].TCDWORD12_.B.SLAST =  0;	    //  last Signed source address adjust                  

   EDMA.CHANNEL[4].TCDWORD16_.B.DADDR = (uint32_t) (RawVoltagesTable);   // Destination address 
   
   EDMA.CHANNEL[4].TCDWORD20_.B.CITER_E_LINK = 0;  //   no link on minor loop 
   EDMA.CHANNEL[4].TCDWORD20_.B.CITER_LINKCH = 0;  //   no link on minor loop                       
   EDMA.CHANNEL[4].TCDWORD20_.B.CITER = 1;	       //   iteration count
   EDMA.CHANNEL[4].TCDWORD20_.B.DOFF = 2;	       //  Signed destination address offset 
   
   EDMA.CHANNEL[4].TCDWORD24_.B.DLAST_SGA = 0;     //  Signed destination address adjust
   
   EDMA.CHANNEL[4].TCDWORD28_.B.BITER_E_LINK = 0;   //   no link on minor loop 
   EDMA.CHANNEL[4].TCDWORD28_.B.BITER_LINKCH = 0;
   EDMA.CHANNEL[4].TCDWORD28_.B.BITER = 1;          //  iteration count 
   EDMA.CHANNEL[4].TCDWORD28_.B.BWC = 0;	        //  Bandwidth control :  No DMA Stalls 
   EDMA.CHANNEL[4].TCDWORD28_.B.MAJOR_LINKCH = 0;	//  No link on major loop
   EDMA.CHANNEL[4].TCDWORD28_.B.DONE = 0;	        //  Channel Done 
   EDMA.CHANNEL[4].TCDWORD28_.B.ACTIVE = 0;	        //  Channel Active
   EDMA.CHANNEL[4].TCDWORD28_.B.MAJOR_E_LINK = 0;   //  No link on major loop
   EDMA.CHANNEL[4].TCDWORD28_.B.E_SG = 0;	    //  Disable Scatter/Gather
   EDMA.CHANNEL[4].TCDWORD28_.B.D_REQ = 0;	    //  TCD still enabled when done  
   EDMA.CHANNEL[4].TCDWORD28_.B.INT_HALF = 0;	//  No interrupt on minor loop count
   EDMA.CHANNEL[4].TCDWORD28_.B.INT_MAJ = 1;    //  Interrupt on major loop completion
  
   /* Channel 5 config */
   EDMA.SERQR.R = 5; //Enable request to channel 5
     
   EDMA.CHANNEL[5].TCDWORD0_.B.SADDR = (uint32_t) (dma_gpio_toggle);   // Source Address 	
   
   EDMA.CHANNEL[5].TCDWORD4_.B.SMOD  = 1;	    //  Source address modulo
   EDMA.CHANNEL[5].TCDWORD4_.B.SSIZE = 0;	    //  Source transfer size : 16 Bits 
   EDMA.CHANNEL[5].TCDWORD4_.B.DMOD  = 0;	    //  Destination address modulo	
   EDMA.CHANNEL[5].TCDWORD4_.B.DSIZE = 0;	    //  Destination transfer size : 16 Bits 
   EDMA.CHANNEL[5].TCDWORD4_.B.SOFF  = 1;	    //  Signed source address offset
   
   EDMA.CHANNEL[5].TCDWORD8_.B.SMLOE = 0;	    //  disable source minor loop offset
   EDMA.CHANNEL[5].TCDWORD8_.B.DMLOE = 0;	    //  disable desti minor loop offset
   EDMA.CHANNEL[5].TCDWORD8_.B.MLOFF = 0;	    //  minor loop offset
   EDMA.CHANNEL[5].TCDWORD8_.B.NBYTES = 1   ;	//  Inner "minor" byte count 
   
   EDMA.CHANNEL[5].TCDWORD12_.B.SLAST =  0;	    //  last Signed source address adjust                  

   EDMA.CHANNEL[5].TCDWORD16_.B.DADDR = (uint32_t) (&(SIU.GPDO[3].R));    // Destination address 
   
   EDMA.CHANNEL[5].TCDWORD20_.B.CITER_E_LINK = 0;  //   no link on minor loop 
   EDMA.CHANNEL[5].TCDWORD20_.B.CITER_LINKCH = 0;  //   no link on minor loop                       
   EDMA.CHANNEL[5].TCDWORD20_.B.CITER = 1;	       //   iteration count
   EDMA.CHANNEL[5].TCDWORD20_.B.DOFF = 0;	       //  Signed destination address offset 
   
   EDMA.CHANNEL[5].TCDWORD24_.B.DLAST_SGA = 0;     //  Signed destination address adjust
   
   EDMA.CHANNEL[5].TCDWORD28_.B.BITER_E_LINK = 0;   // no link on minor loop 
   EDMA.CHANNEL[5].TCDWORD28_.B.BITER_LINKCH = 0;   // no link on minor loop
   EDMA.CHANNEL[5].TCDWORD28_.B.BITER = 1;          // iteration count 
   EDMA.CHANNEL[5].TCDWORD28_.B.BWC = 0;	        // bandwidth control :  No DMA Stalls 
   EDMA.CHANNEL[5].TCDWORD28_.B.MAJOR_LINKCH = 4;	// Link to channel 4 on major loop
   EDMA.CHANNEL[5].TCDWORD28_.B.DONE = 0;	        // Channel Done 
   EDMA.CHANNEL[5].TCDWORD28_.B.ACTIVE = 0;	        // Channel Active
   EDMA.CHANNEL[5].TCDWORD28_.B.MAJOR_E_LINK = 1;   // Link to channel 4 on major loop
   EDMA.CHANNEL[5].TCDWORD28_.B.E_SG = 0;	    //  Disable Scatter/Gather
   EDMA.CHANNEL[5].TCDWORD28_.B.D_REQ = 0;	    //  TCD still enabled when done  
   EDMA.CHANNEL[5].TCDWORD28_.B.INT_HALF = 0;	//  No interrupt on minor loop count
   EDMA.CHANNEL[5].TCDWORD28_.B.INT_MAJ = 0;    //  Interrupt on major loop completion

   DMAMUX.CHCONFIG[5].B.TRIG = 0;
   DMAMUX.CHCONFIG[5].B.SOURCE = 8;   // CTU FIFO0 threshold
   DMAMUX.CHCONFIG[5].B.ENBL = 1;
}
   
/************************************************************************/
/* void ConfigureDMA_Priorities(void)									*/
/************************************************************************/
/* Desc: Configure respective priorities of channels 5,4,3,2            */
/************************************************************************/   
void ConfigureDMA_Priorities(void)
{
   /* Priorities */
   EDMA.DCHPRI0.R = 4;
   EDMA.DCHPRI2.R = 5;
   EDMA.DCHPRI3.R = 3;
   EDMA.DCHPRI4.R = 2;
   EDMA.DCHPRI5.R = 0;   //channel5 should have a lower priority than channel 4
}


/************************************************************************/
/* void ConfigureDMA4ForceOut_test(void)								*/
/************************************************************************/
/* Desc: Configure channels 0 and 1 for force-out reconfiguration       */
/************************************************************************/  
void ConfigureDMA4ForceOut(void)
{
   EDMA.DMACR.B.EMLM = 1;  //enable minor loop mapping

   /* Channel 0 config ***************************************************/
   EDMA.SERQR.R = 0; //Enable request to channel 0
     
   EDMA.CHANNEL[0].TCDWORD0_.B.SADDR = (uint32_t) (forceOut_conf);   // Source address 	
  
   EDMA.CHANNEL[0].TCDWORD4_.B.SMOD  = 5; 	//  Source address modulo
   EDMA.CHANNEL[0].TCDWORD4_.B.SSIZE = 3; 	//  Source transfer size : 64 Bits 
   EDMA.CHANNEL[0].TCDWORD4_.B.DMOD  = 0;	//  Destination address modulo: no modulo features 	
   EDMA.CHANNEL[0].TCDWORD4_.B.DSIZE = 2;	//  Destination transfer size : 32 Bits 
   EDMA.CHANNEL[0].TCDWORD4_.B.SOFF  = 8;	//  Signed source address offset : 64bits


   EDMA.CHANNEL[0].TCDWORD8_.B.SMLOE  = 0;	     //  disable source minor loop offset
   EDMA.CHANNEL[0].TCDWORD8_.B.DMLOE  = 1;	     //  enable desti minor loop offset
   EDMA.CHANNEL[0].TCDWORD8_.B.MLOFF  = 0x01E0;  //  offset at minor loop completion
   EDMA.CHANNEL[0].TCDWORD8_.B.NBYTES = 8;	     //  Inner "minor" byte count: 64bits
   
   EDMA.CHANNEL[0].TCDWORD12_.B.SLAST =  0;	     //  last Signed source address adjust                  

   EDMA.CHANNEL[0].TCDWORD16_.B.DADDR = (uint32_t) (&FLEXPWM_0.SWCOUT.R);   // Destination address 
   
   EDMA.CHANNEL[0].TCDWORD20_.B.CITER_E_LINK = 1;      //  Enable link on minor loop                         
   EDMA.CHANNEL[0].TCDWORD20_.B.CITER_LINKCH = 0;      //  Link to channel 0 on minor loop
   EDMA.CHANNEL[0].TCDWORD20_.B.CITER        = 2; 	   //  2 iterations in major loop
   EDMA.CHANNEL[0].TCDWORD20_.B.DOFF 		 = 0xFF10; //  Signed destination address offset 

   EDMA.CHANNEL[0].TCDWORD24_.B.DLAST_SGA = 0x01E0;  //  signed destination address adjust at major loop completion
  
   EDMA.CHANNEL[0].TCDWORD28_.B.BITER_E_LINK = 1; //  Enable link on minor loop
   EDMA.CHANNEL[0].TCDWORD28_.B.BITER_LINKCH = 0;
   EDMA.CHANNEL[0].TCDWORD28_.B.BITER        = 2; //  2 iterations in major loop
   EDMA.CHANNEL[0].TCDWORD28_.B.BWC = 0;	      //  Bandwidth control :  No DMA Stalls 
   EDMA.CHANNEL[0].TCDWORD28_.B.MAJOR_LINKCH = 1; //  Link on major loop 
   EDMA.CHANNEL[0].TCDWORD28_.B.DONE = 0;	      //  Channel Done 
   EDMA.CHANNEL[0].TCDWORD28_.B.ACTIVE = 0;	      //  Channel Active
   EDMA.CHANNEL[0].TCDWORD28_.B.MAJOR_E_LINK = 1; //  Link on major loop
   EDMA.CHANNEL[0].TCDWORD28_.B.E_SG = 0;	      //  Disable Scatter/Gather
   EDMA.CHANNEL[0].TCDWORD28_.B.D_REQ = 0;	      //  TCD still enabled when done  
   EDMA.CHANNEL[0].TCDWORD28_.B.INT_HALF = 0;     //  No interrupt on minor loop count
   EDMA.CHANNEL[0].TCDWORD28_.B.INT_MAJ = 0;      //  No interrupt on major loop completion
   
   /* Channel 1 config ****************************************************/
   EDMA.SERQR.R = 1;    //Enable request to channel 1
  
   EDMA.CHANNEL[1].TCDWORD0_.B.SADDR = (uint32_t) (&dma_gpio_toggle);   // Source Address 	

   EDMA.CHANNEL[1].TCDWORD4_.B.SMOD  = 1;	    //  Source address modulo
   EDMA.CHANNEL[1].TCDWORD4_.B.SSIZE = 0;	    //  Source transfer size : 8 Bits 
   EDMA.CHANNEL[1].TCDWORD4_.B.SOFF  = 1;	    //  Signed source address offset
   EDMA.CHANNEL[1].TCDWORD4_.B.DMOD  = 0;	    //  Destination address modulo: no modulo features 	
   EDMA.CHANNEL[1].TCDWORD4_.B.DSIZE = 0;	    //  Destination transfer size : 8 Bits 
  
   EDMA.CHANNEL[1].TCDWORD8_.B.SMLOE = 0;	    //  disable source minor loop offset
   EDMA.CHANNEL[1].TCDWORD8_.B.DMLOE = 0;	    //  disable desti minor loop offset
   EDMA.CHANNEL[1].TCDWORD8_.B.MLOFF = 0;	    //  minor loop offset
   EDMA.CHANNEL[1].TCDWORD8_.B.NBYTES = 1;	    //  Inner "minor" byte count 
  
   EDMA.CHANNEL[1].TCDWORD12_.B.SLAST =  0;	    //  last Signed source address adjust                  

   EDMA.CHANNEL[1].TCDWORD16_.B.DADDR = (uint32_t) (&(SIU.GPDO[0].R));         
  
   EDMA.CHANNEL[1].TCDWORD20_.B.DOFF = 0;	      //  Signed destination address offset 
   EDMA.CHANNEL[1].TCDWORD20_.B.CITER_E_LINK = 1; //  link to channel 0 on minor loop
   EDMA.CHANNEL[1].TCDWORD20_.B.CITER_LINKCH = 0; // 
   
   EDMA.CHANNEL[1].TCDWORD20_.B.CITER = 2;  	  //  2 iterations in major loop
  
   EDMA.CHANNEL[1].TCDWORD24_.B.DLAST_SGA = 0;	//  Signed destination address adjust	

   EDMA.CHANNEL[1].TCDWORD28_.B.BITER_E_LINK = 0; // no link on minor loop
   EDMA.CHANNEL[1].TCDWORD28_.B.BITER_LINKCH = 0;
   EDMA.CHANNEL[1].TCDWORD28_.B.BITER = 2;
   EDMA.CHANNEL[1].TCDWORD28_.B.BWC = 0x00;	      //  Bandwidth control :  No DMA Stalls    
   EDMA.CHANNEL[1].TCDWORD28_.B.MAJOR_LINKCH = 0; //  Link to channel 0 on major loop
   EDMA.CHANNEL[1].TCDWORD28_.B.MAJOR_E_LINK = 1; //    
   
   EDMA.CHANNEL[1].TCDWORD28_.B.DONE = 0;	 	  //  Channel Done 
   EDMA.CHANNEL[1].TCDWORD28_.B.ACTIVE = 0;		  //  Channel Active
   EDMA.CHANNEL[1].TCDWORD28_.B.E_SG = 0;	      //  Enable Scatter/Gather : Disabled
   EDMA.CHANNEL[1].TCDWORD28_.B.D_REQ = 0;	      //  Disable TCD When done  
   EDMA.CHANNEL[1].TCDWORD28_.B.INT_HALF = 0;	  //  Interrupt on minor loop count : Disabled
   EDMA.CHANNEL[1].TCDWORD28_.B.INT_MAJ = 0;	  //  Interrupt on major loop completion
   
   DMAMUX.CHCONFIG[1].B.TRIG = 0;
   DMAMUX.CHCONFIG[1].B.SOURCE = 12;   //FlexPWM write request
   DMAMUX.CHCONFIG[1].B.ENBL = 1;
   

}
