/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     math.c
*
* @author   B06050
* 
* @version  1.0.2.0
* 
* @date     Jul-27-2012
* 
* @brief    Mathematical function source file.
*
*******************************************************************************/
#include "math.h"

/*******************************************************************************
*
* Function: tFrac32 MulSat_F32F16F16(tFrac16 f16In1,tFrac16 f16In2)
*
* Description:  This function performs 32-bit fractional multiplication with
*               saturation.
*
* Param[in]:    f32In1      First 32-bit signed fractional operand
*               f32In2      Second 32-bit signed fractional operand
*
* Return:       tFrac16     16-bit fractional multiplication of input arguments
*
********************************************************************************/
#pragma INLINE
tFrac32 MulSat_F32F16F16(tFrac16 f16In1,tFrac16 f16In2)
{
    register int32_t z;

    z = (int32_t)(((int32_t)f16In1*(int32_t)f16In2)<<1);
    z = (z == INT32_MIN) ? INT32_MAX : z;

    return ((tFrac32)z);
}

/*******************************************************************************
*
* Function: tFrac16 ShL_F16(tFrac16 f16In1,uint16_t u16In2)
*
* Description:  This function shifts the first parameter left by number of bits
*               defined by the second argument.
*
* Param[in]:    f16In1      16-bit signed fractional operand to be shifted left
*               u16In2      Shift amount value.
*
* Return:       tFrac16     16-bit fractional value shifted left by the shift
*                           amount. The bits beyond the 16-bit boundary are
*                           discarded.
*
********************************************************************************/
#pragma INLINE
tFrac16 ShL_F16(tFrac16 f16In1,uint16_t u16In2)
{
    return (tFrac16)(f16In1 << u16In2);
}

/*******************************************************************************
*
* Function: tFrac32 ShL_F32(tFrac32 f32In1,uint16_t u16In2)
*
* Description:  This function shifts the first parameter left by number of bits
*               defined by the second argument.
*
* Param[in]:    f32In1      32-bit signed fractional operand to be shifted left
*               u16In2      Shift amount value.
*
* Return:       tFrac16     32-bit fractional value shifted left by the shift
*                           amount. The bits beyond the 32-bit boundary are
*                           discarded.
*
********************************************************************************/
#pragma INLINE
tFrac32 ShL_F32(tFrac32 f32In1,uint16_t u16In2)
{
    return (tFrac32)(f32In1 << u16In2);
}

/*******************************************************************************
*
* Function: tFrac32 ShR_F32(tFrac32 f32In1,uint16_t u16In2)
*
* Description:  This function shifts the first parameter right by number of bits
*               defined by the second argument.
*
* Param[in]:    f32In1      32-bit signed fractional operand to be shifted right
*               u16In2      Shift amount value.
*
* Return:       tFrac16     32-bit fractional value shifted right by the shift
*                           amount. The bits beyond the 32-bit boundary are
*                           discarded.
*
********************************************************************************/
#pragma INLINE
tFrac32 ShR_F32(tFrac32 f32In1,uint16_t u16In2)
{
    return (tFrac32)(f32In1 >> u16In2);
}

/*******************************************************************************
*
* Function: tFrac32 MacSat_F32F16F16(tFrac32 f32In1,tFrac16 f16In2,
*                                    tFrac16 f16In3)
*
* Description:  This function implements the multiply accumulate function with
*               saturation.
*
* Param[in]:    f32In1      Unput value to be added
*               f32In2      First 32-bit signed fractional operand
*               f32In3      Second 32-bit signed fractional operand
*
* Return:       tFrac16     Multiplied second and third input values with
*                           addition of the first input value. The output value
*                           is saturated if necessary.
*
********************************************************************************/
#pragma INLINE
tFrac32 MacSat_F32F16F16(tFrac32 f32In1,tFrac16 f16In2,tFrac16 f16In3)
{
    return(AddSat_F32(f32In1, MulSat_F32F16F16(f16In2, f16In3)));
}

/*******************************************************************************
*
* Function: tFrac16 ControllerPIrAW_F16
*                   (tFrac16 f16InErr, CONTROLLER_PIAW_R_T_F16 * const pParam)
*
* Description:  This function calculates a standard recurrent form
*               of the Proportional-Integral controller with integral
*               anti-windup.
*
*               Discrete time domain controller equation:
*
*               u(k) = u(k-1) + e(k)*CC1 + e(k-1)*CC2
*
*               where u(k) is the controller output, e(k) is controller input
*               error signal, CC1 and CC2 are controller coefficients calculated
*               depending on the discretization method used).
*
*               Calculation of coefficient CC1 and CC2 using various
*               discretization methods:
*
*               Trapeziodal:        CC1 = Kp + Ki*(Ts/2); CC2 = -Kp + Ki*(Ts/2)
*               Backward Rectangle: CC1 = Kp + Ki*Ts;     CC2 = -Kp
*               Forward Rectangle:  CC1 = Kp;             CC2 = -Kp + Ki*Ts
*
*               where Kp is proportional gain, Ki integral gain, and Ts is
*               sampling period.
*
* Param[in]:    f16InErr    16-bit signed fractional error signal in format
*                           Q1.15
* Param[in,out]:*pParam     Pointer to the controller parameters structure.
*
********************************************************************************/
tFrac16 ControllerPIrAW_F16(tFrac16 f16InErr,CONTROLLER_PIAW_R_T_F16 * const pParam)
{
    tFrac32 f32A1;
    tFrac32 f32UpLim;
    tFrac32 f32LoLim;
    tFrac16 f16Acc;

    /*
     * Implemented equation:
     * u(k) = u(k-1) + e(k)*CC1 + e(k-1)*CC2
     *
     * Calculation steps:
     * M1   = e(k)*CC1sc
     * M2   = e(k-1)*CC2sc
     * A1   = M1+M2
     * Acc  = u(k-1)+A1
     */
    /* A1   = u(k-1) + e(k)*CC1sc */
    f32A1 = MacSat_F32F16F16(pParam->f32Acc,f16InErr,pParam->f16CC1sc);

    /* A2   = A1 + e(k-1)*CC2sc */
    f32A1 = MacSat_F32F16F16(f32A1,pParam->f16InErrK1, pParam->f16CC2sc);

    /* Limitation applied during calculation, on internal accumulator */
    f32UpLim = ShR_F32(ShL_F32((int32_t)(pParam->f16UpperLimit),16), pParam->u16NShift);
    f32LoLim = ShR_F32(ShL_F32((int32_t)(pParam->f16LowerLimit),16), pParam->u16NShift);

    f32A1 = ((f32A1 >= f32UpLim) ? f32UpLim : ((f32A1 <= f32LoLim) ? f32LoLim : f32A1));
    
    /* Storing filter states in the buffer */
    pParam->f16InErrK1  = f16InErr;
    pParam->f32Acc = f32A1;
    f16Acc = (tFrac16)(ShR_F32(f32A1, 16));

    return(ShL_F16(f16Acc, pParam->u16NShift));
}

/*******************************************************************************
*
* Function: void FilterMAInit_F16(FILTER_MA_T_F16 *pParam)
*
* Description:  This function clears the internal accumulator of a moving
*               average filter.
*
* Param[in,out]:*pParam     Pointer to the MA filter structure with a filter
*                           accumulator and filter parameters.
*
* Notes:        This function shall be called after filter parameter
*               initialization and whenever the filter initialization is
*               required.
*
********************************************************************************/
#pragma INLINE
void FilterMAInit_F16(FILTER_MA_T_F16 *pParam)
{
    pParam->f32Acc  = (tFrac32) 0;
}

/*******************************************************************************
*
* Function: tFrac16 FilterMA_F16(tFrac16 f16In, FILTER_MA_T_F16 *pParam)
*
* Description:  This function implements a moving average recursive filter.
*
* Param[in]:    f16In       Value of input signal to filtered in step (k).
*                           The value is 16-bit number in Q1.15 format.
* Param[in,out]:*pParam     Pointer to the MA filter structure with a filter
*                           accumulator and filter parameters.
*
* Notes:        This function shall be called after filter parameter
*               initialization and whenever the filter initialization is
*               required.
*
********************************************************************************/
tFrac16 FilterMA_F16(tFrac16 f16In, FILTER_MA_T_F16 *pParam)
{
    tFrac32  f32Temp;
    tFrac32  f32Acc;
    
    f32Acc  = pParam->f32Acc + (tFrac32)f16In;
    f32Temp = f32Acc >> pParam->u16NSamples;
    f32Acc  = f32Acc - f32Temp;

    /* Store new accumulator state */
    pParam->f32Acc    = f32Acc;
    
    return((tFrac16)f32Temp);
}