/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file       MPC5606B_PIT_LLD.h
*
* @author     b06050
*
* @version    1.0.2.0
*
* @date       Aug-3-2012
*
* @brief      PIT low level driver module for MPC5606B.
*
********************************************************************************
*
* This file includes initial setting and MACRO definitions of MPC5606B PIT
* peripheral module.
*
*******************************************************************************/
#ifndef __MPC5606B_PIT_LLD_H
#define __MPC5606B_PIT_LLD_H

#include "typedefs.h"
#include "MPC5606B_LLD_qs.h"

#ifdef __cplusplus
extern "C" {
#endif

#define PIT_BASE                                        0xC3FF0000
#define PIT_CHNL0_BASE                                  0xC3FF0100
#define PIT_CHNL1_BASE                                  0xC3FF0110
#define PIT_CHNL2_BASE                                  0xC3FF0120
#define PIT_CHNL3_BASE                                  0xC3FF0130
#define PIT_CHNL4_BASE                                  0xC3FF0140
#define PIT_CHNL5_BASE                                  0xC3FF0150
#define PIT_CHNL6_BASE                                  0xC3FF0160
#define PIT_CHNL7_BASE                                  0xC3FF0170

#ifndef PIT_CLK
// define PIT CLK in MHz
#define PIT_CLK                     64
#endif
/*******************************************************************************
* PIT ioctl commands definitions
*******************************************************************************/
/***************************************************************************//*!
*
* @brief                Enable/disable clock for PIT module.
*
* @param                PIT_CLK_DISABLE
* @param                PIT_CLK_ENABLE
*
* @return               NULL
*
* @details              The #PIT_MODULE_CLOCK ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_BASE,PIT_MODULE_CLOCK, PIT_DISABLE);
* \endcode
*
*******************************************************************************/
#define PIT_MODULE_CLOCK

/***************************************************************************//*!
*
* @brief                Set behavior in the debug mode.
*
* @param                PIT_RUN
* @param                PIT_STOP
*
* @return               NULL
*
* @details              The #PIT_DEBUG_OPERATION ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_BASE,PIT_DEBUG_OPERATION, PIT_RUN);
* \endcode
*
*******************************************************************************/
#define PIT_DEBUG_OPERATION

/***************************************************************************//*!
*
* @brief                The function read the MCR register value.
*
* @param                NULL
*
* @return               The function return MCR register actual value.
*
* @details              The #PIT_READ_VALUE_REG_MCR ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* vuint32_t u32RetVal;
* u32RetVal = ioctl(PIT_BASE,PIT_READ_VALUE_REG_MCR, NULL);
* \endcode
*
*******************************************************************************/
#define PIT_READ_VALUE_REG_MCR

/***************************************************************************//*!
*
* @brief                The function write parameter value to the MCR register.
*
* @param                (value 32bit unsigned)
*
* @return               NULL
*
* @details              The #PIT_WRITE_VALUE_REG_MCR ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_BASE,PIT_WRITE_VALUE_REG_MCR, (value 32bit unsigned));
* \endcode
*
*******************************************************************************/
#define PIT_WRITE_VALUE_REG_MCR

/***************************************************************************//*!
*
* @brief                The function read the LDVAL register value.
*
* @param                NULL
*
* @return               The function return LDVAL register actual value.
*
* @details              The #PIT_READ_VALUE_REG_LDVAL ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* vuint32_t u32RetVal;
* u32RetVal = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_LDVAL, NULL);
* \endcode
*
*******************************************************************************/
#define PIT_READ_VALUE_REG_LDVAL

/***************************************************************************//*!
*
* @brief                Set timer start value for dedicated channel.
*
* @param                VALUE
*
* @return               NULL
*
* @details              The #PIT_SET_TMR_RELOAD_VALUE ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_CHNL0_BASE,PIT_SET_TMR_RELOAD_VALUE, VALUE);
* \endcode
*
*******************************************************************************/
#define PIT_SET_TMR_RELOAD_VALUE

/***************************************************************************//*!
*
* @brief                Get current timer value for dedicated channel.
*
* @param                NULL
*
* @return               Current channel_x timer value.
*
* @details              The #PIT_GET_TMR_CURRENT_VALUE ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* vuint32_t u32RetVal;
* u32RetVal = ioctl(PIT_CHNL0_BASE,PIT_GET_TMR_CURRENT_VALUE, NULL);
* \endcode
*
*******************************************************************************/
#define PIT_GET_TMR_CURRENT_VALUE

/***************************************************************************//*!
*
* @brief                The function read the TCTRL register value.
*
* @param                NULL
*
* @return               The function return TCTRL register actual value.
*
* @details              The #PIT_READ_VALUE_REG_TCTRL ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* vuint32_t u32RetVal;
* u32RetVal = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_TCTRL, NULL);
* \endcode
*
*******************************************************************************/
#define PIT_READ_VALUE_REG_TCTRL

/***************************************************************************//*!
*
* @brief                The function write parameter value to the TCTRL register.
*
* @param                (value 32bit unsigned)
*
* @return               NULL
*
* @details              The #PIT_WRITE_VALUE_REG_TCTRL ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_CHNL0_BASE,PIT_WRITE_VALUE_REG_TCTRL, (value 32bit unsigned));
* \endcode
*
*******************************************************************************/
#define PIT_WRITE_VALUE_REG_TCTRL

/***************************************************************************//*!
*
* @brief                Enable/disable interrupt from dedicated channel.
*
* @param                PIT_DISABLE
* @param                PIT_ENABLE
*
* @return               NULL
*
* @details              The #PIT_TIMER_INTERRUPT ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_CHNL0_BASE,PIT_TIMER_INTERRUPT, PIT_DISABLE);
* \endcode
*
*******************************************************************************/
#define PIT_TIMER_INTERRUPT

/***************************************************************************//*!
*
* @brief                Enable or disable dedicated timer channel.
*
* @param                PIT_DISABLE
* @param                PIT_ENABLE
*
* @return               NULL
*
* @details              The #PIT_TIMER ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_CHNL0_BASE,PIT_TIMER, PIT_DISABLE);
* \endcode
*
*******************************************************************************/
#define PIT_TIMER

/***************************************************************************//*!
*
* @brief                Read interrupt flag of the dedicated timer channel.
*
* @param                NULL
*
* @return               The function return interrupt flag value for channel x.
*
* @details              The #PIT_GET_FLAG ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* vuint32_t u32RetVal;
* u32RetVal = ioctl(PIT_CHNL0_BASE,PIT_GET_FLAG, NULL);
* \endcode
*
*******************************************************************************/
#define PIT_GET_FLAG

/***************************************************************************//*!
*
* @brief                Clear interrupt flag of the dedicated timer channel.
*
* @param                PIT_CLEAR_FLAG
*
* @return               NULL
*
* @details              The #PIT_CLEAR_FLAG ioctl command TBD
*
* @par Code Example:
*
* \code
* //
* ioctl(PIT_CHNL0_BASE,PIT_CLEAR_FLAG, PIT_CLEAR_FLAG);
* \endcode
*
*******************************************************************************/
#define PIT_CLEAR_FLAG


/*******************************************************************************
* PIT bitfield masks definition
*******************************************************************************/

#define PIT_MDIS_MASK                                   0x00000002
#define PIT_FRZ_MASK                                    0x00000001
#define PIT_TIE_MASK                                    0x00000002
#define PIT_TEN_MASK                                    0x00000001
#define PIT_TIF_MASK                                    0x00000001

/*******************************************************************************
* PIT register offset definition
*******************************************************************************/

#define PIT_PITMCR_OFFSET                               0x0
#define PIT_LDVAL_OFFSET                                0x0
#define PIT_CVAL_OFFSET                                 0x4
#define PIT_TCTRL_OFFSET                                0x8
#define PIT_TFLG_OFFSET                                 0xC

/*******************************************************************************
* PIT MACRO and ioctl parameters definition
*******************************************************************************/

#define PIT_CLK_DISABLE                                 0x1
#define PIT_CLK_ENABLE                                  0x0
#define PIT_DISABLE                                     0x0
#define PIT_ENABLE                                      0x1
#define PIT_RUN                                         0x0
#define PIT_STOP                                        0x1

/*******************************************************************************
* PIT ioctl commands implementation
*******************************************************************************/

#define ioctlPIT_MODULE_CLOCK(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_PITMCR_OFFSET)) = (*((vuint32_t *)(pitAddr + PIT_PITMCR_OFFSET)) & (~PIT_MDIS_MASK)) | (param<<1)

#define ioctlPIT_DEBUG_OPERATION(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_PITMCR_OFFSET)) = (*((vuint32_t *)(pitAddr + PIT_PITMCR_OFFSET)) & (~PIT_FRZ_MASK)) | (param)

#define ioctlPIT_READ_VALUE_REG_MCR(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_PITMCR_OFFSET))

#define ioctlPIT_WRITE_VALUE_REG_MCR(pitAddr,param) \
        *((vuint16_t *)(pitAddr + PIT_PITMCR_OFFSET)) = (param)

#define ioctlPIT_READ_VALUE_REG_LDVAL(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_LDVAL_OFFSET))

#define ioctlPIT_SET_TMR_RELOAD_VALUE(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_LDVAL_OFFSET)) = (param)

#define ioctlPIT_GET_TMR_CURRENT_VALUE(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_CVAL_OFFSET))

#define ioctlPIT_READ_VALUE_REG_TCTRL(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET))

#define ioctlPIT_WRITE_VALUE_REG_TCTRL(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET)) = (param)

#define ioctlPIT_TIMER_INTERRUPT(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET)) = (*((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET)) & (~PIT_TIE_MASK)) | (param<<1)

#define ioctlPIT_TIMER(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET)) = (*((vuint32_t *)(pitAddr + PIT_TCTRL_OFFSET)) & (~PIT_TEN_MASK)) | (param)

#define ioctlPIT_GET_FLAG(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TFLG_OFFSET)) & (PIT_TIF_MASK)

#define ioctlPIT_CLEAR_FLAG(pitAddr,param) \
        *((vuint32_t *)(pitAddr + PIT_TFLG_OFFSET)) = (*((vuint32_t *)(pitAddr + PIT_TFLG_OFFSET)) & (~PIT_TIF_MASK)) | (1)

extern  void delayNanoSec(uint32_t delay_ns);

#ifdef __cplusplus
}
#endif

#endif /* __MPC5606B_PIT_LLD_H */