/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MPC5606B_ADC_BSP.c
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    MPC5606B Board Support Package - ADC module
*
********************************************************************************
*
* This file includes all functions and settings specific for MPC560XB Controller
* Board.
*
*******************************************************************************/

/*******************************************************************************
| Includes
------------------------------------------------------------------------------*/
#include "MPC5606B.h"
#include "gflib.h"
#include "MPC5606B_ADC_BSP.h"
#include "MPC5606B_LLD_qs.h"

/*******************************************************************************
| Function implementations      (scope: module-exported)
------------------------------------------------------------------------------*/

/***************************************************************************//*!
@brief      ADC software initialization routine.

@param[in,out]  *ptr    Pointer to structure of ADC module variables and
                        parameters

@return     # true - ADC initialization successful
            # false - ADC initialization error

@details    This function performs software ADC module initialization.

@note       ADC module as refered to in this function is not actual peripheral
            module but an instance of software defined module of structure
            type holding all necessary ADC parameters and variables.
*******************************************************************************/
bool_t ADCModuleInit(adcModule_t *ptr)
{
    static tU8  i;

    ptr->measured.f32PhA.filt   = 0x0;
    ptr->measured.f32PhA.raw    = 0x0;
    ptr->measured.f32PhB.filt   = 0x0;
    ptr->measured.f32PhB.raw    = 0x0;
    ptr->measured.f32PhC.filt   = 0x0;
    ptr->measured.f32PhC.raw    = 0x0;
    ptr->measured.f32Udcb.filt  = 0x0;
    ptr->measured.f32Udcb.raw   = 0x0;
    ptr->measured.f32Temp.filt  = 0x0;
    ptr->measured.f32Temp.raw   = 0x0;
    ptr->measured.f32Idcb.filt  = 0x0;
    ptr->measured.f32Idcb.raw   = 0x0;

    ptr->flag.R             = 0x0;
    for (i=1;i<=6;i++)
    {
        ptr->offset[i].f32PhA      = 0x0;
        ptr->offset[i].f32PhB      = 0x0;
        ptr->offset[i].f32PhC      = 0x0;
        ptr->offset[i].f32Idcb     = 0x0;
        ptr->offset[i].f32Udcb     = 0x0;
        ptr->offset[i].f32Temp     = 0x0;
    }

    ptr->param.u16CalibSamples  = 0x0;
    ptr->flag.B.calibInitDone   = 0;
    ptr->flag.B.calibDone       = 0;

    return 1;
}

/***************************************************************************//*!
@brief      ADC software initialization routine.

@param[in,out]  *ptr    Pointer to structure of ADC module variables and
                        parameters

@return     # true - ADC initialization successful
            # false - ADC initialization error

@details    This function performs software ADC module initialization.

@note       ADC module as refered to in this function is not actual peripheral
            module but an instance of software defined module of structure
            type holding all necessary ADC parameters and variables.
*******************************************************************************/
bool_t ADCModuleInit16(adcModule16_t *ptr)
{
    static tU8  i;

    ptr->measured.f16BemfA.filt16   = 0x0;
    ptr->measured.f16BemfA.raw16    = 0x0;
    ptr->measured.f16BemfB.filt16   = 0x0;
    ptr->measured.f16BemfB.raw16    = 0x0;
    ptr->measured.f16BemfC.filt16   = 0x0;
    ptr->measured.f16BemfC.raw16    = 0x0;
    ptr->measured.f16Udcb.filt16    = 0x0;
    ptr->measured.f16Udcb.raw16     = 0x0;
    ptr->measured.f16Temp.filt16    = 0x0;
    ptr->measured.f16Temp.raw16     = 0x0;
    ptr->measured.f16Idcb.filt16    = 0x0;
    ptr->measured.f16Idcb.raw16     = 0x0;

    ptr->flag.R                     = 0x0;
    for (i=1;i<=6;i++)
    {
        ptr->offset[i].f16BemfA     = 0x0;
        ptr->offset[i].f16BemfB     = 0x0;
        ptr->offset[i].f16BemfC     = 0x0;
        ptr->offset[i].f16Idcb      = 0x0;
        ptr->offset[i].f16Udcb      = 0x0;
        ptr->offset[i].f16Temp      = 0x0;
    }

    ptr->param.u16CalibSamples  = 0x0;
    ptr->flag.B.calibInitDone   = 0;
    ptr->flag.B.calibDone       = 0;

    return 1;
}

/***************************************************************************//*!
@brief      ADC software calibration routine.

@param[in,out]  *ptr    Pointer to structure of ADC module variables and
                        parameters

@return     # true - when ADC calibration ended successfully
            # false - when ADC calibration is ongoing

@details    This function performs offset calibration during the calibration
            phase of the application. It is not intended to be executed when
            application is in run mode.

@note       ADC module as refered to in this function is not actual peripheral
            module but an instance of software defined module of structure
            type holding all necessary ADC parameters and variables.
*******************************************************************************/
bool_t ADCModuleCalib(adcModule_t *ptr, tU32 svmSector)
{
    static tU16 calibCntr;
    static tFrac16 f16FiltOutput;

    if (!(ptr->flag.B.calibInitDone))
    {
        // +4 in order to accommodate settling time of the filter
        calibCntr = 1<<(ptr->param.u16CalibSamples+4);

        ptr->measured.f32PhA.filt   = 0x0;
        ptr->measured.f32PhB.filt   = 0x0;
        ptr->measured.f32PhC.filt   = 0x0;
        ptr->measured.f32Udcb.filt  = 0x0;

        ptr->flag.B.calibDone       = 0;
        ptr->flag.B.calibInitDone   = 1;
    }

    if (!(ptr->flag.B.calibDone))
    {
        /* --------------------------------------------------------------
         * Phase A - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
        ptr->measured.f32PhA.filt       += (tFrac16)(ptr->measured.f32PhA.raw>>16);
        f16FiltOutput                   = (ptr->measured.f32PhA.filt)>>(ptr->param.u16CalibSamples);
        ptr->measured.f32PhA.filt       -= f16FiltOutput;
        ptr->offset[svmSector].f32PhA   = (tFrac32)(f16FiltOutput<<16);

        /* --------------------------------------------------------------
         * Phase B - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
        ptr->measured.f32PhB.filt       += (tFrac16)(ptr->measured.f32PhB.raw>>16);
        f16FiltOutput                   = (ptr->measured.f32PhB.filt)>>(ptr->param.u16CalibSamples);
        ptr->measured.f32PhB.filt       -= f16FiltOutput;
        ptr->offset[svmSector].f32PhB   = (tFrac32)(f16FiltOutput<<16);

        /* --------------------------------------------------------------
         * Phase C - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
        ptr->measured.f32PhC.filt       += (tFrac16)(ptr->measured.f32PhC.raw>>16);
        f16FiltOutput                   = (ptr->measured.f32PhC.filt)>>(ptr->param.u16CalibSamples);
        ptr->measured.f32PhC.filt       -= f16FiltOutput;
        ptr->offset[svmSector].f32PhC   = (tFrac32)(f16FiltOutput<<16);

        if ((--calibCntr)<=0)
        {
            ptr->flag.B.calibDone       = 1;    // end of DC offset calibration
        }

    }
    return (ptr->flag.B.calibDone);
}

/***************************************************************************//*!
@brief      ADC software calibration routine.

@param[in,out]  *ptr    Pointer to structure of ADC module variables and
                        parameters

@return     # true - when ADC calibration ended successfully
            # false - when ADC calibration is ongoing

@details    This function performs offset calibration during the calibration
            phase of the application. It is not intended to be executed when
            application is in run mode.

@note       ADC module as refered to in this function is not actual peripheral
            module but an instance of software defined module of structure
            type holding all necessary ADC parameters and variables.
*******************************************************************************/
bool_t ADCModuleCalib16(adcModule16_t *ptr, tU16 svmSector)
{
    static tU16 calibCntr;
    static tFrac16 f16FiltOutput;
    static tFrac32 f32FiltAcc;

    if (!(ptr->flag.B.calibInitDone))
    {
        // +4 in order to accommodate settling time of the filter
        calibCntr = 1<<(ptr->param.u16CalibSamples+4);

        ptr->measured.f16BemfA.filt16   = 0x0;
        ptr->measured.f16BemfB.filt16   = 0x0;
        ptr->measured.f16BemfC.filt16   = 0x0;
        ptr->measured.f16Udcb.filt16    = 0x0;
        ptr->measured.f16Idcb.filt16    = 0x0;

        ptr->flag.B.calibDone           = 0;
        ptr->flag.B.calibInitDone       = 1;
    }

    if (!(ptr->flag.B.calibDone))
    {
        switch(svmSector)
        {
        case 1:
        case 4:
                {
                    ptr->measured.f16BemfC.raw16    = (tFrac16)(ADC_1.CDR[BEMFC_CURRENT_CHNL].B.CDATA<<3);
                    ptr->measured.f16Udcb.raw16     = (tFrac16)(ADC_0.CDR[UDCB_VOLTAGE_CHNL].B.CDATA<<5);
                    ptr->measured.f16Idcb.raw16     = (tFrac16)(ADC_0.CDR[IDCB_CURRENT_CHNL].B.CDATA<<5);
                    break;
                }
        case 2:
        case 5:
                {
                    ptr->measured.f16BemfB.raw16    = (tFrac16)(ADC_1.CDR[BEMFB_CURRENT_CHNL].B.CDATA<<3);
                    ptr->measured.f16Udcb.raw16     = (tFrac16)(ADC_0.CDR[UDCB_VOLTAGE_CHNL].B.CDATA<<5);
                    ptr->measured.f16Idcb.raw16     = (tFrac16)(ADC_0.CDR[IDCB_CURRENT_CHNL].B.CDATA<<5);
                    break;
                }
        case 3:
        case 6:
        default:
                {
                    ptr->measured.f16BemfA.raw16    = (tFrac16)(ADC_1.CDR[BEMFA_CURRENT_CHNL].B.CDATA<<3);
                    ptr->measured.f16Udcb.raw16     = (tFrac16)(ADC_0.CDR[UDCB_VOLTAGE_CHNL].B.CDATA<<5);
                    ptr->measured.f16Idcb.raw16     = (tFrac16)(ADC_0.CDR[IDCB_CURRENT_CHNL].B.CDATA<<5);
                    break;
                }
        }

        /* ---------------------------------------------------------------------
         * DCbus current - DC offset data filtering using MA recursive filter
         * -------------------------------------------------------------------*/
        f32FiltAcc                          += (tFrac16)(ptr->measured.f16Idcb.raw16);
        f16FiltOutput                       = (f32FiltAcc)>>(ptr->param.u16CalibSamples);
        f32FiltAcc                          -= f16FiltOutput;
        ptr->offset[svmSector].f16Idcb      = (tFrac16)(f16FiltOutput);
        // BEMF offset definition

        if ((--calibCntr)<=0)
        {
            ptr->flag.B.calibDone       = 1;    // end of DC offset calibration
        }

    }
    return (ptr->flag.B.calibDone);
}

/***************************************************************************//*!
@brief      ADC software BLDC motor BEMF voltage measurement routine.

@param[in,out]  *ptr    Pointer to structure of ADC module variables and
                        parameters

@return     # true - when ADC measurement ended successfully
            # false - when ADC measurement is ongoing, or error occurred.

@details    This function performs measurement of Back EMF voltage from motor
            Phases, which are selected based on active SVM vector. DC bus
            current and voltage is measured also.

            This approach can be used for current measurement up to full PWM
            duty cycle.
*******************************************************************************/
bool_t ADCMeasureBEMF(adcModule16_t *ptr, tU16 svmSector,tFrac16 *pIdcb)
{
    ptr->measured.f16Udcb.raw16     = (tFrac16)(ADC_0.CDR[UDCB_VOLTAGE_CHNL].B.CDATA<<5);
    ptr->measured.f16Idcb.raw16     = (tFrac16)(ADC_0.CDR[IDCB_CURRENT_CHNL].B.CDATA<<5);
    *pIdcb                          = F16Sub(ptr->measured.f16Idcb.raw16, ptr->offset[svmSector].f16Idcb);
    switch(svmSector)
    {
        case 1:
        case 4:
                {
                    ptr->measured.f16BemfC.raw16    = (tFrac16)(ADC_1.CDR[BEMFC_CURRENT_CHNL].B.CDATA<<3);

                    return(1);
                    break;
                }
        case 2:
        case 5:
                {
                    ptr->measured.f16BemfB.raw16    = (tFrac16)(ADC_1.CDR[BEMFB_CURRENT_CHNL].B.CDATA<<3);

                    return(1);
                    break;
                }
        case 3:
        case 6:
        default:
                {
                    ptr->measured.f16BemfA.raw16    = (tFrac16)(ADC_1.CDR[BEMFA_CURRENT_CHNL].B.CDATA<<3);

                    return(1);
                    break;
                }
    }
    return(0);
}

/* End of file */