/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MC33905_routines.h
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    Header file for MC33905 SBC driver.
*
********************************************************************************
*
* This file includes MACRO and user type definitions of MC33905 SBC driver
* for MPC56xx devices.
*
*******************************************************************************/
#ifndef __MC33905_ROUTINES_H  /* Prevent duplicated includes */
#define __MC33905_ROUTINES_H

#include "typedefs.h"
#include "MPC5606B_LLD_qs.h"
#include "MPC5606B_DSPI_LLD.h"
#include "MPC5606B_PIT_LLD.h"

/*******************************************************************************
| Defines and macros            (scope: module-local)
------------------------------------------------------------------------------*/
/* SBC REGISTERS ADDRESSES */
/*! Macro containing MUX register address */
#define SBC_MUX_ADDR                    (0x0000 << 9)
/*! Macro containing RAM A register address */
#define SBC_RAM_A_ADDR                  (0x0001 << 9)
/*! Macro containing RAM B register address */
#define SBC_RAM_B_ADDR                  (0x0002 << 9)
/*! Macro containing RAM C register address */
#define SBC_RAM_C_ADDR                  (0x0003 << 9)
/*! Macro containing RAM D register address */
#define SBC_RAM_D_ADDR                  (0x0004 << 9)
/*! Macro containing Voltage regulator register address */
#define SBC_INIT_VREG_ADDR              (0x0005 << 9)
/*! Macro containing Watchdog initialization register address */
#define SBC_INIT_WDG_ADDR               (0x0006 << 9)
/*! Macro containing LIN I/O initialization register address */
#define SBC_INIT_LIN_IO_ADDR            (0x0007 << 9)
/*! Macro containing MISC initialization register address */
#define SBC_MISC_ADDR                   (0x0008 << 9)
/*! Macro containing Special Mode register address */
#define SBC_SPE_ADDR                    (0x0009 << 9)
/*! Macro containing Timer A register address: I_MCU and Watchdog period */
#define SBC_TIM_A_ADDR                  (0x000A << 9)
/*! Macro containing Timer B register address: Cyclic sense and cyclic int periods */
#define SBC_TIM_B_ADDR                  (0x000B << 9)
/*! Macro containing Timer C register address: Watchdog and forced wake-up period */
#define SBC_TIM_C_ADDR                  (0x000C << 9)
/*! Macro containing Watchdog refresh register address */
#define SBC_WDOG_REFR_ADDR              (0x000D << 9)
/*! Macro containing Mode register address */
#define SBC_MODE_ADDR                   (0x000E << 9)
/*! Macro containing Voltage regulators control register address */
#define SBC_REG_ADDR                  (0x000F << 9)
/*! Macro containing CAN interface register address */
#define SBC_CAN_ADDR                    (0x0010 << 9)
/*! Macro containing I/O terminal register address */
#define SBC_IO_ADDR                     (0x0011 << 9)
/*! Macro containing Interrupts register address */
#define SBC_INT_ADDR                    (0x0012 << 9)
/*! Macro containing LIN 1 interface register address */
#define SBC_LIN1_ADDR                   (0x0013 << 9)
/*! Macro containing LIN 2 interface register address */
#define SBC_LIN2_ADDR                   (0x0014 << 9)


/* SBC SPI CONTROL BITS SETUP */
/*! Macro containing command setup for read back of device control bits */
#define SBC_RD_REG                      0x0100
/*! Macro containing command setup for read back of specific device information */
#define SBC_RD_DEV_INF                  0x0180
/*! Macro containing command setup for write control bits to register */
#define SBC_WR_REG                      0x4000
/*! Macro containing command setup for read back of device control bits */
#define SBC_RD_FLG_L                    0xC000
/*! Macro containing command setup for read back of specific device information */
#define SBC_RD_FLG_H                    0xC180

/* MUX REGISTER */
/* Control bits */
#define SBC_MUX2                        0x0080
#define SBC_MUX1                        0x0040
#define SBC_MUX0                        0x0020
#define SBC_INT2K_EN                    0x0010
/* MUX options */
#define SBC_MUX_DISABLE                 0x0000
#define SBC_VDD_OUT                     SBC_MUX0
#define SBC_INT_VREF                    SBC_MUX1
#define SBC_INT_TEMP                    SBC_MUX1|SBC_MUX0
#define SBC_IO_0                        SBC_MUX2
#define SBC_IO_1                        SBC_MUX2|SBC_MUX0
#define SBC_VSUP1                       SBC_MUX2|SBC_MUX1
#define SBC_VSENSE                      SBC_MUX2|SBC_MUX1|SBC_MUX0
/* Pull down options */
#define SBC_INT2K_ENABLE                SBC_INT2K_EN
#define SBC_INT2K_DISABLE               0x0000


/* VOLTAGE REGULATOR INITIALIZATION REGISTER */
/* Control bits */
#define SBC_VDD_LOW_RST1                0x0040
#define SBC_VDD_LOW_RST0                0x0020
#define SBC_VDD_LOW_RST_D1              0x0010
#define SBC_VDD_LOW_RST_D0              0x0008
#define SBC_AUX5                        0x0004
/* Reset and interrupt conditions */
#define SBC_RST_09VDD                   0x0000
#define SBC_INT_RST                     SBC_VDD_LOW_RST0
#define SBC_RST_07VDD                   SBC_VDD_LOW_RST1
#define SBC_SBC_RESET                   SBC_VDD_LOW_RST1|SBC_VDD_LOW_RST0
/* Reset duration after Vdd recovery or threshold cross */
#define SBC_RST_1ms                     0x0000
#define SBC_RST_5ms                     SBC_VDD_LOW_RST_D0
#define SBC_RST_10ms                    SBC_VDD_LOW_RST_D1
#define SBC_RST_20ms                    SBC_VDD_LOW_RST_D1|SBC_VDD_LOW_RST_D0
/* Vaux  output voltage operation */
#define SBC_VAUX_3V                     0x0000
#define SBC_VAUX_5V                     SBC_AUX5


/* WATCHDOG INITIALIZATION REGISTER */
/* Control bits */
#define SBC_WD2INT                      0x0080
#define SBC_MCU_OC                      0x0040
#define SBC_OC_TIM                      0x0020
#define SBC_WD_SAFE                     0x0010
#define SBC_WD_SPI_1                    0x0008
#define SBC_WD_SPI_0                    0x0004
#define SBC_WD_NOR_WDW                  0x0002
/* Watchdog refresh options in normal mode */
#define SBC_8_16_SPI                    0x0000
#define SBC_ENH_SPI1                    SBC_WD_SPI_0
#define SBC_ENH_SPI2                    SBC_WD_SPI_1
#define SBC_ENH_SPI4                    SBC_WD_SPI_1|SBC_WD_SPI_0


/* LIN INITIALIZATION REGISTER */
/* Control bits */
#define SBC_IO1_OFF                     0x0080
#define SBC_LIN_TERM1_1                 0x0040
#define SBC_LIN_TERM1_0                 0x0020
#define SBC_LIN_TERM0_1                 0x0010
#define SBC_LIN_TERM0_0                 0x0008
#define SBC_IO1_OUT_EN                  0x0004
#define SBC_IO0_OUT_EN                  0x0002
#define SBC_CYC_IO0_INV                 0x0001


/* MISC INITIALIZATION REGISTER */
/* Control bits */
#define SBC_CHNG_MODE_RDM               0x0080
#define SBC_SPI_PARITY                  0x0040
#define SBC_INT_PULSE_LO                0x0020
#define SBC_INT_WIDTH                   0x0010
#define SBC_INT_FLASH50                 0x0008
#define SBC_SAFE_RES_2                  0x0004
#define SBC_SAFE_RES_1                  0x0002
#define SBC_SAFE_RES_0                  0x0001


/* SPECIAL MODE REGISTER */
/* Control bits */
#define SBC_SEL_MODE_1                  0x0080
#define SBC_SEL_MODE_0                  0x0040
#define SBC_RANDOM_MASK                 0x003F
/* Change mode selection */
#define SBC_GOTO_RESET                  0x0000
#define SBC_GOTO_INIT                   SBC_SEL_MODE_0
#define SBC_GOTO_FLASH                  SBC_SEL_MODE_1
#define SBC_NO_CHANGE                   SBC_SEL_MODE_0|SBC_SEL_MODE_0


/* TIMER: I_MCU OVERCURRENT TIME AND WATCHDOG PERIOD IN NORMAL MODE */
/* Control fields */
#define SBC_IMCU_STP_TIM                0x0080
#define SBC_IMCU_STP_PRESC0             0x0000
#define SBC_IMCU_STP_PRESC1             0x0020
#define SBC_IMCU_STP_PRESC2             0x0040
#define SBC_IMCU_STP_PRESC3             0x0060
#define SBC_WD_NOR_TIM0                 0x0000
#define SBC_WD_NOR_TIM1                 0x0008
#define SBC_WD_NOR_TIM2                 0x0010
#define SBC_WD_NOR_TIM3                 0x0018
#define SBC_WD_NOR_PRESC0               0x0000
#define SBC_WD_NOR_PRESC1               0x0001
#define SBC_WD_NOR_PRESC2               0x0002
#define SBC_WD_NOR_PRESC3               0x0003
#define SBC_WD_NOR_PRESC4               0x0004
#define SBC_WD_NOR_PRESC5               0x0005
#define SBC_WD_NOR_PRESC6               0x0006
#define SBC_WD_NOR_PRESC7               0x0007
/* Overcurrent times options */
#define SBC_OVCURR_3ms      SBC_IMCU_STP_PRESC0
#define SBC_OVCURR_4ms      SBC_IMCU_STP_TIM | SBC_IMCU_STP_PRESC0
#define SBC_OVCURR_6ms      SBC_IMCU_STP_PRESC1
#define SBC_OVCURR_8ms      SBC_IMCU_STP_TIM | SBC_IMCU_STP_PRESC1
#define SBC_OVCURR_12ms     SBC_IMCU_STP_PRESC2
#define SBC_OVCURR_16ms     SBC_IMCU_STP_TIM | SBC_IMCU_STP_PRESC2
#define SBC_OVCURR_24ms     SBC_IMCU_STP_PRESC3
#define SBC_OVCURR_32ms     SBC_IMCU_STP_TIM | SBC_IMCU_STP_PRESC3
/* Watchdog period in normal mode options */
#define SBC_WDOG_2_5ms      SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC0
#define SBC_WDOG_3ms        SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC0
#define SBC_WDOG_3_5ms      SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC0
#define SBC_WDOG_4ms        SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC0
#define SBC_WDOG_5ms        SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC1
#define SBC_WDOG_6ms        SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC1
#define SBC_WDOG_7ms        SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC1
#define SBC_WDOG_8ms        SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC1
#define SBC_WDOG_10ms       SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC2
#define SBC_WDOG_12ms       SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC2
#define SBC_WDOG_14ms       SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC2
#define SBC_WDOG_16ms       SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC2
#define SBC_WDOG_20ms       SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC3
#define SBC_WDOG_24ms       SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC3
#define SBC_WDOG_28ms       SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC3
#define SBC_WDOG_32ms       SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC3
#define SBC_WDOG_40ms       SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC4
#define SBC_WDOG_48s        SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC4
#define SBC_WDOG_56ms       SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC4
#define SBC_WDOG_64ms       SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC4
#define SBC_WDOG_80ms       SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC5
#define SBC_WDOG_96ms       SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC5
#define SBC_WDOG_112ms      SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC5
#define SBC_WDOG_128ms      SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC5
#define SBC_WDOG_160ms      SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC6
#define SBC_WDOG_192ms      SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC6
#define SBC_WDOG_224ms      SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC6
#define SBC_WDOG_256ms      SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC6
#define SBC_WDOG_320ms      SBC_WD_NOR_TIM0 | SBC_WD_NOR_PRESC7
#define SBC_WDOG_384ms      SBC_WD_NOR_TIM1 | SBC_WD_NOR_PRESC7
#define SBC_WDOG_448ms      SBC_WD_NOR_TIM2 | SBC_WD_NOR_PRESC7
#define SBC_WDOG_512ms      SBC_WD_NOR_TIM3 | SBC_WD_NOR_PRESC7


/* TIMER: CYCLIC SENSE PERIOD AND CYCLIC INTERRUPT PERIODS (LP MODE) */
/* Control bits */
#define SBC_CYC_SENSE_TIM   0x0080
#define SBC_CYC_SENSE_PRE2  0x0040
#define SBC_CYC_SENSE_PRE1  0x0020
#define SBC_CYC_SENSE_PRE0  0x0010
#define SBC_CYC_INT_TIM     0x0008
#define SBC_CYC_INT_PRE2    0x0004
#define SBC_CYC_INT_PRE1    0x0002
#define SBC_CYC_INT_PRE0    0x0001


/* TIMER: WATCHDOG IN LP VDD ON MODE AND FORCED WAKE-UP PERIOD IN LP MODE */
/* Control bits */
#define SBC_WDOG_STP_TIM    0x0080
#define SBC_WDOG_STP_PRE2   0x0040
#define SBC_WDOG_STP_PRE1   0x0020
#define SBC_WDOG_STP_PRE0   0x0010
#define SBC_FWU_TIM         0x0008
#define SBC_FWU_PRE2        0x0004
#define SBC_FWU_PRE1        0x0002
#define SBC_FWU_PRE0        0x0001


/* MODE REGISTER */
/* Control bits */
#define SBC_MODE_4          0x0080
#define SBC_MODE_3          0x0040
#define SBC_MODE_2          0x0020
#define SBC_MODE_1          0x0010
#define SBC_MODE_0          0x0008
/* Sleep mode options */
#define SBC_SLEEP           SBC_MODE_3 | SBC_MODE_2
#define SBC_SLEEP_CYCSEN    SBC_MODE_3 | SBC_MODE_2 | SBC_MODE_0
#define SBC_SLEEP_FWU       SBC_MODE_3 | SBC_MODE_2 | SBC_MODE_1
#define SBC_SLEEP_CYC_FWU   SBC_MODE_3 | SBC_MODE_2 | SBC_MODE_1 | SBC_MODE_0
/* Stop mode options */
#define SBC_STOP            SBC_MODE_4
#define SBC_STOP_FWU        SBC_MODE_3
#define SBC_STOP_CYCSEN     SBC_MODE_2
#define SBC_STOP_CYCINT     SBC_MODE_1
#define SBC_STOP_WDEN       SBC_MODE_0


/* VOLTAGE REGULATORS CONTROL REGISTER */
/* Control bits */
#define SBC_VAUX_1          0x0080
#define SBC_VAUX_0          0x0040
#define SBC_5VCAN_1         0x0010
#define SBC_5VCAN_0         0x0008
#define SBC_VDD_BAL_EN      0x0004
#define SBC_VDD_BAL_AUTO    0x0002
#define SBC_VDD_OFF_EN      0x0001


/* CAN INTERFACE REGISTER */
/* Control bits */
#define SBC_CAN_MODE_1      0x0080
#define SBC_CAN_MODE_0      0x0040
#define SBC_CAN_SLEWRATE_1  0x0020
#define SBC_CAN_SLEWRATE_0  0x0010
#define SBC_CAN_WAKE_PULSE  0x0008
#define SBC_CAN_INT_MODE    0x0001
/* CAN transceiver modes */
#define SBC_CAN_NO_WU_INT           0x0000
#define SBC_CAN_RX_ONLY             SBC_CAN_MODE_0
#define SBC_CAN_WKUP_INT            SBC_CAN_MODE_1
#define SBC_CAN_TX_RX               SBC_CAN_MODE_1 | SBC_CAN_MODE_0
/* CAN transceiver slew rates */
#define SBC_CAN_SLEW_RATE_FAST      0x0000
#define SBC_CAN_SLEW_RATE_MEDIUM    SBC_CAN_SLEWRATE_0
#define SBC_CAN_SLEW_RATE_SLOW      SBC_CAN_SLEWRATE_1
/* CAN bus failure flags */
#define SBC_CAN_UF          0x0080
#define SBC_CAN_F           0x0040
#define SBC_CANL_VSUP       0x0020
#define SBC_CANL_VDD        0x0010
#define SBC_CANL_GND        0x0008
#define SBC_CANH_VSUP       0x0004
#define SBC_CANH_VDD        0x0002
#define SBC_CANH_GND        0x0001


/* I/O REGISTER */
/* Control bits */
#define SBC_IO3_1           0x0080
#define SBC_IO3_0           0x0040
#define SBC_IO2_1           0x0020
#define SBC_IO2_0           0x0010
#define SBC_IO1_1           0x0008
#define SBC_IO1_0           0x0004
#define SBC_IO0_1           0x0002
#define SBC_IO0_0           0x0001


/* INTERRUPTS REGISTER */
/* Control bits */
#define SBC_CAN_FAIL_INT    0x0080
#define SBC_MCU_REQ_INT     0x0040
#define SBC_LIN1_FAIL_INT   0x0020
#define SBC_LIN2_FAIL_INT   0x0010
#define SBC_IOX_INT         0x0008
#define SBC_SPI_F_INT       0x0002
#define SBC_VMON_F_INT      0x0001


/* LIN 1/2 REGISTERS */
#define SBC_LIN_MODE_1      0x0080
#define SBC_LIN_MODE_0      0x0040
#define SBC_LIN_SLEWRATE_1  0x0020
#define SBC_LIN_SLEWRATE_0  0x0010
#define SBC_LIN_TERM_ON     0x0004
/* LIN transceiver modes */
#define SBC_LIN_NO_WU_INT           0x0000
#define SBC_LIN_SLPWU_NORINT        SBC_LIN_MODE_1
#define SBC_LIN_TX_RX               SBC_LIN_MODE_1 | SBC_LIN_MODE_0
/* LIN transceiver slew rates */
#define SBC_LIN_SLEW_RATE_20KBPS    0x0000
#define SBC_LIN_SLEW_RATE_10KBPS    SBC_LIN_SLEWRATE_0
#define SBC_LIN_SLEW_RATE_FAST      SBC_LIN_SLEWRATE_1

#define DELAY_10MS          10000000

#define SBC_CS              0
#define SBC_CSPCR           4

/* INITIAL CONFIGURATION VALUES */
/*! LIN1 and LIN2 configured as LIN master term, IO output drivers disabled */
#define SBC_LIN_IO_INITVAL              SBC_LIN_TERM1_0 | SBC_LIN_TERM0_0

/*! Watchdog is accessed in simple 8/16 bit SPI, window disabled */
#define SBC_WDT_INITVAL                 SBC_8_16_SPI

/*! VREG: 1ms reset pulse, SBC reset at 0.7Vdd, Vaux = 5V */
#define SBC_VREG_INITVAL                SBC_RST_1ms | SBC_RST_07VDD | SBC_VAUX_5V

/*! MISC: No RANDOM bits for mode change, no SPI parity, 100 us for INT pulse */
#define SBC_MISC_INITVAL                0x00

/*! Watchdog default timeout */
#define SBC_WDT_TIMEOUT                 SBC_WDOG_320ms

/*! All interrupt sources disabled */
#define SBC_INT_INITVAL                 0x00

/*! Enable Vaux and 5V CAN, Vdd PNP on, enable entry to low power modes */
#define SBC_REG_INITVAL                 SBC_VAUX_0 | SBC_5VCAN_0 | SBC_VDD_BAL_EN |  \
                                        SBC_VDD_BAL_AUTO | SBC_VDD_OFF_EN

/*! CAN transceiver enabled, fast slew rate */
#define SBC_CAN_INITVAL                 SBC_CAN_TX_RX | SBC_CAN_SLEW_RATE_FAST

/*! LIN transceivers enabled, 20 kbps */
#define SBC_LIN1_INITVAL                SBC_LIN_MODE_1 | SBC_LIN_MODE_0 |  \
                                        SBC_LIN_TERM_ON
#define SBC_LIN2_INITVAL                SBC_LIN_MODE_1 | SBC_LIN_MODE_0 |  \
                                        SBC_LIN_TERM_ON

/*! I/O0, I/O1 in high impedance state, WAKE/INT disabled */
#define SBC_IO_INITVAL                  0x00

/*! MUX output disabled */
#define SBC_MUX_INITVAL                 0x00

/* SBC SPI communication commands */
/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_MUX_REG_VALUE           (SBC_MUX_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_MUX_REG_VALUE           (SBC_MUX_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_RAMA_REG_VALUE          (SBC_RAM_A_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_RAMA_REG_VALUE          (SBC_RAM_A_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_RAMB_REG_VALUE          (SBC_RAM_B_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_RAMB_REG_VALUE          (SBC_RAM_B_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_RAMC_REG_VALUE          (SBC_RAM_C_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_RAMC_REG_VALUE          (SBC_RAM_C_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_RAMD_REG_VALUE          (SBC_RAM_D_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_RAMD_REG_VALUE          (SBC_RAM_D_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_INITREG_REG_VALUE       (SBC_INIT_VREG_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_INITREG_REG_VALUE       (SBC_INIT_VREG_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_INITWDG_REG_VALUE       (SBC_INIT_WDG_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_INITWDG_REG_VALUE       (SBC_INIT_WDG_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_INIT_LIN_IO_REG_VALUE   (SBC_INIT_LIN_IO_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_INIT_LIN_IO_REG_VALUE   (SBC_INIT_LIN_IO_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_INITMISC_REG_VALUE      (SBC_MISC_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_INITMISC_REG_VALUE      (SBC_MISC_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_SPE_REG_VALUE           (SBC_SPE_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_SPE_REG_VALUE           (SBC_SPE_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_TIMA_REG_VALUE          (SBC_TIM_A_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_TIMA_REG_VALUE          (SBC_TIM_A_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_TIMB_REG_VALUE          (SBC_TIM_B_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_TIMB_REG_VALUE          (SBC_TIM_B_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_TIMC_REG_VALUE          (SBC_TIM_C_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_TIMC_REG_VALUE          (SBC_TIM_C_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_WDG_REG_VALUE           (SBC_WDOG_REFR_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_WDG_REG_VALUE           (SBC_WDOG_REFR_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_MODE_REG_VALUE          (SBC_MODE_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_MODE_REG_VALUE          (SBC_MODE_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_REG_REG_VALUE           (SBC_REG_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_REG_REG_VALUE           (SBC_REG_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_CAN_REG_VALUE           (SBC_CAN_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_CAN_REG_VALUE           (SBC_CAN_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_IO_REG_VALUE            (SBC_IO_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_IO_REG_VALUE            (SBC_IO_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_INT_REG_VALUE           (SBC_INT_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_INT_REG_VALUE           (SBC_INT_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_LIN1_REG_VALUE          (SBC_LIN1_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_LIN1_REG_VALUE          (SBC_LIN1_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_LIN2_REG_VALUE          (SBC_LIN2_ADDR | SBC_RD_REG)

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_SET_LIN2_REG_VALUE          (SBC_LIN2_ADDR | SBC_WR_REG)

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_REG_L               0xDF00

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_REG_H               0xDF80

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_CAN_L               0xE100

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_CAN_H               0xE180

/*! Macro containing command value for reading current CAN status */
#define SBC_GET_CAN_STATUS              0x2180

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_IO_L                0xE300

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_IO_H                0xE380

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_IO_REAL             0x2380

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_INT_L               0xE500

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_INT_H               0xE580

/*! Macro containing command value for reading flags from register */
#define SBC_GET_DEVICE_ID               0x2510

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_LIN1_L              0xE700

/*! Macro containing command value for reading flags from register */
#define SBC_GET_LIN1_STATUS             0x2780

/*! Macro containing command value for reading flags from register */
#define SBC_RD_FLAG_LIN2_L              0xE900

/*! Macro containing command value for reading flags from register */
#define SBC_GET_LIN2_STATUS             0x2980

/*! Macro containing command value for reading flags from register */
#define SBC_SET_REGUL_REG_VALUE         0x5E00

/*! Macro containing command value for reading flags from register */
//#define SBC_SET_IO_REG_VALUE            0x6200

/*! Macro containing command value for reading REGULATOR register value */
#define SBC_GET_MODE_REG_H_FLAGS        (SBC_MODE_ADDR | SBC_RD_FLG_H)

/*******************************************************************************
| Typedefs and structures       (scope: module-local)
------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*//*!
@struct REG_FLAG "\MC33905_routines.h"

@brief  Structure containing REG Flag Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :3;     /*!<  */
        uint16_t    vddThShudown    :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    rstLow          :1;     /*!<  */
        uint16_t    vsupBatFail     :1;     /*!<  */
        uint16_t    iddOcLp         :1;     /*!<  */
        uint16_t    vauxLow         :1;     /*!<  */
        uint16_t    vauxOc          :1;     /*!<  */
        uint16_t    can5vThStdn     :1;     /*!<  */
        uint16_t    can5vOc         :1;     /*!<  */
        uint16_t    vsenseLow       :1;     /*!<  */
        uint16_t    vsupUv          :1;     /*!<  */
        uint16_t    iddOcNorm       :1;     /*!<  */
    } B;
}REG_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct CAN_FLAG_T "\MC33905_routines.h"

@brief  Structure containing CAN Flag Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    canUf           :1;     /*!<  */
        uint16_t    canF            :1;     /*!<  */
        uint16_t    canlToVbat      :1;     /*!<  */
        uint16_t    canlToVdd       :1;     /*!<  */
        uint16_t    canlToGng       :1;     /*!<  */
        uint16_t    canhToVdd       :1;     /*!<  */
        uint16_t    canhToGng       :1;     /*!<  */
        uint16_t    canWakeUp       :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    canOt           :1;     /*!<  */
        uint16_t    rxdLow          :1;     /*!<  */
        uint16_t    rxdHig          :1;     /*!<  */
        uint16_t    tdxDom          :1;     /*!<  */
        uint16_t    busDomClamp     :1;     /*!<  */
        uint16_t    canOc           :1;     /*!<  */
    } B;
}CAN_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct CAN_STAT_T "\MC33905_routines.h"

@brief  Structure containing CAN status Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :8;     /*!<  */
        uint16_t    canDrv          :1;     /*!<  */
        uint16_t    canRcv          :1;     /*!<  */
        uint16_t    canWuEnDis      :1;     /*!<  */
        uint16_t                    :5;     /*!<  */
    } B;
}CAN_STAT_T;

/*-------------------------------------------------------------------------*//*!
@struct IO_FLAG_T "\MC33905_routines.h"

@brief  Structure containing I/O Flag Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    io13WakeUp      :1;     /*!<  */
        uint16_t    io02WakeUp      :1;     /*!<  */
        uint16_t    spiWakeUp       :1;     /*!<  */
        uint16_t    fwu             :1;     /*!<  */
        uint16_t    intSrvTimeout   :1;     /*!<  */
        uint16_t    lpVddOff        :1;     /*!<  */
        uint16_t    rstReq          :1;     /*!<  */
        uint16_t    hwLeaveDbg      :1;     /*!<  */
        uint16_t    hs3ShrtToGnd    :1;     /*!<  */
        uint16_t    hs2ShrtToGnd    :1;     /*!<  */
        uint16_t    spiParErr       :1;     /*!<  */
        uint16_t    csbLow          :1;     /*!<  */
        uint16_t    vsupUv          :1;     /*!<  */
        uint16_t    vsupOv          :1;     /*!<  */
        uint16_t    io0Th           :1;     /*!<  */
        uint16_t    wdgFlashMode    :1;     /*!<  */
    } B;
}IO_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct IO_STAT_T "\MC33905_routines.h"

@brief  Structure containing I/O status Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :8;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    io3Stat         :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    io2Stat         :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    io1Stat         :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
        uint16_t    io0Stat         :1;     /*!<  */
    } B;
}IO_STAT_T;

/*-------------------------------------------------------------------------*//*!
@struct INT_FLAG_T "\MC33905_routines.h"

@brief  Structure containing INT Flag Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :3;     /*!<  */
        uint16_t    vddLow          :1;     /*!<  */
        uint16_t    vddLowRst       :1;     /*!<  */
        uint16_t    rstLow          :1;     /*!<  */
        uint16_t    multRst         :1;     /*!<  */
        uint16_t    wdgRefFail      :1;     /*!<  */
        uint16_t    intRqst         :1;     /*!<  */
        uint16_t    rstHigh         :1;     /*!<  */
        uint16_t    dbgRes          :1;     /*!<  */
        uint16_t    vddTempPrewarn  :1;     /*!<  */
        uint16_t    vddUv           :1;     /*!<  */
        uint16_t    vddOv           :1;     /*!<  */
        uint16_t    vauxOv          :1;     /*!<  */
        uint16_t                    :1;     /*!<  */
    } B;
}INT_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct INT_STAT_T "\MC33905_routines.h"

@brief  Structure containing INT status Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :8;     /*!<  */
        uint16_t    vdd5v33v        :1;     /*!<  */
        uint16_t    devPn1          :1;     /*!<  */
        uint16_t    devPn0          :1;     /*!<  */
        uint16_t    id4             :1;     /*!<  */
        uint16_t    id3             :1;     /*!<  */
        uint16_t    id2             :1;     /*!<  */
        uint16_t    id1             :1;     /*!<  */
        uint16_t    id0             :1;     /*!<  */
    } B;
}INT_STAT_T;

/*-------------------------------------------------------------------------*//*!
@struct INT_FLAG_T "\MC33905_routines.h"

@brief  Structure containing INT Flag Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                    :9;     /*!<  */
        uint16_t    linWakeUp       :1;     /*!<  */
        uint16_t    linTermShrtGnd  :1;     /*!<  */
        uint16_t    linOverTemp     :1;     /*!<  */
        uint16_t    rxdLow          :1;     /*!<  */
        uint16_t    rxdHigh         :1;     /*!<  */
        uint16_t    txdDom          :1;     /*!<  */
        uint16_t    linBusDomClmp   :1;     /*!<  */
    } B;
}LIN_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct INT_STAT_T "\MC33905_routines.h"

@brief  Structure containing INT status Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t                :8;     /*!<  */
        uint16_t    linStat     :1;     /*!<  */
        uint16_t    linWuEnDis  :1;     /*!<  */
        uint16_t                :6;     /*!<  */
    } B;
}LIN_STAT_T;

/*-------------------------------------------------------------------------*//*!
@struct MC33905_FLAG_T "\MC33905_routines.h"

@brief  Structure containing flag registers image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef struct
{
    REG_FLAG_T  regFlg;     /*!< Vreg current flags */
    CAN_FLAG_T  canFlg;     /*!< CAN interface current flags */
    CAN_STAT_T  canStat;    /*!< CAN interface real time status */
    IO_FLAG_T   ioFlg;      /*!< I/O flags and I/O wake-up */
    IO_STAT_T   ioStat;     /*!< I/O real time level */
    INT_FLAG_T  intFlg;     /*!< INT current flags */
    INT_STAT_T  intStat;    /*!< Device Identification */
    LIN_STAT_T  lin1Flg;    /*!< LIN1 interface current flags */
    LIN_STAT_T  lin1Stat;   /*!< LIN1 interface current status flags */
    LIN_STAT_T  lin2Flg;    /*!< LIN2 interface current flags */
    LIN_STAT_T  lin2Stat;   /*!< LIN2 interface current status flags */
}MC33905_FLAG_T;

/*-------------------------------------------------------------------------*//*!
@struct MUX_T "\MC33905_routines.h"

@brief  Structure containing MUX register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    muxSel      :3;    /*!< MUX_OUT pin signal selection */
        uint16_t    int2k       :1;    /*!< Select internal 2.0kOhm resister between AMUX and GND */
        uint16_t    IOAtt       :1;    /*!< Select attenuation or gain */
        uint16_t                :3;
    } B;
}MUX_T;

/*-------------------------------------------------------------------------*//*!
@struct RAM_T "\MC33905_routines.h"

@brief  Structure containing RAM memory register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    bit7        :1;    /*!< RAMx bit7 */
        uint16_t    bit6        :1;    /*!< RAMx bit6 */
        uint16_t    bit5        :1;    /*!< RAMx bit5 */
        uint16_t    bit4        :1;    /*!< RAMx bit4 */
        uint16_t    bit3        :1;    /*!< RAMx bit3 */
        uint16_t    bit2        :1;    /*!< RAMx bit2 */
        uint16_t    bit1        :1;    /*!< RAMx bit1 */
        uint16_t    bit0        :1;    /*!< RAMx bit0  */
    } B;
}RAM_T;

/*-------------------------------------------------------------------------*//*!
@struct INIT_REG_T "\MC33905_routines.h"

@brief  Structure containing INIT register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    ioxSync     :1;    /*!< determine sensing of I/O-1 */
        uint16_t    vddRst      :2;    /*!< Vdd undervoltage threshold */
        uint16_t    vddRstd     :2;    /*!< Select the RESET pin low level duration */
        uint16_t    vauxSel     :1;    /*!< Vauxiliary output voltage */
        uint16_t    cyclicOn    :2;    /*!< cyclic on function */
    } B;
}INIT_REG_T;

/*-------------------------------------------------------------------------*//*!
@struct INIT_WD_T "\MC33905_routines.h"

@brief  Structure containing Init W/D register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    wd2int      :1;    /*!< Maximum time delay */
        uint16_t    mcuOc       :2;    /*!< Select watchdog refresh in LP mode */
        uint16_t    wdSafe      :1;    /*!< select activation of the SAFE pin */
        uint16_t    wdSpi       :2;    /*!< Watchdog refresh operation */
        uint16_t    wdNwin      :1;    /*!< Watchdog window or timeout operation */
        uint16_t    crank       :1;    /*!< Vsup1 threshold to disable Vdd */
    } B;
}INIT_WD_T;

/*-------------------------------------------------------------------------*//*!
@struct INIT_LIN_IO_T "\MC33905_routines.h"

@brief  Structure containing Init LIN and I/O register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    io1Ovoff    :1;    /*!< Select deactivation of I/O-1 in case Vdd or Vaux over-voltage */
        uint16_t    linT1       :2;    /*!< Select pin operation */
        uint16_t    linT0       :2;    /*!< Select pin operation */
        uint16_t    io1OutEn    :1;    /*!< Select I/O-1 operation */
        uint16_t    io0OutEn    :1;    /*!< Select I/O-0 operation */
        uint16_t    cycInv      :1;    /*!< Select I/O-0 operation in LP mode */
    } B;
}INIT_LIN_IO_T;

/*-------------------------------------------------------------------------*//*!
@struct INIT_MISC "\MC33905_routines.h"

@brief  Structure containing Init Miscellaneous Function register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    lpmwRndm    :1;    /*!< enables the usage of random bits of the MODE reg. */
        uint16_t    spiParity   :1;    /*!< Select parity bit usage */
        uint16_t    intPulse    :1;    /*!< Select INT pin operation */
        uint16_t    intWidth    :1;    /*!< Select INT pin pulse duration */
        uint16_t    intFlash    :1;    /*!< Select INT pulse generation */
        uint16_t    DbgRes      :3;    /*!< Select I/O-0 operation in LP mode */
    } B;
}INIT_MISC_T;

/*-------------------------------------------------------------------------*//*!
@struct SPE_MODE "\MC33905_routines.h"

@brief  Structure containing Specific Mode Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    lpmwRndm    :1;    /*!< enables the usage of random bits of the MODE reg. */
        uint16_t    spiParity   :1;    /*!< Select parity bit usage */
        uint16_t    intPulse    :1;    /*!< Select INT pin operation */
        uint16_t    intWidth    :1;    /*!< Select INT pin pulse duration */
        uint16_t    intFlash    :1;    /*!< Select INT pulse generation */
        uint16_t    DbgRes      :3;    /*!< Select I/O-0 operation in LP mode */
    } B;
}SPE_MODE_T;

/*-------------------------------------------------------------------------*//*!
@struct TIM_A "\MC33905_routines.h"

@brief  Structure containing Timer A Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    iMcu        :3;    /*!< Select LP Vdd Over-current blank time [ms] */
        uint16_t    wdN         :5;    /*!< Select watchdog period in device normal mode [ms] */
    } B;
}TIM_A_T;

/*-------------------------------------------------------------------------*//*!
@struct TIM_B "\MC33905_routines.h"

@brief  Structure containing Timer B Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;
        uint16_t    cycSen      :4;    /*!< Select INT pulse generation */
        uint16_t    cycInt      :4;    /*!< Select I/O-0 operation in LP mode */
    } B;
}TIM_B_T;

/*-------------------------------------------------------------------------*//*!
@struct TIM_C "\MC33905_routines.h"

@brief  Structure containing Timer C Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    wdLp        :4;    /*!< Select watchdog in flash mode */
        uint16_t    fwu         :4;    /*!< Select forced wake-up [ms] */
    } B;
}TIM_C_T;

/*-------------------------------------------------------------------------*//*!
@struct WDG_REF "\MC33905_routines.h"

@brief  Structure containing Watchdog refresh Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t                :8;
    } B;
}WDG_REF_T;

/*-------------------------------------------------------------------------*//*!
@struct MODE "\MC33905_routines.h"

@brief  Structure containing MODE Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    mode        :5;    /*!<  */
        uint16_t    rndB        :3;    /*!<  */
    } B;
}MODE_T;

/*-------------------------------------------------------------------------*//*!
@struct REGUL "\MC33905_routines.h"

@brief  Structure containing Regulator Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    vaux        :2;    /*!< Vauxiliary regulator control */
        uint16_t                :1;    /*!<  */
        uint16_t    vccCan      :2;    /*!< 5V-CAN regulator control */
        uint16_t    vddBalEn    :1;    /*!< enable vdd external ballast transistor */
        uint16_t    vddBalAuto  :1;    /*!< Automatically enbl vdd ext.ballast transistor */
        uint16_t    vddOff      :1;    /*!< allow transition into LP */
    } B;
}REGUL_T;

/*-------------------------------------------------------------------------*//*!
@struct CAN "\MC33905_routines.h"

@brief  Structure containing CAN Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    canMod      :2;    /*!< CAN interface mode control */
        uint16_t    slew        :2;    /*!< CAN driver slew rate selection */
        uint16_t    wakeUp      :1;    /*!< CAN wake-up control */
        uint16_t                :2;    /*!<  */
        uint16_t    canInt      :1;    /*!< CAN failure detection reporting */
    } B;
}CAN_T;

/*-------------------------------------------------------------------------*//*!
@struct I/O "\MC33905_routines.h"

@brief  Structure containing I/O Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    io3         :2;    /*!< I/O-3 pin operation */
        uint16_t    io2         :2;    /*!< I/O-2 pin operation */
        uint16_t    io1         :2;    /*!< I/O-1 pin operation */
        uint16_t    io0         :2;    /*!< I/O-0 pin operation */
    } B;
}IO_T;

/*-------------------------------------------------------------------------*//*!
@struct INT "\MC33905_routines.h"

@brief  Structure containing INT Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    canFail     :1;    /*!< enable INT when failure on CAN interface */
        uint16_t    mcuReq      :1;    /*!< enable INT when requested */
        uint16_t    lin2Fail    :1;    /*!< enable INT when failure on LIN2 interface */
        uint16_t    lin1Fail    :1;    /*!< enable INT when failure on LIN1 interface */
        uint16_t    io          :1;    /*!< control I/O interruption */
        uint16_t    safe        :1;    /*!<  */
        uint16_t                :1;    /*!<  */
        uint16_t    vmon        :1;    /*!< enable interrupt by voltage monitoring */
    } B;
}INT_T;

/*-------------------------------------------------------------------------*//*!
@struct LIN "\MC33905_routines.h"

@brief  Structure containing LIN Register image of the MC33905 device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint16_t R;
    struct {
        uint16_t    cmd         :8;    /*!< The bit field specifies control bits and type of command */
        uint16_t    linMode     :2;    /*!< LIN interface mode control */
        uint16_t    slewRate    :2;    /*!< Slew rate selection */
        uint16_t                :1;    /*!<  */
        uint16_t    lintOn      :1;    /*!< LIN Tx on */
        uint16_t                :1;    /*!<  */
        uint16_t    vsupExt     :1;    /*!< Vsup ext */
    } B;
}LIN_T;

/*-------------------------------------------------------------------------*//*!
@struct MC33905_REG_T "\MC33937_routines.h"

@brief  Structure containing status registers image of the MC33937 device
*//*--------------------------------------------------------------------------*/
typedef struct
{
    DSPI_SET_T      dspiSet;    /*!< DSPI settings */
    MC33905_FLAG_T  flags;      /*!< MC33905 flags */
    MUX_T           mux;        /*!< Analog multiplexer register */
    RAM_T           ramA;       /*!< Memory byte A */
    RAM_T           ramB;       /*!< Memory byte B */
    RAM_T           ramC;       /*!< Memory byte C */
    RAM_T           ramD;       /*!< Memory byte D */
    INIT_REG_T      initReg;    /*!< Initialization regulators */
    INIT_WD_T       initWdg;    /*!< Initialization Watchdog */
    INIT_LIN_IO_T   initLinIo;  /*!< Initialization LIN and I/O */
    INIT_MISC_T     initMisc;   /*!< Initialization Misc. function */
    SPE_MODE_T      speMode;    /*!< specific modes */
    TIM_A_T         timA;       /*!< Timer A watchdog & LP MCU consumption */
    TIM_B_T         timB;       /*!< Timer B cyclic sense & cyclic interrupt */
    TIM_C_T         timC;       /*!< Timer C watchdog LP & forced wake-up */
    WDG_REF_T       watchdog;   /*!< Watchdog refresh */
    MODE_T          mode;       /*!< Mode register */
    REGUL_T         reg;        /*!< regulator control */
    CAN_T           can;        /*!< CAN interface control */
    IO_T            io;         /*!< Input Output control */
    INT_T           interrupt;  /*!< Interrupt control */
    LIN_T           lin1;       /*!< LIN1 interface control */
    LIN_T           lin2;       /*!< LIN2 interface control */
}MC33905_REG_T;

#define MPC560XB_LQFP144_CB_SBC_DEFAULT  \
            {SBC_CS,SBC_CSPCR, (DSPI_ASSERT_CS0 | DSPI_USE_CTAR_1), DSPI1_BASE, \
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, \
             (SBC_SET_MUX_REG_VALUE | SBC_MUX_INITVAL), \
             0, \
             0, \
             0, \
             0, \
             (SBC_SET_INITREG_REG_VALUE | SBC_VREG_INITVAL), \
             (SBC_SET_INITWDG_REG_VALUE | SBC_WDT_INITVAL), \
             (SBC_SET_INIT_LIN_IO_REG_VALUE | SBC_LIN_IO_INITVAL), \
             (SBC_SET_INITMISC_REG_VALUE | SBC_MISC_INITVAL), \
             (SBC_SET_SPE_REG_VALUE | SBC_GOTO_FLASH), \
             (SBC_SET_TIMA_REG_VALUE | SBC_WDT_TIMEOUT), \
             (SBC_SET_TIMB_REG_VALUE), \
             (SBC_SET_TIMC_REG_VALUE), \
             (SBC_SET_WDG_REG_VALUE), \
             (SBC_SET_MODE_REG_VALUE | 0x80), \
             (SBC_SET_REG_REG_VALUE | SBC_REG_INITVAL), \
             (SBC_SET_CAN_REG_VALUE | SBC_CAN_INITVAL), \
             (SBC_SET_IO_REG_VALUE | SBC_IO_INITVAL), \
             (SBC_SET_INT_REG_VALUE | SBC_INT_INITVAL), \
             (SBC_SET_LIN1_REG_VALUE | SBC_LIN1_INITVAL), \
             (SBC_SET_LIN2_REG_VALUE | SBC_LIN2_INITVAL) \
            }

/*******************************************************************************
| Exported function prototypes
------------------------------------------------------------------------------*/
extern  bool_t  mc33905ReadFlag(MC33905_REG_T *ptr);
extern  bool_t  mc33905ReadRtDevId(MC33905_REG_T *ptr);
extern  void    mc33905ClearWdt(DSPI_SET_T *ptr);
extern  void    mc33905SetWdtPeriod(MC33905_REG_T *ptr);
extern  bool_t  mc33905Config(MC33905_REG_T *ptr);
extern  void    mc33905CanCfg(MC33905_REG_T *ptr);

#endif /* __MC33905_ROUTINES_H */