/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MPC5606B_BLDC_SENSORLESS.c
*
* @author   b06050
*
* @version  1.0.4.0
*
* @date     Aug-3-2012
*
* @brief    Sensorless BLDC motor control algorithm for MPC5606B.
*
*
*******************************************************************************/
#include "MPC5606B.h"
#include "sys.h"
#include "siu.h"
#include "adc.h"
#include "ctu.h"
#include "emios.h"
#include "pit.h"
#include "MC33905_routines.h"
#include "MC33937_routines.h"
#include "state_machine.h"
#include "gflib.h"
#include "gdflib.h"
#include "MPC5606B_ADC_BSP.h"
#include "MPC5606B_CB_BSP.h"
#include "BLDC_appconfig.h"
#include "MPC5606B_LLD_qs.h"
#include "MPC5606B_SIUL_LLD.h"
#include "MPC5606B_PIT_LLD.h"

/*******************************************************************************
* Defines and macros
*******************************************************************************/
#define DISABLE_IRQ()    asm("wrteei 0")
#define ENABLE_IRQ()     asm("wrteei 1")

#define TIMER_ON    pEMIOS1->CH[EMIOS1_UC_OC].CCR.B.MODE = EMIOS_MODE_SAOC;

#define TIMER_OFF   pEMIOS1->CH[EMIOS1_UC_OC].CCR.B.MODE = EMIOS_MODE_GPO;

/********************  LED Indication defines - Begin *************************/
/* Indication ISR timer period */
#define PER_TMR_LED_INDIC_US        100.0

/* LED toggle period Indication Stop 4Hz */
#define PER_LED_INDIC_STOP          (1.0/2.0)

/* SW counter reload value to get 4Hz LED Indication Stop */
#define SWTMR_LED_INDIC_STOP        (tU16)(PER_LED_INDIC_STOP/ \
                                         (PER_TMR_LED_INDIC_US/1000000) )
/* LED toggle period Indication Fault 8Hz */
#define PER_LED_INDIC_FAULT         (1.0/16.0)

/* SW counter reload value to get 16Hz LED Indication Fault */
#define SWTMR_LED_INDIC_FAULT       (tU16)(PER_LED_INDIC_FAULT/\
                                         (PER_TMR_LED_INDIC_US/(1000000*2)) )
/********************  LED Indication defines - End ***************************/


/*******************************************************************************
* Local function prototypes
*******************************************************************************/
static void stateReset(void);
static void stateFault(void);
static void stateInit(void);
static void stateReady(void);
static void stateCalib(void);
static void stateAlign(void);
static void stateRun(void);
static void stateStart(void);

static void faultDetection(void);
static void CheckSwitchState(void);
static void LED_indication(void);

static void PWM_Alignment(void);
static void PWM_Vector_0(void);
static void PWM_Vector_1(void);
static void PWM_Vector_2(void);
static void PWM_Vector_3(void);
static void PWM_Vector_4(void);
static void PWM_Vector_5(void);
static tPointerFcn PWMCommutationFcn[] = \
{
    PWM_Vector_0,
    PWM_Vector_0,
    PWM_Vector_1,
    PWM_Vector_2,
    PWM_Vector_3,
    PWM_Vector_4,
    PWM_Vector_5,
    PWM_Vector_0
};
static void ZCdetectPhAfalling(void);
static void ZCdetectPhAraising(void);
static void ZCdetectPhBfalling(void);
static void ZCdetectPhBraising(void);
static void ZCdetectPhCfalling(void);
static void ZCdetectPhCraising(void);

/* State machine function pointers */
static tPointerFcn *ZCdetection;
static tPointerFcn ZCdetectionCD[] = \
{
    ZCdetectPhCfalling,
    ZCdetectPhCfalling,
    ZCdetectPhBraising,
    ZCdetectPhAfalling,
    ZCdetectPhCraising,
    ZCdetectPhBfalling,
    ZCdetectPhAraising
};

static tPointerFcn ZCdetectionCCD[] = \
{
    ZCdetectPhBfalling,
    ZCdetectPhBfalling,
    ZCdetectPhCraising,
    ZCdetectPhAfalling,
    ZCdetectPhBraising,
    ZCdetectPhCfalling,
    ZCdetectPhAraising
};

// Interrupt functions
extern void BLDC_Fast_ISR(void);
extern void EMIOS1_CH24_ISR(void);

extern void main(void);

/*******************************************************************************
* Configuration structures variables
*******************************************************************************/
MC33905_REG_T mc33905_config = MPC560XB_LQFP144_CB_SBC_DEFAULT;
MC33937_SET_T mc33937_config = MPC560XB_LQFP144_CB_DEFAULT;

/*******************************************************************************
* Local variables
*******************************************************************************/
static uint16_t  actualPeriodZC;
static vuint16_t periodZC_F_PhA, periodZC_R_PhA;
static vuint16_t periodZC_F_PhB, periodZC_R_PhB;
static vuint16_t periodZC_F_PhC, periodZC_R_PhC;
static vuint16_t periodFalling, periodRaising, periodFiltered;
static uint16_t  periodNextCommutation;
static vuint16_t timeCommutation, nextTimeCommutation;

static uint16_t timeZC, timeZCstep, lastTimeZC;
static uint16_t timeZC_PhA, timeZC_PhB, timeZC_PhC;
static uint16_t timeZC_Phx_help;

static uint16_t alignmentTimer;
static uint16_t startCMTCounter;

static uint8_t cmtDelay;
static bool_t  findZc;

// Variable for slow control counter
static uint16_t SlowContLoopCNT;

static uint32_t wRotElRequired, wRotEl;
static tFrac32  speed_err, torque_err;
static tFrac32  desiredSpeedRamp;
static tFrac16  advance_angle;

static uint8_t direction;

// Voltage variables
static tFrac16 phA_voltage, phB_voltage, phC_voltage;
static tFrac16 u_dc_bus_ZC;

// Current and torque variables
static tFrac16 i_dc_bus = 0;

static tFrac16 torque_F_PhA,torque_F_PhB,torque_F_PhC;
static tFrac16 torque_R_PhA,torque_R_PhB,torque_R_PhC;
static tFrac32 torqueFiltered, torqueRequired, Ureq;


// PI regulator parameters
static GFLIB_CONTROLLER_PIAW_P_T speed_controllerParams = GFLIB_CONTROLLER_PIAW_P_DEFAULT;

static GFLIB_CONTROLLER_PIAW_P_T i_controllerParams1 = GFLIB_CONTROLLER_PIAW_P_DEFAULT;
static GFLIB_CONTROLLER_PIAW_P_T i_controllerParams2 = GFLIB_CONTROLLER_PIAW_P_DEFAULT;

// Ramp generator parameters
static GFLIB_RAMP_T speed_rampParams = GFLIB_RAMP_DEFAULT;

// Filters
static GDFLIB_FILTER_MA_T Udcb_filt = GDFLIB_FILTER_MA_DEFAULT;

// Variables for check switch states
static uint16_t switchCounter, switchCounterUp, switchCounterDown;
static tU16     dc_help;
static tFrac32  torque_help;
static uint16_t duty_cycle;
static tFrac32  duty_cycle_speed;

static AppProcessStatus processStatus;

static PFCN_VOID_VOID state_table[17][8]={
    /* Actual state ->       'Reset'       'Init'           'Fault'         'Ready'         'Calib'         'Align'         'start'           'Run'      */
    /* e_reset          */ { stateReset,    stateFault,     stateFault,     stateReset,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_reset_done     */ { stateInit,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_fault          */ { stateFault,    stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_fault_clear    */ { stateFault,    stateFault,     stateInit,      stateFault,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_init_done      */ { stateFault,    stateReady,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_ready          */ { stateFault,    stateFault,     stateFault,     stateReady,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_app_on         */ { stateFault,    stateFault,     stateFault,     stateCalib,     stateFault,     stateFault,     stateFault,     stateFault},
    /* e_calib          */ { stateFault,    stateFault,     stateFault,     stateFault,     stateCalib,     stateFault,     stateFault,     stateFault},
    /* e_calib_done     */ { stateFault,    stateFault,     stateFault,     stateFault,     stateAlign,     stateFault,     stateFault,     stateFault},
    /* e_align          */ { stateFault,    stateFault,     stateFault,     stateFault,     stateFault,     stateAlign,     stateFault,     stateFault},
    /* e_align_done     */ { stateFault,    stateFault,     stateFault,     stateFault,     stateFault,     stateStart,     stateFault,     stateFault},
    /* e_run            */ { stateFault,    stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateFault,     stateRun  },
    /* e_app_off        */ { stateFault,    stateFault,     stateFault,     stateFault,     stateInit,      stateInit,      stateInit,      stateInit },
    /* e_start          */ { stateFault,    stateFault,     stateFault,     stateFault,     stateInit,      stateInit,      stateStart,     stateInit },
    /* e_start_done     */ { stateFault,    stateFault,     stateFault,     stateFault,     stateInit,      stateInit,      stateRun,       stateInit }
};

static AppStates state;
static AppEvents event;
static bool_t userAppSwitch, userAppSwitchInp, switchFaultClear, \
              switchAlignDone, switchAppReset, switchSlowControl, \
              switchUp, switchDown;

static AppFaultStatus faultID,faultIDp;
static bool_t         switchAppOnOff;

static bool_t        bFcnStatus = false;
static adcModule16_t adc = ADC_MODULE_DEFAULT;
static tU16          svmSector = 1;

/*------------------------------------
 * General use variables
 * ----------------------------------*/
static bool_t bFcnStatus;
static bool_t curPISet = false;
static tU16   countSWtmrLEDIndic;

/*------------------------------------
 * FREEMASTER SCALES
 * ----------------------------------*/
const tFrac32 fm_current     = FM_I_SCALE;
const tFrac32 fm_voltage     = FM_U_SCALE;
const tFrac32 fm_dcb_voltage = FM_UDCB_SCALE;

/***************************************************************************//*!
*
* @brief   Special assembly function
*
* @param   void
*
* @return  none
*
* @Details Following assembly code enables eMIOS0_CH0 (Modulus Up Counter mode)
           and PIT2 counters with minimal delay between related control
           registers byte writes

           pEMIOS0->CH[EMIOS0_UC_MCB].CCR = 0xXXXXXX10;
           pPIT->CH[2].TCTRL.R = 0xXXXXXX01;
*
*******************************************************************************/
asm void eMIOS_PIT2_Start(void)
{
    // Load address of eMIOS0_CH[EMIOS0_UC_MCB].CCR reg. least significant byte
    // into r3
    e_lis  r3,((EMIOS0_BASE_ADDRESS+0x2C+(EMIOS0_UC_MC*0x20))+3)@h
    e_or2i r3,((EMIOS0_BASE_ADDRESS+0x2C+(EMIOS0_UC_MC*0x20))+3)@l

    // Load address of PIT2.TCTRL register least significant byte into r5
    e_lis  r5,(PIT2_TCTRL_ADDRESS+3)@h
    e_or2i r5,(PIT2_TCTRL_ADDRESS+3)@l

    // Load CCR least significant byte value into r4
    se_li  r4,0x10
    // Load TCTRL least significant byte value into r6
    se_li  r6,0x01

    // Write CCR least significant byte
    se_stb r4,0x0(r3)
    // Write TCTRL least significant byte
    se_stb r6,0x0(r5)
}

/***************************************************************************//*!
*
* @brief   Application main function
*
* @param   void
*
* @return  none
*
*******************************************************************************/
void main (void)
{
    state  = reset;
    event  = e_reset;

    // Initiate RESET state first
    state_table[event][state]();

    while(1)
    {
        // Reading faults from MC33937A pre-driver, PF[13]
        if(ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,MC33937_INT_PIN))
        {
            mc33937ReadSr(&mc33937_config);
        }
        FMSTR_Poll();
    }
}

/***************************************************************************//*!
*
* @brief   RESET state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateReset(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state   = reset;

    // Configure Flash
    CFLASH.PFCR0.R = 0x10C580ED;

    // Disable all interrupts
    DISABLE_IRQ();

    // Initializing of clock, reset, core mode etc.
    sys_init();

    // SIUL initialize
    siu_init();

    // Initialize DSPI module
    dspi_init();

    // Initialize PIT timer module
    pit_init();

    // Initialize MC33905D System Basis Chip
    mc33905Config(&mc33905_config);

    // Initialize UART
    uart_init();

    // FreeMASTER initialization
    FMSTR_Init();

    // Initialize CTU module
    ctu_init();

    // Initialize ADC_0 module
    adc0_init();

    // Initialize ADC_1 module
    adc1_init();

    // Initialize eMIOS_0 module
    emios0_init();

    // Initialize eMIOS_1 module
    emios1_init();

    // Initialize MC33937A driver on 3-phase Low-voltage Power Stage
    bFcnStatus  = mc33937Config(&mc33937_config);
    // Enable MC33937A
    mc33937Enable(&mc33937_config);

    /*------------------------------------
     * SVC-PWM variables
     * ----------------------------------*/
    svmSector = 1;

    /*------------------------------------
     * Application state machine variables
     * ----------------------------------*/
    switchAppOnOff = false;
    switchFaultClear = false;
    switchAppReset = false;
    faultID.R = 0x0;
    faultIDp.R = 0x0;

    // Reset sequence has finished
    event = e_reset_done;

    // Enable all interrupts
    // Ensure INTC current priority is 0
    INTC.CPR.B.PRI = 0;

    //Enable external interrupt and exceptions (MSR reg. CE, EE, ME bits)
    asm ("e_lis r3,  0x0002");
    asm ("e_or2i r3, 0x9000");
    asm ("mfmsr r4");
    asm ("se_or r4, r3");
    asm ("mtmsr r4");

     // Clear Fault PIN flags manually
    ioctlSIUL_SET_PAD(SIUL_BASE,FAULT_RESET_PIN);
    ioctlSIUL_CLEAR_PAD(SIUL_BASE,FAULT_RESET_PIN);
    // Enable Faults
    ioctlSIUL_SET_PAD(SIUL_BASE,FAULT_EN_PIN);

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();
    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);
}

/***************************************************************************//*!
*
* @brief   FAULT state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateFault(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state = fault;

    // Disable PWM outputs
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_OFF;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set PWM duty cycle to 0% before disabling PWM outputs
    duty_cycle = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CBDR.R = 0;

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));
    eMIOS_PIT2_Start();
    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Disable user application switch
    userAppSwitch = false;

    if(switchFaultClear)
    {
        // Clear Fault PIN flags manually
        ioctlSIUL_SET_PAD(SIUL_BASE,FAULT_RESET_PIN);
        ioctlSIUL_CLEAR_PAD(SIUL_BASE,FAULT_RESET_PIN);

        // Clear MC33937A status registers
        mc33937ClearFault(&mc33937_config);

        // Re-enable MC33937A
        mc33937Enable(&mc33937_config);
        switchFaultClear = false;

        // Clear fault status
        faultID.R = 0x0;
        faultIDp.R = 0x0;
        event = e_fault_clear;
    }
}

/***************************************************************************//*!
*
* @brief   INIT state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateInit(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state                                   = init;

    // Disable PWM outputs
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_OFF;

    // Set PWM duty cycle to 0%
    duty_cycle = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CBDR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CADR.R = 0;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CBDR.R = 0;

    TIMER_OFF;

    bFcnStatus = false;

    switchAlignDone = false;

    svmSector = 1;
    duty_cycle = 0;
    direction = C_DIRECTION;

    // Define advance angle for next commutation
    advance_angle = ADVANCE_ANGLE;

    // i_dc_bus controller initialization, used only by alignment
    i_controllerParams1.s32PropGain       = I_AL_REG_PROP_GAIN;
    i_controllerParams1.s32IntegGain      = I_AL_REG_INTEG_GAIN;
    i_controllerParams1.s16PropGainShift  = I_AL_REG_PROP_GAIN_SHIFT;
    i_controllerParams1.s16IntegGainShift = I_AL_REG_INTEG_GAIN_SHIFT;
    i_controllerParams1.s32LowerLimit     = I_AL_REG_LOWER_LIMIT;
    i_controllerParams1.s32UpperLimit     = I_AL_REG_UPPER_LIMIT;
    i_controllerParams1.s32InK_1          = 0;
    i_controllerParams1.s32IntegPartK_1   = 0;

    // i_dc_bus controller initialization, used in closed loop
    i_controllerParams2.s32PropGain       = I_REG_PROP_GAIN;
    i_controllerParams2.s32IntegGain      = I_REG_INTEG_GAIN;
    i_controllerParams2.s16PropGainShift  = I_REG_PROP_GAIN_SHIFT;
    i_controllerParams2.s16IntegGainShift = I_REG_INTEG_GAIN_SHIFT;
    i_controllerParams2.s32LowerLimit     = I_REG_LOWER_LIMIT;
    i_controllerParams2.s32UpperLimit     = I_REG_UPPER_LIMIT;
    i_controllerParams2.s32InK_1          = 0;
    i_controllerParams2.s32IntegPartK_1   = 0;

    // Speed controller in parallel form - initialization
    speed_controllerParams.s32PropGain       = SPEED_REG_PROP_GAIN;
    speed_controllerParams.s32IntegGain      = SPEED_REG_INTEG_GAIN;
    speed_controllerParams.s16PropGainShift  = SPEED_REG_PROP_GAIN_SHIFT;
    speed_controllerParams.s16IntegGainShift = SPEED_REG_INTEG_GAIN_SHIFT;
    speed_controllerParams.s32LowerLimit     = SPEED_REG_LOWER_LIMIT;
    speed_controllerParams.s32UpperLimit     = SPEED_REG_UPPER_LIMIT;
    speed_controllerParams.s32InK_1          = 0;
    speed_controllerParams.s32IntegPartK_1   = 0;

    Udcb_filt.u16NSamples                       = 5;
    GDFLIB_FilterMAInitANSIC(&Udcb_filt);

    // Speed ramp initialization, used only by alignment
    speed_rampParams.s32State    = 0;
    speed_rampParams.s32RampUp   = SPEED_RAMP_UP;
    speed_rampParams.s32RampDown = SPEED_RAMP_UP;
    desiredSpeedRamp             = GFLIB_Ramp(0, &speed_rampParams);

    duty_cycle_speed = 0;
    duty_cycle = 0;

    ZCdetection = ZCdetectionCD;

    bFcnStatus = 1;
    if(!bFcnStatus)
    {
        faultID.B.ADC_Error = 1;
        event = e_fault;
    }
    else
    {
        // Initialization phase finished
        event = e_init_done;
    }
}

/***************************************************************************//*!
*
* @brief   READY state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateReady(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state           = ready;

    // User accessible switch for enabling the application.
    if(userAppSwitch)
    {
        event = e_app_on;
        // ADC module initialization
        bFcnStatus = ADCModuleInit16(&adc);
        adc.param.u16CalibSamples = 8;    // number of samples = 2^8
        svmSector = 1;
        desiredSpeedRamp = GFLIB_Ramp(0, &speed_rampParams);

        // Enable PWM outputs (0% duty cycle)
        // Set new PIT2 reload value
        ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

        // Start eMIOS modulus counter and PIT2 counter
        eMIOS_PIT2_Start();

        // Enable eMIOS0_ch1 to generate CTU trigger
        pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF  | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);

        // Set next PIT2 reload value
        ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);
    }
    else
    {
        event = e_ready;
    }

    if(switchAppReset)
    {
        event = e_reset;
    }
}

/***************************************************************************//*!
*
* @brief   CALIBRATION state - ADC calibration state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateCalib(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state = calib;

    bFcnStatus = ADCModuleCalib16(&adc, svmSector);

    // User accessible switch for stopping the application.
    if(userAppSwitch)
    {
        event = e_calib;
    }
    else
    {
        event = e_app_off;
    }

    // Test whether the DC offset calibration for one sector ended successfully.
    if(bFcnStatus)
    {
        svmSector += 1;
        adc.flag.B.calibInitDone = 0;
        adc.flag.B.calibDone = 0;
    }

    // Exit the calibration state when DC calibration is done for all sectors.
    if(svmSector>6)
    {
        alignmentTimer = ALIGNMENT_TIME;
        duty_cycle = 0;

        if(direction == C_DIRECTION)
        {
            ZCdetection = ZCdetectionCD;
        }
        else
        {
            ZCdetection = ZCdetectionCCD;
        }
        wRotElRequired = MIN_SPEED;

        // Desired motor current (torque) for alignment
        torqueRequired = ALIGNMENT_DESIRED_CURRENT;
        torqueFiltered = 0;

        // Initiate alignment vector
        PWM_Alignment();

        svmSector = 1;
        event = e_calib_done;
    }
}

/***************************************************************************//*!
*
* @brief   ALIGNMENT state - motor control axes alignment
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateAlign(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state = align;

    // Slow controller loop execution
    if(switchSlowControl == true)
    {
        // Decrement alignment counter
        if(alignmentTimer > 0)
        {
            alignmentTimer--;
        }
        else
        {
            switchAlignDone             = true;     // end of alignment
        }
    }
    // Torque error limitation
    torqueFiltered = (tFrac32)(i_dc_bus<<16);
    torque_err     = F32SubSat(torqueRequired, torqueFiltered);

    // Required voltage calculation
    Ureq = GFLIB_ControllerPIpAW(torque_err, &i_controllerParams1);
    dc_help = (tFrac16)(Ureq >> 16);
    // Duty cycle calculation block duty_cycle = Ureq * (Udcb_MAX / (2 * DC_MAX))
    duty_cycle = (uint16_t)F16Mul(dc_help, DUTY_CYCLE_SCALE);
    duty_cycle = (duty_cycle < DC_MIN) ? DC_MIN : duty_cycle;

    // User accessible switch for stopping the application.
    if(userAppSwitch)
    {
        event = e_align;
    }
    else
    {
        event = e_app_off;
    }

    if(switchAlignDone)
    {    // Align sequence has successfully finished
        processStatus.B.AfterCMT = 0;
        processStatus.B.EnableZC = 0;
        torque_R_PhA = torque_F_PhA = torque_R_PhB = torque_F_PhB = \
        torque_R_PhC = torque_F_PhC = torqueFiltered;

        if(direction == C_DIRECTION)
        {
            periodZC_F_PhA = F32Mul(START_PERIOD, START_ACCELERATION);
            periodZC_R_PhC = F32Mul(periodZC_R_PhB, START_ACCELERATION);
            periodZC_F_PhB = F32Mul(periodZC_F_PhA, START_ACCELERATION);
            periodZC_R_PhA = F32Mul(periodZC_F_PhC, START_ACCELERATION);
            periodZC_F_PhC = F32Mul(periodZC_F_PhB, START_ACCELERATION);
            periodZC_R_PhB = F32Mul(periodZC_R_PhA, START_ACCELERATION);
        }
        else
        {
            periodZC_F_PhA = F32Mul(START_PERIOD, START_ACCELERATION);
            periodZC_R_PhB = F32Mul(periodZC_R_PhA, START_ACCELERATION);
            periodZC_F_PhC = F32Mul(periodZC_F_PhB, START_ACCELERATION);
            periodZC_R_PhA = F32Mul(periodZC_F_PhC, START_ACCELERATION);
            periodZC_F_PhB = F32Mul(periodZC_F_PhA, START_ACCELERATION);
            periodZC_R_PhC = F32Mul(periodZC_R_PhB, START_ACCELERATION);
        }

        // Setup first commutation period
        svmSector = 1;
        startCMTCounter = START_CMT_NUMBER;
        torqueRequired = START_TORQUE;
        timeZCstep = START_PERIOD;

        PWM_Vector_0();

        // Save commutation time
        timeCommutation = 0;
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = timeCommutation + timeZCstep;
        TIMER_ON;

        event = e_align_done;
    }
}

/***************************************************************************//*!
*
* @brief   RUN state
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateRun(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state = run;

    // Delay for skipping zero-cross detection during freewheeling
    if(findZc)
    {
        if(cmtDelay)
        {
            cmtDelay--;
        }
        else
        {
            ZCdetection[svmSector]();
        }
    }

    // Calculate filtered value of current from one electrical period
    torque_help = ((tFrac32)(torque_R_PhA) + (tFrac32)(torque_R_PhB) + (tFrac32)(torque_R_PhC) + \
                   (tFrac32)(torque_F_PhA) + (tFrac32)(torque_F_PhB) + (tFrac32)(torque_F_PhC));
    // torquefiltered = sum of phase torque divided by 6
    torqueFiltered = (F32MulSat(torque_help, TORQUE_FILT_CONST)<<16);

    torque_err = F32SubSat(torqueRequired, torqueFiltered);
    // Motor current PI regulator
    Ureq = GFLIB_ControllerPIpAW(torque_err, &i_controllerParams2);

    // Slow controller loop execution
    if(switchSlowControl == true)
    {
        // Desired speed ramp
        desiredSpeedRamp = GFLIB_Ramp((tFrac32)wRotElRequired, &speed_rampParams);
        speed_err = (tFrac32)(desiredSpeedRamp - (tFrac32)wRotEl);
        // Speed PI controller
        duty_cycle_speed = GFLIB_ControllerPIpAW(speed_err, &speed_controllerParams);
    }
    // Torque limitation
    if((Ureq > duty_cycle_speed) && !curPISet)
    {
        // Speed control
        duty_cycle = (uint16_t)(duty_cycle_speed >> 16);
    }
    else
    {
        // Current control
        duty_cycle = (uint16_t)(Ureq >> 16);
    }
    duty_cycle = (uint16_t)F16Mul(duty_cycle, DUTY_CYCLE_SCALE);
    duty_cycle = (duty_cycle < DC_MIN) ? DC_MIN : duty_cycle;
    // User accessible switch for stopping the application.
    if(userAppSwitch)
    {
        event  = e_run;
    }
    else
    {
        // Change application state
        periodZC_F_PhA = 0;
        periodZC_R_PhC = 0;
        periodZC_F_PhB = 0;
        periodZC_R_PhA = 0;
        periodZC_F_PhC = 0;
        periodZC_R_PhB = 0;
        wRotElRequired = 0;
        wRotEl = 0;
        periodFiltered = 0;
        torqueFiltered = 0;

        duty_cycle = 0;
        dc_help = 0;
        event = e_app_off;
    }
}

/***************************************************************************//*!
*
* @brief   START state - open loop
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void stateStart(void)
{
    /*-----------------------------------------------------
    Application State Machine - state identification
    ----------------------------------------------------- */
    state = start;

    if(processStatus.B.DisableADC == 1)
    {
        processStatus.B.DisableADC = 0;
        processStatus.B.NewZC = 0;
        processStatus.B.EnableZC = 0;
    }
    torqueFiltered = (tFrac32)(i_dc_bus<<16);
    torque_err = F32SubSat(torqueRequired, torqueFiltered);

    Ureq = GFLIB_ControllerPIpAW(torque_err, &i_controllerParams1);
    dc_help = (uint16_t)(Ureq >> 16);
    duty_cycle = (uint16_t)F16Mul(dc_help, DUTY_CYCLE_SCALE);
    duty_cycle = (duty_cycle < DC_MIN) ? DC_MIN : duty_cycle;
    // User accessible switch for stopping the application.
    if(userAppSwitch)
    {
        event                           = e_start;
    }
    else
    {
        event                           = e_app_off;
    }
    // Update actual period according to calculated ZC step
    actualPeriodZC                      = timeZCstep;
    if(!startCMTCounter)
    {
        torque_R_PhA = torque_F_PhA = torque_R_PhB = torque_F_PhB = \
        torque_R_PhC = torque_F_PhC = i_dc_bus;

        periodZC_F_PhA = timeZCstep;
        periodZC_R_PhA = periodZC_F_PhA;
        periodZC_F_PhB = periodZC_F_PhA;
        periodZC_R_PhB = periodZC_F_PhA;
        periodZC_F_PhC = periodZC_F_PhA;
        periodZC_R_PhC = periodZC_F_PhA;

        timeZCstep = 10;
        // Initialize torque limit for run state
        torqueRequired = MAX_TORQUE;
        event = e_start_done;
    }
}

/*******************************************************************************
*
* Function: void PWM_Alignment(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Alignment(void)
{
    // A = +Voltage, B = -Voltage, C = -Voltage

    // A is complementary DT is controlled
    // B TOP is set to off, BOTTOM is set to on
    // C TOP is set to off, BOTTOM is set to on

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_DUTY;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);
    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);
}

/*******************************************************************************
*
* Function: void PWM_Vector_0(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_0(void)
{
    // A = +Voltage, B = -Voltage, C off

    // A is complementary, DT is controlled
    // B TOP is set to off, BOTTOM is set to on
    // C TOP is set to off, BOTTOM is set to off

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_DUTY | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_OFF;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);
    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFC_CURRENT_CHNL;
}

/*******************************************************************************
*
* Function: void PWM_Vector_1(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_1(void)
{
    // A +Voltage, B = off, C = -Voltage

    // A is complementary, DT is controlled
    // B TOP is set to off, BOTTOM is set to off
    // C TOP is set to off, BOTTOM is set to on

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_DUTY | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_ON;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFB_CURRENT_CHNL;
}

/*******************************************************************************
*
* Function: void PWM_Vector_2(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_2(void)
{
    // A off, B = +Voltage, C = -Voltage

    // A TOP is set to off, BOTTOM is set to off
    // B is complementary, DT is controlled
    // C TOP is set to off, BOTTOM is set to on

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_ON;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFA_CURRENT_CHNL;
}

/*******************************************************************************
*
* Function: void PWM_Vector_3(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_3(void)
{
    // A -Voltage, B = +Voltage, C = off

    // A TOP is set to off, BOTTOM is set to on
    // B is complementary, DT is controlled
    // C TOP is set to off, BOTTOM is set to off

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF  | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_LS_OFF;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFC_CURRENT_CHNL;
}

/*******************************************************************************
*
* Function: void PWM_Vector_4(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_4(void)
{
    // A -Voltage, B = off, C = +Voltage

    // A TOP is set to off, BOTTOM is set to on
    // B TOP is set to off, BOTTOM is set to off
    // C is complementary, DT is controlled

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF  | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_DUTY;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFB_CURRENT_CHNL;
}

/*******************************************************************************
*
* Function: void PWM_Vector_5(void)
*
* Description: Generates the 6-channel complementary bipolar PWM output
*              for 3-phase power stages. For voltage vector see function comment
*              rotation direction: clockwise
*
* Returns: None
*
* Arguments: None
*
* Range Issues: None
*
*******************************************************************************/
void PWM_Vector_5(void)
{
    // A = off, B = -Voltage, C = +Voltage

    // A TOP is set to off, BOTTOM is set to off
    // B TOP is set to off, BOTTOM is set to on
    // C is complementary, DT is controlled

    // Disable and reset eMIOS modulus counter
    pEMIOS0->CH[EMIOS0_UC_MC].CCR.B.MODE = EMIOS_MODE_GPO;

    // Setup PWM
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CCR.R = (EMIOS_CCR_HS_OFF  | EMIOS_CCR_DMA_MASK | EMIOS_CCR_FEN_MASK);
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CCR.R = EMIOS_CCR_LS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CCR.R = EMIOS_CCR_HS_OFF;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CCR.R = EMIOS_CCR_LS_ON;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CCR.R = EMIOS_CCR_DUTY;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CCR.R = EMIOS_CCR_DUTY;

    // Disable PIT2
    ioctlPIT_WRITE_VALUE_REG_TCTRL(PIT_CHNL2_BASE,0x0);

    // Set new PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,((PERIOD >> 1) + DELAY));

    // Start eMIOS modulus counter and PIT2 counter
    eMIOS_PIT2_Start();

    // Set next PIT2 reload value
    ioctlPIT_SET_TMR_RELOAD_VALUE(PIT_CHNL2_BASE,PERIOD);

    // Update CTU trigger source
    pCTU->EVTCFGR[EMIOS0_UC_PWMA_HS].B.CHANNELVALUE = BEMFA_CURRENT_CHNL;
}

/***************************************************************************//*!
@brief          static void ZCdetectPhAfalling(void)

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase A falling
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    u_dc_bus            - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phA_voltage         - Back-EMF voltage of phase A
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhA          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_F_PhA      - commutation period of phase A
    periodNextCommutation - time difference for new commutation
    torque_F_PhA        - actual current/torque at ZC of phase A
*******************************************************************************/
static void ZCdetectPhAfalling(void)
{
    phA_voltage = adc.measured.f16BemfA.raw16;

    if(phA_voltage < u_dc_bus_ZC)
    {
        timeZC_PhA = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhA;
        periodZC_F_PhA = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_F_PhA;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        // Save motor phase current
        torque_F_PhA = i_dc_bus;
        // Disable checking of ZC event for next periods till commutation
        processStatus.B.NewZC = 1;
        findZc = false;
    }
}

/***************************************************************************//*!
@brief          static void ZCdetectPhAraising(void)

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase A raising
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    dcb_voltage         - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phA_voltage         - Back-EMF voltage of phase A
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhA          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_R_PhA      - commutation period of phase A
    periodNextCommutation - time difference for new commutation
    torque_R_PhA        - actual current/torque at ZC of phase A
*******************************************************************************/
static void ZCdetectPhAraising(void)
{
    phA_voltage = adc.measured.f16BemfA.raw16;

    if(phA_voltage > u_dc_bus_ZC)
    {
        timeZC_PhA = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhA;
        periodZC_R_PhA = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_R_PhA;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        torque_R_PhA = i_dc_bus;
        processStatus.B.NewZC = 1;
        findZc = false;
    }
}

/***************************************************************************//*!
@brief          static void ZCdetectPhBfalling(void)

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase B falling
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    dcb_voltage         - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phB_voltage         - Back-EMF voltage of phase B
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhB          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_F_PhB      - commutation period of phase B
    periodNextCommutation - time difference for new commutation
    torque_F_PhB        - actual current/torque at ZC of phase B
*******************************************************************************/
static void ZCdetectPhBfalling(void)
{
    phB_voltage = adc.measured.f16BemfB.raw16;

    if(phB_voltage < u_dc_bus_ZC)
    {
        timeZC_PhB = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhB;
        periodZC_F_PhB = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_F_PhB;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        torque_F_PhB = i_dc_bus;
        processStatus.B.NewZC = 1;
        findZc        = false;
    }
}

/***************************************************************************//*!
@brief          static void ZCdetectPhBfalling(void)

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase B raising
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    dcb_voltage         - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phB_voltage         - Back-EMF voltage of phase B
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhB          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_R_PhB      - commutation period of phase B
    periodNextCommutation - time difference for new commutation
    torque_R_PhB        - actual current/torque at ZC of phase B
*******************************************************************************/
static void ZCdetectPhBraising(void)
{
    phB_voltage = adc.measured.f16BemfB.raw16;

    if(phB_voltage > u_dc_bus_ZC)
    {
        timeZC_PhB = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhB;
        periodZC_R_PhB = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_R_PhB;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        torque_R_PhB = i_dc_bus;
        processStatus.B.NewZC = 1;
        findZc = false;
    }
}

/***************************************************************************//*!
@brief          static void ZCdetectPhCfalling(void)

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase C falling
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    dcb_voltage         - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phB_voltage         - Back-EMF voltage of phase C
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhC          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_F_PhC      - commutation period of phase C
    periodNextCommutation - time difference for new commutation
    torque_F_PhC        - actual current/torque at ZC of phase C
*******************************************************************************/
static void ZCdetectPhCfalling(void)
{
    phC_voltage = adc.measured.f16BemfC.raw16;

    if(phC_voltage < u_dc_bus_ZC)
    {
        timeZC_PhC = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhC;
        periodZC_F_PhC = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_F_PhC;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        torque_F_PhC = i_dc_bus;
        processStatus.B.NewZC = 1;
        findZc = false;
    }
}

/***************************************************************************//*!
@brief          static void ZCdetectPhCraising

@param[in,out]  void
@param[in]      void

@return         void

@details        The function detects the zero crossing of phase C raising
                Back EMF voltage and sets the new commutation event using
                eMIOS1_CH[24]

@note
    u_dc_bus_ZC         - corrected DC-bus voltage
    dcb_voltage         - DC-bus voltage
    dcb_offset          - offset of DC-bus voltage
    processStatus       - process status byte
    phB_voltage         - Back-EMF voltage of phase C
    timeCommutation     - time of last commutation
    timeZCstep          - time difference from last commutation
                          when ZC detection is enabled
    timeZC_PhC          - time of current commutation
    timeZC              - time of current commutation
    lastTimeZC          - time of last commutation
    actualPeriodZC      - commutation period
    periodZC_R_PhC      - commutation period of phase C
    periodNextCommutation - time difference for new commutation
    torque_R_PhC        - actual current/torque at ZC of phase C
*******************************************************************************/
static void ZCdetectPhCraising(void)
{
    phC_voltage = adc.measured.f16BemfC.raw16;

    if(phC_voltage > u_dc_bus_ZC)
    {
        timeZC_PhC = timeZC_Phx_help;
        lastTimeZC = timeZC;
        timeZC = timeZC_PhC;
        periodZC_R_PhC = (timeZC - lastTimeZC);
        actualPeriodZC = periodZC_R_PhC;
        periodNextCommutation = (((uint16_t)(F16Mul((tFrac16)(((uint16_t)(actualPeriodZC)>>1)),(tFrac16)(advance_angle)))) << 1);
        periodNextCommutation = (periodNextCommutation < 20) ? 20 : periodNextCommutation;
        // Setup compare value for next commutation
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeZC + periodNextCommutation);
        torque_R_PhC = i_dc_bus;
        processStatus.B.NewZC = 1;
        findZc = false;
    }
}

/***************************************************************************//*!
*
* @brief   Fault Detection function
*
* @param   void
*
* @return  none
*
*******************************************************************************/
static void faultDetection(void)
{
    //-----------------------------
    // Pending Faults
    //-----------------------------
    // Fault:   DC-bus over-current, PC[5]
    faultIDp.B.OverDCBusCurrent = ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,FAULT_DCBI_PIN);

    // Fault:   DC-bus under-voltage, PC[4]
    faultIDp.B.UnderDCBusVoltage = ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,FAULT_DCBV_PIN);

     // Clear Fault flags
    ioctlSIUL_SET_PAD(SIUL_BASE,FAULT_RESET_PIN);
    ioctlSIUL_CLEAR_PAD(SIUL_BASE,FAULT_RESET_PIN);

    // Fault:   DC-bus over-voltage
    faultIDp.B.OverDCBusVoltage = (adc.measured.f16Udcb.filt16 > DCBUS_OVERVOLT_THRESH) ? true:false;

    faultID.R |= faultIDp.R;

    if(faultIDp.R != 0x0)
    {
        event = e_fault;
    }
}

/***************************************************************************//*!
@brief              Check switch state

@param[in,out]      void
@param[in]          void

@return             none

@details            The function provides reading of run/stop switch
                    from port E[1], speed-up/speed-down switches (E[9]/E[10]),
                    and FreeMASTER-controlled switches

@note               none
*******************************************************************************/
static void CheckSwitchState(void)
{
    // Check UP switch port E[9]
    if((ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,UP_SW_PIN) == 0) && \
       (ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,UP_SW_PIN) == switchUp))
    {
        switchCounterUp++;

        if(switchCounterUp > SW_PRESS_DEBOUNCE)
        {
            wRotElRequired += SPEED_STEP;
            if(wRotElRequired > MAX_SPEED)
            {
                wRotElRequired = MAX_SPEED;
            }
            switchCounterUp = 0;
        }
    }

    // Check DOWN switch port E[10]
    if((ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,DOWN_SW_PIN) == 0) && \
       (ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,DOWN_SW_PIN) == switchDown))
    {
        switchCounterDown++;

        if(switchCounterDown > SW_PRESS_DEBOUNCE)
        {
            wRotElRequired -= SPEED_STEP;
            if(wRotElRequired < MIN_SPEED)
            {
                wRotElRequired = MIN_SPEED;
            }
            switchCounterDown = 0;
        }
    }

    // If UP & DOWN switches are pressed clear faults
    if((ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,UP_SW_PIN) == 0) && (ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,DOWN_SW_PIN) == 0))
    {
        switchFaultClear = true;
    }
    else
    {
        switchFaultClear = false;
    }

    switchUp = ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,UP_SW_PIN);
    switchDown = ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,DOWN_SW_PIN);

    // Check ON/OFF switch port E[1]
    if(ioctlSIUL_READ_INP_VALUE_PAD(SIUL_BASE,ON_OFF_SW_PIN) == 1)
    {
        if(switchCounter < SW_PRESS_DEBOUNCE)
        {
            switchCounter++;
        }
    }
    else
    {
        if(switchCounter > 0)
        {
            switchCounter--;
        }
    }

    userAppSwitch = (switchAppOnOff) ? true : false;;
    if((switchCounter == SW_PRESS_DEBOUNCE) && !userAppSwitchInp)
    {
        userAppSwitchInp = true;
        switchAppOnOff = true;
    }
    if((switchCounter == 0) && userAppSwitchInp)
    {
        userAppSwitchInp = false;
        switchAppOnOff = false;
    }
}

/***************************************************************************//*!
@brief          LED indication function

@param[in,out]  void
@param[in]      void

@return         void

@details        The function is called in fast loop Zero cross detection
                function to indicate application status
*******************************************************************************/
static void LED_indication(void)
{
    if((countSWtmrLEDIndic)-- <= 1)
    {
        // Setup default value
        countSWtmrLEDIndic = SWTMR_LED_INDIC_STOP;

        switch (state)
        {
            case run:
            {
                // Set LED indicator RUN, B[0]
                ioctlSIUL_SET_PAD(SIUL_BASE,LED_RUN_PIN);
                break;
            }
            case init:
            case ready:
            case calib:
            case align:
            {
                // Toggle pin B[0] = LED D506
                ioctlSIUL_TOGGLE_PAD(SIUL_BASE,LED_RUN_PIN);
                countSWtmrLEDIndic = SWTMR_LED_INDIC_STOP;
                break;
            }
            case fault:
            {
                // Toggle pin B[0] = LED D506
                ioctlSIUL_TOGGLE_PAD(SIUL_BASE,LED_RUN_PIN);
                countSWtmrLEDIndic = SWTMR_LED_INDIC_FAULT;
                break;
            }
            default:
                break;
        }
    }
}

/***************************************************************************//*!
@brief          ADC_0 End of Injected Conversion Chain ISR

@param[in,out]  void
@param[in]      void

@return         void

@details        The function performs fast loop Zero cross detection and
                provides duty cycle value for PWM generation using eMIOS
*******************************************************************************/
void BLDC_Fast_ISR(void)
{
    ioctlSIUL_SET_PAD(SIUL_BASE,SIUL_PAD_105);

    // Read captured ZC time from eMIOS1.channel25
    timeZC_Phx_help = pEMIOS1->CH[EMIOS1_UC_IC].CADR.R;

    bFcnStatus = ADCMeasureBEMF(&adc, svmSector, &i_dc_bus);
    /* --------------------------------------------------------------
     * DC-bus Voltage - DC data filtering using MA recursive filter
     * ------------------------------------------------------------ */
    adc.measured.f16Udcb.filt16 = (tFrac16)(GDFLIB_FilterMA((tFrac32)(adc.measured.f16Udcb.raw16<<16), &Udcb_filt)>>16);
    // Braking resistor control - limitation of the DC-bus voltage by braking, PA[4]
    if(state == run)
    {
        pSIU->GPDO[UNI3_BRAKE_PIN].B.PDO = (adc.measured.f16Udcb.raw16 > FRAC16(U_DCB_TRIP/VOLT_RANGE_MAX)) ? (vuint8_t)true : (vuint8_t)false;
    }

    // DC-bus Zero cross voltage is 1/2*Udcbus
    u_dc_bus_ZC = adc.measured.f16Udcb.filt16 >> 1;

    // Fault detection routine, must be executed prior application state machine
    faultDetection();

    // Counter for slow control loop, code is executed in state machine routines
    // must be executed prior application state machine
    if(SlowContLoopCNT != 0)
    {
        SlowContLoopCNT--;
    }
    else
    {
        SlowContLoopCNT = CONTROLLER_LOOP_TIME;
        switchSlowControl = true;
        // Calculate motor electrical period for speed calculation
        periodRaising = (periodZC_R_PhA + periodZC_R_PhB + periodZC_R_PhC);
        periodFalling = (periodZC_F_PhA + periodZC_F_PhB + periodZC_F_PhC);
        periodFiltered = periodRaising + periodFalling;

        if(periodFiltered == 0)
        {
            wRotEl = 0;
        }
        else
        {
            // Electrical speed in RPM
            wRotEl = ((uint32_t)SPEED_CONST / (uint32_t)(periodFiltered));
        }
        // Input detection
        CheckSwitchState();
    }

    // Application state machine
    state_table[event][state]();

    // Clear End of Injected Chain Conversion interrupt flag
    ADC_0.ISR.R = 0x00000004;

    // Update duty cycle
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CADR.R = (PERIOD >> 1) - duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMA_HS].CBDR.R = (PERIOD >> 1) + duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CADR.R = (PERIOD >> 1) - duty_cycle - DEADTIME_F;
    pEMIOS0->CH[EMIOS0_UC_PWMA_LS].CBDR.R = (PERIOD >> 1) + duty_cycle + DEADTIME_R;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CADR.R = (PERIOD >> 1) - duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMB_HS].CBDR.R = (PERIOD >> 1) + duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CADR.R = (PERIOD >> 1) - duty_cycle - DEADTIME_F;
    pEMIOS0->CH[EMIOS0_UC_PWMB_LS].CBDR.R = (PERIOD >> 1) + duty_cycle + DEADTIME_R;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CADR.R = (PERIOD >> 1) - duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMC_HS].CBDR.R = (PERIOD >> 1) + duty_cycle;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CADR.R = (PERIOD >> 1) - duty_cycle - DEADTIME_F;
    pEMIOS0->CH[EMIOS0_UC_PWMC_LS].CBDR.R = (PERIOD >> 1) + duty_cycle + DEADTIME_R;

    switchSlowControl = false;

    LED_indication();

    // FreeMASTER recorder
    FMSTR_Recorder();

    ioctlSIUL_CLEAR_PAD(SIUL_BASE,SIUL_PAD_105);
}

/***************************************************************************//*!
@brief          eMIOS_1 channel 24 compare event

@param[in,out]  void
@param[in]      void

@return         void

@details        Commutation function
*******************************************************************************/
void EMIOS1_CH24_ISR(void)
{
    ioctlSIUL_SET_PAD(SIUL_BASE,SIUL_PAD_104);

    if(state != fault)
    {
        // Switch PWM to new sector
        PWMCommutationFcn[svmSector+1]();
    }

    // Save commutation time
    timeCommutation = pEMIOS1->CH[EMIOS1_UC_IC].CADR.R;
    // Setup new sector
    if(svmSector == 6)
    {
        svmSector = 1;
    }
    else
    {
        svmSector++;
    }

    // No ZC detected
    if(processStatus.B.NewZC == 0)
    {
        timeZC = F16Sub(timeCommutation, (actualPeriodZC >> 1));
    }
    processStatus.B.DisableADC = 1;
    processStatus.B.AfterCMT = 1;
    processStatus.B.StallCheck = 1;

    cmtDelay = 2;
    findZc = 1;

    pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)((vuint16_t)(timeCommutation) + (uint16_t)(START_PERIOD));
    // Decrement alignment counter
    if(startCMTCounter > 0)
    {
        startCMTCounter--;
        timeZCstep = F16Mul(timeZCstep, START_ACCELERATION);
        pEMIOS1->CH[EMIOS1_UC_OC].CADR.R = (vuint16_t)(timeCommutation + timeZCstep);
    }
    // Clear FLAG bit
    pEMIOS1->CH[EMIOS1_UC_OC].CSR.R = 1;

    ioctlSIUL_CLEAR_PAD(SIUL_BASE,SIUL_PAD_104);
}