/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     BLDC_appconfig.h
*
* @author   b06050
*
* @version  1.0.8.0
*
* @date     Aug-20-2012
*
* @brief    BLDC motor application header file
*
********************************************************************************
*
* This file includes initial setting and MACRO definitions of BLDC motor control
* application.
*
*******************************************************************************/
#ifndef __BLDC_APPCONFIG_H
#define __BLDC_APPCONFIG_H

#include "gflib.h"

/*******************************************************************************
* Type definition
*******************************************************************************/
typedef void (*tPointerFcn)(void); /* pointer to function */

/*******************************************************************************
* STRUCTURES
*******************************************************************************/
typedef union
{
    vuint16_t R;
    struct
    {
        vuint16_t AfterAlignment : 1;
        vuint16_t AfterCMT       : 1;
        vuint16_t NewZC          : 1;
        vuint16_t DisableADC     : 1;
        vuint16_t EnableZC       : 1;
        vuint16_t StallCheck     : 1;
        vuint16_t AdcCalib       : 1;
        vuint16_t : 9;  /* RESERVED */
    } B;
} AppProcessStatus;     /* Application process status user type */

/******************************************************************************/
/* DEFINES                                                                    */
/******************************************************************************/

// Application maximum values used for scaling
// determined by HW voltage sensing
// Vref=3.246V => Vdcb=<35.436>V, Idcb=<-9.836,9.836>A
#define VOLT_RANGE_MAX           35.436
#define VOLT_RANGE_MIN           0.0
#define CURR_MAX                 9.85
#define CURR_MIN                -9.85

// DC-bus overvoltage fault limit
#define DC_OVERVOLTAGE          30.0
// DC-bus braking threshold
#define U_DCB_TRIP              27.0
// DC-Bus braking trip voltage hysteresis [V]*/
#define U_DCB_TRIP_HYST         1
// DC-bus over-voltage threshold
#define DCBUS_OVERVOLT_THRESH   FRAC16(DC_OVERVOLTAGE/VOLT_RANGE_MAX)
// Brake resistor turn ON time; 1 tick equals 50us; 4000 equals 200ms ON
#define BRAKE_ON_COUNT_MAX      4000

// Voltage offset HW
#define OFFSET_BEMF_A           0
#define OFFSET_BEMF_B           0
#define OFFSET_BEMF_C           0

/****************** Global Applications defines - Begin ***********************/
// Debounce timer for filtering inputs, filtering period is N*t_SlowLoop
#define SW_PRESS_DEBOUNCE           50
#define SW_PRESS_SHORT_MAX          300

#define C_DIRECTION                 0
#define CC_DIRECTION                1

// PWM frequency in Hz
#define PWM_FREQ                    20000
// PWM module clock in Hz
#define PWM_CLK                     64000000
// Define period of PWM
#define PERIOD                      (PWM_CLK/PWM_FREQ)
// Define ADC trigger delay in PWM_CLK ticks (from the centre of PWM)
#define DELAY                       40
// Define minimal duty cycle with respect to i_dc_bus and BEMF stabilization
// it should be higher than 2-3us -> Tstab * PWM_CLK/2 = DC_MIN
#define DC_MIN                      100 //64
// Define maximal duty cycle, has to be DC < PERIOD/2
#define DC_MAX                      1500
// Deadtime is defined as DT = deadtime [s] * PWM_CLK [Hz]
// PWM deadtime is setup to 400ns by 64MHz it is 26
#define DEADTIME_R                  32
#define DEADTIME_F                  32
// Duty cycle scale value for outputs from regulators
// reg.outputs <-1, 1) and DC <0, PERIOD/2)
#define DUTY_CYCLE_SCALE            FRAC16(0.048828125)
#define UREQ_MIN                    FRAC32(0.1/VOLT_RANGE_MAX)
#define UREQ_MAX                    FRAC32(VOLT_RANGE_MAX/VOLT_RANGE_MAX)
// Define alignment state time N * period of slow control loop (N * 2.5ms)
#define ALIGNMENT_TIME              500
// Desired current by alignment
#define ALIGNMENT_DESIRED_CURRENT   FRAC32(1.0/CURR_MAX)
// Maximum of the motor current in the run state
#define MAX_TORQUE                  FRAC32(3.0/CURR_MAX)
#define MIN_TORQUE                  FRAC32(0.05/CURR_MAX)

/* Ramp generator parameters
 */
#define SPEED_RAMP_UP               FRAC32(0.000005)

/* --------- BLDC motor algorithm defines
 */
// Define advance angle for next commutation, depends on motor quality
#define ADVANCE_ANGLE               FRAC16(0.374)//(0.323)//0.374)
// Timeout period for PIT timer0 interrupts, PIT clock = SYS_CLK = 64MHz
// Counter value = (period * PWM_FREQ)
// period = 2ms
#define CONTROLLER_LOOP_TIME        40
// Constant for speed calc, Nel = 1s / Tel [us]
// Tel value from eMIOS1_CH24, f = 64MHz/16/4, SPEED_CONST = (60 * f) / 4
#define SPEED_CONST                 15000000
// ZC timer frequency 1MHz -> N * 1us = START_PERIOD
#define START_PERIOD                25000   // 25ms
#define START_ACCELERATION          FRAC16(0.84668245)
// Motor start torque
#define START_TORQUE                FRAC32(0.5/CURR_MAX)

#define START_CMT_NUMBER            18

#define MAX_SPEED                   10000   // 10000 RPM
#define MIN_SPEED                   1000    // 1000 RPM
#define SPEED_STEP                  100     // 100 RPM

// Constant for calculate filtered torque for one el. period
#define TORQUE_FILT_CONST           FRAC32(0.16666667)

/****************** PI Controller defines - begin *****************************/

// i_dc_bus controller initialization, used only by alignment
// PI controller proportional gain, see documentation
#define I_AL_REG_PROP_GAIN          FRAC32(0.1)
// PI controller integral gain, see documentation
#define I_AL_REG_INTEG_GAIN         FRAC32(0.4)
// PI controller proportional gain shift, see documentation
#define I_AL_REG_PROP_GAIN_SHIFT    (-4)
// PI controller integral gain shift, see documentation
#define I_AL_REG_INTEG_GAIN_SHIFT   (-4)
// PI controller lower limit
#define I_AL_REG_LOWER_LIMIT        UREQ_MIN
// PI controller upper limit
#define I_AL_REG_UPPER_LIMIT        FRAC32(0.8)

// i_dc_bus controller initialization, used in closed loop
// PI controller proportional gain, see documentation
#define I_REG_PROP_GAIN             FRAC32(0.1)
// PI controller integral gain, see documentation
#define I_REG_INTEG_GAIN            FRAC32(0.18)
// PI controller proportional gain shift, see documentation
#define I_REG_PROP_GAIN_SHIFT       (-6)
// PI controller integral gain shift, see documentation
#define I_REG_INTEG_GAIN_SHIFT      (-6)
// PI controller lower limit
#define I_REG_LOWER_LIMIT           UREQ_MIN
// PI controller upper limit
#define I_REG_UPPER_LIMIT           FRAC32(0.9)

// Speed controller constants
// PI controller proportional gain, see documentation
#define SPEED_REG_PROP_GAIN         FRAC32(0.40)
// PI controller integral gain, see documentation
#define SPEED_REG_INTEG_GAIN        FRAC32(0.56)
// PI controller proportional gain shift, see documentation
#define SPEED_REG_PROP_GAIN_SHIFT   11
// PI controller integral gain shift, see documentation
#define SPEED_REG_INTEG_GAIN_SHIFT  11
// PI controller lower limit
#define SPEED_REG_LOWER_LIMIT       UREQ_MIN
// PI controller upper limit
#define SPEED_REG_UPPER_LIMIT       UREQ_MAX

/****************** PI Controller defines - end *******************************/

/*
* FreeMASTER Scale Variables
* note: scaled at input = 1000
*/
#define FM_I_SCALE                               (9850)
#define FM_U_SCALE                               (35436)
#define FM_UDCB_SCALE                            (35436)

#endif /* __BLDC_APPCONFIG_H */