/**************************************************************************
* PROJECT: Interrupts in Decoupled Parallel Mode for MPC5675K	          *
* File name: main.c							  *
* Interrupt handling in DPM mode                        	    	  *
* Date   : 16 February 2012                                              *
* Author : Tomas Kulig                                                   *
/**************************************************************************/
/*  LICENSE: 
 *  Copyright (c) 2012 Freescale Semiconductor
 *  
 *  Permission is hereby granted, free of charge, to any person 
 *  obtaining a copy of this software and associated documentation 
 *  files (the "Software"), to deal in the Software without 
 *  restriction, including without limitation the rights to use, 
 *  copy, modify, merge, publish, distribute, sublicense, and/or 
 *  sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following 
 *  conditions:
 *  
 *  The above copyright notice and this permission notice 
 *  shall be included in all copies or substantial portions 
 *  of the Software.
 *  
 *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
***************************************************************************/

/******************************************************************************
 * LIBRARIES
 *****************************************************************************/
#include "..\header\mpc5675k-0200.h"

/******************************************************************************
 * CONSTANT
 *****************************************************************************/
#define		Disable		0
#define		Enable		1

#define 	VLE		2			 
#define 	DPM_KEY1        0x5AF0
#define 	DPM_KEY2        0xA50F

#define		PLLS_RC		0x0 				//input for PLL is internal RC oscilator
#define		PLLS_CRY	0x1 				//input for PLL is crystal oscillator

#define		INTC_SHARED 	0				//both core handling the same interrupt request
#define		INTC_INDEPE 	1				//each core has own interrupts
#define		intc_sel	INTC_INDEPE

enum MODDE {M_REST=0, M_TEST=1, M_SAFE=2, M_DRUN=3, M_RUN0=4, M_RUN1=5, M_RUN2=6, M_RUN3=7, M_HLAT=8, M_STOP=0xA};
enum CFLASH {CF_None=0,CF_Power_down=1,CF_Low_power=2,CF_Normal=3};
enum DFLASH {DF_None=0,DF_Power_down=1,DF_None1=2,DF_Normal=3};
enum SYS_CLOCK {SC_RC=0,SC_PLL=4,SC_TEST=15};


/******************************************************************************
 * EXTERN FUNCTION
 *****************************************************************************/
extern void     _start          (void);
extern void	xcptn_xmpl	(void);
extern void	xcptn_xmpl1	(void);

/******************************************************************************
 * PROTOTYPES of FUNCTION
 *****************************************************************************/
void            main0           (void);				//Core0
void            main1           (void);				//Core1
void 		Init_ADC0	(int8_t chan);			//Core0
void 		ADC0_EOC	(void);				//Core0
void 		ADC01_EOC	(void);				//Core1
void 		Init_ADC1	(int8_t chan);			//Core1
void 		ADC1_EOC	(void);				//Core1
//modes
void 		Init_REST	(void);				//Core0
void 		Init_DRUN	(void);				//Core0
void 		MODE_GO		(uint8_t mode);			//Core0
//clocks
void		Init_PLL0	(uint8_t mode);			//Core0
void		Init_PLL1	(uint8_t mode);			//Core0
void 		Init_clock_man	(void);				//Core0

/******************************************************************************
 * GLOBAL VARIABLE
 *****************************************************************************/
uint8_t flag = 0;						//flag which inform core0 that core1 is running


/******************************************************************************
 * FUNCTION     : main0
 * CORE		: core0
 * DESCRIPTION  : main C entry point for core 0
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void main0(void)
{
    //flag = 0;						//init global variables
    //intc_sel=INTC_INDEPE;

    RGM.FES.B.F_FCCU_SAFE = 0x1;                        // Clear flag for enter to SAFE mode
    Init_clock_man();					// init clock distribution 
    Init_DRUN();					// init DRUN mode and go into it
    MODE_GO(M_DRUN);
    Init_REST();					// init RESET mode
    
    SSCM.DPMBOOT.R = (unsigned int) _start + VLE;       // start address for core 1 and set up for VLE code
    
    SSCM.DPMKEY.R  = DPM_KEY1;                          // include direct key for start core1             
    SSCM.DPMKEY.R  = DPM_KEY2;                          // include inverse key for start core1
  
    xcptn_xmpl();					// enable and init interrupt
    SIU.PCR[59].R = 0x0220;                             // Enable LED1
    SIU.PCR[62].R = 0x0220;                             // Enable LED2+
    SIU.GPDO[59].B.PDO = 1;                       	// LED1 turn off
    SIU.GPDO[62].B.PDO = 1;                       	// LED2 turn off
   
    INTC_0.PSR[62].R = 0x1;				// Enable interrupt from ADC.EOC

    if (intc_sel) Init_ADC0(10);			// different interrupt 
    
    while(1)
    {
	if ((!intc_sel)&&(flag==1))			// shared interrupt, waiting for start core1
	{
	    Init_ADC0(10);
	    flag=2;
	}
    }
} 

/******************************************************************************
 * FUNCTION     : main1
 * CORE		: core1
 * DESCRIPTION  : main C entry point for core 1
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void main1(void)
{
    xcptn_xmpl1();					// enable and init interrupt
    SIU.PCR[99].R = 0x0220;				// Enable LED3
    SIU.PCR[102].R = 0x0220;				// Enable LED4
    SIU.GPDO[99].B.PDO = 1;				// LED3 turn off
    SIU.GPDO[102].B.PDO = 1;				// LED4 turn off

    //interrupt
    if (intc_sel == INTC_SHARED) INTC_1.PSR[62].R = 0x1;// shared interrupt, waitin for start core1
    else						// different interrupt
    {
	INTC_1.PSR[82].R = 0x1;
	Init_ADC1(15);
    }
    
    while(1)
    {
	if ((flag==0)&&(!intc_sel)) flag=1;		//give information that core1 is ready
    }
}  

/******************************************************************************
 * FUNCTION     : Init_ADC0
 * CORE		: core0 
 * DESCRIPTION  : initialization of ADC0
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Init_ADC0(int8_t chan)
{
    ADC_0.CIMR0.R = (1<<chan);				//enabled interrupt for "chan" channel
    ADC_0.IMR.B.MSKEOC = 0x1;				//enabled the EOC interrupt
    ADC_0.MCR.B.MODE = 0x0;				//one shot mode
    ADC_0.MCR.B.PWDN = 0x0;				//wake-up ADC from low-power

    ADC_0.NCMR0.R|=(1<<chan);				//select "chan" channel for conversation	
    ADC_0.MCR.B.NSTART = 0x1;				//start conversation
}

/******************************************************************************
 * FUNCTION     : Init_ADC1
 * CORE		: core1 
 * DESCRIPTION  : initialization of ADC1
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Init_ADC1(int8_t chan)
{
    ADC_1.CIMR0.R = (1<<chan);				//enabled interrupt for "chan" channel
    ADC_1.IMR.B.MSKEOC = 0x1;				//enabled the EOC interrupt
    ADC_1.MCR.B.MODE = 0x0;				//one shot mode
    ADC_1.MCR.B.PWDN = 0x0;				//wake-up ADC from low-power

    ADC_1.NCMR0.R|=(1<<chan);				//select "chan" channel for conversation
    ADC_1.MCR.B.NSTART = 0x1;				//start conversation
}

/******************************************************************************
 * FUNCTION     : ADC0_EOC
 * CORE		: core0
 * DESCRIPTION  : Interrupt service routine for ADC0
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void ADC0_EOC(void)
{
    ADC_0.ISR.B.EOC = 0x1;				// clear interrupt flag
    SIU.GPDO[59].B.PDO = 0x0;                       	// LED1 turn on
    SIU.GPDO[62].B.PDO = 0x0;                       	// LED2 turn on
    if (intc_sel == INTC_SHARED) SIU.PCR[59].R = 0x0;	// LED1 disconnect
    else SIU.PCR[62].R = 0x0;				// LED2 disconnect
}

/******************************************************************************
 * FUNCTION     : ADC01_EOC
 * CORE		: core0
 * DESCRIPTION  : Interrupt service routine for ADC0
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void ADC01_EOC(void)
{
    ADC_0.ISR.B.EOC = 0x1;				// clear interrupt flag
    SIU.GPDO[99].B.PDO = 0;                    		// LDE3 turn on
    SIU.GPDO[102].B.PDO = 0;                   		// LED4 turn on
    if (intc_sel == INTC_SHARED) SIU.PCR[99].R = 0x0;	// LED3 disconnect
    else SIU.PCR[102].R = 0x0;				// LED4 disconnect
}
/******************************************************************************
 * FUNCTION     : ADC1_EOC
 * CORE		: core1
 * DESCRIPTION  : Interrupt service routine for ADC1
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void ADC1_EOC(void)
{
    ADC_1.ISR.B.EOC = 0x1;				// clear interrupt flag
    SIU.GPDO[99].B.PDO = 0;                    		// LED3 turn on
    SIU.PCR[102].R = 0x0;				// LED4 disconnect
}

/******************************************************************************
 * FUNCTION     : Init_REST()
 * CORE		: core0
 * DESCRIPTION  : Initialization of RESET mode
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Init_REST()
{
    ME.RUNPC[0].B.RESET = Enable;			// enabled clock for peripherals under RUNPC[0]
    ME.MER.B.RESET = Enable;				// enabled RESET mode
}

/******************************************************************************
 * FUNCTION     : Init_DRUN()
 * CORE		: core0
 * DESCRIPTION  : Initialization of DRUN mode
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Init_DRUN()
{
    ME.DRUN.B.CFLAON = CF_Normal;			// Code flash power mode definition
    ME.DRUN.B.DFLAON = DF_Normal;			// Data flash power mode definition
    ME.DRUN.B.XOSCON = Enable;				// allow external crystal for PLL
//this is place for setting pll, for calling the function
    MODE_GO(M_DRUN);
    ME.DRUN.B.PLL0ON = Enable;
    Init_PLL0(M_DRUN);
    ME.DRUN.B.PLL1ON = Enable;
    Init_PLL1(M_DRUN);
    ME.DRUN.B.SYSCLK = SC_PLL;				// PLL as system clock

    ME.RUNPC[0].B.DRUN = Enable;			// enabled clock for peripherals under RUNPC[0]
    
    ME.MER.B.DRUN = Enable;				// enabled DRUN mode
}

/******************************************************************************
 * FUNCTION     : MODE_GO()
 * CORE		: core0
 * DESCRIPTION  : change the actual mode to "mode" mode
 * INPUTS       : mode	- name of mode which will be used
 * OUTPUTS      : None
 *****************************************************************************/
void MODE_GO(uint8_t mode)
{
    ME.MCTL.R = ((mode<<28)|(0x00005AF0));		// key for changing mode - direction form        		
    ME.MCTL.R = ((mode<<28)|(0x0000A50F));		// key for changing mode - inverze mode
    while (ME.GS.B.S_MTRANS!=0);			// waiting for succesfull change mode
    while (ME.GS.B.S_CURRENTMODE!=mode);		// chack the swap mode
}

/******************************************************************************
 * FUNCTION     : Init_PLL0()
 * CORE		: core0
 * DESCRIPTION  : set PLL0 to 180 MHz
 * INPUTS       : mode	-  name of mode which start the PLL0
 * OUTPUTS      : None
 *****************************************************************************/
void Init_PLL0(uint8_t mode)
{
    //180 MHz external
    CGM.AC3_SC.B.SELCTL = 0x1;				// source of PLL0 is external oscillator
    CGM.FMPLL[0].CR.B.IDF = 0x3;			// IDF = 4, divided by 4
    CGM.FMPLL[0].CR.B.ODF = 0x0;//23.1			// ODF = 2, divided by 2
    CGM.FMPLL[0].CR.B.NDIV = 36;			// NDIV = 36. dived by 36
    CGM.FMPLL[0].CR.B.EN_PLL_SW = 1;			// gradually increasing the frequency of the PLL
        
    MODE_GO(mode);					// for star PLL must change mode, upadate mode
    while (CGM.FMPLL[0].CR.B.S_LOCK==0);		// waiting for capture of loop PLL phase
}

/******************************************************************************
 * FUNCTION     : Init_PLL1()
 * CORE		: core0
 * DESCRIPTION  : set PLL1 to 80 MHz
 * INPUTS       : mode	-  name of mode which start the PLL1
 * OUTPUTS      : None
 *****************************************************************************/
void Init_PLL1(uint8_t mode)
{
    //80 MHz external
    CGM.AC4_SC.B.SELCTL = 0x1;				// source of PLL1 is external oscillator
    CGM.FMPLL[1].CR.B.IDF = 0x3;			// IDF = 4, divided by 4
    CGM.FMPLL[1].CR.B.ODF = 0x1;			// ODF = 4, divided by 4
    CGM.FMPLL[1].CR.B.NDIV = 32;			// NDIV = 32. divided by 32
    
    MODE_GO(mode);					// for star PLL must change mode, upadate mode
    while (CGM.FMPLL[1].CR.B.S_LOCK==0);		// waiting for capture of loop PLL phase
}


/******************************************************************************
 * FUNCTION     : Init_clock_man()
 * CORE		: core0
 * DESCRIPTION  : initialization of clock architecture
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Init_clock_man(void)
{
    //Peripheral set 0 clock - 45 MHz
    CGM.SC_DC0.B.DIV0 = 1;				//divide by 2
    CGM.SC_DC0.B.DE0 = Enable;				//enable divider

    //Peripheral set 1 clock - 22,5 MHz    
    //minimal value must be set so that peri 1 clock is smaller then 45 MHz
    CGM.SC_DC0.B.DIV1 = 3;				//divide by 4
    CGM.SC_DC0.B.DE1 = Enable;				//enable divider

    //Motor control clock - input PLL0, 15 MHz
    CGM.AC0_SC.B.SELCTL = 4;				//PLL0
    CGM.AC0_DC0.B.DIV0 = 11; 				//divide by 12
    CGM.AC0_DC0.B.DE0 = Enable;				//enable divider

    //FlexRay clock - input PLL0, 30 MHz
    CGM.AC1_SC.B.SELCTL = 4;				//PLL0
    CGM.AC1_DC0.B.DIV0 = 5;				//divide by 6
    CGM.AC1_DC0.B.DE0 = Enable;				//enable divider

    //FlexCAN clock - input PLL0, 12,5 MHz
    CGM.AC2_SC.B.SELCTL = 4;				//PLL0
    CGM.AC2_DC0.B.DIV0 = 7;				//divide by 8
    CGM.AC2_DC0.B.DE0 = Enable;				//enable divider
    
}
