/********************************************************************/
/* Demo code for Dual Core Mode for MPC5675K                        */
/* Modified from MPC5643L Code                                      */
/* 21 April 2010                                                    */
/* Mark Ruthenbeck                                                  */
/********************************************************************/
#/*
# *  LICENSE: 
# *  Copyright (c) 2006 Freescale Semiconductor
# *  
# *  Permission is hereby granted, free of charge, to any person 
# *  obtaining a copy of this software and associated documentation 
# *  files (the "Software"), to deal in the Software without 
# *  restriction, including without limitation the rights to use, 
# *  copy, modify, merge, publish, distribute, sublicense, and/or 
# *  sell copies of the Software, and to permit persons to whom the 
# *  Software is furnished to do so, subject to the following 
# *  conditions:
# *  
# *  The above copyright notice and this permission notice 
# *  shall be included in all copies or substantial portions 
# *  of the Software.
# *  
# *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
# *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
# *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
# *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
# *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
# *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
# *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
# *  DEALINGS IN THE SOFTWARE.
# *  
# */


#include "..\header\mpc5675k-0200.h"

#define 	VLE		2			 
#define 	DPM_KEY1        0x5AF0
#define 	DPM_KEY2        0xA50F
#define 	DRUN_MODE  	0x03

#define		BANDGAP		0
#define		LVD12V		1
#define		HVD12V		2
#define		REG12V		7
#define		LVDADC33V	3
#define		LVDIO33V	5
#define		LVDFLA33V	4
#define		LVDREG33V	6

extern void     _start          (void);
void            main0           (void);
void            main1           (void);
void            Toggle_LED      (int flag,unsigned long delay);
void 		Init_ADC0	(void);
int16_t 	Meas_ADC0	(int8_t chan);
void 		Init_ADC1	(void);
int16_t 	Meas_ADC1	(int8_t chan);
void            CLKOUT		(void);
void 		PLL		(void);
void 		ADC0_EOC	(void);
void 		ADC1_EOC	(void);

uint16_t i = 0, j = 0, Ptat = 0, Ctat = 0, VOL[8] = {0,0,0,0,0,0,0,0}, Out = 0;


/******************************************************************************
 * FUNCTION     : main0
 * DESCRIPTION  : main C entry point for core 0
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void main0(void)
{
    RGM.FES.B.F_FCCU_SAFE = 0x1;                        // Clear any fails 

    CLKOUT();                                           // Make clock signal visible on J45 pin 4
                                                        // 
    PLL();
    SSCM.DPMBOOT.R = (unsigned int) _start + VLE;       // start address for core 1
                                                        // and set up for VLE code

    SSCM.DPMKEY.R  = DPM_KEY1;                          // kick starts core1             
    SSCM.DPMKEY.R  = DPM_KEY2;                          // with these values
                                                        // core 1 now running

    xcptn_xmpl();					//interrupt

    INTC_0.PSR[62].R = 0x5;
//    INTC_1.PSR[62].R = 0x5;
    ADC_0.CIMR0.B.CIM0 = 0x1;
    ADC_0.IMR.B.MSKEOC = 0x1;
       
    Init_ADC0();

    ADC_0.CTR1.B.TSENSOR_SEL=0x0;
    Ptat = Meas_ADC0(15);
    ADC_0.CTR1.B.TSENSOR_SEL=0x1;
    Ctat = Meas_ADC0(15);
    ADC_0.CTR1.B.TSENSOR_SEL=0x0;
    
    
    
    PMC.CFGR.B.ADC_CHANNEL_SEL = BANDGAP;
    VOL[0] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = LVD12V;
    VOL[1] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = HVD12V;
    VOL[2] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = REG12V;
    VOL[3] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = LVDADC33V;
    VOL[4] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = LVDIO33V;
    VOL[5] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = LVDFLA33V;
    VOL[6] = Meas_ADC0(9);
    PMC.CFGR.B.ADC_CHANNEL_SEL = LVDREG33V;
    VOL[7] = Meas_ADC0(9);

    PMC.CFGR.B.ADC_CHANNEL_SEL = 0xF;

    SIU.PCR[33].B.APC = 0x1;
    Out = Meas_ADC0(2);
    SIU.PCR[34].B.APC = 0x1;
    Out = Meas_ADC0(3);
    SIU.PCR[70].B.APC = 0x1;
    Out = Meas_ADC0(4);
    SIU.PCR[66].B.APC = 0x1;
    Out = Meas_ADC0(5);
    SIU.PCR[71].B.APC = 0x1;
    Out = Meas_ADC0(6);
    
    SIU.PCR[59].R = 0x0220;                             // Enable LED1
    SIU.PCR[62].R = 0x0220;                             // Enable LED2

    SIU.PCR[23].B.APC = 0x1;
    while(1)
    {
        Toggle_LED(0,0x4FFFFF);
	j = Meas_ADC0(0);
    }

} 


/******************************************************************************
 * FUNCTION     : main1
 * DESCRIPTION  : main C entry point for core 1
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void main1(void)
{
    xcptn_xmpl1();					//interrupt
    Init_ADC1();
    INTC_1.PSR[82].R = 0x5;
    ADC_1.CIMR0.B.CIM0 = 0x1;
    ADC_1.IMR.B.MSKEOC = 0x1;
   
    SIU.PCR[99].R = 0x0220;                             // Enable LED3
    SIU.PCR[102].R = 0x0220;                             // Enable LED4

    Meas_ADC1(0);
    while(1)
    {
	i++;
        Toggle_LED(1,0x100000);
    }
}  


/******************************************************************************
 * FUNCTION     : Toggle_LED
 * DESCRIPTION  : toggles the motherboard LEDs [1..4]
 * INPUTS       : None
 * OUTPUTS      : None
 *****************************************************************************/
void Toggle_LED(int flag,unsigned long delay)
{
    unsigned long count;
    // Toggle LEDs on ports D(4,5,6,7)
    if (flag == 0)
    {
	SIU.GPDO[59].B.PDO ^= 1;                       // LED1
	SIU.GPDO[62].B.PDO ^= 1;                       // LED2
    }
    else if (flag ==1)
    {
        SIU.GPDO[99].B.PDO ^= 1;                    // LDE3
        SIU.GPDO[102].B.PDO ^= 1;                   // LED4
    }

    for(count=0;count<=delay;count++)
    {
        // delay duration provided by calling function 
    }
}

void Init_ADC0(void)
{
    ADC_0.MCR.B.MODE = 0x0;
    ADC_0.MCR.B.PWDN = 0x0;
}

void Init_ADC1(void)
{
    ADC_1.MCR.B.MODE = 0x0;
    ADC_1.MCR.B.PWDN = 0x0;
}

int16_t Meas_ADC0(int8_t chan)
{
    ADC_0.NCMR0.R|=(1<<chan);
    ADC_0.MCR.B.NSTART = 0x1;
    while(ADC_0.CDR[chan].B.VALID!=0x1);
    ADC_0.NCMR0.R&=(~(1<<chan));
    return ADC_0.CDR[chan].B.CDATA;
}

int16_t Meas_ADC1(int8_t chan)
{
    ADC_1.NCMR0.R|=(1<<chan);
    ADC_1.MCR.B.NSTART = 0x1;
    while(ADC_1.CDR[chan].B.VALID!=0x1);
    ADC_1.NCMR0.R&=(~(1<<chan));
    return ADC_1.CDR[chan].B.CDATA;
}

void ADC0_EOC(void)
{
	SIU.PCR[59].R = 0x0;
	ADC_0.ISR.B.EOC = 0x1;
}

void ADC1_EOC(void)
{
	SIU.PCR[99].R = 0x0;
	ADC_1.ISR.B.EOC = 0x1;
}

/***************************************************************
* FUNCTION : CLKOUT
* DESCRIPTION : Outputs chosen sysclk on PB6
* INPUTS : None
* OUTPUTS : None
****************************************************************/
void CLKOUT(void)
{

    SIU.PCR[22].R      = 0x0600;        /* Configure CLKOUT on B3 */
//    SIU.PCR[233].R     = 0x0600;        /* also send cloout on E20 */		
/*          Select 1 from below          */
//    CGM.OCDSSC.B.SELCTL = 3; 	   	 	// PLL1 is Output Clock
//    CGM.OCDSSC.B.SELCTL = 2; 	   	 	// PLL0 is Output Clock 
//    CGM.OCDSSC.B.SELCTL = 1; 	   	 	// OSC is Output Clock
    CGM.OCDS_SC.B.SELCTL = 2; 	   	 	// IRC is Output Clock 
    CGM.OCDS_SC.B.SELDIV = 3; 	   	 	// Divide Output Clock by 8
    CGM.OC_EN.B.EN       = 1; 	   	  	// Enable Output Clock

}
void PLL(void)
{
  ME.DRUN.B.XOSCON=1;			         /* Switch on external osc */
  
   CGM.CMU_0_CSR.B.RCDIV = 3;		    /* to avoid CMU reset in case of Fmosc<Frcosc */
   CGM.CMU_0_LFREFR_A.B.LFREF_A = 0x1;
   CGM.CMU_0_HFREFR_A.B.HFREF_A = 0xFFE;
  /* RE  enter the drun mode, to update the configuration */
  ME.MCTL.R = 0x30005AF0;                /* Mode & Key */
  ME.MCTL.R = 0x3000A50F;                /* Mode & Key inverted */
  
  while(ME.GS.B.S_MTRANS == 1);            /* Wait for mode entry to complete */
  while(ME.GS.B.S_CURRENTMODE != 3);       /* Check DRUN mode has been entered */

  while(!ME.GS.B.S_XOSC);		 		// Wait for clock to stabilise

  /* Selecting XOSC as the FMPLL_0 Source */
  CGM.AC3_SC.B.SELCTL = 0x1;
    /* Selecting XOSC as the FMPLL_1 Source */
  CGM.AC4_SC.B.SELCTL = 0x1;
  
  /* Max bus 120MHz, VCO 256-512Mhz */
  /* Fpll = XTAL * NDIV / IDF / ODF = 180MHz */
  /* Fvco = XTAL * NDIV / IDF = 360 MHz */
  /* Generation of 180MHz clock */
  CGM.FMPLL[0].CR.B.IDF = 0x3;			// Divide by 4
  CGM.FMPLL[0].CR.B.ODF = 0x0;		        // Divide by 2
  CGM.FMPLL[0].CR.B.NDIV = 36;		        // Multiply by 20  
  CGM.FMPLL[0].CR.B.EN_PLL_SW = 1;		

  /* Configure FMPLL_1 to generate 80MHz */
  CGM.FMPLL[1].CR.B.IDF = 0x3;	
  CGM.FMPLL[1].CR.B.ODF = 0x1;
  CGM.FMPLL[1].CR.B.NDIV = 32; 
  

  //CGM.SC_DC0.B.DE1 = 1;
  //CGM.SC_DC0.B.DIV1 = 1;    /*divide by 2 */

  /* Switching both the PLLs ON */
  ME.DRUN.B.PLL0ON  = 1;
  ME.DRUN.B.PLL1ON  = 1;
  
  /* RE enter the drun mode, to update the configuration */
  ME.MCTL.R = 0x30005AF0;     	         /* Mode & Key */
  ME.MCTL.R = 0x3000A50F;                /* Mode & Key inverted */

  while(ME.GS.B.S_MTRANS == 1);		         /* Wait for mode entry to complete */
  while(ME.GS.B.S_CURRENTMODE != 0x3);       /* Check DRUN mode has been entered */

  /* wait for PLL to lock - will not lock before DRUN re-entry */
  while(CGM.FMPLL[0].CR.B.S_LOCK == 0);
  while(CGM.FMPLL[1].CR.B.S_LOCK == 0);
  
  ME.DRUN.B.SYSCLK=0x4;                  	/* system clock is PLL */
  
  ME.MCTL.R = 0x30005AF0;     	         	/* Mode & Key */
  ME.MCTL.R = 0x3000A50F;                	/* Mode & Key inverted */
  while(ME.GS.B.S_MTRANS == 1);		  		/* Wait for mode entry to complete */
  while(ME.GS.B.S_CURRENTMODE != 0x3);      /* Check DRUN mode has been entered */

    /* ME_GS Poll Global status register to get current system clock
       fail if system clock is not pll
       0000 16MHz internal RC oscillator
       0001 divided 16MHz internal RC oscillator
       0010 4MHz crystal oscillator
       0011 divided 4MHz crystal oscillator
       0100 system PLL
       1111 system clock is disabled */
  while(ME.GS.B.S_SYSCLK != 4){};          /* fail if stuck here	 */
 
  /* Select Motor Clock Source */
   CGM.AC0_SC.B.SELCTL = 0x4;
   CGM.AC0_DC0.B.DE0 = 0x1;
   CGM.AC0_DC0.B.DIV0 = 0x0;
  
   /* Select Flexray Clock Source */
   CGM.AC1_SC.B.SELCTL = 0x5;				/* Select PLL_1 PHI */
   CGM.AC1_DC0.B.DE0 = 0x1;
   CGM.AC1_DC0.B.DIV0 = 0x1;
        
   /* Select CAN Clock Source */
   CGM.AC2_SC.B.SELCTL = 0x4;
   CGM.AC2_DC0.B.DE0 = 0x1;
   CGM.AC2_DC0.B.DIV0 = 0x0;

   /* Re-enter in DRUN mode to update */
   ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
   ME.MCTL.R = 0x3000A50F;     	  /* Mode & Key */

     /* waiting for end of mode change */
   while(ME.GS.B.S_MTRANS == 1){}; 	
   /* ME_GS Check DRUN mode has successfully been entered */
   while(ME.GS.B.S_CURRENTMODE != DRUN_MODE ){};

   	ME.RUNPC[0].B.DRUN = 1;
	ME.RUNPC[0].B.RUN0 = 1;
	ME.RUNPC[1].B.DRUN = 1;
	ME.RUNPC[1].B.RUN0 = 1;

	ME.DRUN.B.CFLAON = 3;
	ME.DRUN.B.PLL0ON = 1;		/* PLL1 enabled */
	ME.DRUN.B.PLL1ON = 1;		/* PLL2 enabled */
	ME.DRUN.B.XOSCON = 1;
	ME.DRUN.B.SYSCLK =0x4;		/* sysclk = PLL1 */


	ME.RUN[0].B.CFLAON = 3;

	ME.RUN[0].B.XOSCON = 1;
	ME.RUN[0].B.PLL0ON = 1;		/* PLL1 enabled */
	ME.RUN[0].B.PLL1ON = 1;		/* PLL2 enabled */

	ME.RUN[0].B.SYSCLK =0x4;	/* sysclk = PLL1 */

    ME.MCTL.R = 0x30005AF0;        		
    ME.MCTL.R = 0x3000A50F; 

}
