/**************************************************************************
* FILE NAME: etpu_rsvl.h                    COPYRIGHT (c) FREESCALE 2009  *
* DESCRIPTION:                                     All Rights Reserved    *
* This file contains the prototypes and defines for the eTPU Resolver     *
* Software Interface API.                                                 *
*=========================================================================*
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE
* ---   -----------  ----------    ---------------------
* 0.1      r29302     30/03/09     Initial file version.
* 1.0      r29302     25/04/09     First official release.
* 1.1   r54529       12/JAN/11     Separate dc offsets for sin and cos.
* 1.2   r54529       31/JAN/11     Resolver fault diagnostics added.
**************************************************************************/

#ifndef _ETPU_RSLV_H_
#define _ETPU_RSLV_H_

#include "etpu_rslv_auto.h"   /* Auto generated header file for eTPU RSLV */

/* Autocalibration constant */
#define FS_ETPU_RSLV_CALIB_AMPL          (0x007FFFFF * 8/10)  /* 0.8 */

/* Diagnostics constants */
#define FS_ETPU_RSLV_DIAG_ZERO_MAX       (0x007FFFFF * 1/10)  /* 0.1 */
#define FS_ETPU_RSLV_DIAG_MEAN_MAX       (0x007FFFFF * 1/10)  /* 0.1 */
#define FS_ETPU_RSLV_DIAG_AMPL_MAX       (0x007FFFFF * 88/100)  /* 0.88 */
#define FS_ETPU_RSLV_DIAG_ERROR_MAX      (0x007FFFFF * 1/10000)  /* 0.0001 */
#define FS_ETPU_RSLV_DIAG_UNIT_CIRC_MIN  (0x007FFFFF * (32-4)/100)  /* 0.28 */
#define FS_ETPU_RSLV_DIAG_UNIT_CIRC_MAX  (0x007FFFFF * (32+3)/100)  /* 0.35 */

/* Modes of operation */
/* Note that the operating modes flags are the same as state flags */
#define FS_ETPU_RSLV_MODE_PERIODIC_MSK   FS_ETPU_RSLV_STATE_PERIODIC_MSK
#define FS_ETPU_RSLV_MODE_PERIODIC       FS_ETPU_RSLV_STATE_PERIODIC
#define FS_ETPU_RSLV_MODE_NOPERIODIC     FS_ETPU_RSLV_STATE_NOPERIODIC
#define FS_ETPU_RSLV_MODE_ONREQUEST      FS_ETPU_RSLV_STATE_ONREQUEST

#define FS_ETPU_RSLV_MODE_LINK_MSK       FS_ETPU_RSLV_STATE_LINK_MSK
#define FS_ETPU_RSLV_MODE_LINK           FS_ETPU_RSLV_STATE_LINK
#define FS_ETPU_RSLV_MODE_NOLINK         FS_ETPU_RSLV_STATE_NOLINK

#define FS_ETPU_RSLV_MODE_CONTROL_MSK    FS_ETPU_RSLV_STATE_CONTROL_MSK
#define FS_ETPU_RSLV_MODE_CONTROL_MASTER FS_ETPU_RSLV_STATE_CONTROL_MASTER
#define FS_ETPU_RSLV_MODE_CONTROL_SLAVE  FS_ETPU_RSLV_STATE_CONTROL_SLAVE
#define FS_ETPU_RSLV_MODE_MASTER         FS_ETPU_RSLV_STATE_CONTROL_MASTER
#define FS_ETPU_RSLV_MODE_SLAVE          FS_ETPU_RSLV_STATE_CONTROL_SLAVE

#define FS_ETPU_RSLV_MODE_START_MSK      FS_ETPU_RSLV_STATE_START_MSK
#define FS_ETPU_RSLV_MODE_START_REL      FS_ETPU_RSLV_STATE_START_REL
#define FS_ETPU_RSLV_MODE_START_ABS      FS_ETPU_RSLV_STATE_START_ABS

#define FS_ETPU_RSLV_MODE_TIMEBASE_MSK   FS_ETPU_RSLV_STATE_TIMEBASE_MSK
#define FS_ETPU_RSLV_MODE_TIMEBASE_TCR1  FS_ETPU_RSLV_STATE_TIMEBASE_TCR1
#define FS_ETPU_RSLV_MODE_TIMEBASE_TCR2  FS_ETPU_RSLV_STATE_TIMEBASE_TCR2
#define FS_ETPU_RSLV_MODE_TCR1           FS_ETPU_RSLV_STATE_TIMEBASE_TCR1
#define FS_ETPU_RSLV_MODE_TCR2           FS_ETPU_RSLV_STATE_TIMEBASE_TCR2


/* Operating mode aliases */

#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK)
#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_LINK)

#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK)
#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_LINK)

#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR1  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK \
     | FS_ETPU_RSLV_MODE_TCR1)
#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR1  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_LINK \
     | FS_ETPU_RSLV_MODE_TCR1)

#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR1  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK \
     | FS_ETPU_RSLV_MODE_TCR1)
#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR1  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_LINK \
     | FS_ETPU_RSLV_MODE_TCR1)

#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR2  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK \
     | FS_ETPU_RSLV_MODE_TCR2)
#define FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR2  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_PERIODIC \
     | FS_ETPU_RSLV_MODE_LINK \
     | FS_ETPU_RSLV_MODE_TCR2)

#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR2  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_NOLINK \
     | FS_ETPU_RSLV_MODE_TCR2)
#define FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR2  \
    (  FS_ETPU_RSLV_MODE_MASTER \
     | FS_ETPU_RSLV_MODE_NOPERIODIC \
     | FS_ETPU_RSLV_MODE_LINK \
     | FS_ETPU_RSLV_MODE_TCR2)

#define FS_ETPU_RSLV_MODE_SLAVE_NOLINK \
    (  FS_ETPU_RSLV_MODE_SLAVE \
     | FS_ETPU_RSLV_MODE_NOLINK)
#define FS_ETPU_RSLV_MODE_SLAVE_LINK \
    (  FS_ETPU_RSLV_MODE_SLAVE \
     | FS_ETPU_RSLV_MODE_LINK)


/* Diagnostics - faults definition */
#define FS_ETPU_RSLV_FAULT_DYNAMICS               0x00000001
#define FS_ETPU_RSLV_FAULT_RATIONALITY            0x00000002
#define FS_ETPU_RSLV_FAULT_PLAUSABILITY_SIN       0x00000004
#define FS_ETPU_RSLV_FAULT_PLAUSABILITY_COS       0x00000008

#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_SINREFSIN 0x00000010
#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_COSREFCOS 0x00000020
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFSIN       0x00000040
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFCOS       0x00000080
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_COS          0x00000100
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFCOS       0x00000200
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFSIN       0x00000400
#define FS_ETPU_RSLV_FAULT_SHORT_REFSIN_REFCOS    0x00000800
#define FS_ETPU_RSLV_FAULT_OPEN_SINREFSIN         0x00001000
#define FS_ETPU_RSLV_FAULT_OPEN_COSREFCOS         0x00002000
#define FS_ETPU_RSLV_FAULT_GND_SINREFSIN          0x00004000
#define FS_ETPU_RSLV_FAULT_GND_COSREFCOS          0x00008000
#define FS_ETPU_RSLV_FAULT_OPENGND_REZGEN         0x00010000

/* Diagnostics - general fault masks */
#define FS_ETPU_RSLV_DYNAMICS_FAULT_MASK         (FS_ETPU_RSLV_DIAG_OBSERVER_ERROR)
#define FS_ETPU_RSLV_RATIONALITY_FAULT_MASK      (FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX)
#define FS_ETPU_RSLV_PLAUSABILITY_SIN_FAULT_MASK (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS)
#define FS_ETPU_RSLV_PLAUSABILITY_COS_FAULT_MASK (FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS)
#define FS_ETPU_RSLV_ALL_FAULTS_MASK             (FS_ETPU_RSLV_DYNAMICS_FAULT_MASK \
                                                | FS_ETPU_RSLV_RATIONALITY_FAULT_MASK \
                                                | FS_ETPU_RSLV_PLAUSABILITY_SIN_FAULT_MASK \
                                                | FS_ETPU_RSLV_PLAUSABILITY_COS_FAULT_MASK)

/* Diagnostics - specific fault masks and values */
#define FS_ETPU_RSLV_FAULT_OPEN_SINREFSIN_MASK   (FS_ETPU_RSLV_DIAG_SINA_POS \
                                                | FS_ETPU_RSLV_DIAG_SINA_NEG \
                                                | FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_OPEN_SINREFSIN_VALUE  (FS_ETPU_RSLV_DIAG_SINA_NEG \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_OPEN_COSREFCOS_MASK   (FS_ETPU_RSLV_DIAG_COSA_POS \
                                                | FS_ETPU_RSLV_DIAG_COSA_NEG \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_OPEN_COSREFCOS_VALUE  (FS_ETPU_RSLV_DIAG_COSA_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_GND_SINREFSIN_MASK    (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_GND_SINREFSIN_VALUE   (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_GND_COSREFCOS_MASK    (FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_GND_COSREFCOS_VALUE   (FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_OPENGND_REZGEN_MASK   (FS_ETPU_RSLV_DIAG_SINA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_SINA_POS \
                                                | FS_ETPU_RSLV_DIAG_SINA_NEG \
                                                | FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_COSA_POS \
                                                | FS_ETPU_RSLV_DIAG_COSA_NEG \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_OPENGND_REZGEN_VALUE  (FS_ETPU_RSLV_DIAG_SINA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_COSA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN )

#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_SINREFSIN_MASK (FS_ETPU_RSLV_DIAG_SINA_POS \
                                                | FS_ETPU_RSLV_DIAG_SINA_NEG \
                                                | FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_SINREFSIN_VALUE (FS_ETPU_RSLV_DIAG_SINA_POS \
                                                | FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_COSREFCOS_MASK (FS_ETPU_RSLV_DIAG_COSA_POS \
                                                | FS_ETPU_RSLV_DIAG_COSA_NEG \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_REZGEN_COSREFCOS_VALUE (FS_ETPU_RSLV_DIAG_COSA_POS \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFSIN_MASK (FS_ETPU_RSLV_DIAG_SINA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_SINA_POS \
                                                | FS_ETPU_RSLV_DIAG_SINA_NEG \
                                                | FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFSIN_VALUE (FS_ETPU_RSLV_DIAG_SINA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN)
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFCOS_MASK (FS_ETPU_RSLV_DIAG_COSA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_COSA_POS \
                                                | FS_ETPU_RSLV_DIAG_COSA_NEG \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFCOS_VALUE (FS_ETPU_RSLV_DIAG_COSA_ZERO \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN )
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_COS_MASK    (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_COS_VALUE   (FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN)
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFCOS_MASK (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_SIN_REFCOS_VALUE (FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFSIN_MASK (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_COS_REFSIN_VALUE (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_REFSIN_REFCOS_MASK (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_SIN_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_POS \
                                                | FS_ETPU_RSLV_DIAG_COS_MEAN_NEG \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MIN \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )
#define FS_ETPU_RSLV_FAULT_SHORT_REFSIN_REFCOS_VALUE (FS_ETPU_RSLV_DIAG_SINA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_COSA_AMPL \
                                                | FS_ETPU_RSLV_DIAG_UNIT_CIRCLE_MAX )


/* General macro for checking for an operating mode
 * FS_ETPU_RSLV_STATE_ISSET
 *   returns 1 if within the operating mode category pointed
 *   by msk is the operating mode is equal to what.
 *
 * Note that the macro is not the most effiecient and faster
 * methods exist.
 */
#define FS_ETPU_RSLV_STATE_ISSET(state, what, msk) \
    (((state) & (msk)) == what)

/**************************************************************************/
/*                       Function Prototypes                              */
/**************************************************************************/

/* Initialization */
int32_t fs_etpu_rslv_init( uint8_t channel,
                           uint8_t priority,
                           uint32_t mode,
                           uint32_t period,
                           uint32_t duty,
                           uint32_t delay,
                           int32_t start_offset,
                           uint32_t sample_delay,
                           int32_t gain_sin,
                           int32_t gain_cos,
                           int32_t dc_offset_sin,
                           int32_t dc_offset_cos,
                           uint8_t link_chan,
                           uint32_t rate_irq,
                           uint32_t rate_link,
                           int32_t k1_d,
                           int32_t k1_scale,
                           int32_t k2_d,
                           int32_t k2_scale);
/* Set duty cycle */
void fs_etpu_rslv_set_duty(uint8_t channel, uint32_t duty);

/* Set duty cycle, duty as percentage */
void fs_etpu_rslv_set_dutypercent(uint8_t channel, uint32_t dutypercent);

/* Set period */
void fs_etpu_rslv_set_period(uint8_t channel, uint32_t period);

/* Set delay */
void fs_etpu_rslv_set_delay(uint8_t channel, uint32_t delay);

/* Set delay, delay as percentage */
void fs_etpu_rslv_set_delaypercent(uint8_t channel, uint32_t delaypercent);

/* Update PWM parameters */
void fs_etpu_rslv_update(uint8_t channel);

/* Update PWM parameters - extended */
void fs_etpu_rslv_update_ext(uint8_t channel,
    uint32_t period, uint32_t duty, uint32_t delay);

/* Stop wave generation */
void fs_etpu_rslv_stop(uint8_t channel);

/* Compute once */
void fs_etpu_rslv_compute(uint8_t channel);

/* Reset Angle Tracking Observer */
void fs_etpu_rslv_reset(uint8_t channel);

/* Supply with sine and cosine resolver signals */
void fs_etpu_rslv_set_sincos(uint8_t channel, int32_t sina, int32_t cosa);

/* Get the address of sine, cosine parameters */
uint32_t* fs_etpu_rslv_get_sincos_addr(uint8_t channel);

/* Get position */
int32_t fs_etpu_rslv_get_position(uint8_t channel);

/* Set position */
void fs_etpu_rslv_set_position(uint8_t channel, int32_t position);

/* Get Angle */
int32_t fs_etpu_rslv_get_angle(uint8_t channel);

/* Get speed */
int32_t fs_etpu_rslv_get_speed(uint8_t channel);

/* Get number of revolutions */
int32_t fs_etpu_rslv_get_revolutions(uint8_t channel);

/* Set number of revolutions */
void fs_etpu_rslv_set_revolutions(uint8_t channel, int32_t revolutions);

/* Set steady-state */
void fs_etpu_rslv_set_steadystate(uint8_t channel,
    int32_t angle, int32_t sina, int32_t cosa);

/* Get Error */
int32_t fs_etpu_rslv_get_error(uint8_t channel);

/* get state */
uint32_t fs_etpu_rslv_get_state(uint8_t channel);

/* set state */
void fs_etpu_rslv_set_state(uint8_t channel, uint32_t state);

/* enables periodic computation */
void fs_etpu_rslv_enable_periodic(uint8_t channel);

/* disables periodic computation */
void fs_etpu_rslv_disable_periodic(uint8_t channel);

/* enables link at the end of computation */
void fs_etpu_rslv_enable_link(uint8_t channel);

/* disables link at the end of computation */
void fs_etpu_rslv_disable_link(uint8_t channel);

/* autocalibration of dc_offsets and gains */
uint8_t fs_etpu_rslv_autocalibrate(uint8_t channel);

/* turn on/off diagnostics */
void fs_etpu_rslv_diagnostics_on(uint8_t channel);
void fs_etpu_rslv_diagnostics_off(uint8_t channel);

/* read diagnostics */
uint32_t fs_etpu_rslv_get_diags_actual(uint8_t channel);
uint32_t fs_etpu_rslv_get_diags_cumulative(uint8_t channel);

/* read/clear fault */
uint32_t fs_etpu_rslv_get_fault(uint8_t channel);
void fs_etpu_rslv_clear_fault(uint8_t channel);

/* set Angle Tracking Algorithm coeficients */
void fs_etpu_rslv_set_coefs(uint8_t  channel, 
                            uint32_t k1_d,
                            uint32_t k1_scale,
                            uint32_t k2_d,
                            uint32_t k2_scale);


/* Operating mode aliases for which GCT load computation is defined
 *
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR1
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR1
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR1
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR1
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR2
 * FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR2
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR2
 * FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR2
 * FS_ETPU_RSLV_MODE_SLAVE_NOLINK
 * FS_ETPU_RSLV_MODE_SLAVE_LINK
 *
 * The default computation takes care of all other modes.
 *
 * NOTES:
 *   The computed cycle count is approximate. The differences of a few cycles
 *   due to testing LINK/NOLINK conditions are simply ignorred (anyway the
 *   performance figures provided in application note are 5..10% larger
 *   than the actual ones).
 */

/*******************************************************************************
*        Information for eTPU Graphical Configuration Tool
*******************************************************************************/
/* full function name: RSLV - Resolver Software Interface */
/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK); load: rslv_master_periodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK); load: rslv_master_periodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK); load: rslv_master_noperiodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK); load: rslv_master_noperiodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR1); load: rslv_master_periodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR1); load: rslv_master_periodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR1); load: rslv_master_noperiodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR1); load: rslv_master_noperiodic_nolink_tcr1
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_NOLINK_TCR2); load: rslv_master_periodic_nolink_tcr2
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_PERIODIC_LINK_TCR2); load: rslv_master_periodic_nolink_tcr2
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_NOLINK_TCR2); load: rslv_master_noperiodic_nolink_tcr2
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_MASTER_NOPERIODIC_LINK_TCR2); load: rslv_master_noperiodic_nolink_tcr2
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_SLAVE_NOLINK); load: rslv_slave_nolink
   channel=channel
*/

/* channel assignment when (mode=FS_ETPU_RSLV_MODE_SLAVE_LINK); load: rslv_slave_nolink
   channel=channel
*/

/* channel assignment ; load: rslv_default
   channel=channel
*/

/* load expression rslv_master_periodic_nolink_tcr1
var SIMULEVENT {0..1} [DO NOT EXIST 0, EXIST 1]
etpu_base_freq = etpu_tcr1_freq
period_ccnt = period
base = 100*etpu_base_freq/(period_ccnt*etpu_clock_freq)
cmpt_ccnt = 160
pwm_ccnt = if(SIMULEVENT==1,45,28)
channel = (cmpt_ccnt + pwm_ccnt)*base
*/

/* load expression rslv_master_noperiodic_nolink_tcr1
var COMPUTE_FREQUENCY {0..100000000} [Hz]
var SIMULEVENT {0..1} [DO NOT EXIST 0, EXIST 1]
etpu_base_freq = etpu_tcr1_freq
period_ccnt = period
base_pwm = 100*etpu_base_freq/(period_ccnt*etpu_clock_freq)
base_cmpt = 100*COMPUTE_FREQUENCY/etpu_clock_freq
cmpt_ccnt = 160
pwm_ccnt = if(SIMULEVENT==1,45,28)
channel = cmpt_ccnt*base_cmpt + pwm_ccnt*base_pwm
*/

/* load expression rslv_master_periodic_nolink_tcr2
var SIMULEVENT {0..1} [DO NOT EXIST 0, EXIST 1]
etpu_base_freq = etpu_tcr2_freq
period_ccnt = period
base = 100*etpu_base_freq/(period_ccnt*etpu_clock_freq)
cmpt_ccnt = 160
pwm_ccnt = if(SIMULEVENT==1,45,28)
channel = (cmpt_ccnt + pwm_ccnt)*base
*/

/* load expression rslv_master_noperiodic_nolink_tcr2
var COMPUTE_FREQUENCY {0..100000000} [Hz]
var SIMULEVENT {0..1} [DO NOT EXIST 0, EXIST 1]
etpu_base_freq = etpu_tcr2_freq
period_ccnt = period
base_pwm = 100*etpu_base_freq/(period_ccnt*etpu_clock_freq)
base_cmpt = 100*COMPUTE_FREQUENCY/etpu_clock_freq
cmpt_ccnt = 160
pwm_ccnt = if(SIMULEVENT==1,45,28)
channel = cmpt_ccnt*base_cmpt + pwm_ccnt*base_pwm
*/

/* load expression rslv_slave_nolink
var COMPUTE_FREQUENCY {0..100000000} [Hz]
period_ccnt = period
base = 100*COMPUTE_FREQUENCY/etpu_clock_freq
cmpt_ccnt = 160
channel = cmpt_ccnt*base
*/

/* load expression rslv_default
var MASTER {0..1} [MASTER 1, SLAVE 0]
var TCR {1..2} [TCR1 1, TCR2 2]
var COMPUTE_FREQUENCY {0..100000000} [Hz]
var PERIODIC {0..1} [NOPERIODIC 0, PERIODIC 1]
var SIMULEVENT {0..1} [DO NOT EXIST 0, EXIST 1]
etpu_base_freq = if(TCR==1,etpu_tcr1_freq,etpu_tcr2_freq)
period_ccnt = period
base_pwm = if(MASTER==1,100*etpu_base_freq/(period_ccnt*etpu_clock_freq),0)
base_cmpt_1 = 100*COMPUTE_FREQUENCY/etpu_clock_freq
base_cmpt = if(MASTER==1,if(PERIODIC==1,base_pwm,base_cmpt_1),base_cmpt_1)
cmpt_ccnt = 160
pwm_ccnt = if(SIMULEVENT==1,45,28)
channel = cmpt_ccnt*base_cmpt + pwm_ccnt*base_pwm
*/

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 *
 ********************************************************************/

#endif /* _ETPU_RSLV_H_ */