/*******************************************************************************
* FILE NAME: etpu_app_pmsmrsvc.h      COPYRIGHT (c) Freescale Semiconductor 2012  
*                                               All Rights Reserved     
*                                                                        
* DESCRIPTION:                                                          
* This file contains the prototypes and defines for the ETPU application
* PMSM with Resolver, Speed Vector Control API.
================================================================================
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE                 
* ---   -----------  ----------    ---------------------                 
* 1.0   M. Brejl     25/Apr/12     Initial version.  
*******************************************************************************/

#ifndef _FS_ETPU_APP_PMSMRSVC_H_
#define _FS_ETPU_APP_PMSMRSVC_H_

#include "etpu_pwmmac.h" /* PWMMAC */
#include "etpu_rslv.h"   /* RSLV */
#include "etpu_sc.h"     /* SC */
#include "etpu_bc.h"     /* BC */
#include "etpu_pmsmvc.h" /* PMSMVC */
#include "etpu_asac.h"   /* ASAC */

/*******************************************************************************
*                            Definitions                                 
*******************************************************************************/
/* PWM_phases_type options */
#define FS_ETPU_APP_PMSMRSVC_SINGLE_CHANNELS   FS_ETPU_PWMMAC_FULL_RANGE_SINGLE_CHANNELS  /* single channels */
#define FS_ETPU_APP_PMSMRSVC_COMPL_PAIRS       FS_ETPU_PWMMAC_FULL_RANGE_COMPL_PAIRS      /* complementary pairs */

/* direction options */
#define FS_ETPU_APP_PMSMRSVC_DIRECTION_INC     0  /* incremental direction */
#define FS_ETPU_APP_PMSMRSVC_DIRECTION_DEC     1  /* decremental direction */

/* SC saturation options */
#define FS_ETPU_APP_PMSMRSVC_SC_SATURATION_NO    FS_ETPU_SC_SATURATION_NO  /* no saturation */
#define FS_ETPU_APP_PMSMRSVC_SC_SATURATION_POS   FS_ETPU_SC_SATURATION_POS /* saturation to positiov limit */
#define FS_ETPU_APP_PMSMRSVC_SC_SATURATION_NEG   FS_ETPU_SC_SATURATION_NEG /* saturation to negative limit */

/* PMSMVC saturation options */
#define FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NO    FS_ETPU_PMSMVC_SATURATION_NO  /* no saturation */
#define FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_POS   FS_ETPU_PMSMVC_SATURATION_POS /* saturation to positiov limit */
#define FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NEG   FS_ETPU_PMSMVC_SATURATION_NEG /* saturation to negative limit */

/* SC configuration options */
#define FS_ETPU_APP_PMSMRSVC_SPEED_LOOP_OPENED   FS_ETPU_SC_RAMP_ON_PID_OFF /* speed loop opened */
#define FS_ETPU_APP_PMSMRSVC_SPEED_LOOP_CLOSED   FS_ETPU_SC_RAMP_ON_PID_ON  /* speed loop closed */

/* BC_mode options */
#define FS_ETPU_APP_PMSMRSVC_BC_MODE_ON_OFF    FS_ETPU_BC_INIT_SLAVE_ON_OFF  /* ON/OFF mode */
#define FS_ETPU_APP_PMSMRSVC_BC_MODE_PWM       FS_ETPU_BC_INIT_SLAVE_PWM     /* PWM mode */

/* BC_polarity options */
#define FS_ETPU_APP_PMSMRSVC_BC_ON_HIGH        ETPU_BC_ON_HIGH  /* ON high */
#define FS_ETPU_APP_PMSMRSVC_BC_ON_LOW         ETPU_BC_ON_LOW   /* ON low */

/* ASAC_polarity options */
#define FS_ETPU_APP_PMSMRSVC_ASAC_PULSE_HIGH     FS_ETPU_ASAC_PULSE_HIGH  /* Pulse high, low-high edge triggers */
#define FS_ETPU_APP_PMSMRSVC_ASAC_PULSE_LOW      FS_ETPU_ASAC_PULSE_LOW   /* Pulse low, high-low edge triggers

/* ASAC_bit_shift options */
#define FS_ETPU_APP_PMSMRSVC_ASAC_SHIFT_LEFT_BY_8    FS_ETPU_ASAC_SHIFT_LEFT_BY_8    /* left shift by 8 */
#define FS_ETPU_APP_PMSMRSVC_ASAC_SHIFT_LEFT_BY_10   FS_ETPU_ASAC_SHIFT_LEFT_BY_10   /* left shift by 10 */
#define FS_ETPU_APP_PMSMRSVC_ASAC_SHIFT_LEFT_BY_12   FS_ETPU_ASAC_SHIFT_LEFT_BY_12   /* left shift by 12 */
#define FS_ETPU_APP_PMSMRSVC_ASAC_SHIFT_LEFT_BY_16	  FS_ETPU_ASAC_SHIFT_LEFT_BY_16   /* left shift by 16 */

/* constant */
#define TWO_PI_0X200000   0x00C90FDB  /* 2 * pi * 2^21 */

/*******************************************************************************
*                pmsmrsvc_instance_t Structure Definition
********************************************************************************
*  This structure must be declared in the user application. 
*  In case there is more instances of the application running simultaneously
*  there must be a separate pmsmrsvc_instance_t structure for each one.
*
*  The structure is filled by the fs_etpu_app_pmsmrsvc_init function.
*  Consequently all the other functions use it.
*******************************************************************************/
typedef struct {
   uint8_t   PWM_master_channel;
   uint8_t   PWM_phaseA_channel;
   uint8_t   PWM_phaseB_channel;
   uint8_t   PWM_phaseC_channel;
   uint8_t   RSLV_channel;
   uint8_t   SC_channel;
   uint8_t   BC_channel;
   uint8_t   PMSMVC_channel;
   uint8_t   ASAC_channel;
   int32_t   speed_range_rpm;
   int32_t   speed_min_rpm;
   int32_t   dc_bus_voltage_range_mv;
   int32_t   phase_current_range_ma;
   uint8_t   motor_pole_pairs;
   uint8_t   rslv_pole_pairs;
   int32_t   PMSM_Ke_mv_per_krpm;
   uint32_t  BC_u_dc_bus_ON_perc;
   uint32_t  BC_u_dc_bus_OFF_perc;
   fract24_t ASAC_forget_factor_i;
   uint32_t  PWM_freq_hz;
} pmsmrsvc_instance_t;
   
/*******************************************************************************
*                  pmsmrsvc_data_t Structure Definition
********************************************************************************
*  speed_ramp_rpm      - This is the output from required speed ramp in rpm.
*  speed_actual_rpm    - This is the actual speed in rpm.
*  torque_required_mnm - This is the Speed Controller output - required
*                        torque in mili-Nm.
*  torque_actual_mnm   - This is the actual torque in mili-Nm.
*  applied_voltage_mv  - This is the applied motor voltage in mV.
*  revolution_counter - This is the bidirectional counter of motor 
*                       mechanical revolutions. 
*                       The counter is in signed 24-bit integer format,
*                       the bits 24-31 are always 0x00 or 0xFF.
*  direction          - This is the motor direction. It can be either
*                       FS_ETPU_APP_PMSMRSVC_DIRECTION_INC (0) or
*                       FS_ETPU_APP_PMSMRSVC_DIRECTION_DEC (1)
*  sc_saturation      - This is the Speed Controller saturation flag.
*                       It can be one of
*                       FS_ETPU_APP_PMSMRSVC_SC_SATURATION_NO 
*                       FS_ETPU_APP_PMSMRSVC_SC_SATURATION_POS
*                       FS_ETPU_APP_PMSMRSVC_SC_SATURATION_NEG
*  d_saturation       - This is the D-current saturation flag.
*                       It can be one of
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NO 
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_POS
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NEG
*  q_saturation       - This is the Q-current saturation flag.
*                       It can be one of
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NO 
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_POS
*                       FS_ETPU_APP_PMSMRSVC_PMSMVC_SATURATION_NEG
*******************************************************************************/
typedef struct {
   int32_t   speed_ramp_rpm;      /* output from required speed ramp in rpm */
   int32_t   speed_actual_rpm;    /* actual speed in rpm */
   int32_t   torque_required_mnm; /* required torque in mili-Nm */
   int32_t   torque_actual_mnm;   /* actual torque in mili-Nm */
   int32_t   applied_voltage_mv;  /* applied motor voltage in mV */
   int32_t   revolution_counter;  /* bidirectional 24-bit revolution counter */
   uint8_t   direction;           /* motor direction */
   uint8_t   sc_saturation;       /* speed controller saturation flag */
   uint8_t   d_saturation;        /* current d controller saturation flag */
   uint8_t   q_saturation;        /* current q controller saturation flag */
} pmsmrsvc_data_t;


/*******************************************************************************
*                       Function Prototypes                              
*******************************************************************************/
/* Initialize */
int32_t fs_etpu_app_pmsmrsvc_init(
  pmsmrsvc_instance_t * pmsmrsvc_instance,
              uint8_t   PWM_master_channel,
              uint8_t   PWM_phaseA_channel,
              uint8_t   PWM_phaseB_channel,
              uint8_t   PWM_phaseC_channel,
              uint8_t   RSLV_channel,
              uint8_t   SC_channel,
              uint8_t   BC_channel,
              uint8_t   PMSMVC_channel,
              uint8_t   ASAC_channel,
              uint8_t   PWM_phases_type,
              uint32_t  PWM_freq_hz,
              uint32_t  PWM_dead_time_ns,
              int32_t   speed_range_rpm,
              int32_t   speed_min_rpm,
              int32_t   dc_bus_voltage_range_mv,
              uint8_t   motor_pole_pairs,
              uint8_t   rslv_pole_pairs,
              uint32_t  SC_freq_hz,
              int32_t   SC_PID_gain_permil,
              int32_t   SC_I_time_const_us,
              uint32_t  SC_ramp_time_ms,
              int32_t   PMSMVC_D_PID_gain_permil,
              int32_t   PMSMVC_D_I_time_const_us,
              int32_t   PMSMVC_Q_PID_gain_permil,
              int32_t   PMSMVC_Q_I_time_const_us,
              int32_t   PMSM_Ke_mv_per_krpm,
              int32_t   PMSM_L_uH,
              int32_t   phase_current_range_ma,
              uint8_t   BC_mode,
              uint8_t   BC_polarity,
              uint8_t   BC_u_dc_bus_ON_perc,
              uint8_t   BC_u_dc_bus_OFF_perc,
              uint8_t   ASAC_polarity,
              uint32_t  ASAC_measure_time_us,
              uint32_t *ASAC_result_queue,
              uint8_t   ASAC_bit_shift,
              uint32_t  ASAC_filter_time_constant_i_us,
              uint32_t  ASAC_filter_time_constant_u_us);

/* Calibration Start */
int32_t fs_etpu_app_pmsmrsvc_calib_start( pmsmrsvc_instance_t * pmsmrsvc_instance);

/* Calibration Finish */
void fs_etpu_app_pmsmrsvc_calib_finish( pmsmrsvc_instance_t * pmsmrsvc_instance);

/* Alignment Start */
int32_t fs_etpu_app_pmsmrsvc_align_start( pmsmrsvc_instance_t * pmsmrsvc_instance,
                                                      int24_t   alignment_u_d);

/* Alignment Finish */
void fs_etpu_app_pmsmrsvc_align_finish( pmsmrsvc_instance_t * pmsmrsvc_instance,
                                                    uint8_t   sc_configuration);

/* Disable */
int32_t fs_etpu_app_pmsmrsvc_disable( pmsmrsvc_instance_t * pmsmrsvc_instance);

/* Set required speed */
void fs_etpu_app_pmsmrsvc_set_speed_required( pmsmrsvc_instance_t * pmsmrsvc_instance,
                                                          int32_t   speed_required_rpm);

/* Get data */
void fs_etpu_app_pmsmrsvc_get_data( pmsmrsvc_instance_t * pmsmrsvc_instance,
                                        pmsmrsvc_data_t * pmsmrsvc_data);
                                  
/*******************************************************************************
*        Information for eTPU Graphical Configuration Tool 
*******************************************************************************/
/* eTPU functions used: PWMMAC, PWMF, RSLV, SC, BC, PMSMVC, ASAC */
/* full function name: PMSM with Resolver, Speed Vector Control */
/* channel assignment when (PWM_phases_type=FS_ETPU_APP_PMSMRSVC_SINGLE_CHANNELS)
   PWM_master = PWM_master_channel; priority=Middle
   PWM_phaseA = PWM_phaseA_channel; priority=Middle
   PWM_phaseB = PWM_phaseB_channel; priority=Middle
   PWM_phaseC = PWM_phaseC_channel; priority=Middle
   RSLV = RSLV_channel; priority=Middle
   SC = SC_channel; priority=Low
   BC = BC_channel; priority=Low
   PMSMVC = PMSMVC_channel; priority=Low
   ASAC = ASAC_channel; priority=Middle
*/
/* channel assignment when (PWM_phases_type=FS_ETPU_APP_PMSMRSVC_COMPL_PAIRS)
   PWM_master = PWM_master_channel; priority=Middle
   PWM_phaseA_base_channel = PWM_phaseA_channel; priority=Middle
   PWM_phaseA_compl_channel = PWM_phaseA_channel + 1; priority=Middle
   PWM_phaseB_base_channel = PWM_phaseB_channel; priority=Middle
   PWM_phaseB_compl_channel = PWM_phaseB_channel + 1; priority=Middle
   PWM_phaseC_base_channel = PWM_phaseC_channel; priority=Middle
   PWM_phaseC_compl_channel = PWM_phaseC_channel + 1; priority=Middle
   RSLV = RSLV_channel; priority=Middle
   SC = SC_channel; priority=Low
   BC = BC_channel; priority=Low
   PMSMVC = PMSMVC_channel; priority=Low
   ASAC = ASAC_channel; priority=Middle
*/

#endif

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
