/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2010 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file     MC33937_routines.h
*
* @author   B15651
* 
* @version  1.0.1.0
* 
* @date     May-28-2010
* 
* @brief    MC33937 MOSFET pre-driver Init function header file, with routines
*           for setup and diagnostic.
*
*******************************************************************************
*
* Detailed Description of the file.
*
******************************************************************************/
#ifndef _MC33937_PARAM_H
#define _MC33937_PARAM_H

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

#define MPC5554

// deadtime value in [ns]
#define DEADTIME                    800

// define commands for MC33937 predriver

// MASK command
    /*
    MASK 0
    -------------------------------------------------------------------------------
    | 7 | 6 | 5 | 4 |     bit3     |       bit2       |     bit1     |    bit0    |
    -------------------------------------------------------------------------------
    | 0 | 0 | 1 | 0 | Over current | VLS undervoltage | Desaturation | Over Temp. |
    -------------------------------------------------------------------------------
    */
#define MASK0_CMD                   0x20

/*
    MASK 1
    ---------------------------------------------------------------------------
    | 7 | 6 | 5 | 4 |     bit3    |     bit2    |     bit1      |    bit0     |
    ---------------------------------------------------------------------------
    | 0 | 0 | 1 | 1 | Reset Event | Write Error | Framing Error | Phase Error |
    ---------------------------------------------------------------------------
    */
#define MASK1_CMD                   0x30

// CLINT command
// Clears portion of the fault latch corresponding to MASK0
#define CLINT0_CMD                  0x6F
// Clears portion of the fault latch corresponding to MASK1
#define CLINT1_CMD                  0x7F

/*
    MODE command
    ---------------------------------------------------------
    | 7 | 6 | 5 | 4 |     bit3    | 2 |  bit1   |    bit0   |
    ---------------------------------------------------------
    | 0 | 1 | 0 | 0 | Desat Fault | 0 | FULLON  | Mode Lock |
    |   |   |   |   |    mode     |   |  mode   |           |
    ---------------------------------------------------------
    */
#define MODE_CMD                    0x41

// DEADTIME command
#define DEADTIME_CMD                0x81

// STATUS register 0
#define STATUS0_CMD                 0x00
// STATUS register 0
#define STATUS1_CMD                 0x01
// STATUS register 0
#define STATUS2_CMD                 0x02
// STATUS register 0
#define STATUS3_CMD                 0x03

// define PIT CLK in MHz multiplyed by 16
#define PIT_CLK                     60 * 16

#define PIT_DELAY_2MS               0x0001F400
// timeout for stabilize power on MC33937 device = 2ms
#define MC33937_POWER_UP_TIME       0x00033000

#ifdef LEOPARD
/*! Macro containing motor CS values for MPC5643L Dual Motor Controller board */
#define MOTOR1_CS    0
#define MOTOR2_CS    1

/*! Macro containing number of the PCR register of the CS for motor 1 and 2 */
#define MOTOR1_CSPCR                36
#define MOTOR2_CSPCR                79
#endif

#ifdef PICTUS
/*! Macro containing motor CS values for MPC5604P Dual Motor Controller board */
#define MOTOR1_CS    0

/*! Macro containing number of the PCR register of the CS for motor 1 and 2 */
#define MOTOR1_CSPCR                83

#endif

#ifdef MPC5554
/*! Macro containing motor CS values for MPC5554 Motor Controller board */
#define MOTOR1_CS                    0
/*! Macro containing number of the DSPI module used for communication with predriver */
#define MOTOR1_DSPI_NB               0

/*! Macro containing number of the PCR register of the CS for motor 1 */
#define MOTOR1_CSPCR                96
/*! Macro containing number of the PCR register of the RESET input for motor 1 predriver */
#define PCR_RESET_NB               191
/*! Macro containing number of the PCR register of the ENEBLE input for motor 1 predriver */
#define PCR_ENABLE_NB              192

#endif



/*! Macro containing default value for setup MPC5643L dual motor controller board.*/
#define MPC5643L_BGA257_CB_M1_DEFAULT      {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR1_CS,0,MOTOR1_CSPCR,10,11,2}
#define MPC5643L_BGA257_CB_M2_DEFAULT      {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR2_CS,0,MOTOR2_CSPCR,10,11,2}

/*! Macro containing default values for setup MPC5604P controller board.*/
#define MPC5604P_TQFP144_CB_DEFAULT        {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR1_CS,3,MOTOR1_CSPCR,42,96,1}

/*! Macro containing default values for setup MPC5554 controller board.*/
#define MPC5554_BGA416_CB_DEFAULT        {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR1_CS,MOTOR1_DSPI_NB,MOTOR1_CSPCR,PCR_RESET_NB,PCR_ENABLE_NB,1}

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*//*!
@struct SR0_T "\MC33937_routines.h"

@brief  Structure containing status register 0 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    reset_event  :1;    /*!< reset event flag, is set upon exiting /RST */
        uint8_t    write_err    :1;    /*!< Write Error Afther the Lock flag */
        uint8_t    framing_err  :1;    /*!< Framing error flag */
        uint8_t    phase_err    :1;    /*!< Phase error flag */
        uint8_t    overcurrent  :1;    /*!< Over-current event flag */
        uint8_t    lowvls       :1;    /*!< Low VLS voltage flag */
        uint8_t    desat        :1;    /*!< DESAT flag detected on any channel */
        uint8_t    tlim         :1;    /*!< TLIM flag detected on any channel */
    } B;
}SR0_T;

/*------------------------------------------------------------------------*//*!
@struct SR1_T "\MC33937_routines.h"

@brief  Structure containing status register 1 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    :1;                  /*!< reserve */
        uint8_t    desat_mode   :1;     /*!< current state of the Desaturation/Phase Error turn-off mode */
        uint8_t    zds          :1;     /*!< Zero deadtime is commanded */
        uint8_t    calib_overfl :1;     /*!< flag for a Deadtime Calibration Overflow */
        uint8_t    deadtime_cal :1;     /*!< Deadtime calibration occurred */
        uint8_t    :1;                  /*!< reserve */
        uint8_t    fullon       :1;     /*!< present status of FULLON MODE */
        uint8_t    lockbit      :1;     /*!< LockBit indicates the IC regs are locked */
    } B;
}SR1_T;

/*------------------------------------------------------------------------*//*!
@struct SR2_T "\MC33937_routines.h"

@brief  Structure containing status register 2 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    mask1_3      :1;     /*!< status of the MASK1.3 bit */
        uint8_t    mask1_2      :1;     /*!< status of the MASK1.2 bit */
        uint8_t    mask1_1      :1;     /*!< status of the MASK1.1 bit */
        uint8_t    mask1_0      :1;     /*!< status of the MASK1.0 bit */
        uint8_t    mask0_3      :1;     /*!< status of the MASK0.3 bit */
        uint8_t    mask0_2      :1;     /*!< status of the MASK0.2 bit */
        uint8_t    mask0_1      :1;     /*!< status of the MASK0.1 bit */
        uint8_t    mask0_0      :1;     /*!< status of the MASK0.0 bit */
    } B;
}SR2_T;

/*------------------------------------------------------------------------*//*!
@struct MC33937_SR_T "\MC33937_routines.h"

@brief  Structure containing status registers image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef struct
{
    SR0_T       sr0;    /*!< status register 0 */
    SR1_T       sr1;    /*!< status register 1 */
    SR2_T       sr2;    /*!< status register 2 */
    uint8_t     sr3;    /*!< status register 3 */
}MC33937_SR_T;

/*------------------------------------------------------------------------*//*!
@struct MC33937_SET_T "\MC33937_routines.h"

@brief  Structure containing configuration setup for the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef struct
{
    MC33937_SR_T    status;         /*!< image of the Status registers */
    uint16_t        Set_deadtime;   /*!< real dead time of HS and LS transistors */
    uint16_t        deadtime;       /*!< define dead time of HS and LS transistors */
    uint8_t         mode;           /*!< define value of the MODE register */
    uint8_t         mask0;          /*!< define value of the MASK0 register */
    uint8_t         mask1;          /*!< define value of the MASK1 register */
    uint8_t         PITChannel;     /*!< define number of the PIT channel used for generation delays */
    uint8_t         DspiCSsignal;   /*!< define number of the CS signal used for communication */
    uint8_t         DspiNb;         /*!< define number of the DSPI module used for communication */
    uint8_t         CSPcrNb;        /*!< define number of the PCR register used for CS signal on the CB */
    uint8_t         RstPcrNb;       /*!< define number of the PCR register used for /RESET signal on the CB */
    uint8_t         EnPcrNb;        /*!< define number of the PCR register used for ENABLE signal on the CB */
    uint8_t         CBtype;         /*!< define type of the used CB */
}MC33937_SET_T;




// ****************************************************************************
/*! 
* @ingroup MC33937_GROUP 
*/ 
extern  void       MC33937_clear_fault(MC33937_SET_T *ptr);
extern  void       MC33937_enable(MC33937_SET_T *ptr);
extern  void       MC33937_disable(MC33937_SET_T *ptr);
extern  void       MC33937_read_SR(MC33937_SET_T *SetPtr);
extern  void       MC33937_deadtime_setup(MC33937_SET_T *SetPtr);
extern  void       MC33937_config(MC33937_SET_T *SetPtr);
static	void 	     initFIT(void);
static  uint32_t   readFIT(void);

#endif  /* _MC33937_PARAM_H */
// ****************************************************************************
