/******************************************************************************
*                                                                     
*       COPYRIGHT (c) Freescale 2009            
*       Freescale Confidential Proprietary								              
*														              
* Filename:     main.c                
*														              
* Revision:      										              
*														              
* Functions:    Sample application for Real Time Software Clock for LG32
*         
* Description:  .
*
* Notes:       
*
******************************************************************************/

/****************************************************************
 * HEADERS
 ****************************************************************/
#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "RTC.h"

/****************************************************************
 * MACROS
 ****************************************************************/
/*Register value Macros*/
#define SOPTI_VALUE         0x63 /*Disable Watchdog, Enable Stop Mode, BKGD and 
                                  Reset Pin Enable*/
#define SPMSC2_VALUE        0x30 /*Selecting the Low-Voltage Warning and Detect 
                                  Voltages*/
#define RTCSC_RTCLKS_VALUE  0x01 /*Real-time clock srce is the external clock*/

#define RTC_PRESCALER_VALUE 0x01 /*Prescaler value = 2^10*/

#define CLEAR_RTC_INTERRUPT 0x1  /*Clear the RTC interrupt*/

#define RTCMOD_NORMAL_VALUE 0x1F /*RTCMOD value in normal mode*/

#define RTCMOD_LP_VALUE     0xFF /*RTCMOD value in Stop3 mode*/

/****************************************************************
 * ENUM
 ****************************************************************/
typedef enum 
{
  STOP_MODE   = 0 ,  
  NORMAL_MODE = 1
}OPERATING_MODE_E;

typedef enum 
{
  DISABLE = 0,
  ENABLE
}CONFIGURE_BITS_E;

/****************************************************************
 * STRUCTURES
 ****************************************************************/
typedef struct 
{
  unsigned char hours;
  unsigned char minutes;
  unsigned char seconds;
  float milliseconds;
}CLK_S;

/****************************************************************
 * GLOBAL VARIABLES (STATIC)
 ****************************************************************/
static CLK_S rtclock;
static OPERATING_MODE_E code_state = STOP_MODE;

/****************************************************************
 * GLOBAL VARIABLES
 ****************************************************************/
unsigned char CorrectionFlag = 0;

/****************************************************************
 * FUNCTIONS
 ****************************************************************/

/*****************************************************************************
 * Function:		System_Init
 *
 * Description: Initializes the overall system    
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void System_Init(void)
{
  /*System Options Register 1*/
  SOPT1 = SOPTI_VALUE; /*Disable Watchdog, Enable Stop Mode, BKGD and Reset
                       Pin Enable*/

  /*System Power Management Status and Control registers*/
  SPMSC1_LVDRE = DISABLE;   /*Low-Voltage Detect Reset Disable*/
  SPMSC2 = SPMSC2_VALUE;    /*Selecting the Low-Voltage Warning and Detect 
                              Voltages*/

  /*Wait for oscillator to Initialize*/
  while(!ICSSC_OSCINIT);
}

/*****************************************************************************
 * Function:		RTC_Init
 *
 * Description: Initializes the system and RTC clock
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void RTC_Init(void)
{
  /*External Crystal Confiuration - PLL*/
  ICSC2_EREFS = ENABLE;   /*Oscillator requested*/
  ICSC2_ERCLKEN = ENABLE; /*External Reference Enabled*/
  ICSC2_EREFSTEN = 1;     /*External reference clock src stays enabled in stop*/

  /*Internal Crystal Confiuration - Disable*/  
  ICSC1_IREFS = 0;        /*External reference clock selected*/
  ICSC1_IRCLKEN = 0;      /*ICSIRCLK inactive*/
  ICSC1_IREFSTEN = 0;     /*Internal reference clock is disabled in stop*/
  
  /*Clock Gating Register*/
  SCGC1_RTC = ENABLE;     /*Bus clock to the RTC module is enabled.*/
  RTCSC_RTCLKS = RTCSC_RTCLKS_VALUE;    /*external clock src selected*/
  RTCSC_RTCPS = RTC_PRESCALER_VALUE;    /*load the prescaler value*/
  RTCSC_RTIF = CLEAR_RTC_INTERRUPT;     /*Clear the interrupt*/
  RTCSC_RTIE = ENABLE;                  /*Enable the interrupt*/
  
  if(code_state == NORMAL_MODE)
  {
    RTCMOD = RTCMOD_NORMAL_VALUE;
  }
  else
  {
    RTCMOD = RTCMOD_LP_VALUE;
  }

  rtclock.hours   = RTC_HOURS;
  rtclock.minutes = RTC_MINUTES;
  rtclock.seconds = RTC_SECONDS;
  rtclock.milliseconds = 0;
}

/*****************************************************************************
 * Function:		IRQ_Init
 *
 * Description: Initializes the IRQ line, which is connected to Ignition, 
 *              thorugh a porper singal conditioning on the pin
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void IRQ_Init()
{
  
  SCGC2_IRQ = ENABLE;   /*Bus clock to the IRQ is enabled.*/
  
  IRQSC_IRQACK = 0x1;
  IRQSC = 0x56;
}

/*****************************************************************************
 * Function:		ClockCorrection
 *
 * Description: Provides a correction, and should be changed as per requirement 
 *              of the crystal
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void ClockCorrection(void) 
{
  if(rtclock.hours % 8 == 0 && CorrectionFlag == 1)
  {
    CorrectionFlag = 0;
    rtclock.seconds++;  /*Compensation of 1s per 8 hours*/
  }
}

/*****************************************************************************
 * Function:		UpdateAndDisplayTime
 *
 * Description: Updates the Clock structure, everytime RTC interrupt occurs 
 *              or ignition is received 
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void UpdateAndDisplayTime(void)
{
  if(1000 <= rtclock.milliseconds)
  {
    rtclock.milliseconds -= 1000;
    rtclock.seconds += 1;
  }
  
  if(60 <= rtclock.seconds)
  {
    rtclock.minutes += 1;
    rtclock.seconds -= 60;
  }
  if(60 <= rtclock.minutes)
  {
    rtclock.hours += 1;
    rtclock.minutes = 0;
    if(  rtclock.hours %8 == 0)  /*Compensation of 1s per 8 hours*/
      CorrectionFlag = 1;
  }
  if(24 <= rtclock.hours)
  {
    rtclock.hours = 0;
  }

  ClockCorrection();
}

/*****************************************************************************
 * Function:		main
 *
 * Description: MAIN FUNCTION
 *              
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void main(void) 
{

  RTC_Init();
  
  EnableInterrupts; /* enable interrupts */

  code_state = NORMAL_MODE;

  System_Init();
  
  IRQ_Init();
  
  UpdateAndDisplayTime();
  
  for(;;) {
  if(STOP_MODE == code_state)
  {
    if(RTCMOD_NORMAL_VALUE == RTCMOD)
    {
      rtclock.milliseconds += ((RTCCNT+1)*31.25);
      RTCMOD = RTCMOD_LP_VALUE;
    }
    asm
    {
      stop
    }
  }
  else
  {
    
  }
    __RESET_WATCHDOG(); /* feeds the dog */
  } /* loop forever */
  /* please make sure that you never leave main */
}

/*****************************************************************************
 * Function:		RTC_ISR
 *
 * Description: ISR for RTC
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void interrupt VectorNumber_Vrtc RTC_ISR()
{
  if(STOP_MODE == code_state)
  {
    rtclock.seconds += 8;
  }
  else
  {
    rtclock.seconds += 1;
  }

  UpdateAndDisplayTime();
  RTCSC_RTIF = 1;
}

/*****************************************************************************
 * Function:		IRQ_ISR
 *
 * Description: ISR for ignition
 *              
 *
 * Returns: 		None 
 *
 * Notes:
 *
 *****************************************************************************/
void interrupt VectorNumber_Virq IRQ_ISR()
{
  rtclock.milliseconds += ((RTCCNT+1)*31.25);
  RTCMOD = RTCMOD_NORMAL_VALUE;

  UpdateAndDisplayTime();

  IRQSC_IRQACK = 0x1;  /*Clear the flag*/

  if(code_state == NORMAL_MODE)
  {
    code_state = STOP_MODE;
  }
  else
  {
    code_state = NORMAL_MODE;
  }
}

/*****************************************************************************
 * END OF FILE
 *****************************************************************************/