/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2010 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
********************************************************************************
*
* File Name:  sys.h
*
* $Date:      Oct-12-2010$
*
* $Version:   2.5.6.0$
*
* Description: SIM, low voltage interrupt (LVI) and various system 
*              support macros and definitions for MC56F824x/5x
*
*****************************************************************************/

#ifndef __SYS_H
#define __SYS_H

/* qs.h is a master header file, which must be included */
#if !defined(__ARCH_H) || !defined(__PERIPH_H) || !defined(__APPCONFIG_H)
#error Please include qs.h before sys.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************
* system module identifiers, correspond to module base address
*************************************************************/

#define SYS (&ArchIO.Sim)
#define SIM (&ArchIO.Sim)
#define LVI (&ArchIO.Lvi)

/***********************************************************************
* SYS static configuration items for appconfig.h
***********************************************************************/

/*
    initial register values:

        #ifdef SIM_CONROL_INIT
        #ifdef SIM_CLKOSR_INIT
        #ifdef SIM_GPS0_INIT
        #ifdef SIM_GPS1_INIT
        #ifdef SIM_GPS2_INIT
        #ifdef SIM_GPS3_INIT
        #ifdef SIM_PCR_INIT
        #ifdef SIM_SD0_INIT
        #ifdef SIM_SD1_INIT
        #ifdef SIM_SD2_INIT
        #ifdef SIM_PCE0_INIT
        #ifdef SIM_PCE1_INIT
        #ifdef SIM_PCE2_INIT
        #ifdef SIM_PROT_INIT
        #ifdef LVI_CONTROL_INIT

    #define INTERRUPT_VECTOR_ADDR_yy  userISR //specify interrupt handler routine : e.g. lviISR
    #define INT_PRIORITY_LEVEL_yy     INTC_xxx // interrupt level : (INTC_DISABLED, INTC_LEVEL0,
                                               //                    INTC_LEVEL1 or INTC_LEVEL2)
  where:
         yy  is interrupt vector number
            e.g. On 56F8257
                14 - Low Voltage Detector (Power sense)

*/

/*****************************************************************
* Single-Instruction  ioctl() commands
*
*  Word16 ioctl( SYS_module_identifier, Command_name, Parameter );
*
******************************************************************

        COMMAND NAME                      PARAMETERS, COMMENTS
------------------------------------------------------------------ */
#define SYS_SOFTWARE_RESET         /* NULL, issue software reset (CTRL->SWRST)*/
#define SYS_ONCE                   /* SYS_ENABLE/SYS_DISABLE, OnCE module enable (CTRL->ONCEEBL) */

#define SYS_ENABLE_IN_STOP_REG2    /* combination of SYS_xxx_MOD2, enable  modules in STOP mode (SD2) */
#define SYS_DISABLE_IN_STOP_REG2   /* combination of SYS_xxx_MOD2, disable modules in STOP mode (SD2) */

#define SYS_PERIPH_CLK_REG2_ENABLE  /* combination of SYS_xxx_MOD2, enable peripheral clock  (PCE2) */
#define SYS_PERIPH_CLK_REG2_DISABLE /* combination of SYS_xxx_MOD2, disable peripheral clock (PCE2) */

#define SYS_HS_CLOCK_ENABLE        /* combination of SYS_HS_xxx (TMRA/TMRB/SCI0/SCI1), enable high-speed clock */
#define SYS_HS_CLOCK_DISABLE       /* combination of SYS_HS_xxx (TMRA/TMRB/SCI0/SCI1), enable high-speed clock */
                                      
#define SYS_SET_A0PAD_FUNCTION     /* one of SYS_A0PAD_xxx (ANA0_VREFHA/CMPC_O), package pin function selection */

#define SYS_SET_C15PAD_FUNCTION    /* one of SYS_C15PAD_xxx (SCL0/XB_OUT1), package pin function selection */
#define SYS_SET_C14PAD_FUNCTION    /* one of SYS_C14PAD_xxx (SDA0/XB_OUT0), package pin function selection */
#define SYS_SET_C13PAD_FUNCTION    /* one of SYS_C13PAD_xxx (TA3/XB_IN6), package pin function selection */
#define SYS_SET_C9PAD_FUNCTION     /* one of SYS_C9PAD_xxx (SCLK0/XB_IN4), package pin function selection */
#define SYS_SET_C8PAD_FUNCTION     /* one of SYS_C8PAD_xxx (MISO0/RXD0), package pin function selection */
#define SYS_SET_C7PAD_FUNCTION     /* one of SYS_C7PAD_xxx (SSB0_B/TXD0), package pin function selection */
#define SYS_SET_C5PAD_FUNCTION     /* one of SYS_C5PAD_xxx (DACO/XB_IN7), package pin function selection */
#define SYS_SET_C4PAD_FUNCTION     /* one of SYS_C4PAD_xxx (TA1/CMPB_O), package pin function selection */
#define SYS_SET_C0PAD_FUNCTION     /* one of SYS_C0PAD_xxx (XTAL/CLKIN), package pin function selection */

#define SYS_SET_E7PAD_FUNCTION     /* one of SYS_E7PAD_xxx (PWMA3/XB_IN5), package pin function selection */
#define SYS_SET_E6PAD_FUNCTION     /* one of SYS_E6PAD_xxx (PWMB3/XB_IN4), package pin function selection */
#define SYS_SET_E5PAD_FUNCTION     /* one of SYS_E5PAD_xxx (PWMA2/XB_IN3), package pin function selection */
#define SYS_SET_E4PAD_FUNCTION     /* one of SYS_E4PAD_xxx (PWMB2/XB_IN2), package pin function selection */

#define SYS_SET_F8PAD_FUNCTION     /* one of SYS_F8PAD_xxx (RXD0/TB1), package pin function selection */
#define SYS_SET_F6PAD_FUNCTION     /* one of SYS_F6PAD_xxx (PWMX3/TB2), package pin function selection */
#define SYS_SET_F5PAD_FUNCTION     /* one of SYS_F5PAD_xxx (RXD1/XB_OUT5), package pin function selection */
#define SYS_SET_F4PAD_FUNCTION     /* one of SYS_F4PAD_xxx (XB_OUT4/TXD1), package pin function selection */
#define SYS_SET_F3PAD_FUNCTION     /* one of SYS_F3PAD_xxx (SDA1/XB_OUT3), package pin function selection */
#define SYS_SET_F2PAD_FUNCTION     /* one of SYS_F2PAD_xxx (SCL1/XB_OUT2), package pin function selection */
#define SYS_SET_F1PAD_FUNCTION     /* one of SYS_F1PAD_xxx (CLKOUT/XB_IN7), package pin function selection */

#define SYS_SET_TMRB3_INPUT        /* one of SYS_TMRB3_xxx (GPIOF7/XB_OUT29), TMRB3 input selection */
#define SYS_SET_TMRB2_INPUT        /* one of SYS_TMRB2_xxx (GPIOF6/XB_OUT28), TMRB2 input selection */
#define SYS_SET_TMRB1_INPUT        /* one of SYS_TMRB1_xxx (GPIOF8/XB_OUT27), TMRB1 input selection */
#define SYS_SET_TMRB0_INPUT        /* one of SYS_TMRB0_xxx (GPIOC2/XB_OUT26), TMRB0 input selection */

#define SYS_WRITE_SW_CONTROL_REG0  /* UWord16, write SIM software control register 0 (SCR0) */
#define SYS_READ_SW_CONTROL_REG0   /* NULL, returns UWord16 (SCR0) */
#define SYS_WRITE_SW_CONTROL_REG1  /* UWord16, write SIM software control register 1 (SCR1) */
#define SYS_READ_SW_CONTROL_REG1   /* NULL, returns UWord16 (SCR1) */
#define SYS_WRITE_SW_CONTROL_REG2  /* UWord16, write SIM software control register 2 (SCR2) */
#define SYS_READ_SW_CONTROL_REG2   /* NULL, returns UWord16 (SCR2) */
#define SYS_WRITE_SW_CONTROL_REG3  /* UWord16, write SIM software control register 3 (SCR3) */
#define SYS_READ_SW_CONTROL_REG3   /* NULL, returns UWord16 (SCR3) */

#define SYS_READ_LSH_JTAG_ID       /* NULL, returns JTAG ID as UWord16 (MSHID,LSHID) */
#define SYS_READ_MSH_JTAG_ID       /* NULL, returns JTAG ID as UWord16 (MSHID,LSHID) */

#define SYS_CLKOUT                 /* SYS_ENABLE/SYS_DISABLE, enable CLKOUT pin */

#define LVI_INT_ENABLE               /* LVI_22V_LEVEL|LVI_27V_LEVEL, low voltage interrupt enable */
#define LVI_INT_DISABLE              /* LVI_22V_LEVEL|LVI_27V_LEVEL, low voltage interrupt disable */
#define LVI_CLEAR_LOW_VOLTAGE_INT    /* LVI_INT|LVI_22V_LEVEL|LVI_27V_LEVEL, clear interrupt flags */



/*****************************************************************
* Multi-Instruction ioctl() commands
*
*  Word16 ioctl( SYS_module_identifier, Command_name, Parameter );
*
******************************************************************

        COMMAND NAME                      PARAMETERS, COMMENTS
------------------------------------------------------------------ */
#define SYS_STOP                   /* SYS_ENABLE(_PERMANENT)/SYS_DISABLE(_PERMANENT); "stop" instruction ena/dis/perm (CTRL->STOPDIS) */
#define SYS_WAIT                   /* SYS_ENABLE(_PERMANENT)/SYS_DISABLE(_PERMANENT); "wait" instruction ena/dis/perm (CTRL->WAITDIS) */

#define SYS_ENABLE_IN_STOP         /* combination of SYS_xxx_MOD, enable  modules in STOP mode (SD0, SD1) */
#define SYS_DISABLE_IN_STOP        /* combination of SYS_xxx_MOD, disable modules in STOP mode (SD0, SD1) */

#define SYS_PERIPH_CLK_ENABLE      /* combination of SYS_xxx_MOD, enable peripheral clock  (PCE0, PCE1) */
#define SYS_PERIPH_CLK_DISABLE     /* combination of SYS_xxx_MOD, disable peripheral clock (PCE0, PCE1) */

#define SYS_SET_C12PAD_FUNCTION    /* one of SYS_C12PAD_xxx (RX0/SDA1/RXD1), package pin function selection */
#define SYS_SET_C11PAD_FUNCTION    /* one of SYS_C11PAD_xxx (TX0/SCL1/TXD1), package pin function selection */
#define SYS_SET_C10PAD_FUNCTION    /* one of SYS_C10PAD_xxx (MOSI0/XB_IN5/MISO0), package pin function selection */
#define SYS_SET_C6PAD_FUNCTION     /* one of SYS_C6PAD_xxx (TA2/XB_IN3/CMPREF), package pin function selection */
#define SYS_SET_C3PAD_FUNCTION     /* one of SYS_C3PAD_xxx (TA0/CMPA_O/RXD0), package pin function selection */
#define SYS_SET_C2PAD_FUNCTION     /* one of SYS_C2PAD_xxx (TXD0/TB0/XB_IN2/CLKO), package pin function selection */

#define SYS_SET_POWER_MODE         /* SYS_NORMAL_POWER/SYS_REDUCED_POWER | SYS_POWER_MODE_PERMANENT */
#define SYS_GET_POWER_MODE         /* NULL, returns UWord16 */

#define SYS_TEST_RESET_SOURCE      /* any of SYS_xxx_RESET (SW/COP/COP_TOR/COP_LOR/EXTERN/POWER_ON/ANY), get/test type of the previous RESET  (RSTAT) */
#define SYS_CLEAR_RESET_SOURCE     /* any of SYS_xxx_RESET (SW/COP/COP_TOR/COP_LOR/EXTERN/POWER_ON/ANY), clear apropriate bit in reset staus register (RSTAT) */

#define SYS_CLKOUT_SELECT          /* one of SYS_CLKOUT_xxx (SYSCLK/IPBCLK/HSCLK/MSTRCLK), select CLSKOUT source */

#define SYS_WRITE_IO_SHORT_ADDR_LOCATION_REG  /* UWord32, set I/O short address mode base address */
#define SYS_READ_IO_SHORT_ADDR_LOCATION_REG   /* NULL, returns UWord32 */

#define SYS_WPROTECT_CLOCK_SETTINGS /* SYS_ENABLE(_PERMANENT)/SYS_DISABLE(_PERMANENT); write protect PCE, SD and PCR */
#define SYS_WPROTECT_SIGNALS_ROUTING /* SYS_ENABLE(_PERMANENT)/SYS_DISABLE(_PERMANENT); write protect GPSx, XBAR, GPIO_X_PER, GPIO_X_PPMODE, GPIO_X_DRIVE and GIO_X_IFE registers}  */

#define LVI_GET_LOW_VOLTAGE          /* LVI_22V_LEVEL|LVI_27V_LEVEL, test low voltage interrupt flags */
#define LVI_GET_NONSTICKY_INT_SOURCE /* LVI_22V_LEVEL|LVI_27V_LEVEL */



/*****************************************************************
* Software Layer ioctl() commands
*
*  Word16 ioctl( SYS_module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define SYS_INIT                   /* NULL, SIM & LVI registers initialization acc.to appconfig.h */



/****************************************************************************
* SYS constants and type declarations used in the ioctl functions
*****************************************************************************/

#define SYS_DISABLE           0
#define SYS_ENABLE            1

/* for permanent enable/disable */
#define SYS_DISABLE_PERMANENT 2
#define SYS_ENABLE_PERMANENT  3

/* High Speed clock enable in modules */
#define SYS_HS_TMRB  SIM_PCR_TMRB
#define SYS_HS_TMRA  SIM_PCR_TMRA
#define SYS_HS_SCI0  SIM_PCR_SCI0
#define SYS_HS_SCI1  SIM_PCR_SCI1

/* reset sources */
#define SYS_SW_RESET        SIM_RSTSTS_SWR  
#define SYS_COP_RESET       (SIM_RSTSTS_COP_TOR | SIM_RSTSTS_COP_LOR)
#define SYS_COP_TOR_RESET   SIM_RSTSTS_COP_TOR
#define SYS_COP_LOR_RESET   SIM_RSTSTS_COP_LOR
#define SYS_EXTERN_RESET    SIM_RSTSTS_EXTR
#define SYS_POWER_ON_RESET  SIM_RSTSTS_POR  
#define SYS_ALL_RESETS      (SIM_RSTSTS_SWR | SIM_RSTSTS_COP_TOR | SIM_RSTSTS_COP_LOR | SIM_RSTSTS_EXTR | SIM_RSTSTS_POR)
#define SYS_ANY_RESET       SYS_ALL_RESETS

/* large regulator modes */
#define SYS_NORMAL_POWER            0
#define SYS_REDUCED_POWER           SIM_POWER_LRSTDBY0
#define SYS_POWER_MODE_PERMANENT    SIM_POWER_LRSTDBY1

/* module clock-enables and modules in stop */
#define SYS_GPIOF_MOD   ((UWord32)(SIM_PCE0_GPIOF))
#define SYS_GPIOE_MOD   ((UWord32)(SIM_PCE0_GPIOE))
#define SYS_GPIOD_MOD   ((UWord32)(SIM_PCE0_GPIOD))
#define SYS_GPIOC_MOD   ((UWord32)(SIM_PCE0_GPIOC))
#define SYS_GPIOB_MOD   ((UWord32)(SIM_PCE0_GPIOB))
#define SYS_GPIOA_MOD   ((UWord32)(SIM_PCE0_GPIOA))
#define SYS_ADC_MOD     ((UWord32)(SIM_PCE0_ADC))
#define SYS_TB3_MOD     ((UWord32)(SIM_PCE0_TB3))
#define SYS_TB2_MOD     ((UWord32)(SIM_PCE0_TB2))
#define SYS_TB1_MOD     ((UWord32)(SIM_PCE0_TB1))
#define SYS_TB0_MOD     ((UWord32)(SIM_PCE0_TB0))
#define SYS_TB0_MOD     ((UWord32)(SIM_PCE0_TB0))
#define SYS_TA3_MOD     ((UWord32)(SIM_PCE0_TA3))
#define SYS_TA2_MOD     ((UWord32)(SIM_PCE0_TA2))
#define SYS_TA1_MOD     ((UWord32)(SIM_PCE0_TA1))
#define SYS_TA0_MOD     ((UWord32)(SIM_PCE0_TA0))
#define SYS_MSCAN_MOD   (((UWord32)(SIM_PCE1_MSCAN))<<16)
#define SYS_HFM_MOD     (((UWord32)(SIM_PCE1_HFM))<<16)
#define SYS_REFC_MOD    (((UWord32)(SIM_PCE1_REFC))<<16)
#define SYS_REFB_MOD    (((UWord32)(SIM_PCE1_REFB))<<16)
#define SYS_REFA_MOD    (((UWord32)(SIM_PCE1_REFA))<<16)
#define SYS_CRC_MOD     (((UWord32)(SIM_PCE1_CRC))<<16)
#define SYS_IIC1_MOD    (((UWord32)(SIM_PCE1_IIC1))<<16)
#define SYS_IIC0_MOD    (((UWord32)(SIM_PCE1_IIC0))<<16)
#define SYS_QSPI0_MOD   (((UWord32)(SIM_PCE1_QSPI0))<<16)
#define SYS_SCI0_MOD    (((UWord32)(SIM_PCE1_SCI0))<<16)
#define SYS_SCI1_MOD    (((UWord32)(SIM_PCE1_SCI1))<<16)
#define SYS_CMPC_MOD    (((UWord32)(SIM_PCE1_CMPC))<<16)
#define SYS_CMPB_MOD    (((UWord32)(SIM_PCE1_CMPB))<<16)
#define SYS_CMPA_MOD    (((UWord32)(SIM_PCE1_CMPA))<<16)
#define SYS_DAC_MOD     (((UWord32)(SIM_PCE1_DAC))<<16)
#define SYS_PWMCH3_MOD2  SIM_PCE2_PWMCH3
#define SYS_PWMCH2_MOD2  SIM_PCE2_PWMCH2
#define SYS_PWMCH1_MOD2  SIM_PCE2_PWMCH1
#define SYS_PWMCH0_MOD2  SIM_PCE2_PWMCH0

/* clock-enables parameters for bacwardward comaptibility */
#define SYS_GPIOF_CLK   SYS_GPIOF_MOD  
#define SYS_GPIOE_CLK   SYS_GPIOE_MOD  
#define SYS_GPIOD_CLK   SYS_GPIOD_MOD  
#define SYS_GPIOC_CLK   SYS_GPIOC_MOD  
#define SYS_GPIOB_CLK   SYS_GPIOB_MOD  
#define SYS_GPIOA_CLK   SYS_GPIOA_MOD  
#define SYS_ADC_CLK     SYS_ADC_MOD    
#define SYS_TB3_CLK     SYS_TB3_MOD    
#define SYS_TB2_CLK     SYS_TB2_MOD    
#define SYS_TB1_CLK     SYS_TB1_MOD    
#define SYS_TB0_CLK     SYS_TB0_MOD    
#define SYS_TB0_CLK     SYS_TB0_MOD    
#define SYS_TA3_CLK     SYS_TA3_MOD    
#define SYS_TA2_CLK     SYS_TA2_MOD    
#define SYS_TA1_CLK     SYS_TA1_MOD    
#define SYS_TA0_CLK     SYS_TA0_MOD    
#define SYS_MSCAN_CLK   SYS_MSCAN_MOD  
#define SYS_HFM_CLK     SYS_HFM_MOD    
#define SYS_REFC_CLK    SYS_REFC_MOD   
#define SYS_REFB_CLK    SYS_REFB_MOD   
#define SYS_REFA_CLK    SYS_REFA_MOD   
#define SYS_CRC_CLK     SYS_CRC_MOD    
#define SYS_IIC1_CLK    SYS_IIC1_MOD   
#define SYS_IIC0_CLK    SYS_IIC0_MOD   
#define SYS_QSPI0_CLK   SYS_QSPI0_MOD  
#define SYS_SCI0_CLK    SYS_SCI0_MOD   
#define SYS_SCI1_CLK    SYS_SCI1_MOD   
#define SYS_CMPC_CLK    SYS_CMPC_MOD   
#define SYS_CMPB_CLK    SYS_CMPB_MOD   
#define SYS_CMPA_CLK    SYS_CMPA_MOD   
#define SYS_DAC_CLK     SYS_DAC_MOD    
#define SYS_PWMCH3_CLK2 SYS_PWMCH3_MOD2 
#define SYS_PWMCH2_CLK2 SYS_PWMCH2_MOD2 
#define SYS_PWMCH1_CLK2 SYS_PWMCH1_MOD2 
#define SYS_PWMCH0_CLK2 SYS_PWMCH0_MOD2 


/* pad functions */
#define SYS_A0PAD_ANA0_VREFHA SIM_GPS3_A0_ANA0_VREFHA 
#define SYS_A0PAD_CMPC_O      SIM_GPS3_A0_CMPC_O      
#define SYS_C15PAD_SCL0       SIM_GPS1_C15_SCL0       
#define SYS_C15PAD_XB_OUT1    SIM_GPS1_C15_XB_OUT1    
#define SYS_C14PAD_SDA0       SIM_GPS1_C14_SDA0       
#define SYS_C14PAD_XB_OUT0    SIM_GPS1_C14_XB_OUT0    
#define SYS_C13PAD_TA3        SIM_GPS1_C13_TA3        
#define SYS_C13PAD_XB_IN6     SIM_GPS1_C13_XB_IN6     
#define SYS_C12PAD_RX0        SIM_GPS1_C12_RX0        
#define SYS_C12PAD_SDA1       SIM_GPS1_C12_SDA1       
#define SYS_C12PAD_RXD1       SIM_GPS1_C12_RXD1       
#define SYS_C11PAD_TX0        SIM_GPS1_C11_TX0        
#define SYS_C11PAD_SCL1       SIM_GPS1_C11_SCL1       
#define SYS_C11PAD_TXD1       SIM_GPS1_C11_TXD1       
#define SYS_C10PAD_MOSI0      SIM_GPS1_C10_MOSI0      
#define SYS_C10PAD_XB_IN5     SIM_GPS1_C10_XB_IN5     
#define SYS_C10PAD_MISO0      SIM_GPS1_C10_MISO0      
#define SYS_C9PAD_SCLK0       SIM_GPS1_C9_SCLK0       
#define SYS_C9PAD_XB_IN4      SIM_GPS1_C9_XB_IN4      
#define SYS_C8PAD_MISO0       SIM_GPS1_C8_MISO0       
#define SYS_C8PAD_RXD0        SIM_GPS1_C8_RXD0        
#define SYS_C7PAD_SSB0_B      SIM_GPS0_C7_SSB0_B      
#define SYS_C7PAD_TXD0        SIM_GPS0_C7_TXD0        
#define SYS_C6PAD_TA2         SIM_GPS0_C6_TA2         
#define SYS_C6PAD_XB_IN3      SIM_GPS0_C6_XB_IN3      
#define SYS_C6PAD_CMPREF      SIM_GPS0_C6_CMPREF      
#define SYS_C5PAD_DACO        SIM_GPS0_C5_DACO        
#define SYS_C5PAD_XB_IN7      SIM_GPS0_C5_XB_IN7      
#define SYS_C4PAD_TA1         SIM_GPS0_C4_TA1         
#define SYS_C4PAD_CMPB_O      SIM_GPS0_C4_CMPB_O      
#define SYS_C3PAD_TA0         SIM_GPS0_C3_TA0         
#define SYS_C3PAD_CMPA_O      SIM_GPS0_C3_CMPA_O      
#define SYS_C3PAD_RXD0        SIM_GPS0_C3_RXD0        
#define SYS_C2PAD_TXD0        SIM_GPS0_C2_TXD0        
#define SYS_C2PAD_TB0         SIM_GPS0_C2_TB0         
#define SYS_C2PAD_XB_IN2      SIM_GPS0_C2_XB_IN2      
#define SYS_C2PAD_CLKO        SIM_GPS0_C2_CLKO        
#define SYS_C0PAD_XTAL        SIM_GPS0_C0_XTAL        
#define SYS_C0PAD_CLKIN       SIM_GPS0_C0_CLKINO      
#define SYS_E7PAD_PWMA3       SIM_GPS3_E7_PWMA3       
#define SYS_E7PAD_XB_IN5      SIM_GPS3_E7_XB_IN5      
#define SYS_E6PAD_PWMB3       SIM_GPS3_E6_PWMB3       
#define SYS_E6PAD_XB_IN4      SIM_GPS3_E6_XB_IN4      
#define SYS_E5PAD_PWMA2       SIM_GPS3_E5_PWMA2       
#define SYS_E5PAD_XB_IN3      SIM_GPS3_E5_XB_IN3      
#define SYS_E4PAD_PWMB2       SIM_GPS3_E4_PWMB2       
#define SYS_E4PAD_XB_IN2      SIM_GPS3_E4_XB_IN2      
#define SYS_F8PAD_RXD0        SIM_GPS3_F8_RXD0       
#define SYS_F8PAD_TB1         SIM_GPS3_F8_TB1      
#define SYS_F6PAD_TB2         SIM_GPS2_F6_TB2         
#define SYS_F6PAD_PWMX3       SIM_GPS2_F6_PWMX3       
#define SYS_F5PAD_RXD1        SIM_GPS2_F5_RXD1        
#define SYS_F5PAD_XB_OUT5     SIM_GPS2_F5_XB_OUT5     
#define SYS_F4PAD_XB_OUT4     SIM_GPS2_F4_XB_OUT4     
#define SYS_F4PAD_TXD1        SIM_GPS2_F4_TXD1        
#define SYS_F3PAD_SDA1        SIM_GPS2_F3_SDA1        
#define SYS_F3PAD_XB_OUT3     SIM_GPS2_F3_XB_OUT3     
#define SYS_F2PAD_SCL1        SIM_GPS2_F2_SCL1        
#define SYS_F2PAD_XB_OUT2     SIM_GPS2_F2_XB_OUT2     
#define SYS_F1PAD_CLKOUT      SIM_GPS2_F1_CLKOUT      
#define SYS_F1PAD_XB_IN7      SIM_GPS2_F1_XB_IN7      
#define SYS_TMRB3_GPIOF7      SIM_GPS3_TMRB3_GPIOF7   
#define SYS_TMRB3_XB_OUT29    SIM_GPS3_TMRB3_XB_OUT29 
#define SYS_TMRB2_GPIOF6      SIM_GPS3_TMRB2_GPIOF6   
#define SYS_TMRB2_XB_OUT28    SIM_GPS3_TMRB2_XB_OUT28 
#define SYS_TMRB1_GPIOF8      SIM_GPS3_TMRB1_GPIOF8   
#define SYS_TMRB1_XB_OUT27    SIM_GPS3_TMRB1_XB_OUT27 
#define SYS_TMRB0_GPIOC2      SIM_GPS3_TMRB0_GPIOC2   
#define SYS_TMRB0_XB_OUT26    SIM_GPS3_TMRB0_XB_OUT26 

/* clkout mux */
#define SYS_CLKOUT_SYSCLK   SIM_CLKOSR_CLKOSEL_SYSCLK    /* sys_clk */
#define SYS_CLKOUT_IPBCLK   SIM_CLKOSR_CLKOSEL_IPBCLK    /* periph_clk */
#define SYS_CLKOUT_HSCLK    SIM_CLKOSR_CLKOSEL_HSCLK     /* hs_perf clk */
#define SYS_CLKOUT_MSTRCLK  SIM_CLKOSR_CLKOSEL_MSTRCLK   /* mstr_osc */

/* LVI levels */
#define LVI_22V_LEVEL       LVI_CONTROL_LVIE22 
#define LVI_27V_LEVEL       LVI_CONTROL_LVIE27
#define LVI_INT             0x0004

/****************************************************************************
* register bit names and masks
*****************************************************************************/

/* SIM_CONTROL */
#define SIM_CONTROL_ONCEEBL             0x0020
#define SIM_CONTROL_SWRST               0x0010
#define SIM_CONTROL_STOPDIS_MASK        0x000c
#define SIM_CONTROL_STOPDIS_ENABLE      0x0000
#define SIM_CONTROL_STOPDIS_SOFT        0x0004
#define SIM_CONTROL_STOPDIS_PERMANENT   0x0008
#define SIM_CONTROL_WAITDIS_MASK        0x0003
#define SIM_CONTROL_WAITDIS_ENABLE      0x0000
#define SIM_CONTROL_WAITDIS_SOFT        0x0001
#define SIM_CONTROL_WAITDIS_PERMANENT   0x0002

/* SIM_RSTSTS */
#define SIM_RSTSTS_SWR          0x0040
#define SIM_RSTSTS_COP_TOR      0x0020
#define SIM_RSTSTS_COP_LOR      0x0010
#define SIM_RSTSTS_EXTR         0x0008
#define SIM_RSTSTS_POR          0x0004

/* SIM_POWER */
#define SIM_POWER_LRSTDBY0      0x0001
#define SIM_POWER_LRSTDBY1      0x0002
#define SIM_POWER_LRSTDBY_MASK  0x0003

/* SIM_CLKOSR */
#define SIM_CLKOSR_CLKDIS           0x0020
#define SIM_CLKOSR_CLKOSEL_MASK     0x001f
#define SIM_CLKOSR_CLKOSEL_SYSCLK   0x0000
#define SIM_CLKOSR_CLKOSEL_IPBCLK   0x0001
#define SIM_CLKOSR_CLKOSEL_HSCLK    0x0002
#define SIM_CLKOSR_CLKOSEL_MSTRCLK  0x0003

/* SIM_PCR */
#define SIM_PCR_TMRB            0x8000
#define SIM_PCR_TMRA            0x4000
#define SIM_PCR_SCI0            0x2000
#define SIM_PCR_SCI1            0x1000

/* SIM_PCE0 */
#define SIM_PCE0_TA0            0x8000
#define SIM_PCE0_TA1            0x4000
#define SIM_PCE0_TA2            0x2000
#define SIM_PCE0_TA3            0x1000
#define SIM_PCE0_TB0            0x0800
#define SIM_PCE0_TB1            0x0400
#define SIM_PCE0_TB2            0x0200
#define SIM_PCE0_TB3            0x0100
#define SIM_PCE0_ADC            0x0080
#define SIM_PCE0_GPIOA          0x0040
#define SIM_PCE0_GPIOB          0x0020
#define SIM_PCE0_GPIOC          0x0010
#define SIM_PCE0_GPIOD          0x0008
#define SIM_PCE0_GPIOE          0x0004
#define SIM_PCE0_GPIOF          0x0002

/* SIM_PCE1 */
#define SIM_PCE1_DAC            0x4000
#define SIM_PCE1_CMPA           0x2000
#define SIM_PCE1_CMPB           0x1000
#define SIM_PCE1_CMPC           0x0800
#define SIM_PCE1_SCI0           0x0400
#define SIM_PCE1_SCI1           0x0200
#define SIM_PCE1_QSPI0          0x0100
#define SIM_PCE1_IIC0           0x0080
#define SIM_PCE1_IIC1           0x0040
#define SIM_PCE1_CRC            0x0020
#define SIM_PCE1_REFA           0x0010
#define SIM_PCE1_REFB           0x0008
#define SIM_PCE1_REFC           0x0004
#define SIM_PCE1_HFM            0x0002
#define SIM_PCE1_MSCAN          0x0001

/* SIM_PCE2 */
#define SIM_PCE2_PWMCH0         0x0008
#define SIM_PCE2_PWMCH1         0x0004
#define SIM_PCE2_PWMCH2         0x0002
#define SIM_PCE2_PWMCH3         0x0001


/* SIM_SD0 */
#define SIM_SD0_TA0             0x8000
#define SIM_SD0_TA1             0x4000
#define SIM_SD0_TA2             0x2000
#define SIM_SD0_TA3             0x1000
#define SIM_SD0_TB0             0x0800
#define SIM_SD0_TB1             0x0400
#define SIM_SD0_TB2             0x0200
#define SIM_SD0_TB3             0x0100
#define SIM_SD0_ADC             0x0080
#define SIM_SD0_GPIOA           0x0040
#define SIM_SD0_GPIOB           0x0020
#define SIM_SD0_GPIOC           0x0010
#define SIM_SD0_GPIOD           0x0008
#define SIM_SD0_GPIOE           0x0004
#define SIM_SD0_GPIOF           0x0002

/* SIM_SD1 */
#define SIM_SD1_DAC             0x4000
#define SIM_SD1_CMPA            0x2000
#define SIM_SD1_CMPB            0x1000
#define SIM_SD1_CMPC            0x0800
#define SIM_SD1_SCI0            0x0400
#define SIM_SD1_SCI1            0x0200
#define SIM_SD1_QSPI0           0x0100
#define SIM_SD1_IIC0            0x0080
#define SIM_SD1_IIC1            0x0040
#define SIM_SD1_CRC             0x0020
#define SIM_SD1_REFA            0x0010
#define SIM_SD1_REFB            0x0008
#define SIM_SD1_REFC            0x0004
#define SIM_SD1_HFM             0x0002
#define SIM_SD1_MSCAN           0x0001

/* SIM_SD2 */
#define SIM_SD2_PWMCH0          0x0008
#define SIM_SD2_PWMCH1          0x0004
#define SIM_SD2_PWMCH2          0x0002
#define SIM_SD2_PWMCH3          0x0001

/* SIM_PROT */
#define SIM_PROT_PCEP_MASK      0x000c
#define SIM_PROT_PCEP_PROT      0x0004
#define SIM_PROT_PCEP_LOCK      0x0008
#define SIM_PROT_GIPSP_MASK     0x0003
#define SIM_PROT_GIPSP_PROT     0x0001
#define SIM_PROT_GIPSP_LOCK     0x0002

/* SIM_GPS0 */
#define SIM_GPS0_C7             0x4000
#define SIM_GPS0_C7_SSB0_B      0x0000
#define SIM_GPS0_C7_TXD0        0x4000
#define SIM_GPS0_C6_MASK        0x3000
#define SIM_GPS0_C6_TA2         0x0000
#define SIM_GPS0_C6_XB_IN3      0x1000
#define SIM_GPS0_C6_CMPREF      0x2000
#define SIM_GPS0_C5             0x0400
#define SIM_GPS0_C5_DACO        0x0000
#define SIM_GPS0_C5_XB_IN7      0x0400
#define SIM_GPS0_C4             0x0100
#define SIM_GPS0_C4_TA1         0x0000
#define SIM_GPS0_C4_CMPB_O      0x0100
#define SIM_GPS0_C3_MASK        0x00C0
#define SIM_GPS0_C3_TA0         0x0000
#define SIM_GPS0_C3_CMPA_O      0x0040
#define SIM_GPS0_C3_RXD0        0x0080
#define SIM_GPS0_C2_MASK        0x0030
#define SIM_GPS0_C2_TXD0        0x0000
#define SIM_GPS0_C2_TB0         0x0010
#define SIM_GPS0_C2_XB_IN2      0x0020
#define SIM_GPS0_C2_CLKO        0x0030
#define SIM_GPS0_C0             0x0001
#define SIM_GPS0_C0_XTAL        0x0000
#define SIM_GPS0_C0_CLKINO      0x0001


/* SIM_GPS1 */
#define SIM_GPS1_C15            0x4000
#define SIM_GPS1_C15_SCL0       0x0000
#define SIM_GPS1_C15_XB_OUT1    0x4000
#define SIM_GPS1_C14            0x1000
#define SIM_GPS1_C14_SDA0       0x0000
#define SIM_GPS1_C14_XB_OUT0    0x1000
#define SIM_GPS1_C13            0x0400
#define SIM_GPS1_C13_TA3        0x0000
#define SIM_GPS1_C13_XB_IN6     0x0400
#define SIM_GPS1_C12_MASK       0x0300
#define SIM_GPS1_C12_RX0        0x0000
#define SIM_GPS1_C12_SDA1       0x0100
#define SIM_GPS1_C12_RXD1       0x0200
#define SIM_GPS1_C11_MASK       0x00C0
#define SIM_GPS1_C11_TX0        0x0000
#define SIM_GPS1_C11_SCL1       0x0040
#define SIM_GPS1_C11_TXD1       0x0080
#define SIM_GPS1_C10_MASK       0x0030
#define SIM_GPS1_C10_MOSI0      0x0000
#define SIM_GPS1_C10_XB_IN5     0x0010
#define SIM_GPS1_C10_MISO0      0x0020
#define SIM_GPS1_C9             0x0004
#define SIM_GPS1_C9_SCLK0       0x0000
#define SIM_GPS1_C9_XB_IN4      0x0004
#define SIM_GPS1_C8             0x0001
#define SIM_GPS1_C8_MISO0       0x0000
#define SIM_GPS1_C8_RXD0        0x0001

/* SIM_GPS2 */
#define SIM_GPS2_F6             0x1000
#define SIM_GPS2_F6_TB2         0x0000
#define SIM_GPS2_F6_PWMX3       0x1000
#define SIM_GPS2_F5             0x0400
#define SIM_GPS2_F5_RXD1        0x0000
#define SIM_GPS2_F5_XB_OUT5     0x0400
#define SIM_GPS2_F4             0x0100
#define SIM_GPS2_F4_XB_OUT4     0x0000
#define SIM_GPS2_F4_TXD1        0x0100
#define SIM_GPS2_F3             0x0040
#define SIM_GPS2_F3_SDA1        0x0000
#define SIM_GPS2_F3_XB_OUT3     0x0040
#define SIM_GPS2_F2             0x0010
#define SIM_GPS2_F2_SCL1        0x0000
#define SIM_GPS2_F2_XB_OUT2     0x0010
#define SIM_GPS2_F1             0x0004
#define SIM_GPS2_F1_CLKOUT      0x0000
#define SIM_GPS2_F1_XB_IN7      0x0004

/* SIM_GPS3 */
#define SIM_GPS3_TMRB3          0x8000
#define SIM_GPS3_TMRB3_GPIOF7   0x0000
#define SIM_GPS3_TMRB3_XB_OUT29 0x8000
#define SIM_GPS3_TMRB2          0x4000
#define SIM_GPS3_TMRB2_GPIOF6   0x0000
#define SIM_GPS3_TMRB2_XB_OUT28 0x4000
#define SIM_GPS3_TMRB1          0x2000
#define SIM_GPS3_TMRB1_GPIOF8   0x0000
#define SIM_GPS3_TMRB1_XB_OUT27 0x2000
#define SIM_GPS3_TMRB0          0x1000
#define SIM_GPS3_TMRB0_GPIOC2   0x0000
#define SIM_GPS3_TMRB0_XB_OUT26 0x1000
#define SIM_GPS3_E7             0x0400
#define SIM_GPS3_E7_PWMA3       0x0000
#define SIM_GPS3_E7_XB_IN5      0x0400
#define SIM_GPS3_E6             0x0100
#define SIM_GPS3_E6_PWMB3       0x0000
#define SIM_GPS3_E6_XB_IN4      0x0100
#define SIM_GPS3_E5             0x0040
#define SIM_GPS3_E5_PWMA2       0x0000
#define SIM_GPS3_E5_XB_IN3      0x0040
#define SIM_GPS3_E4             0x0010
#define SIM_GPS3_E4_PWMB2       0x0000
#define SIM_GPS3_E4_XB_IN2      0x0010
#define SIM_GPS3_F8             0x0004
#define SIM_GPS3_F8_RXD0        0x0000
#define SIM_GPS3_F8_TB1         0x0004
#define SIM_GPS3_A0             0x0001
#define SIM_GPS3_A0_ANA0_VREFHA 0x0000
#define SIM_GPS3_A0_CMPC_O      0x0001

/* LVI_CONTROL */
#define LVI_CONTROL_LVIE27      0x0002
#define LVI_CONTROL_LVIE22      0x0001

/* LVI_STATUS */
#define LVI_STATUS_LVI          0x0010
#define LVI_STATUS_LVIS27S      0x0008
#define LVI_STATUS_LVIS22S      0x0004
#define LVI_STATUS_LVIS27       0x0002
#define LVI_STATUS_LVIS22       0x0001

/***********************************************************************
* SYS control command macros
***********************************************************************/

/* SYS initialization */
void sysInit(arch_sSIM *pSimBase);
#define ioctlSYS_INIT(pSimBase, param) sysInit(pSimBase)

/* reprogramable "stop" instruction disable */ 
#define ioctlSYS_STOP(pSimBase, param) \
  periphBitGrpSet(SIM_CONTROL_STOPDIS_MASK, (param)<<2, &((pSimBase)->sim_control))

/* reprogramable "wait" instruction disable */ 
#define ioctlSYS_WAIT(pSimBase, param) \
  periphBitGrpSet(SIM_CONTROL_WAITDIS_MASK, param, &((pSimBase)->sim_control))

/* subset of teh ones above, maintained for backward compatibility */
#define ioctlSYS_STOP_PERMANENT_DISABLE(pSimBase, param) \
  periphBitGrpSet(SIM_CONTROL_STOPDIS_MASK, (SYS_DISABLE_PERMANENT)<<2, &((pSimBase)->sim_control))
#define ioctlSYS_WAIT_PERMANENT_DISABLE(pSimBase, param) \
  periphBitGrpSet(SIM_CONTROL_WAITDIS_MASK, SYS_DISABLE_PERMANENT, &((pSimBase)->sim_control))

/* issue software reset */
#define ioctlSYS_SOFTWARE_RESET(pSimBase, param) \
  periphBitSet(SIM_CONTROL_SWRST, &((pSimBase)->sim_control))

/* OnCE enable/disable */
#define ioctlSYS_ONCE(pSimBase, param) \
  if(param) periphBitSet(SIM_CONTROL_ONCEEBL, &((pSimBase)->sim_control));\
  else periphBitClear(SIM_CONTROL_ONCEEBL, &((pSimBase)->sim_control))

/* backward compatibility macros */
#define ioctlSYS_ONCE_ENABLE(pSimBase, param) \
  periphBitSet(SIM_CONTROL_ONCEEBL, &((pSimBase)->sim_control))
#define ioctlSYS_ONCE_DISABLE(pSimBase, param) \
  periphBitClear(SIM_CONTROL_ONCEEBL, &((pSimBase)->sim_control))

/* enable modules in STOP mode */
#define ioctlSYS_ENABLE_IN_STOP(pSimBase, param) { \
  if((param) & 0xffff0000L) periphBitSet((UWord16)((param)>>16), &((pSimBase)->sim_sd1));   \
  if((param) & 0x0000ffff)  periphBitSet((UWord16)((param)&0xffff), &((pSimBase)->sim_sd0)); }
  
/* disable modules in STOP mode */
#define ioctlSYS_DISABLE_IN_STOP(pSimBase, param) { \
  if((param) & 0xffff0000L) periphBitClear((UWord16)((param)>>16), &((pSimBase)->sim_sd1));   \
  if((param) & 0x0000ffff)  periphBitClear((UWord16)((param)&0xffff), &((pSimBase)->sim_sd0)); }


/* enable peripheral clock */
#define ioctlSYS_PERIPH_CLK_ENABLE(pSimBase, param) { \
  if((param) & 0x0ffff0000L) periphBitSet((UWord16)((param)>>16), &((pSimBase)->sim_pce1));    \
  if((param) & 0x00000ffff)  periphBitSet((UWord16)((param)&0xffff), &((pSimBase)->sim_pce0)); }

/* disable peripheral clock */
#define ioctlSYS_PERIPH_CLK_DISABLE(pSimBase,  param)  { \
  if((param) & 0xffff0000L) periphBitClear((UWord16)((param)>>16), &((pSimBase)->sim_pce1));   \
  if((param) & 0x0000ffff)  periphBitClear((UWord16)((param)&0xffff), &((pSimBase)->sim_pce0)); }

/* enable modules in STOP mode, second register  */
#define ioctlSYS_ENABLE_IN_STOP_REG2(pSimBase, param) \
  periphBitSet((param), &((pSimBase)->sim_sd2)); 
  
/* disable modules in STOP mode, second register  */
#define ioctlSYS_DISABLE_IN_STOP_REG2(pSimBase, param) \
  periphBitClear((param), &((pSimBase)->sim_sd2)); 

/* enable peripheral clock, second register */
#define ioctlSYS_PERIPH_CLK_REG2_ENABLE(pSimBase, param) \
  periphBitSet((param), &((pSimBase)->sim_pce2));

/* disable peripheral clock, second register  */
#define ioctlSYS_PERIPH_CLK_REG2_DISABLE(pSimBase,  param) \
  periphBitClear((param), &((pSimBase)->sim_pce2));


/* enable High_Speed clock for peripheral modules */
#define ioctlSYS_HS_CLOCK_ENABLE(pSimBase, param) \
  periphBitSet(param, &((pSimBase)->sim_pcr))

/* disable High_Speed clock for peripheral modules */
#define ioctlSYS_HS_CLOCK_DISABLE(pSimBase, param) \
  periphBitClear(param, &((pSimBase)->sim_pcr))

/* pins configuration usign GPS registers */
#define ioctlSYS_SET_A0PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_A0, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_A0, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_C0PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS0_C0, &((pSimBase)->sim_gps0)); \
  else periphBitClear(SIM_GPS0_C0, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C2PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS0_C2_MASK, param, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C3PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS0_C3_MASK, param, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C4PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS0_C4, &((pSimBase)->sim_gps0)); \
  else periphBitClear(SIM_GPS0_C4, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C5PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS0_C5, &((pSimBase)->sim_gps0)); \
  else periphBitClear(SIM_GPS0_C5, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C6PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS0_C6_MASK, param, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C7PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS0_C7, &((pSimBase)->sim_gps0)); \
  else periphBitClear(SIM_GPS0_C7, &((pSimBase)->sim_gps0))

#define ioctlSYS_SET_C8PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS1_C8, &((pSimBase)->sim_gps1)); \
  else periphBitClear(SIM_GPS1_C8, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C9PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS1_C9, &((pSimBase)->sim_gps1)); \
  else periphBitClear(SIM_GPS1_C9, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C10PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS1_C10_MASK, param, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C11PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS1_C11_MASK, param, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C12PAD_FUNCTION(pSimBase, param) \
  periphBitGrpSet(SIM_GPS1_C12_MASK, param, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C13PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS1_C13, &((pSimBase)->sim_gps1)); \
  else periphBitClear(SIM_GPS1_C13, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C14PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS1_C14, &((pSimBase)->sim_gps1)); \
  else periphBitClear(SIM_GPS1_C14, &((pSimBase)->sim_gps1))

#define ioctlSYS_SET_C15PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS1_C15, &((pSimBase)->sim_gps1)); \
  else periphBitClear(SIM_GPS1_C15, &((pSimBase)->sim_gps1))


#define ioctlSYS_SET_E4PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_E4, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_E4, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_E5PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_E5, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_E5, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_E6PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_E6, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_E6, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_E7PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_E7, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_E7, &((pSimBase)->sim_gps3))


#define ioctlSYS_SET_F8PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_F8, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_F8, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_F6PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F6, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F6, &((pSimBase)->sim_gps2))

#define ioctlSYS_SET_F5PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F5, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F5, &((pSimBase)->sim_gps2))

#define ioctlSYS_SET_F4PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F4, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F4, &((pSimBase)->sim_gps2))

#define ioctlSYS_SET_F3PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F3, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F3, &((pSimBase)->sim_gps2))

#define ioctlSYS_SET_F2PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F2, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F2, &((pSimBase)->sim_gps2))

#define ioctlSYS_SET_F1PAD_FUNCTION(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS2_F1, &((pSimBase)->sim_gps2)); \
  else periphBitClear(SIM_GPS2_F1, &((pSimBase)->sim_gps2))


#define ioctlSYS_SET_TMRB3_INPUT(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_TMRB3, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_TMRB3, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_TMRB2_INPUT(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_TMRB2, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_TMRB2, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_TMRB1_INPUT(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_TMRB1, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_TMRB1, &((pSimBase)->sim_gps3))

#define ioctlSYS_SET_TMRB0_INPUT(pSimBase, param) \
  if(param) periphBitSet(SIM_GPS3_TMRB0, &((pSimBase)->sim_gps3)); \
  else periphBitClear(SIM_GPS3_TMRB0, &((pSimBase)->sim_gps3))


/* set large-regulator mode */
#define ioctlSYS_SET_POWER_MODE(pSimBase, param) \
  periphBitGrpZS(SIM_POWER_LRSTDBY_MASK, param, &((pSimBase)->sim_power))

/* get large-regulator mode */
#define ioctlSYS_GET_POWER_MODE(pSimBase, param) \
  periphBitTest(SIM_POWER_LRSTDBY_MASK, &((pSimBase)->sim_power))
  
/* test reset status SWR, COPR, EXTR, POR bits */
#define ioctlSYS_TEST_RESET_SOURCE(pSimBase, param) \
  periphBitTest(param, &((pSimBase)->sim_rststs))

/* clear reset status SWR, COPR, EXTR, POR bits */
#define ioctlSYS_CLEAR_RESET_SOURCE(pSimBase, param) \
  periphMemInvBitSet(param, &((pSimBase)->sim_rststs))

/* SIM software control registers */ 
#define ioctlSYS_WRITE_SW_CONTROL_REG0(pSimBase, param) \
  periphMemWrite(param, &((pSimBase)->sim_scr0))

#define ioctlSYS_WRITE_SW_CONTROL_REG1(pSimBase, param) \
  periphMemWrite(param, &((pSimBase)->sim_scr1))

#define ioctlSYS_WRITE_SW_CONTROL_REG2(pSimBase, param) \
  periphMemWrite(param, &((pSimBase)->sim_scr2))

#define ioctlSYS_WRITE_SW_CONTROL_REG3(pSimBase, param) \
  periphMemWrite(param, &((pSimBase)->sim_scr3))

#define ioctlSYS_READ_SW_CONTROL_REG0(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_scr0))

#define ioctlSYS_READ_SW_CONTROL_REG1(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_scr1))

#define ioctlSYS_READ_SW_CONTROL_REG2(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_scr2))

#define ioctlSYS_READ_SW_CONTROL_REG3(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_scr3))

/* enable/disable CLKOUT pin */
#define ioctlSYS_CLKOUT(pSimBase, param) \
  if(param) periphBitClear(SIM_CLKOSR_CLKDIS, &((pSimBase)->sim_clkosr));\
  else periphBitSet(SIM_CLKOSR_CLKDIS, &((pSimBase)->sim_clkosr))
  
/* enable/disable CLKOUT pin, backward comaptibility only */
#define ioctlSYS_CLKOUT_ENABLE(pSimBase, param)  \
  periphBitClear(SIM_CLKOSR_CLKDIS, &((pSimBase)->sim_clkosr))
#define ioctlSYS_CLKOUT_DISABLE(pSimBase, param) \
  periphBitSet(SIM_CLKOSR_CLKDIS, &((pSimBase)->sim_clkosr))

/* select CLSKOUT source */
#define ioctlSYS_CLKOUT_SELECT(pSimBase, param)  \
  periphBitGrpSet(SIM_CLKOSR_CLKOSEL_MASK, param, &((pSimBase)->sim_clkosr))

/* JTAG ID */
#define ioctlSYS_READ_LSH_JTAG_ID(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_lsh_id))

#define ioctlSYS_READ_MSH_JTAG_ID(pSimBase, param) \
  periphMemRead(&((pSimBase)->sim_msh_id))

/* memory reference via the I/O short address mode */
#define ioctlSYS_WRITE_IO_SHORT_ADDR_LOCATION_REG(pSimBase, param) \
  { periphMemWrite((UWord16)(((UWord32)(param)) & 0xFFFF), &((pSimBase)->sim_isall));   \
    periphMemWrite((UWord16)(((UWord32)(param)) >> 16), &((pSimBase)->sim_isalh));  }

#define ioctlSYS_READ_IO_SHORT_ADDR_LOCATION_REG(pSimBase, param) \
  (  (UWord32) periphMemRead(&((pSimBase)->sim_isall)) | \
    ((UWord32)(periphMemRead(&((pSimBase)->sim_isalh)) & 3) << 16) )

/* write protect PCE, SD and PCR */
#define ioctlSYS_WPROTECT_CLOCK_SETTINGS(pSimBase, param)    \
  periphBitGrpSet(SIM_PROT_PCEP_MASK, ((param)<<2), &((pSimBase)->sim_prot))

/* write protect IPS and GPSxx */
#define ioctlSYS_WPROTECT_SIGNALS_ROUTING(pSimBase, param)    \
  periphBitGrpSet(SIM_PROT_GIPSP_MASK, (param), &((pSimBase)->sim_prot))

/********** LVI **********/

/* LVI low voltage interrupt enable */
#define ioctlLVI_INT_ENABLE(pLviBase, param) \
  periphBitSet(param, &((pLviBase)->lvictlr))

/* LVI low voltage interrupt disable */
#define ioctlLVI_INT_DISABLE(pLviBase, param) \
  periphBitClear(param, &((pLviBase)->lvictlr))

/* LVI low voltage interrupt selection */
#define ioctlLVI_INT_SELECT(pLviBase, param) \
  periphBitGrpSR(LVI_CONTROL_LVIE22|LVI_CONTROL_LVIE27, param, &((pLviBase)->lvictlr))

/* test LVI Status LVIS27, LVIS22 bits */
#define ioctlLVI_GET_LOW_VOLTAGE(pLviBase, param) \
  periphBitTest(((param) << 2), &((pLviBase)->lvisr))

#define ioctlLVI_GET_NONSTICKY_INT_SOURCE(pLviBase, param) \
  periphBitTest(param, &((pLviBase)->lvisr))

/* clear LVI Status LVIS27, LVIS22 bits (cleared by writing "1") */

/* on 802x/3x, the LVI INT flag sometimes fails to get cleared. This typically
   happens when result of LVI equation: 
            LVI = (LVIS27 AND LVIE27) or (LVIS22 AND LVIE22) 
   is TRUE during clearing. It is more likely to happen when PLL is off. 
   Also, rising of the LVI interrupt is sensitive to edges of the LVI bit, 
   which means the interrupt is never raised again if clearing LVI bit fails.
*/    
#if 0  
#define ioctlLVI_CLEAR_LOW_VOLTAGE_INT(pLviBase, param) \
  periphMemWrite(((param) << 2), &((pLviBase)->lvisr))
#else
/* a workaround is to clear the LVI INT flag during LVIExx temporarily zeroed */
#define ioctlLVI_CLEAR_LOW_VOLTAGE_INT(pLviBase, param) \
  if((param) & LVI_INT) \
  { UWord16 tmp = periphMemRead(&((pLviBase)->lvictlr)); \
    periphMemWrite(0, &((pLviBase)->lvictlr)); \
    periphMemWrite(((param) << 2), &((pLviBase)->lvisr)); \
    periphMemWrite(tmp, &((pLviBase)->lvictlr)); } \
  else periphMemWrite(((param) << 2), &((pLviBase)->lvisr));
#endif

#ifdef __cplusplus
}
#endif

#endif

