/*
 * File:		uart_lon_col.c
 * Purpose:		LON mode collision.
 *
 */

#include "common.h"
#include "uart.h"


/********************************************************************/
/* UART module LON mode collision test.
 *
 * Tests the LON mode collision detect feature. The test will use a GPIO 
 * pin connected to the UART_COL signal to activate the collision
 * signal during different parts of the transfer. The delay loops
 * in the test are important because the are used to force the collision
 * toggle to occur during specific parts of the transfer. It is a good 
 * idea to use a scope to double check the timing of these delays anytime
 * this test is ported to a new processor. 
 *
 * Test will check both polarities of the COL signal and all four possible
 * states for collision detection available in S4[CDET]. Test returns the 
 * number of errors detected.
 */
int 
uart_lon_col_test(UART_MemMapPtr module)
{
    volatile char ch;
    uint32 i;
    int error = 0;
    
    /*Enable UART0_COL function during startup*/
    
    /* Configure the UART for LON mode */
    UART_C6_REG(module) = UART_C6_EN709_MASK;
    
    UART_PRE_REG(module) = 0x4; /* Set preamble length */
    
    UART_TPL_REG(module) = 0x01; /* Set packet length */

    /* Set beta1 timer to 1 as 0 seems to max out delay */
    UART_B1T_REG(module) = 1;
    
    /* Enable collision - active low COL is default */
    UART_C6_REG(module) |= UART_C6_CE_MASK;
    
#if 0
    /* Make sure CDET field is cleared to start */
    if ( UART_S4_REG(module) != 0)
    {
        error++;
        printf("\nERR! S4 value is incorrect. Expected: 0x0 Read: 0x%02X", UART_S4_REG(module));
    }
#endif
    
    UART_C6_REG(module) |= UART_C6_TX709_MASK;
    
    uart_putchar (module, 0xAA); 
    
    /* Small delay before driving COL low */
    for(i=0; i < 0x400; i++)
    {;}
    
    /* Toggle COL pin */
    
    /* Wait for transmit to complete */
    while( !(UART_S3_REG(module) & UART_S3_PTXF_MASK))
    {;}

    /* Disable transmit to prevent a second packet from attempting to go out */
    UART_C6_REG(module) &= ~UART_C6_TX709_MASK;
    
    /* Clear the PTXF flag */
    UART_S3_REG(module) = UART_S3_PTXF_MASK;    
    
    /* Make sure CDET field is 01 indicating COL during preamble */
    if ( (UART_S4_REG(module) & UART_S4_CDET(0x3)) != 0x04)
    {
        error++;
        printf("\nERR! S4 value is incorrect. Expected: 0x04 Read: 0x%02X", UART_S4_REG(module));
    }
    
    /* Write 1s to S4 to clear CDET field */
    UART_S4_REG(module) = 0xF;
    
    UART_C6_REG(module) |= UART_C6_TX709_MASK;
    
    uart_putchar (module, 0xAA); 
    
    for(i=0; i < 0x800; i++)
    {;}
    
    /* Toggle COL low */
    
    /* Wait for transmit to complete */
    while( !(UART_S3_REG(module) & UART_S3_PTXF_MASK))
    {;}

    /* Disable transmit to prevent a second packet from attempting to go out */
    UART_C6_REG(module) &= ~UART_C6_TX709_MASK;
    
    /* Clear the PTXF flag */
    UART_S3_REG(module) = UART_S3_PTXF_MASK;    
    
    /* Make sure CDET field is 10 indicating COL during data */
    if ( (UART_S4_REG(module) & UART_S4_CDET(0x3)) != 0x08)
    {
        error++;
        printf("\nERR! S4 value is incorrect. Expected: 0x08 Read: 0x%02X", UART_S4_REG(module));
    } 
    
    /* Write 1s to S4 to clear CDET field */
    UART_S4_REG(module) = 0xF;

    /* All tests complete. Return registers to default values */
    UART_C6_REG(module) = 0;   
    UART_PRE_REG(module) = 0;    
    UART_TPL_REG(module) = 0;  
        
    return error;    
} 
/********************************************************************/
