/***********************************************************************************************\
* Freescale MMA8651,2Q Driver
*
* Filename: terminal.c
*
*
* (c) Copyright 2010, Freescale, Inc.  All rights reserved.
*
*
\***********************************************************************************************/

#include "system.h"

/***********************************************************************************************\
* Private macros
\***********************************************************************************************/

/***********************************************************************************************\
* Private type definitions
\***********************************************************************************************/

/***********************************************************************************************\
* Private prototypes
\***********************************************************************************************/

extern byte ProcessHexInput (byte *in, byte *out);
extern void CopyXYZ (byte *ptr);
extern void CopyXYZ8 (byte *ptr);
extern void PrintXYZdec14 (void);
extern void PrintXYZdec12 (void);
extern void PrintXYZdec10(void);
extern void PrintXYZdec8 (void);
extern void PrintXYZfrac (void);
extern void ClearDataFlash (void);
extern void PrintBuffering (void);
extern void PrintFIFO (void);
extern void ReadDataFlashXYZ (void);
extern void WriteFlashBuffer (void);

/***********************************************************************************************\
* Private memory declarations
\***********************************************************************************************/

#pragma DATA_SEG __SHORT_SEG _DATA_ZEROPAGE

BIT_FIELD StreamMode;                       // stream mode control flags

extern BIT_FIELD RegisterFlag;
extern byte value[];                        // working value result scratchpad

static byte temp;                          //  byte variables
//static byte index;                          // input character string index

static tword x_value;                       // 16-bit X accelerometer value
static tword y_value;                       // 16-bit Y accelerometer value
static tword z_value;                       // 16-bit Z accelerometer value

extern byte SlaveAddressIIC;                // accelerometer slave I2C address

extern byte functional_block;               // accelerometer function
                                        
extern byte address_in[3];                  // Data Flash input address pointer
extern byte address_out[3];                 // Data Flash output address pointer

extern BIT_FIELD tilt;

#pragma DATA_SEG DEFAULT

extern tfifo_sample fifo_data[FIFO_BUFFER_SIZE];   // FIFO sample buffer


byte *const ODRvalues [] = 
{
  "800", "400", "200", "100", "50", "12.5", "6.25", "1.563"

};

byte *const HPvaluesN [] =
{
  "16", "8", "4", "2",  //800
  "16", "8", "4", "2",  //400
  "8", "4", "2", "1",   //200
  "4", "2", "1", "0.5",   //100
  "2", "1", "0.5", "0.25", //50
  "2", "1", "0.5", "0.25", //12.5
  "2", "1", "0.5", "0.25", //6.25
  "2", "1", "0.5", "0.25"
};

byte *const HPvaluesLNLP [] =
{
  "16", "8", "4", "2",
  "16", "8", "4", "2",
  "8", "4", "2", "1",
  "4", "2", "1", "0.5",
  "2", "1", "0.5", "0.25",
  "0.5", "0.25", "0.125", "0.063",
  "0.25", "0.125", "0.063", "0.031",
  "0.063", "0.031", "0.016", "0.008"
};

byte *const HPvaluesHiRes [] =
{
  "16", "8", "4", "2"
};

byte *const HPvaluesLP [] =
{
 "16", "8", "4", "2",
  "8", "4", "2", "1",
  "4", "2", "1", "0.5",
  "2", "1", "0.5", "0.25",
  "1", "0.5", "0.25", "0.125",
  "0.25", "0.125", "0.063", "0.031",
  "0.125", "0.063", "0.031", "0.016",
  "0.031", "0.016", "0.008", "0.004" 
};

byte *const GRange [] =
{
  "2g", "4g", "8g"
};


/***********************************************************************************************\
* Public functions
\***********************************************************************************************/

/*********************************************************\
**  Terminal Strings
\*********************************************************/
const byte string_Prompt []       = {"\r\nMMA8491Q> "};
const byte string_What []         = {" <-- what?"};
const byte string_Streaming []    = {" - Streaming XYZ data"};
const byte string_Counts []       = {" as signed counts\r\n"};  
const byte string_Gs []           = {" as signed g's\r\n"};
const byte string_Interrupts []   = {" via Int Mode "};
const byte string_FIFO []         = {" via FIFO"};

/*********************************************************\
**  Initialize Terminal Interface
\*********************************************************/
void TerminalInit (void)
{

  SCISendString("\r\n\n**  Freescale Semiconductor  **");
  SCISendString  ("\r\n**  MMA8491Q Driver          **"); 
  SCISendString  ("\r\n**      using the MC9S08QE8  **");
  SCISendString  ("\r\n**                           **");
  SCISendString  ("\r\n**  "__DATE__"    "__TIME__"  **\r\n\n");
  /*
  **  Prepare Data Flash.
  */
  DATAFLASH_Unprotect();
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;

  SPI_SS_SELECT;
  SPI_ChrShift(0x01);
  SPI_ChrShift(0x00);
  SPI_SS_DESELECT;

  functional_block = FBID_MAX;
  StreamMode.Byte = 0;
  tilt.Byte = 0;
  
  TILT_APP = 0;       // tilt application is OFF
  DATA_APP = 0;       // data application is OFF
  sample_num = 666;   // unlimited sample
  STREAM_FULLC =1;    // output in counts
  
}


/*********************************************************\
**  Process Terminal Interface
\*********************************************************/

void ProcessTerminal (void)
{
  /*
  **  Output command prompt if required.
  */
  if (PROMPT_WAIT == FALSE)
  {
  
    SCISendString((byte*)string_Prompt);
   
    PROMPT_WAIT = TRUE;
  }
  /*
  **  Get input from terminal.
  */
  if (SCI_INPUT_READY == TRUE)
  {
    INPUT_ERROR = FALSE;
    /*
    **  Use command line input if not streaming data
    */
    if (XYZ_STREAM == FALSE)
    {
      PROMPT_WAIT = FALSE;
      /*
      **  Get first input character - only uppercase
      */
      switch (BufferRx[0])
      {
        /***************************************************************************************/                 
        case '?':
          /*
          **  Help : list valid commands
          */              
          SCISendString("\r\n");
          SCISendString(" ---- Function ---- \r\n");
          SCISendString("T     : Tilt Sensing Application: \r\n");
          SCISendString("         0 = off (Default), 1 = on\r\n");
          SCISendString("D     : Raw Data: \r\n");
          SCISendString("         0 = off (Default), 1 = on\r\n");
          SCISendString(" ---- Sampling Configuration ---- \r\n");
          SCISendString("Nn    : Taking <n> samples.\r\n");
          SCISendString("         x = unlimited (Default)\r\n");  
          SCISendString("         0 = 1 sample, 1 = 100 samples,2 = 500 samples\r\n");
          SCISendString("Fn    : Data output format. \r\n");
          SCISendString("         0 = sign counts (Default), 1 = sign frac g values\r\n");
          SCISendString(" ---- Sampling ---- \r\n");
          SCISendString("S     : Start sampling\r\n");
          break;
          
        /***************************************************************************************/
        case 'T':
          switch (BufferRx[1]){
            case  '0': 
              TILT_APP = FALSE;
              SCISendString("Tilt application is OFF. \r\n");
              break;
            case  '1': 
              TILT_APP = TRUE;
              SCISendString("Tilt application is ON. \r\n");  
              break;
            default: 
              SCISendString((byte*)string_What);
              break;
          }
          break;
        
        /***************************************************************************************/
        case 'D':
          switch (BufferRx[1]){
            case  '0': 
              DATA_APP = FALSE;
              SCISendString("Data reading is OFF. \r\n");
              break;
            case  '1': 
              DATA_APP = TRUE;
              SCISendString("Data reading is ON. \r\n");  
              break;
            default: 
              SCISendString((byte*)string_What);
              break;
          }
          break;
        /***************************************************************************************/                 
        case 'N':
          switch (BufferRx[1]){
            case  '0': 
              sample_num = 1;
              SCISendString("Sample number = 1\r\n");  
              break;
            case  '1': 
              sample_num = 100;
              SCISendString("Sample number = 100\r\n");   
              break;
            case  '2': 
              sample_num = 10000;
              SCISendString("Sample number = 10000\r\n");    
              break;
            case  'X':
              sample_num = 666;
              SCISendString("Sample number = stream\r\n");
              break;  
            default: 
              SCISendString((byte*)string_What);
              break;
          }
          break;

        /***************************************************************************************/
        case 'F':
          switch (BufferRx[1]){
            case  '0': 
              STREAM_FULLC = 1;
              SCISendString("Output format = Signed counts\r\n");
              break;
            case  '1': 
              STREAM_FULLC = 0;
              SCISendString("Output format = Signed gs\r\n");  
              break;
            default: 
              SCISendString((byte*)string_What);
              break;
          }
          break;
        
        /***************************************************************************************/
        case 'S':
          //ODR_400 = TRUE ; //FL
          functional_block = FBID_FULL_XYZ_SAMPLE;
          sample_num_dec = sample_num;
          Tmr_counter = 0;
          Print_Setting();
          SCISendString("******    Start to take samples    *****\r\n");
          XYZ_STREAM = TRUE;
          RTC_ENABLED;         
          PROMPT_WAIT = TRUE;
          break;

        /***************************************************************************************/
        default:
          /*
          **  Undefined inputs are ignored.
          */
          INPUT_ERROR = TRUE;
          break;
      } //end of case statement for main list of options  
      
      if (INPUT_ERROR == TRUE)
      {
        SCISendString((byte*)string_What);
      }
      SCI_INPUT_READY = FALSE;
      temp = 0;
    }
      
    /*
    **  Data streaming is stopped by any character input.
    */
    else
    {
      /*
      **  Turn off data streaming
      */
      INT_PINS_DISABLED;
      POLL_ACTIVE = FALSE;
      XYZ_STREAM = FALSE;
      SCI_INPUT_READY = FALSE;
      PROMPT_WAIT = FALSE;
      RTCSC_RTIE = FALSE;
      
      /*
      **  If samples were being buffered, output them now.
      */
      if (ODR_400 == TRUE) 
      {
        SCI_putCRLF();
        address_out[0] = 0;
        address_out[1] = 0;
        address_out[2] = 0;

        /*
        **  Read samples stored in Data Flash
        */
        while (ODR_400 == TRUE)
        {
          ReadDataFlashXYZ();
          /*
          **  Output formats are:
          **    - Stream Full Resolution XYZ data as signed counts
          **    - Stream Full Resolution XYZ data as signed g's
          */
          if (STREAM_FULLC == 1) 
          {
              PrintXYZdec14();            
          }
          else
          {
              PrintXYZfrac();
          }
          SCI_putCRLF();
          /*
          **  Adjust Data Flash address pointer
          */
          if (address_out[2] > 245)
          {
            address_out[2] = 0;
            address_out[1]++;
          }
          else
          {
            address_out[2] += 6;
          }
          /*
          **  Check if we're done.
          */
          if (address_out[0] == address_in[0])
          {
            if (address_out[1] == address_in[1])
            {
              if (address_out[2] == address_in[2])
              {
                ODR_400 = FALSE;
              }
            }
          }
        }
        /*
        **  Erase the Data Flash
        */
        ODR_400 = TRUE;
        SCISendString  ("\r\nErasing Data Flash ... Please wait ");
        while (SCItxActive);
        ClearDataFlash();
      }
    }
  }
}


/*********************************************************\
**  Terminal Output
\*********************************************************/
void OutputTerminal (byte BlockID, byte *ptr)
{
  switch (BlockID)
  {
    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_FULL_XYZ_SAMPLE:
      /*
      **  Full Resolution XYZ Sample Registers (0x00 - 0x06)
      */      
      
      CopyXYZ(ptr); 
      
      if (STREAM_FULLC == 1)
      {
          PrintXYZdec14();
      }
      else
      {
          PrintXYZfrac();
      }
            
      SCI_putCRLF();
      
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case MESSAGE:
      SCISendString(ptr);
      break;
    
    default:
      break;
  }
}


/***********************************************************************************************\
* Private functions
\***********************************************************************************************/

/*********************************************************\
* Print accelerometer's ODR, Sample number and Output format
\*********************************************************/
void Print_Setting (void)
{
  SCISendString("\r\n");
  SCISendString("\r\n**   MMA8491Q Setting   **\r\n");
  SCISendString("\r\nMMA8491Q IIC address = 0xAA");
  SCISendString("\r\n");

  SCISendString("Application is ON: ");
  if (TILT_APP == TRUE)
    SCISendString("Tilt  ");  
  if (DATA_APP == TRUE)
    SCISendString("DATA  ");
  SCISendString("\r\n");
  
  SCISendString("Sampling config is: \r\n");
  SCISendString("  Sample Number = ");
  switch (sample_num_dec)
  {
    case 1:   SCISendString("1"); break;
    case 100: SCISendString("100"); break;
    case 500: SCISendString("500"); break;
    case 666: SCISendString("unlimited"); break;
    default:  SCISendString("unlimited");break;
  }
  SCISendString("\r\n");
  
  SCISendString("  ODR = 0.25Hz\r\n");
  //SCISendString((byte*)ODRvalues);
  //SCISendString(" Hz\r\n");
  
  SCISendString("  Output format = ");
  if (STREAM_FULLC == 1)
  {
    SCISendString("Signed counts ");
  } 
  else
  {    
    SCISendString("Signed gs ");
  }
  SCISendString("\r\n");      
}
  

/*********************************************************\
*
\*********************************************************/
void WriteFlashBuffer (void)
{
  /*
  **  Save sample to Data Flash
  */
  DATAFLASH_WriteEnableLatch();
  SPI_SS_SELECT;
  SPI_ChrShift(0x02);
  SPI_ChrShift(address_in[0]);
  SPI_ChrShift(address_in[1]);
  SPI_ChrShift(address_in[2]);
  SPI_ChrShift(x_value.Byte.hi);  
  SPI_ChrShift(x_value.Byte.lo);
  SPI_ChrShift(y_value.Byte.hi);
  SPI_ChrShift(y_value.Byte.lo);
  SPI_ChrShift(z_value.Byte.hi);
  SPI_ChrShift(z_value.Byte.lo);
  SPI_SS_DESELECT;
  /*
  **  Adjust Data Flash address pointer
  */
  if (address_in[2] > 245)
  {
    address_in[2] = 0;
    address_in[1]++;
  }
  else
  {
    address_in[2] += 6;
  }
}


/*********************************************************\
*
\*********************************************************/
void PrintBuffering (void)
{
  if (temp > 50)
  {
    SCI_putCRLF();
    temp = 0;
    address_out[2] += 50;
    if (address_out[2] < 50)
    {
      address_out[1]++;
      if (address_out[1] == 0)
      {
        address_out[0]++;
      }
    }
  }
  if (temp++ == 0)
  {
    hex2ASCII(address_out[0],&value[0]);
    hex2ASCII(address_out[1],&value[2]);
    value[4] = 0;
    SCISendString(&value[0]);
    hex2ASCII(address_out[2],&value[0]);
    value[2] = 0;
    SCISendString(&value[0]);
  }
  SCI_CharOut('*');
}


/*********************************************************\
*
\*********************************************************/
void PrintFIFO (void)
{
}


/*********************************************************\
*
\*********************************************************/
void ReadDataFlashXYZ (void)
{
  SPI_SS_SELECT;
  SPI_ChrShift(0x0B);
  SPI_ChrShift(address_out[0]);
  SPI_ChrShift(address_out[1]);
  SPI_ChrShift(address_out[2]);
  SPI_ChrShift(0x00);
  x_value.Byte.hi = SPI_ChrShiftR(0x00);
  x_value.Byte.lo = SPI_ChrShiftR(0x00);
  y_value.Byte.hi = SPI_ChrShiftR(0x00);
  y_value.Byte.lo = SPI_ChrShiftR(0x00);
  z_value.Byte.hi = SPI_ChrShiftR(0x00);
  z_value.Byte.lo = SPI_ChrShiftR(0x00);
  SPI_SS_DESELECT;
}


/*********************************************************\
*
\*********************************************************/
byte ProcessHexInput (byte *in, byte *out)
{
  byte data;

  data = *in++;
  if (ishex(data) == TRUE)
  {
    data = tohex(data);
  }
  else
  {
    return (0);
  }
  if (ishex(*in) == TRUE)
  {
    data <<= 4;
    data += tohex(*in);
    *out = data;
    return (2);
  }
  else if ((*in == ' ') || (*in == 0))
  {
    *out = data;
    return (1);
  }
  return (0);
}


/*********************************************************\
*
\*********************************************************/
void CopyXYZ (byte *ptr)
{
  x_value.Byte.hi = *ptr++;
  x_value.Byte.lo = *ptr++;
  y_value.Byte.hi = *ptr++;
  y_value.Byte.lo = *ptr++;
  z_value.Byte.hi = *ptr++; 
  z_value.Byte.lo = *ptr;
       
}

void CopyXYZ8 (byte *ptr) 
{
  x_value.Byte.hi = *ptr++;
  x_value.Byte.lo = 0;
  y_value.Byte.hi = *ptr++;
  y_value.Byte.lo = 0;
  z_value.Byte.hi = *ptr; 
  z_value.Byte.lo = 0;
}
  


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec14 (void)
{
  SCISendString("X= ");
  SCI_s14dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s14dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s14dec_Out(z_value);
}
/*********************************************************\
*
\*********************************************************/
 void PrintXYZdec12 (void)
{
  SCISendString("X= ");
  SCI_s12dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s12dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s12dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
 void PrintXYZdec10 (void)
{
  SCISendString("X= ");
  SCI_s10dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s10dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s10dec_Out(z_value);
}

/*********************************************************\
*
\*********************************************************/
void PrintXYZdec8 (void)
{
  SCISendString("X= ");
  SCI_s8dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s8dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s8dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZfrac (void)
{
  SCISendString("X= ");
  SCI_s14frac_Out(x_value);
  SCISendString("  Y= ");
  SCI_s14frac_Out(y_value);
  SCISendString("  Z= ");
  SCI_s14frac_Out(z_value);
}

/*********************************************************\
*
\*********************************************************/
void ClearDataFlash (void)
{
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;
  DATAFLASH_Erase();
}

