/**##########################################################################**/
/*     Author    : Jose Cisneros based on Matthew Grant version               */
/*     Device    : MC9S12XHY256                                               */
/*     DataSheet : MC9S12XHY256RMV0.11.pdf                                    */
/*     Compiler  : Code Warrior C compiler 5.01                               */
/*     Date      : 10.05.2010                                                 */
/*     Company   : Freescale Semiconductor Inc.                               */
/*     Purpose   : THIS IS A PROJECT, FOR FREESCALE'S SSD0 MODULE             */
/*                 ON THE MC9S12XHY256 MCU. THE FORMAT IS CONSISTENT WITH     */
/*                 PROJECTS GENERATED BY THE CODE WARRIOR WIZARD. THIS        */
/*                 PROJECT IS INTENDED TO BE A WORKING EXAMPLE UPON WHICH     */
/*                 USERS MAY GAIN FAMILIARITY WITH THE SSD MODULE. THIS       */
/*                 PROJECT IS PROVIDED AS IS WITH NO PROMISE OF SUPPORT,      */
/*                 AND FREESCALE MAKES NO GUARANTEE THIS PROJECT OR THE       */
/*                 FUNCTIONS INCLUDED WILL PERFORM AS DESIRED OR MEET THE     */
/*                 DEMANDS OF ANY REAL APPLICATION.                           */
/*                                                                            */
/*     Setup     : THIS PROJECT IS WRITTEN TO WORK WITH A SMALL STEPPER       */
/*                 MOTOR CONNECTED TO THE SSD0 MODULE PINS ON THE             */
/*                 DEMO9S12XHY256 . IT IS ASSUMED THE STEPPER MOTOR HAS A     */
/*                 POINTER ATTACHED TO THE SHAFT COMING OUT OF THE MOTOR.     */
/*                 WHEN THE PROJECT IS WORKING, THE USER SHOULD SEE THE       */
/*                 POINTER ROTATE IN ONE DIRECTION UNTIL IT BUMPS INTO AN     */
/*                 OBJECT. WHEN THE POINTER BUMPS INTO AN OBJECT, THE SSD     */
/*                 ACCUMULATOR SHOULD YIELD AN INTEGRATION RESULT WITH A      */
/*                 LOWER MAGNITUDE VALUE THAT IS BETWEEN THE STALL            */
/*                 MAGNITUDE AND ZERO. THE CODE MAY INTERPRET THIS AS A       */
/*                 STALL, AND REVERSE THE ROTATIONAL DIRECTION. THE PROCESS   */
/*                 REPEATS BY BEGINNING ANOTHER SEQUENCE OF STEPS UNTIL       */
/*                 ANOTHER STALL IS DETECTED AND THE ROTATIONAL DIRECTION     */
/*                 IS AGAIN REVERSED.                                         */
/*                                                                            */
/*                 PER APPLICATION, THE FOLLOWING CONSTANTS MAY NEED TO       */
/*                 BE ADJUSTED.                                               */
/*                                                                            */
/*                 INTEGRATION_COUNT                                          */
/*                 BLANKING_COUNT                                             */
/*                 STALL                                                      */
/**##########################################################################**/

#include <hidef.h>        /* common defines and macros */
#include <MC9S12XHY256.h> /* derivative information */

/*********CONSTANTS AND DEFINES*********/

typedef enum
{
   NO_STALL = 0,
   STALLED,
   OVERFLOW
	 
} SM_STATE; /* Stepper Motor State */


/* # of modulus counts during which integration is performed. */
const INTEGRATION_COUNT = 1000;

/* # of modulus counts during which blanking is performed.    */
const BLANKING_COUNT    = 200;

/* 
 Stall level. Adjust to match specific setup. If the SSD takes steps
 with integration and the pointer bumps into a stop device, and
 continues to do so, the stall level may be set too low. 
*/
const STALL = 100;

/* -------------------------- Function Definitions ------------------------- */

/***************************************************************
*	CRG_8MHz
*
*	Setup clock module for PLL Engaged using externals XTAL
*	For DEMO9S12XHY256 with 8MHz XTAL 
*	Crystal  -  8MHz
* VCOCLK   - 32MHz, 
* PLL      - 16MHz, 
* Bus Clk  -  8MHz
* Suggestion: You can use iPLL calculator to speed up CRG 
*             registers configuration (look for S12XESW.zip)
*             at www.freescale.com.
****************************************************************/
void CRG_8MHz(void)
{

   SYNR             = 0x01; /* 32 < VCO freq < 48 MHz,      */
   REFDV            = 0x80; /* Freq ref within 6 and 12 MHz */
   POSTDIV          = 0x01; /* PLL = 32MHz, FBus = 16 MHz   */     
    
   while(!CRGFLG_LOCK)
   {
      ;                     /* Intentionally  left empty    */
   }			 
   CLKSEL_PLLSEL = 1;       /* wait for VCO to stabilize    */
}

/*******************  Global Variables Definition   ***************************/
signed int  integrated_value; /* Holds result of the SSD accumulator                 */
signed char step_state;       /* Keeps track of the 4 possible step states:0,1,2,3   */
signed char clockwise;        /* Indicates the motor's relative rotational direction */
SM_STATE    result;           /* Holds the result of the stall detection function	   */
signed char * step;           /* Used to point to the step_state character variable  */

/* The following variables will allow to calibrate the SSD module on the fly  */
int Integration_Count = INTEGRATION_COUNT;
int Blanking          = BLANKING_COUNT;
int Stall             = STALL;
int Stall_Counter     = 0;

/*********************  Function Prototypes   *********************************/
SM_STATE SSD0_Step(int integration_count, int blanking_count,
                          int stall_level, signed char * step);
                          
/*********************  Function DEfinitions  *********************************/

/*******************************************************************************
*	Name: main()
*
*	Description: Main function caller in charge to change motor direction. 
*	Parameters:  None.
*	Returns:     void.
*	
*******************************************************************************/
void main(void)
{

   EnableInterrupts;

   CRG_8MHz();
   /* 
   Initialize the step state to state 0. In a real application,
   the initial position/starting state of the motor pointer may
   not be perfectly aligned with state 0. Software may need to
   take this into consideration if it is critical that the
   initial step state be aligned with the physical state of
   the motor.
   */
   step_state = 0;       /* Arbitrarily initialized to 0.                */

   step = &step_state;   /* Point the char * to the step_state variable. */
   clockwise = 1;        /* Arbitrarily init rotational direction to 1.  */
   for(;;)
   {
      while((step_state != 4) && (step_state != -1))
      {
         /* Call the SSD0 step function, which uses Port V[0:3] */
         result = SSD0_Step(Integration_Count, Blanking, Stall, step);
         if(OVERFLOW == result)
         {
            /********* ACCUMULATOR OVERFLOW DETECTED, TAKE ACTION *********/
            for(;;); /* Loop forever. User may change to perform some other action */
         }
         if(STALLED == result)
         {
            /******* SINCE STALL DETECTED, REVERSE MOTOR DIRECTION *******/
            
            /*              NOTE:
                    Not checking for overflow
            */
            
            Stall_Counter ++;
            
            if (clockwise)
            {        
               clockwise = 0;
            }        
            else
            {
               clockwise = 1;        
            }        
         }
         
         /****************** MOVE TO NEXT STEP STATE ********************/
         if (clockwise)
         {
            /*
            Move clockwise (CW)through step states. This
            is all relative and dependent upon the arrangement of
            the MCU to motor pin connections.
            */
            step_state++;
         }
         else
         {
            /*
            Move counter-clockwise (CCW)through step states. This
            is all relative and dependent upon the arrangement of
            the MCU to motor pin connections.
            */
            step_state--;
         }
      }/* while */
    
      /******************* CYCLE THE STEP STATES ************************/
      /*
      The valid step states cycle from 0-3. When code has incremented
      above, or decremented below the valid step states, cycle the
      step_state variable to the next valid value. This should be
      either state 0, or state 3.
      */
      if (clockwise)
      {
         step_state = 0;
      }
      else
      {
         step_state = 3;
      }
   }/* for */
 
   for(;;); /* wait forever */
   /* please make sure that you never leave the Main function */
}

/*******************************************************************************
*	Name: SSD0_Step()
*
*	Description: This function will take care of SSD logic, by configuring 
*	             microcontroller registers according to the SSD logic state
*	             as follows:
*              1.- SSD registers initialization to perform a step.
*              2.- If enabled, configure registers to perform blanking time.
*              3.- Configure registers to perform integration time.
*              4.- Read and compare integration result against stall value.
*
*	Parameters:  integration_count:
*	             blanking_count:
*	             stall_level:
*	             step:.
*	Returns:     Stepper motor state: NO_STALL, STALLED or integration OVERFLOW.
*	
*******************************************************************************/
SM_STATE SSD0_Step(int integration_count, int blanking_count,
               int stall_level, signed char * step)
{

   /**************INIT THE SDD FOR THE NEXT STEP****************/
   RTZ0CTL_RCIR  = 0;    /* Perform recirculation on low side.                */
   RTZ0CTL_POL   = 0;    /* Set polarity.                                     */
   RTZ0CTL_SMS   = 0;    /* Always keep SMS bit cleared.                      */
   RTZ0CTL_STEP  = *step;/* Set step state before stepping with integration.  */
   MDC0CTL_PRE   = 0;    /* Clear or set the SSD prescaler as desired.        */
   SSD0CTL_ACLKS = 0;    /* Setup the SSD sample freq as desired.             */
   SSD0CTL_RTZE  = 1;    /* Enable SSD. SSD now controls the port pins.       */
   SSD0CTL_SDCPU = 1;    /* Power up the sigma delta converter.               */
  
   /******************IF BLANKING IS DESIRED********************/
   /* In some cases, an application may work even when blanking is not  	    */
   /* used. To skip blanking, and go straight to integration, set the	        */
   /* function parameter blanking_count to 0.													        */           
   if(0 < blanking_count)
   {
      /*******************BEGIN TAKING A STEP**********************/
      SSD0FLG_MCZIF = 1;       /* Clear the modulus down counter zero flag.   */
      RTZ0CTL_ITG   = 0;       /* Disable integration before blanking.        */

 
   	  MDC0CTL_MODMC = 0;              /* Set the modulus mode to 0.           */
      MDC0CTL_MCEN  = 1;              /* Enable modulus down-counter.         */
      MDC0CNT       = blanking_count; /* Load the blanking count.             */
      RTZ0CTL_DCOIL = 1;              /* Turn on the SSD channel coil drivers */
      
      /******************WAIT FOR END OF BLANKING******************/
      while(0 == SSD0FLG_MCZIF);
   }
  
   /********************START INTEGRATION***********************/
   SSD0FLG_MCZIF = 1;           /* Clear the modulus down counter zero flag. */
   /* MDC0CTL_MCZIE  = 1; */    /* Uncomment to allow zero flag interrupt.   */
	 MDC0CTL_MODMC = 0;           /* Set the modulus mode to 0.                */
   SSD0FLG_AOVIF = 1;           /* Clear the overflow flag.                  */
   MDC0CTL_MCEN  = 1;           /* Enable modulus down-counter.              */
   MDC0CNT       = integration_count; /* Load the integration count.         */

   /* 
   NOTE: If integration is enabled before the modulus counter is set up,
   the SSD module may not perform offset cancelation. This increases
   the possibility of large offset errors corrupting the integration value.
   It is recommended to enable integration AFTER the modulus counter has
   been setup and enabled.
   */

   RTZ0CTL_ITG   = 1;   /* Begin integration.                    */
   RTZ0CTL_DCOIL = 1;   /* Turn on the SSD channel coil drivers. */
   
   /****************WAIT FOR END OF INTEGRATION*****************/
   while(0 == SSD0FLG_MCZIF);
   
   /*************READ ITGACC & DISABLE INTEGRATION**************/
   integrated_value = ITG0ACC; /* Immediately read-store the integration result. */
	 RTZ0CTL_ITG      = 0;   /* Turn off integration.                   */
   SSD0FLG_MCZIF    = 1;   /* Clear MCZIF flag for future interrupts. */
   RTZ0CTL_DCOIL    = 0;   /* Turn off the DCOIL.                     */
   MDC0CTL_MCEN     = 0;   /* Disable the modulus down-counter        */
   
   /* Since the calling function will use the SSD again to take
    another step, RTZE can be left enabled. If disabling is   
    desired, uncomment the line below.
   */
   /* SSD2CTL_RTZE = 0; */

   /* NOW THAT INTEGRATION HAS COMPLETED, CHECK THE RESULT */
   /***************CHECK FOR ACCUMULATOR OVERFLOW***************/
	 if(1 == SSD0FLG_AOVIF)
	 {
      return(OVERFLOW);     /* Overflow detected. */
   }         
   /*********CHECK IF INTEGRATION VALUE IN STALL RANGE**********/
	 if(((integrated_value <= stall_level)    && (integrated_value >= 0)) ||
	    ((integrated_value >= (-stall_level)) && (integrated_value <= 0)))
   {        	  	  
      return(STALLED);    /* Stall detected. */
   }
   else
   {
      return(NO_STALL); /* No stall detected. */
   }
}
