/****************************************************************/
/* AES-CMAC with AES-128 bit                                    */
/* CMAC     Algorithm described in SP800-38B                    */
/* Author: Junhyuk Song (junhyuk.song@samsung.com)              */
/*         Jicheol Lee  (jicheol.lee@samsung.com)               */
/* Modified by Sri Mahadevan -Sri.Mahadevan@freescale.com       */
/****************************************************************/

  #include <stdio.h>
  #include <string.h>

  /* For CMAC Calculation */
  unsigned char const_Rb[16] = {
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x87
  };
  unsigned char const_Zero[16] = {
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };

// The number of columns comprising a state in AES. This is a constant in AES. Value=4
#define Nb 4

// The number of rounds in AES Cipher. It is simply initiated to zero. The actual value is recieved in the program.
int Nr=0;
int Nk=0;

/*************************************************************************/
// in - it is the array that holds the plain text to be encrypted.
// out - it is the array that holds the key for encryption.
// state - the array that holds the intermediate results during encryption.
// Roundkey - The array that stores the round keys.
// Key - The Key input to the AES Program
/*************************************************************************/

unsigned char in[16], out[16], state[4][4];
unsigned char RoundKey[240];
unsigned char Key[32];

int getSBoxValue(int num)
{
	int sbox[256] =   {
	//0     1    2      3     4    5     6     7      8    9     A      B    C     D     E     F
	0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76, //0
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0, //1
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15, //2
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75, //3
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84, //4
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf, //5
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8, //6
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2, //7
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73, //8
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb, //9
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79, //A
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08, //B
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a, //C
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e, //D
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf, //E
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 }; //F
	return sbox[num];
}

// The round constant word array, Rcon[i], contains the values given by 
// x to th e power (i-1) being powers of x (x is denoted as {02}) in the field GF(28)
// Note that i starts at 1, not 0).
int Rcon[255] = {
	0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 
	0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 
	0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 
	0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 
	0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 
	0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc,
	0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 
	0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 
	0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 
	0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 
	0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 
	0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 
	0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 
	0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 
	0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 
	0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb  };

// This function produces Nb(Nr+1) round keys. The round keys are used in each round to encrypt the states. 
void KeyExpansion()
{
	int i,j;
	unsigned char temp[4],k;
	
	// The first round key is the key itself.
	for(i=0;i<Nk;i++)
	{
		RoundKey[i*4]=Key[i*4];
		RoundKey[i*4+1]=Key[i*4+1];
		RoundKey[i*4+2]=Key[i*4+2];
		RoundKey[i*4+3]=Key[i*4+3];
	}

	// All other round keys are found from the previous round keys.
	while (i < (Nb * (Nr+1)))
	{
					for(j=0;j<4;j++)
					{
						temp[j]=RoundKey[(i-1) * 4 + j];
					}
					if (i % Nk == 0)
					{
						// This function rotates the 4 bytes in a word to the left once.
						// [a0,a1,a2,a3] becomes [a1,a2,a3,a0]

						// Function RotWord()
						{
							k = temp[0];
							temp[0] = temp[1];
							temp[1] = temp[2];
							temp[2] = temp[3];
							temp[3] = k;
						}

						// SubWord() is a function that takes a four-byte input word and 
						// applies the S-box to each of the four bytes to produce an output word.

						// Function Subword()
						{
							temp[0]=getSBoxValue(temp[0]);
							temp[1]=getSBoxValue(temp[1]);
							temp[2]=getSBoxValue(temp[2]);
							temp[3]=getSBoxValue(temp[3]);
						}

						temp[0] =  temp[0] ^ Rcon[i/Nk];
					}
					else if (Nk > 6 && i % Nk == 4)
					{
						// Function Subword()
						{
							temp[0]=getSBoxValue(temp[0]);
							temp[1]=getSBoxValue(temp[1]);
							temp[2]=getSBoxValue(temp[2]);
							temp[3]=getSBoxValue(temp[3]);
						}
					}
					RoundKey[i*4+0] = RoundKey[(i-Nk)*4+0] ^ temp[0];
					RoundKey[i*4+1] = RoundKey[(i-Nk)*4+1] ^ temp[1];
					RoundKey[i*4+2] = RoundKey[(i-Nk)*4+2] ^ temp[2];
					RoundKey[i*4+3] = RoundKey[(i-Nk)*4+3] ^ temp[3];
					i++;
	}
}

// This function adds the round key to state.
// The round key is added to the state by an XOR function.
void AddRoundKey(int round) 
{
	int i,j;
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			state[j][i] ^= RoundKey[round * Nb * 4 + i * Nb + j];
		}
	}
}

// The SubBytes Function Substitutes the values in the
// state matrix with values in an S-box.
void SubBytes()
{
	int i,j;
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			state[i][j] = getSBoxValue(state[i][j]);

		}
	}
}

// The ShiftRows() function shifts the rows in the state to the left.
// Each row is shifted with different offset.
// Offset = Row number. So the first row is not shifted.
void ShiftRows()
{
	unsigned char temp;

	// Rotate first row 1 columns to left	
	temp=state[1][0];
	state[1][0]=state[1][1];
	state[1][1]=state[1][2];
	state[1][2]=state[1][3];
	state[1][3]=temp;

	// Rotate second row 2 columns to left	
	temp=state[2][0];
	state[2][0]=state[2][2];
	state[2][2]=temp;

	temp=state[2][1];
	state[2][1]=state[2][3];
	state[2][3]=temp;

	// Rotate third row 3 columns to left
	temp=state[3][0];
	state[3][0]=state[3][3];
	state[3][3]=state[3][2];
	state[3][2]=state[3][1];
	state[3][1]=temp;
}

// xtime is a macro that finds the product of {02} and the argument to xtime modulo {1b}  
#define xtime(x)   ((x<<1) ^ (((x>>7) & 1) * 0x1b))

// MixColumns function mixes the columns of the state matrix
void MixColumns()
{
	int i;
	unsigned char Tmp,Tm,t;
	for(i=0;i<4;i++)
	{	
		t=state[0][i];
		Tmp = state[0][i] ^ state[1][i] ^ state[2][i] ^ state[3][i] ;
		Tm = state[0][i] ^ state[1][i] ; Tm = xtime(Tm); state[0][i] ^= Tm ^ Tmp ;
		Tm = state[1][i] ^ state[2][i] ; Tm = xtime(Tm); state[1][i] ^= Tm ^ Tmp ;
		Tm = state[2][i] ^ state[3][i] ; Tm = xtime(Tm); state[2][i] ^= Tm ^ Tmp ;
		Tm = state[3][i] ^ t ; Tm = xtime(Tm); state[3][i] ^= Tm ^ Tmp ;
	}
}

// Cipher is the main function that encrypts the PlainText.
void Cipher()
{
	int i,j,round=0;

	//Copy the input PlainText to state array.
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			state[j][i] = in[i*4 + j];
		}
	}

	// Add the First round key to the state before starting the rounds.
	AddRoundKey(0); 
	
	// There will be Nr rounds.
	// The first Nr-1 rounds are identical.
	// These Nr-1 rounds are executed in the loop below.
	for(round=1;round<Nr;round++)
	{
		SubBytes();
		ShiftRows();
		MixColumns();
		AddRoundKey(round);
	}
	
	// The last round is given below.
	// The MixColumns function is not here in the last round.
	SubBytes();
	ShiftRows();
	AddRoundKey(Nr);

	// The encryption process is over.
	// Copy the state array to output array.
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			out[i*4+j]=state[j][i];
		}
	}
}


  /* Basic Functions */

  /*  XOR Inputs a & b and generates output out */

  void xor_128(unsigned char *a, unsigned char *b, unsigned char *out)
  {
      int i;
      for (i=0;i<16; i++)
      {
          out[i] = a[i] ^ b[i];
      }
  }


  void print_hex(FILE *fname,char *str, unsigned char *buf, int len)
  {
      int     i;

      for ( i=0; i<len; i++ ) {
          if ( (i % 16) == 0 && i != 0 ) fprintf(fname,str);
          fprintf(fname,"%02x", buf[i]);
          if ( (i % 4) == 3 ) fprintf(fname," ");
          if ( (i % 16) == 15 ) fprintf(fname,"\n");
      }
      if ( (i % 16) != 0 ) fprintf(fname,"\n");
  }

  void print128(unsigned char *bytes)
  {
      int         j;
      for (j=0; j<16;j++) {
          printf("%02x",bytes[j]);
      }
  }

  void print96(unsigned char *bytes)
  {
      int         j;
      for (j=0; j<12;j++) {
          printf("%02x",bytes[j]);
          if ( (j%4) == 3 ) printf(" ");
      }
  }

  /* AES-CMAC Generation Function */

  void leftshift_onebit(unsigned char *input,unsigned char *output)
  {
      int         i;
      unsigned char overflow = 0;

      for ( i=15; i>=0; i-- ) {
          output[i] = input[i] << 1;
          output[i] |= overflow;
          overflow = (input[i] & 0x80)?1:0;
      }
      return;
  }

  void generate_subkey(unsigned char *key, unsigned char *K1, unsigned
                       char *K2)
  {
      unsigned char L[16];
      unsigned char Z[16];
      unsigned char tmp[16];
      int i;

      for ( i=0; i<16; i++ ) Z[i] = 0;

      // Perform AES_128 on Input Data Z with Key "Key" and store output in L
      for ( i=0; i<16; i++) {
          Key[i] = key[i];
	  in[i]  = Z[i];
      }
      KeyExpansion();
      Cipher();
      for ( i=0; i<16; i++) {
          L[i] =out[i];
      }


      if ( (L[0] & 0x80) == 0 ) { /* If MSB(L) = 0, then K1 = L << 1 */
          leftshift_onebit(L,K1);
      } else {    /* Else K1 = ( L << 1 ) (+) Rb */
          leftshift_onebit(L,tmp);
          xor_128(tmp,const_Rb,K1);
      }

      if ( (K1[0] & 0x80) == 0 ) {
          leftshift_onebit(K1,K2);
      } else {
          leftshift_onebit(K1,tmp);
          xor_128(tmp,const_Rb,K2);
      }
      return;
  }

  void padding ( unsigned char *lastb, unsigned char *pad, int length )
  {
      int         j;

      /* original last block */
      for ( j=0; j<16; j++ ) {
          if ( j < length ) {
              pad[j] = lastb[j];
          } else if ( j == length ) {
              pad[j] = 0x80;
          } else {
              pad[j] = 0x00;
          }
      }
  }

  void AES_CMAC ( unsigned char *key, unsigned char *input,
                  int flag, unsigned char *mac,int msg_len_bits)
  {
      unsigned char       X[16],Y[16], M_last[16], padded[16];
      unsigned char       K1[16], K2[16];
      int         num_blocks, i ,j;


      generate_subkey(key,K1,K2);

      num_blocks = (msg_len_bits+127)/128;

      if ( num_blocks == 0 ) {
          num_blocks = 1;
      } 

      if ( flag == 0) { /* last block is complete block */
          xor_128(&input[16*(num_blocks-1)],K1,M_last);
      } else {
          xor_128(&input[16*(num_blocks-1)],K2,M_last);
      }


      for ( i=0; i<16; i++ ) X[i] = 0;
      for ( i=0; i<num_blocks-1; i++ ) {
          xor_128(X,&input[16*i],Y); /* Y := Mi (+) X  */

	//    Perform AES on Value Y withe Key "key" and output is stored in X. 
	//    X := AES-128(KEY, Y); 

          for ( j=0; j<16; j++) {
              Key[j] = key[j];
              in[j]  = Y[j];
          }
          KeyExpansion();
          Cipher();
          for ( j=0; j<16; j++) {
              X[j] =out[j];
          }
      }

      xor_128(X,M_last,Y);

      //    Perform AES on Value Y withe Key "key" and output is stored in X. 
      //    X := AES-128(KEY, Y); 

      for ( i=0; i<16; i++) {
          Key[i] = key[i];
          in[i]  = Y[i];
      }
      KeyExpansion();
      Cipher();
      for ( i=0; i<16; i++) {
          X[i] =out[i];
      }

      for ( i=0; i<16; i++ ) {
          mac[i] = X[i];
      }
  }

// get128 function determines and passes back the actual value
// of and argv string (as opposed its ascii character value
// representation)
void get128(char *argv_array, char out[16]) {
      int i;
      char tmp, tmp2;
      char upper, lower;

      for(i=0;i<Nk*4;i++) {
        tmp = argv_array[i*2];
        if (tmp >= '0' && tmp <= '9')
          tmp2 = tmp - '0';
        else if (tmp >= 'a' && tmp <= 'f')
          tmp2 = tmp - 'W';
        else if (tmp >= 'A' && tmp <= 'F')
          tmp2 = tmp - '7';
        upper = tmp2 << 4;

        tmp = argv_array[i*2+1];
        if (tmp >= '0' && tmp <= '9')
          tmp2 = tmp - '0';
        else if (tmp >= 'a' && tmp <= 'f')
          tmp2 = tmp - 'W';
        else if (tmp >= 'A' && tmp <= 'F')
          tmp2 = tmp - '7';
        lower = tmp2;
        out[i] = upper | lower;
      }
}


  int main(int argc, char *argv[])
  {
      unsigned char L[16],K1[16],K2[16],T[16],TT[12];
      unsigned char M[2048],key[16],MLB[16];
      unsigned char T_bit[16];
      unsigned char MLB_pad, MLB_ext_bit;
      unsigned char user_key[16];

      int i,msg_len_bytes,mlb_mod_bytes,j,num_blks,flag;
      int msg_len_bits,mlb_mod_bits,num_ext_bits;
      FILE *fp1_infile,*fp2_outfile;

      // KEY SIZE is FIXED TO 128bits
      Nr = 128;
      // Calculate Nk and Nr from the recieved value.
      Nk = Nr / 32;
      Nr = Nk + 6;

      ///////////////////////////////////////////////////////
      // Input file contents:
      //    Message (128 bits Per line)
      //    .
      //    .
      ////////////////////////////////////////////////////


     if (argc != 4) { 
        printf("ERROR: Usage AES_CMAC_CMD.out <Key Value> <Message Length in Bits> <Message File name> \n");
	return 1;
     }
      
     if (strlen(argv[1]) != 32)  {
	printf("ERROR: Key Value is not 128 bits wide \n");
	return 1;
     }

       
      // Load argv[1] (key data) into "Key" variable
      get128(argv[1], user_key);
	
      // Load Argv[2] in to message length in bits
      msg_len_bits = atoi(argv[2]);

      num_blks = msg_len_bits/128;   // Compute number of 128 bit chunks

      if (msg_len_bits%128 == 0) 
	flag = 0;
      else 
	flag = 1;
         

      // argv[3] is file name where the Message for which CMAC needs
      // to be calculated is present

      if ((fp1_infile  = fopen(argv[3], "r")) == NULL) {
	 printf("ERROR: Input File Does not Exist \n");
         return 1;
      }
   
      // Read the number of 128bits chunks from the input file

       for(j=0; j<num_blks; j++) {
	  // Read the Message from the Input file. 
	  // Message is expected to be 128 bits per line
          for(i=0;i<16;i++) 
	  {
          	fscanf(fp1_infile,"%02x",&M[i+j*16]);
          }
	      fscanf(fp1_infile,"\n");
	}


      // Read the Message from the Input file
      // Last chunk of Message if not 128 bits 
      if (flag) 
      {
        mlb_mod_bits  = msg_len_bits%128;	// Compute the number of Last Block Bits
        num_ext_bits = mlb_mod_bits%8;		// Compute the number of Last Block Extra Bits that dont fill a byte 
        mlb_mod_bytes = (mlb_mod_bits)/8;	// Compute the number of Last Block Bytes	
	for (i=0; i<mlb_mod_bytes; i++)
	     fscanf(fp1_infile,"%02x",&MLB[i]);
	fscanf(fp1_infile,"%02x",&MLB_ext_bit);
        MLB_ext_bit = MLB_ext_bit >> (8-num_ext_bits);
	MLB_ext_bit = MLB_ext_bit << (8-num_ext_bits);
        MLB_pad = 0x80;
	MLB_pad = MLB_pad >> num_ext_bits;
	MLB_ext_bit = MLB_ext_bit | MLB_pad;
	MLB[i] = MLB_ext_bit;
	for (j=i+1; j<16; j++)
	     MLB[j] = 0x00;
	for (j=0; j<16; j++) {
	   M[16*num_blks+j] = MLB[j];
	}
      }   
      
//    Compute the CMAC Value for the above message and print it to the Output file

      AES_CMAC(user_key,M,flag,T,msg_len_bits);
      print128(T); printf("\n");


      return 0;
  }
