/**************************************************************************/
/* FILE NAME: main.c                       COPYRIGHT (c) Freescale 2009   */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* Used to load data arrays into QSPI module                         */
/* NOTE: All images must be divisible by 4bytes! Please pad with 0xFF to ensure this */
/* 			*/
/*                                                                        */
/* SETUP:                                                                 */
/* Software tested on MPC560xS EVB using Lauterbach and GHS 5.0.5 compiler with Winbond and Spansion QSPI Memory  */
/*                                                                        */
/*========================================================================*/
/* REV      AUTHOR        DATE        DESCRIPTION OF CHANGE               */
/* ---   -----------    ----------    ---------------------               */
/* 0.1    D. Mckenna       5/May/09    Initial Version                     */
/**************************************************************************/
#include "56xxS_0204.h"
#include "Ports.h"


/*========================================================================*/
/*                          DEFINES                                       */
/*========================================================================*/
#define ERASE_FIRST 1 //set to 1 to fully erase part before programming
#define START_ADDRESS 0x80000000 //address to program first image MUST BE 256BYTE ALIGNED
#define IMAGE_COUNT 1 //number of images (data arrays) to copy to memory - MAXIMUM = 9
#define CHECK_DATA 1 //set to 1 to validate wrtitten data
#define QUAD_READ 0 //set to 1 to use QUAD reads, 0 will use single line reads


#define QUADSPI_ARDB (*(vuint32_t *)(0x87FFFFFC));

/*========================================================================*/
/*                      GLOBAL VARIABLES                                  */
/*========================================================================*/

extern const uint8_t Image0[0x1000];
//extern const uint8_t Image1[512];
//extern const uint16_t Image2[512];
// extern const uint8_t Image3[];
// extern const uint8_t Image4[];
// extern const uint8_t Image5[];
// extern const uint8_t Image6[];
// extern const uint8_t Image7[];
// extern const uint8_t Image8[];

vuint32_t Image_Location[9];
vuint32_t Current_Address;
vuint32_t *current_image;
uint32_t image_size;
vuint32_t qspi_receive_data[16];


/*========================================================================*/
/*                      INPUTS PARAMETERS                                 */
/*========================================================================*/

/*========================================================================*/
/*                      OUTPUT PARAMETERS                                 */
/*========================================================================*/

/*========================================================================*/
/*                          PROTOTYPES                                    */
/*========================================================================*/
void MC_MODE_INIT();
void DISABLE_WATCHDOG();
uint16_t SetupPLL(uint16_t);
void CONFIG_QSPI(void);
void qspi_erase_flash(void);
void qspi_flash_write_enable(void);
void wait_while_flash_busy(void);
void qspi_program_page(void);  
void set_quad_bit(void);
void dma_read_example(void);
/*========================================================================*/
/*                          FUNCTIONS                                     */
/*========================================================================*/

int main(void)
{
	DISABLE_WATCHDOG();

	MC_MODE_INIT();   // select 16MHz IRC

	SetupPLL(64); // sets system clock to 64MHz. Assumes 4MHz XTAL

	CONFIG_QSPI();//configure QSPI module	

	
	#if ERASE_FIRST==1
		qspi_erase_flash(); /* erase contents of QSPI flash */   
	#endif
	
	QUADSPI_0.SFAR = START_ADDRESS;
	Current_Address = START_ADDRESS;	
	
	#if IMAGE_COUNT > 0
		current_image = (uint32_t *)Image0;
		image_size = sizeof(Image0);
		Image_Location[0] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 1
		current_image = (uint32_t *)Image1;
		image_size = sizeof(Image1);
		Image_Location[1] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 2
		current_image = (uint32_t *)Image2;
		image_size = sizeof(Image2);
		Image_Location[2] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 3
		current_image = (uint32_t *)Image3;
		image_size = sizeof(Image3);
		Image_Location[3] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 4
		current_image = (uint32_t *)Image4;
		image_size = sizeof(Image4);
		Image_Location[4] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 5
		current_image = (uint32_t *)Image5;
		image_size = sizeof(Image5);
		Image_Location[5] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 6
		current_image = (uint32_t *)Image6;
		image_size = sizeof(Image6);
		Image_Location[6] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT > 7
		current_image = (uint32_t *)Image7;
		image_size = sizeof(Image7);
		Image_Location[7] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	#if IMAGE_COUNT == 8
		current_image = (uint32_t *)Image8;
		image_size = sizeof(Image8);
		Image_Location[8] = QUADSPI_0.SFAR;
		if((image_size - ((image_size/4)*4))!=0)
			while(1); /*error! images must be divisable by 4bytes please pad with 0xFF */
		qspi_program_page();
	#endif
	
	
	while(1)
	{ Image_Location[0]; //loaction of images is stored in Image_location global array!
		dma_read_example();
		qspi_receive_data[0]; //data retrieved by DMA stored in this array.
	};
} /* main */


void MC_MODE_INIT()
{
        uint8_t i;
    /* Enable all peripheral clocks */
	for (i=0;i<4;i++) CGM.SC_DC[i].R = 0x80;
	/* Setting RUN Configuration Register ME_RUN_PC[0] */
	ME.RUNPC[0].R=0x000000FE; /* Peripheral ON in every mode */
	//CGM.OCDS_SC.B.SELCTL=1;    	   //Select Fast IRC (16 MHz) for CLKOUT

	/* Re-enter in DRUN mode to update */
	ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
	ME.MCTL.R = 0x3000A50F;     	  /* Mode & Key */  
	while(ME.GS.B.S_CURRENTMODE!=3); /* Check DRUN mode has been entered */
}                                                                                                                                                                                                                        

void DISABLE_WATCHDOG()
{
	SWT.SR.R = 0x0000c520; /* key */
	SWT.SR.R = 0x0000d928; /* key */
	SWT.CR.R = 0x8000010A; /* disable WEN */
}                                                                                                                                                                                                                                                                   

//Initial PLL to freqMHz and select as system bus - assume 8MHz crystal
//freq must be >= 16 and <= 64
//Returns value actually set
uint16_t SetupPLL(uint16_t freq)
{
    if (freq > 64) freq = 64;
    if (freq < 16) freq = 16;
    ME.DRUN.B.FXOSCON=1;			       /* Switch on external osc */
    ME.DRUN.B.SYSCLK=2;			       /* Select external osc */
/* RE enter the drun mode, to update the configuration */
    ME.MCTL.R = 0x30005AF0;        /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;        /* Mode & Key inverted */
    while(ME.GS.B.S_FXOSC==0);			   /* Wait for external osc to stabilize */
    while(ME.GS.B.S_MTRANS==1);      /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=3); /* Check DRUN mode has been entered */
	
/* Max bus 64Mhz, VCO 256-512Mhz */
/* Fpll = XTAL * NDIV / IDF / ODF = 64MHz */
/* Fvco = XTAL * NDIV / IDF = 512MHz */
    if (freq >= 32)
    {
	CGM.FMPLL[0].CR.B.IDF=0x0000; 	/* Divide by 1 */
	CGM.FMPLL[0].CR.B.ODF=0x2;	/* Divide by 8 */
	CGM.FMPLL[0].CR.B.NDIV=freq;	/* Loop divide by freq */
    }
    else if (freq >= 16)
    {
	CGM.FMPLL[0].CR.B.IDF=0x0000;	/* Divide by 1 */
	CGM.FMPLL[0].CR.B.ODF=0x3;	/* Divide by 16 */
	CGM.FMPLL[0].CR.B.NDIV=freq*2;	/* Loop divide by freq*2 */
    }

//    CGM.FMPLL[1].CR.B.EN_PLL_SW	= 1;	//Enable progressive PLL switching
	
    ME.DRUN.B.FMPLL0ON=1;     /* enable pll */      
      
/* RE enter the drun mode, to update the configuration */
    ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;         /* Mode & Key inverted */
    while(ME.GS.B.S_MTRANS==1);		    /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=0x3);/* Check DRUN mode has been entered */
		
    while(CGM.FMPLL[0].CR.B.S_LOCK==0);
	
    ME.DRUN.B.SYSCLK=0x4;  /* system clock is PLL */
    ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;         /* Mode & Key inverted */
    while(ME.GS.B.S_MTRANS==1);		    /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=0x3);/* Check DRUN mode has been entered */

/* ME_GS Poll Global status register to get current system clock
   fail if system clock is not pll
   0000 16MHz internal RC oscillator
   0001 divided 16MHz internal RC oscillator
   0010 4MHz crystal oscillator
   0011 divided 4MHz crystal oscillator
   0100 system PLL
   1111 system clock is disabled */

    while(ME.GS.B.S_SYSCLK != 4){}; /* fail is stuck here	 */
    return freq;
}

void CONFIG_QSPI(void)
{
	/* configure ports */
	SIU.PCR[80].R = 0x0A00;          //QSPI pin config QPCS0
    SIU.PCR[81].R = 0x0B00;          //QSPI pin config QSPI_SDO2
    SIU.PCR[82].R = 0x0B00;          //QSPI pin config QSPI_SDO3
    SIU.PCR[83].R = 0x0700;          //QSPI pin config QSPI_SDO1
    SIU.PCR[84].R = 0x0700;          //QSPI pin config QSPI_SDO0
    SIU.PCR[85].R = 0x0600;          //QSPI pin config QSPI_SCK
	
	CGM.AC3_SC.B.SELCTL = 1; //set QSPI clock as sysclk/2
	
	// Enable sfm mode for winbond
    QUADSPI_0.MCR.R = 0x00000088;
	
	#if QUAD_READ == 1
		//enable quad reads
		QUADSPI_0.ICR.R = 0x000000a3; //Send instruction to enable high performance mode
		while(QUADSPI_0.SFMSR.B.BUSY);
		QUADSPI_0.ACR.R = 0x00007ceb; //Use Quad I/O reads - Command: 0xEB, Read Size:64bytes
	#endif
}

void qspi_erase_flash(void)
{
    qspi_flash_write_enable();
    QUADSPI_0.ICR.R = 0x00000060;
    while(QUADSPI_0.SFMSR.B.BUSY);
	wait_while_flash_busy();
}

void qspi_flash_write_enable(void)
{
    //send write enable instruction
    QUADSPI_0.ICR.R = 0x00000006;
    while(QUADSPI_0.SFMSR.B.BUSY);
}

void wait_while_flash_busy(void)
{
    vuint32_t read_data;

	read_data = 0x01000000;
	while((read_data & 0x01000000)==0x01000000)
	{
		QUADSPI_0.ICR.R = 0x00010005;
	    while(QUADSPI_0.SFMSR.B.BUSY);
		while(QUADSPI_0.SFMSR.B.RXNE == 0);
		read_data = QUADSPI_ARDB;
		QUADSPI_0.SFMFR.R = 0x10000; /* read complete */
	}
}

void qspi_program_page(void)
{ 
    uint32_t len, i;
    vuint32_t read_data[64];
	vuint32_t *fromaddr;
	
	fromaddr  = current_image;
    
	/* Split transfer into aligned 256 Byte pages */
	do
	{	
		len=0x100-((QUADSPI_0.SFAR)&0xFF); /* only write till end of 256 page ...  */
		if (len>image_size)            /* ... and not more then size bytes   */
			len=image_size;			 /* length and size are in bytes - writes should be done as longs (32bits) */

		for(i=0; ((i<15)&(i<(len/4))); i++) {
			QUADSPI_0.TBDR = ((unsigned int)(fromaddr[i]));
		}  

		/* enable the flash for writing */
		qspi_flash_write_enable();
		
		QUADSPI_0.ICR.R = 0x00000002|((len)<<16); /* trigger single line write of correct size */

		for(i=15;(i<(len/4));i++) {
			while(QUADSPI_0.SFMSR.B.TXFULL);
			QUADSPI_0.TBDR = ((unsigned int)(fromaddr[(i)]));
		}  

		while(QUADSPI_0.SFMSR.B.BUSY);
		wait_while_flash_busy();
		
		#if CHECK_DATA == 1
			QUADSPI_0.MCR.B.CLR_RXF = 1;//empty Rx Buffer
			QUADSPI_0.ICR.R = 0x0000000B|(len<<16); //fast read 'length' bytes
			for(i=0; (i<(len/4)); i++) {
		        while(QUADSPI_0.SFMSR.B.RXNE == 0);
		        read_data[i] = QUADSPI_ARDB;
		        QUADSPI_0.SFMFR.R = 0x10000; /* read complete, clear RBDF */
		    }
			
			for(i=0; (i<(len/4)); i++) {
		        if(read_data[i] != fromaddr[i])
		            while(1); //error! - Data Read from memory is incorrect
		    }
		#endif
		
		image_size-=len;
		QUADSPI_0.SFAR +=len;
		fromaddr+=(len/4); /* as from addr is a pointer we must also divide by the size of int (i.e. 4) */
		
	} while (image_size>0);
}


// function to set QUAD bit on external QuadSPI memory
void set_quad_bit(void)
{
	/* change to QuadSPI mode */
	qspi_flash_write_enable();
	QUADSPI_0.TBDR = 0x00020000; //data to be writen
	QUADSPI_0.ICR.R = 0x00020001; //write config register
	while(QUADSPI_0.SFMSR.B.BUSY);
	wait_while_flash_busy();
}

void qspi_config_dma(void)
{
    // Route requests to DMA channel 12
    DMAMUX.CHCONFIG[12].R    = 0x86;         // QSPI_RFDF  channel
   
    EDMA.TCD[12].SADDR       = 0x87FFFFFC;  /* source address of data (ARDB register) */
	EDMA.TCD[12].SMOD        = 0;			/* source address modulo disabled */
	EDMA.TCD[12].SSIZE       = 0x2;			/* 32-bit Source size */
	EDMA.TCD[12].DMOD        = 0;			/* Destination address modulo disabled */
	EDMA.TCD[12].DSIZE       = 0x2;			/* 32-bit destination size */
	EDMA.TCD[12].SOFF        = 0x0;			/* source address offset = 0 */
	EDMA.TCD[12].NBYTESu.R   = 0x4;			/* transfer 4 bytes per channel activation */
	EDMA.TCD[12].SLAST       = 0x0;			/* Restore Source address by 0 */
   	EDMA.TCD[12].DADDR       = (vuint32_t)(& qspi_receive_data);	/* destination address */
	EDMA.TCD[12].CITERE_LINK = 0;			/* disabled chanel2chanel linking */
	EDMA.TCD[12].CITER       = 1;			/* current major iteration */
	EDMA.TCD[12].DOFF        = 0;			/* destination address offset = 0 */
	EDMA.TCD[12].DLAST_SGA   = 0x4;			/* restore destination address by 4 */
	EDMA.TCD[12].BITERE_LINK = 0;			/* disabled chanel2chanel linking */
	EDMA.TCD[12].BITER       = 1;			/* starting major iteration count */
	EDMA.TCD[12].BWC         = 0;			/* no bandwidth control */
	EDMA.TCD[12].MAJORLINKCH = 0;			/* disabled chanel2chanel linking */
	EDMA.TCD[12].E_SG         = 0x0;         /* Disable scatter gather on major loop      */
    EDMA.TCD[12].D_REQ        = 0x0;         /* Do not dsiable hardware request on citer=0 */
    EDMA.TCD[12].INT_HALF     = 0x0;         /* Disable half point interrupt               */
    EDMA.TCD[12].INT_MAJ      = 0x0;         /* Disable interrupt on major loop complete   */  
	EDMA.TCD[12].DONE        = 0;			/* Channel Done */
	EDMA.TCD[12].ACTIVE      = 0;			/* Channel Active */
	EDMA.TCD[12].MAJORE_LINK = 0;			/* Major Channel Link : Disabled */
	EDMA.TCD[12].E_SG        = 0;			/* Disable Scatter/Gather */
	EDMA.TCD[12].D_REQ       = 0;			/* Disable the DMA channel when Done */
	EDMA.TCD[12].INT_HALF    = 0;			/* Disable Interrupt on Half Major Loop */
	EDMA.TCD[12].INT_MAJ     = 0;			/* Disable Interrupt on  Major Loop completion */
	EDMA.TCD[12].START       = 0;			/* Start bit for channel */
	
	EDMA.ERQL.R             = 0x00001000; //enable requests from channel 12
}

void dma_read_example()
{
	qspi_config_dma();
	
    QUADSPI_0.MCR.B.CLR_RXF = 1; // Clear RX FIFO
	QUADSPI_0.SFMRSER.B.RBDDE = 1;//enable DMA mode

    //normal read
	QUADSPI_0.SFAR = 0x80000000; 
	QUADSPI_0.ICR.R = 0x0b|((64)<<16); //read 64bytes
	
	while(QUADSPI_0.SFMSR.B.BUSY);
	while(EDMA.TCD[12].DONE == 0);
}
