#include "rtc_driver_int.h"
#include "rtc_driver_ext.h"

void (*callback)(unsigned int interrupt_mask, unsigned char tamper_mask);

/******************************************************************************/
/* FUNCTION          : rtc_setConfig                                          */
/******************************************************************************/
/* Abstract          : This function configs the control register of IRTC     */
/*                                                                            */
/* Input Parameters  : rtc_clkout: RTC clock out                              */
/*                     daylightsavings_enable: DST enabled/disabled           */
/*                     softreset: if 1 then RTC is soft reset                 */
/*                     callback: function ptr for ISR callback                */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setConfig(unsigned char rtc_clkout, unsigned char daylightsavings_enable
                            , unsigned char softreset, void (*func)(unsigned int, unsigned char))
{
  SCGC5_IRTC = 1;

  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_CTRL_H_RTC_CLKOUT  = rtc_clkout;
  IRTC_CTRL_L_DSTEN = daylightsavings_enable;
  IRTC_CTRL_H_SWR = softreset;
  IRTC_IER_L_TMPR = 1;

  //protect registers
  irtc_protectRegisters();
  
  if(rtc_clkout != IRTC_CTRL_H_RTC_CLKOUT)
  {
    return 1;
  }
  
 
  callback = func;

  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_setAlarmConfig                                     */
/******************************************************************************/
/* Abstract          : This function configs the alarm control register       */
/*                                                                            */
/* Input Parameters  : alarm_match: alarm configuration be used               */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setAlarmConfig(unsigned char alarm_match)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_CTRL_L_ALARM_MATCH = alarm_match;

  //protect registers
  irtc_protectRegisters();

  if(alarm_match != IRTC_CTRL_L_ALARM_MATCH)
  {
    return 1;
  }

  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_setCompensation                                    */
/******************************************************************************/
/* Abstract          : This function sets the compensation settings for RTC   */
/*                                                                            */
/* Input Parameters  : comp_interval: compensation interval window            */
/*                     comp_value: 2's complement form of compensating value  */
/*                                                                            */
/* Return Parameter  : none                                                   */
/******************************************************************************/
void rtc_setCompensation(unsigned char comp_interval, unsigned char comp_value)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_COMPEN_H  = comp_interval;
  IRTC_COMPEN_L = comp_value;

  //protect registers
  irtc_protectRegisters();
}

/******************************************************************************/
/* FUNCTION          : rtc_setYear                                            */
/******************************************************************************/
/* Abstract          : This function sets the year in IRTC                    */
/*                                                                            */
/* Input Parameters  : year: year to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setYear(unsigned int year)
{
  unsigned char offset = (year - 2112);

  //unprotect registers
  irtc_unprotectRegisters();

  //set the year offset
  offset = irtc_setYearOffset(offset);

  //protect registers
  irtc_protectRegisters();

  return offset;
}

/******************************************************************************/
/* FUNCTION          : rtc_setMonth                                           */
/******************************************************************************/
/* Abstract          : This function sets the month in IRTC                   */
/*                                                                            */
/* Input Parameters  : month: month to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setMonth(enum month_names month)
{
  unsigned char result;
  
  if( (month < January) || (month > December) )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the month
  result = irtc_setMonth((unsigned char) month);

  //protect registers
  irtc_protectRegisters();

  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_setDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function sets the day and weekday in IRTC         */
/*                                                                            */
/* Input Parameters  : weekday: week day to be set                            */
/*                     date: date to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setDayAndDate(enum weekday_names weekday, unsigned char date)
{
  unsigned char result;
  
  if( (weekday < Sunday) || (weekday > Saturday) || (date > 31) || (date < 1) )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the day and date
  result = irtc_setDayAndDate((unsigned char) weekday, date);

  //protect registers
  irtc_protectRegisters();

  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_setHour                                            */
/******************************************************************************/
/* Abstract          : This function sets the hour in IRTC in 24H time format */
/*                                                                            */
/* Input Parameters  : hour: hour to be set                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setHour(unsigned char hour)
{
  unsigned char result;
  
  if( (hour < 0) || (hour > 23) )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the hour
  result = irtc_setHour(hour);

  //protect registers
  irtc_protectRegisters();

  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_setMin                                             */
/******************************************************************************/
/* Abstract          : This function sets the minutes in IRTC                 */
/*                                                                            */
/* Input Parameters  : min: minutes to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setMin(unsigned char min)
{
  unsigned char result;
  
  if( (min < 0) || (min > 59) )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the minutes
  result = irtc_setMin(min);

  //protect registers
  irtc_protectRegisters();

  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_setSeconds                                         */
/******************************************************************************/
/* Abstract          : This function sets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : sec: seconds to be set                                 */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setSeconds(unsigned char sec)
{
  unsigned char result;
  
  if( (sec < 0) || (sec > 59) )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the seconds
  result = irtc_setSeconds(sec);

  //protect registers
  irtc_protectRegisters();

  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_setDateAndTime                                     */
/******************************************************************************/
/* Abstract          : This is one function that sets the date and time both  */
/*                                                                            */
/* Input Parameters  : year: year to be programmed                            */
/*                     month: month to be programmed                          */
/*                     date:  date  to be programmed                          */
/*                     weekday: day to be programmed                          */
/*                     hour : hour  to be programmed                          */
/*                     minutes: minutes to be programmed                      */
/*                     seconds: seconds to be programmed                      */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setDateAndTime(unsigned int year, enum month_names month, unsigned char date,
                                 enum weekday_names weekday, unsigned char hour, unsigned char minutes,
                                 unsigned char seconds)
{
  unsigned char result, offset = (year - 2112);

  if( (month < January) || (month > December) || (weekday < Sunday)
      || (weekday > Saturday) || (date > 31) || (date < 1)
      || (hour < 0) || (hour > 23) || (minutes < 0) || (minutes > 59)
      || (seconds < 0) || (seconds > 59)
    )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the year offset
  result = irtc_setYearOffset(offset);

  //set the month
  result += irtc_setMonth((unsigned char) month);

  //set the day and date
  result += irtc_setDayAndDate((unsigned char) weekday, date);

  //set the hour
  result += irtc_setHour(hour);

  //set the minutes
  result += irtc_setMin(minutes);

  //set the seconds
  result += irtc_setSeconds(seconds);

  //protect registers
  irtc_protectRegisters();
  
  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_getYear                                            */
/******************************************************************************/
/* Abstract          : This function gets the year in IRTC                    */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getYear(void)
{
  unsigned char offset;

  //while(!IRTC_STATUS_INVAL);
  offset = (unsigned char)irtc_getYearOffset();

  return (2112 + (signed char)offset);
}

/******************************************************************************/
/* FUNCTION          : rtc_getMonth                                           */
/******************************************************************************/
/* Abstract          : This function gets the month in IRTC                   */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : month value                                            */
/******************************************************************************/
enum month_names rtc_getMonth(void)
{
  //while(!IRTC_STATUS_INVAL);
  return irtc_getMonth();
}

/******************************************************************************/
/* FUNCTION          : rtc_getDayAndDate                                      */
/******************************************************************************/
/* Abstract          : This function gets the day and weekday in IRTC         */
/*                                                                            */
/* Output Parameters  : weekday: week day to be get                           */
/*                     date: date to be get                                   */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
void rtc_getDayAndDate(enum weekday_names *weekday, unsigned char * date)
{
  //while(!IRTC_STATUS_INVAL);
  *weekday = irtc_getDay();
  
  *date = (unsigned char)irtc_getDate();
}

/******************************************************************************/
/* FUNCTION          : rtc_getHour                                            */
/******************************************************************************/
/* Abstract          : This function gets the hour in IRTC in 24H time format */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : hour value                                             */
/******************************************************************************/
unsigned char rtc_getHour(void)
{
  //while(!IRTC_STATUS_INVAL);
  return (unsigned char)irtc_getHour();
}

/******************************************************************************/
/* FUNCTION          : rtc_getMin                                             */
/******************************************************************************/
/* Abstract          : This function gets the minutes in IRTC                 */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : minutes value                                          */
/******************************************************************************/
unsigned char rtc_getMin(void)
{
  //while(!IRTC_STATUS_INVAL);
  return (unsigned char)irtc_getMin();
}

/******************************************************************************/
/* FUNCTION          : rtc_getSeconds                                         */
/******************************************************************************/
/* Abstract          : This function gets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : sec: seconds to be get                                 */
/*                                                                            */
/* Return Parameter  : seconds value                                          */
/******************************************************************************/
unsigned char rtc_getSeconds(void)
{
  //while(!IRTC_STATUS_INVAL);
  return (unsigned char)irtc_getSeconds();
}

/******************************************************************************/
/* FUNCTION          : rtc_setAlarm                                           */
/******************************************************************************/
/* Abstract          : This is one function that sets alarm in IRTC           */
/*                                                                            */
/* Input Parameters  : year: year to be programmed                            */
/*                     month: month to be programmed                          */
/*                     date:  date  to be programmed                          */
/*                     hour : hour  to be programmed                          */
/*                     minutes: minutes to be programmed                      */
/*                     seconds: seconds to be programmed                      */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setAlarm(unsigned int year, enum month_names month, unsigned char date,
                           unsigned char hour, unsigned char minutes, unsigned char seconds)
{
  unsigned char result, offset = (year - 2112);

  if( (month < January) || (month > December)
      || (date > 31) || (date < 1) || (hour < 0) || (hour > 23)
      || (minutes < 0) || (minutes > 59) || (seconds < 0) || (seconds > 59)
    )
  return 1;

  //unprotect registers
  irtc_unprotectRegisters();

  //set the year offset
  result = irtc_setAlarmYearOffset(offset);

  //set the month
  result = irtc_setAlarmMonth((unsigned char) month);

  //set the day and date
  result = irtc_setAlarmDate(date);

  //set the hour
  result = irtc_setAlarmHour(hour);

  //set the minutes
  result = irtc_setAlarmMin(minutes);

  //set the seconds
  result = irtc_setAlarmSeconds(seconds);
  
  return result;
}

/******************************************************************************/
/* FUNCTION          : rtc_getAlarmYear                                       */
/******************************************************************************/
/* Abstract          : This function gets alarm year in IRTC                  */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getAlarmYear(void)
{
  unsigned char offset = (unsigned char)irtc_getAlarmYearOffset();

  return (2112 + (signed char)offset);
}

/******************************************************************************/
/* FUNCTION          : rtc_setBatteryTamperConfig                             */
/******************************************************************************/
/* Abstract          : This function configs battery tamper control           */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setBatteryTamperConfig(unsigned char state)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_TAMPER_SCR_L_BAT_TAMPER_CTRL = state;
  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_setTamper1Config                                   */
/******************************************************************************/
/* Abstract          : This function configs the Tamper1 control and filter   */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                     filter_clk: clock select for filter delay              */
/*                     filter_duration: duration for filter delay             */
/*                     pol: polarity, 1-active low, 0-active high             */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setTamper1Config(unsigned char state, unsigned char filter_clk
                                  , unsigned char filter_duration, unsigned char pol)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_TAMPER_SCR_L_TAMPER1_CTRL = state;
  
  IRTC_FILTER01_CTRL_L_CLKSEL = filter_clk;
  IRTC_FILTER01_CTRL_L_CLKSEL = filter_duration;
  IRTC_FILTER01_CTRL_L_POL = pol;
  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_setTamper2Config                                   */
/******************************************************************************/
/* Abstract          : This function configs the Tamper2 control and filter   */
/*                                                                            */
/* Input Parameters  : state: whether ON or OFF                               */
/*                     filter_clk: clock select for filter delay              */
/*                     filter_duration: duration for filter delay             */
/*                     pol: polarity, 1-active low, 0-active high             */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setTamper2Config(unsigned char state, unsigned char filter_clk
                                  , unsigned char filter_duration, unsigned char pol)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_TAMPER_SCR_L_TAMPER2_CTRL = state;
  
  IRTC_FILTER23_CTRL_H_CLKSEL = filter_clk;
  IRTC_FILTER23_CTRL_H_CLKSEL = filter_duration;
  IRTC_FILTER23_CTRL_H_POL = pol;

  //protect registers
  irtc_protectRegisters();

  return 0;
}

#if 0       //not required as RAM is accesible using 'StandbyRAM' variable

/******************************************************************************/
/* FUNCTION          : rtc_writeRAM                                           */
/******************************************************************************/
/* Abstract          : This function writes the data into RTC RAM             */
/*                                                                            */
/* Input Parameters  : address: RAM address                                   */
/*                     length: number of bytes to be written                  */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_writeRAM(unsigned char* ram_address, unsigned char length)
{
  unsigned char count = 0;
  if( (ram_address < StandbyRAM) || (ram_address > (&StandbyRAM[31]))
      || (length > (&StandbyRAM[31] - ram_address + 1))
    )
  {
    return 1;
  }
  //unprotect registers
  irtc_unprotectRegisters();

  while(count < length)
  {
    *ram_address

  //protect registers
  irtc_protectRegisters();

  return 0;
}

#endif

/******************************************************************************/
/* FUNCTION          : rtc_getTamperStampYear                                 */
/******************************************************************************/
/* Abstract          : This function gets tamper stamp year in IRTC           */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : year value                                             */
/******************************************************************************/
unsigned int rtc_getTamperStampYear(void)
{
  unsigned char offset = (unsigned char)irtc_getTamperStampYearOffset();

  return (2112 + (signed char)offset);
}

/******************************************************************************/
/* FUNCTION          : rtc_setCountDownTimer                                  */
/******************************************************************************/
/* Abstract          : This function sets the seconds in IRTC                 */
/*                                                                            */
/* Input Parameters  : minutes: no. of minutes to be countdown                */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_setCountDownTimer(unsigned char minutes)
{
  //unprotect registers
  irtc_unprotectRegisters();

  //set the countdown
  IRTC_COUNT_DN = minutes;

  //protect registers
  irtc_protectRegisters();

  if(minutes != IRTC_COUNT_DN)
  {
    return 1;
  }

  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_EnableInterrupt                                    */
/******************************************************************************/
/* Abstract          : This function sets particular interrupt in IRTC        */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be enabled           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_EnableInterrupt(unsigned int interrupt_byte)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_IER |= interrupt_byte;

  //protect registers
  irtc_protectRegisters();
  
  if(IRTC_IER & interrupt_byte)
  {
    return 0;
  }
  return 1;
}

/******************************************************************************/
/* FUNCTION          : rtc_DisableInterrupt                                   */
/******************************************************************************/
/* Abstract          : This function disables particular interrupt in IRTC    */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be enabled           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_DisableInterrupt(unsigned int interrupt_byte)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_IER &= ~interrupt_byte;

  //protect registers
  irtc_protectRegisters();
  
  if(IRTC_IER & interrupt_byte)
  {
    return 1;
  }
  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_clearInterrupt                                     */
/******************************************************************************/
/* Abstract          : This function clears particular interrupt in IRTC      */
/*                                                                            */
/* Input Parameters  : interrupt_byte: interrupt mask to be cleared           */
/*                                                                            */
/* Return Parameter  : 0 - success, 1 - failure                               */
/******************************************************************************/
unsigned char rtc_clearInterrupt(unsigned int interrupt_byte)
{
  //unprotect registers
  irtc_unprotectRegisters();

  IRTC_ISR|= interrupt_byte;

  //protect registers
  irtc_protectRegisters();
  
  if(IRTC_ISR & interrupt_byte)
  {
    return 1;
  }
  return 0;
}

/******************************************************************************/
/* FUNCTION          : rtc_ISR                                               */
/******************************************************************************/
/* Abstract          : This function is interrupt service routine for IRTC    */
/*                                                                            */
/* Input Parameters  : none                                                   */
/*                                                                            */
/* Return Parameter  : none                                                   */
/******************************************************************************/
void interrupt VectorNumber_Virtc rtc_ISR()
{
  unsigned int int_mask;
  unsigned char tamper_mask = 0;

  int_mask = IRTC_ISR;

  if(IRTC_ISR_L_TMPR)
  {
    if(IRTC_TAMPER_SCR_H_TAMPER1_STATUS)
    {
      tamper_mask |= RTC_ISR_TMPR1;
    }
    if(IRTC_TAMPER_SCR_H_TAMPER2_STATUS)
    {
      tamper_mask |= RTC_ISR_TMPR2;
    }
    if(IRTC_TAMPER_SCR_H_BAT_TAMPER_STATUS)
    {
      tamper_mask |= RTC_ISR_BAT_TMPR;
    }
  }

  irtc_unprotectRegisters();
  IRTC_ISR = int_mask;    //clear interrupts

  IRTC_TAMPER_SCR_H_TAMPER1_STATUS = 1;
  IRTC_TAMPER_SCR_H_TAMPER2_STATUS = 1;
  IRTC_TAMPER_SCR_H_BAT_TAMPER_STATUS = 1;

  irtc_protectRegisters();
  if(callback)
  {
    (*callback)(int_mask, tamper_mask);
  }
}
