/////////////////////////////////////////////////////////////////////////////////////////
//
// LCD Library for Freescale DEMO9S08LC60
//
// S-TEK Display (model GD-3980)
// Driving Voltage: 3V
// Duty: 1/4
// Bias: 1/3
// Frequency: 64Hz
//
// --------------------------------------------------------------------------------------
//
// CodeWarrior V5.1 for HC(S)08
// Registers Files version 2.87.175
//
// --------------------------------------------------------------------------------------
//
// Copyright (c) 2006 SofTec Microsystems
// http://www.softecmicro.com/
//
/////////////////////////////////////////////////////////////////////////////////////////
#include "lcd.h"
#include "lcd_config.h"
#include "MC9S08GW64.h"

unsigned int qseconds = 0;

/////////////////////////////////////////////////////////////////////////////////////////
// Variables
/////////////////////////////////////////////////////////////////////////////////////////

#if LCD_PIN_44

const struct lcd_char_segs lcd_char_table[] = {

  // ' ' (Space)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '"' (Quotation Mark)
  {
  LCD_SEG_H,
  LCD_SEG_NONE,
  LCD_SEG_K,
  LCD_SEG_NONE
  },

  // ''' (Apostrophe)
  {
  LCD_SEG_H,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '(' (Left Parenthesis)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // ')' (Right Parenthesis)
  {
  LCD_SEG_H | LCD_SEG_N,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '*' (Asterisk)
  {
  LCD_SEG_H | LCD_SEG_N,
  LCD_SEG_G | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // '+' (Plus Sign)
  {
  LCD_SEG_NONE,
  LCD_SEG_G | LCD_SEG_J |LCD_SEG_M,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // ',' (Comma)
  {
  LCD_SEG_N,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '-' (Minus Sign)
  {
  LCD_SEG_NONE,
  LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '.' (Period)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '/' (Slash)
  {
  LCD_SEG_N,
  LCD_SEG_NONE,
  LCD_SEG_K,
	LCD_SEG_NONE
  },

  // '0'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // '1'
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_B | LCD_SEG_C
  },

  // '2'
  {
  LCD_SEG_E,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_B
  },

  // '3'
  {
  LCD_SEG_NONE,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // '4'
  {
  LCD_SEG_F,
  LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_B | LCD_SEG_C
  },

  // '5'
  {
  LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_C
  },

  // '6'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_C
  },

  // '7'
  {
  LCD_SEG_NONE,
  LCD_SEG_A,
  LCD_SEG_NONE,
  LCD_SEG_B | LCD_SEG_C
  },

  // '8'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // '9'
  {
  LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // ':'
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '=' (Equal Sign)
  {
  LCD_SEG_NONE,
  LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // 'A'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'B'
  {
  LCD_SEG_NONE,
  LCD_SEG_A | LCD_SEG_G | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'C'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // 'D'
  {
  LCD_SEG_NONE,
  LCD_SEG_A | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'E'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // 'F'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // 'G'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A,
  LCD_SEG_D | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // 'H'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_B | LCD_SEG_C,
  },

  // 'I'
  {
  LCD_SEG_NONE,
  LCD_SEG_A | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // 'J'
  {
  LCD_SEG_E,
  LCD_SEG_NONE,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C 
  },
  
  // 'K'
  {
  LCD_SEG_NONE,
  LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // 'L'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_NONE,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // 'M'
  {
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_H,
  LCD_SEG_NONE,
  LCD_SEG_K,  
  LCD_SEG_B | LCD_SEG_C
  },

  // 'N'
  {
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_H,
  LCD_SEG_NONE,
  LCD_SEG_L,  
  LCD_SEG_B | LCD_SEG_C
  },

  // 'O'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'P'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_NONE,
  LCD_SEG_B
  },

  // 'Q'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A,
  LCD_SEG_D | LCD_SEG_L,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'R'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_L,
  LCD_SEG_B
  },

  // 'S'
  {
  LCD_SEG_F,
  LCD_SEG_A | LCD_SEG_G,
  LCD_SEG_D,
  LCD_SEG_C
  },

  // 'T'
  {
  LCD_SEG_NONE,
  LCD_SEG_A | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // 'U'
  {
  LCD_SEG_E | LCD_SEG_F,
  LCD_SEG_NONE,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'V'
  {
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_N,
  LCD_SEG_NONE,
  LCD_SEG_K,
  LCD_SEG_NONE
  },

  // 'W'
  {
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_N,
  LCD_SEG_NONE,
  LCD_SEG_L,
  LCD_SEG_B | LCD_SEG_C
  },

  // 'X'
  {
  LCD_SEG_H | LCD_SEG_N, 
  LCD_SEG_NONE,
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // 'Y'
  {
  LCD_SEG_H,
  LCD_SEG_M,
  LCD_SEG_K,
  LCD_SEG_NONE
  },

  // 'Z'
  {
  LCD_SEG_N,
  LCD_SEG_A,
  LCD_SEG_D | LCD_SEG_K,
  LCD_SEG_NONE
  },

  // '[' (Left Bracket)
  {
  LCD_SEG_F | LCD_SEG_E,
  LCD_SEG_A,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // '\' (Backslash)
  {
  LCD_SEG_H,
  LCD_SEG_NONE,
  LCD_SEG_L,
  LCD_SEG_NONE
  },

  // ']' (Right Bracket)
  {
  LCD_SEG_NONE,
  LCD_SEG_A,
  LCD_SEG_D,
  LCD_SEG_B | LCD_SEG_C
  },

  // '^' (Exponent)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_K,
  LCD_SEG_B
  },

  // '_' (Underline)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE,
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // '|' (Vertical Bar)
  {
  LCD_SEG_NONE,
  LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // Test
  {
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_H | LCD_SEG_N,
  LCD_SEG_A | LCD_SEG_G | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_D | LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_B | LCD_SEG_C
  }
  
};

#else

const struct lcd_char_segs lcd_char_table[] = {

  // ' ' (Space)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '"' (Quotation Mark)
  {
  LCD_SEG_K,
  LCD_SEG_H
  },

  // ''' (Apostrophe)
  {
  LCD_SEG_NONE,
  LCD_SEG_H
  },

  // '(' (Left Parenthesis)
  {
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // ')' (Right Parenthesis)
  {
  LCD_SEG_NONE,
  LCD_SEG_H | LCD_SEG_N
  },

  // '*' (Asterisk)
  {
  LCD_SEG_J | LCD_SEG_K | LCD_SEG_L | LCD_SEG_M, 
  LCD_SEG_H | LCD_SEG_G | LCD_SEG_N
  },

  // '+' (Plus Sign)
  {
  LCD_SEG_J |LCD_SEG_M,
  LCD_SEG_G 
  },

  // ',' (Comma)
  {
  LCD_SEG_NONE,
  LCD_SEG_N
  },

  // '-' (Minus Sign)
  {
  LCD_SEG_NONE,
  LCD_SEG_G
  },

  // '.' (Period)
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '/' (Slash)
  {
  LCD_SEG_K,
  LCD_SEG_N
  },

  // '0'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A  
  },

  // '1'
  {
  LCD_SEG_B | LCD_SEG_C,
  LCD_SEG_NONE
  },

  // '2'
  {
  LCD_SEG_D | LCD_SEG_B,
  LCD_SEG_A | LCD_SEG_G | LCD_SEG_E
  },

  // '3'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_G
  },

  // '4'
  {
  LCD_SEG_B | LCD_SEG_C,
  LCD_SEG_F | LCD_SEG_G
  },

  // '5'
  {
  LCD_SEG_D | LCD_SEG_C,
  LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // '6'
  {
  LCD_SEG_D | LCD_SEG_C,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // '7'
  {
  LCD_SEG_B | LCD_SEG_C,
  LCD_SEG_A 
  },

  // '8'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // '9'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_G | LCD_SEG_F
  },

  // ':'
  {
  LCD_SEG_NONE,
  LCD_SEG_NONE
  },

  // '=' (Equal Sign)
  {
  LCD_SEG_D,
  LCD_SEG_G
  },

  // 'A'
  {
  LCD_SEG_B | LCD_SEG_C,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // 'B'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_J | LCD_SEG_M | LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_G 
  },

  // 'C'
  {
  LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_E | LCD_SEG_F
  },

  // 'D'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D | LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_A 
  },

  // 'E'
  {
  LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // 'F'
  {
  LCD_SEG_NONE,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // 'G'
  {
  LCD_SEG_D | LCD_SEG_L,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A
  },

  // 'H'
  {
  LCD_SEG_B | LCD_SEG_C,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_G
  },

  // 'I'
  {
  LCD_SEG_J | LCD_SEG_D |  LCD_SEG_M,
  LCD_SEG_A 
  },

  // 'J'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_E
  },
  
  // 'K'
  {
  LCD_SEG_J | LCD_SEG_M | LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_NONE
  },

  // 'L'
  {
  LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F
  },

  // 'M'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_K,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_H
  },

  // 'N'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_L,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_H
  },

  // 'O'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A
  },

  // 'P'
  {
  LCD_SEG_B,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
   },

  // 'Q'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D | LCD_SEG_L,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A
  },

  // 'R'
  {
  LCD_SEG_B | LCD_SEG_L,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_A | LCD_SEG_G
  },

  // 'S'
  {
  LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_G | LCD_SEG_F
   },

  // 'T'
  {
  LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_A  
  },

  // 'U'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_E | LCD_SEG_F
  },

  // 'V'
  {
  LCD_SEG_K,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_N
  },

  // 'W'
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_L,
  LCD_SEG_E | LCD_SEG_F | LCD_SEG_N
  },

  // 'X'
  {
  LCD_SEG_K | LCD_SEG_L,
  LCD_SEG_H | LCD_SEG_N 
  },

  // 'Y'
  {
  LCD_SEG_M | LCD_SEG_K,
  LCD_SEG_H
  },

  // 'Z'
  {
  LCD_SEG_D | LCD_SEG_K,
  LCD_SEG_N | LCD_SEG_A
  },

  // '[' (Left Bracket)
  {
  LCD_SEG_D,
  LCD_SEG_F | LCD_SEG_E | LCD_SEG_A
  },

  // '\' (Backslash)
  {
  LCD_SEG_L,
  LCD_SEG_H
  },

  // ']' (Right Bracket)
  {
  LCD_SEG_B | LCD_SEG_C | LCD_SEG_D,
  LCD_SEG_A
  },

  // '^' (Exponent)
  {
  LCD_SEG_K | LCD_SEG_B,
  LCD_SEG_NONE
  },

  // '_' (Underline)
  {
  LCD_SEG_D,
  LCD_SEG_NONE
  },

  // '|' (Vertical Bar)
  {
  LCD_SEG_J | LCD_SEG_M,
  LCD_SEG_NONE
  },

  // Test
  {
  LCD_SEG_J | LCD_SEG_K | LCD_SEG_B | LCD_SEG_C | LCD_SEG_L | LCD_SEG_M | LCD_SEG_D,
  LCD_SEG_A | LCD_SEG_H | LCD_SEG_F | LCD_SEG_G | LCD_SEG_E | LCD_SEG_N
  }
  
};

#endif 

void Delay(unsigned int no_of_qseconds)
{
  qseconds = 0;
  while(qseconds < no_of_qseconds);
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDInit
// --------------------------------------------------------------------------------------
// LCD Peripheral Initialization
/////////////////////////////////////////////////////////////////////////////////////////
void LCDInit(void)
{ 

  SCGC2_LCD = 1;
 
  #if !LCD_CLOCK 
  // Select External Crystal clock (32.768 KHz) as LCD Clock Source. LCDCLK=32.768 KHz
  LCDC0_SOURCE = 0x00;
  
  #else
  // Select fBUS (8MHz) as LCD Clock Source. Enable prescaler by 16 and CLKADJ=14.
  // LCDCLK about 32.768 KHz
  LCDC0_SOURCE = 0x01;
  
  #endif
    
  
  
  #if LCD_PIN_44
  LCDC0_DUTY = 0x3;
  CONFIG_LCD_40x4_MODE
  
  #else 
  CONFIG_LCD_21x8_MODE
  LCDC0_DUTY = 0x7;
  
 // LCDSUPPLY_VSUPPLY = 0x0;  //derive VLL1,2 and 3 from VLCD
 
//  LCDSUPPLY_CPSEL = 1;
  
  #endif

  //LCDC0_LCLK = 6;
  //LCDSUPPLY_HREFSEL = 1;
  
  //LCDSUPPLY_LADJ = 1;
  //LCDRVC_RVEN = 1;
  LCDSUPPLY_CPSEL = 1;
  LCDSUPPLY_LADJ = 1;
  LCDSUPPLY_VSUPPLY = 1;
 // LCDSUPPLY_VSUPPLY = 0x1;  //derive VLL1,2 and 3 from VLCD
  LCDC0_LCDEN = 1;
    
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDBlinkState
// --------------------------------------------------------------------------------------
// Enable LCD Segments Blink State Mode
/////////////////////////////////////////////////////////////////////////////////////////
void LCDBlinkState(void)
{
  
  //Selects LCDRAM registers that control LCD segments blink state (LCDDRMS=1)
  LCDBCTL = 0x2;
  
  //Starts LCD module blinking (BLINK=1)
  LCDBCTL_BLINK = 0x1;
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDOnOffState
// --------------------------------------------------------------------------------------
// Enable LCD Segments ON/OFF State Mode
/////////////////////////////////////////////////////////////////////////////////////////
void LCDOnOffState(void)
{

  //Stop LCD module blinking (BLINK=0)
  LCDBCTL &= ~0x80;
  
  //Selects LCDRAM registers that control LCD segments on/off state (LCDDRMS=0)
  //LCDCMD = 0x00;

}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDClear
// --------------------------------------------------------------------------------------
// Clear LCDRAM Registers
/////////////////////////////////////////////////////////////////////////////////////////
void LCDClear(void)
{

  LCD_ALLFP_OFF
    
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDFindChar
// --------------------------------------------------------------------------------------
// Finds the given character in the characters table
/////////////////////////////////////////////////////////////////////////////////////////
unsigned char LCDFindChar(unsigned char character)
{
  if(character == ' ')
    return(LCD_CHAR_SPACE);
  else if(character == '"')
    return(LCD_CHAR_QUOTMARK);
  else if(character >= '\'' && character <= ':')
    return(character - '\'' + LCD_CHAR_APOSTROPHE);
/*  else if(character == '/')
    return(LCD_CHAR_SLASH);
  else if(character >= '0' && character <= '9')
    return(character - '0' + LCD_CHAR_0);*/
  else if(character == '=')
    return(LCD_CHAR_EQUAL);
  else if(character >= 'A' && character <= '_')
    return(character - 'A' + LCD_CHAR_A);
/*  else if(character == '\\')
    return(LCD_CHAR_BACKSLASH);
  else if(character == '_')
    return(LCD_CHAR_UNDERLINE);*/
  else if(character >= 'a' && character <= 'z')
    return(character - 'a' + LCD_CHAR_A);
  else if(character == '|')
    return(LCD_CHAR_VERTBAR);
  else
    return(LCD_CHAR_TEST);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDPutChar
// --------------------------------------------------------------------------------------
// Put Char in the LCDRAM Register
/////////////////////////////////////////////////////////////////////////////////////////
void LCDPutChar(unsigned char character, unsigned char digit)
{

  unsigned char *pLCDPIN5 = &LCDPIN5, table_pos, mytempreg;
  unsigned char *pLCDPIN9 = &LCDPIN9;
  unsigned char *pLCDPIN28= &LCDPIN28;

  if(digit > NUM_DIGITS)
    return;

  table_pos = LCDFindChar(character);
  
  if(table_pos == LCD_CHAR_TEST)
  {
    return;
  }
  
  if(character == '.')
  {
    LCDWriteSegment(DTx,digit);
    return;
  }
  if(character == ':')
  {
    LCDWriteSegment(COLx,digit);
    return;
  }
 
 #if LCD_PIN_44
      
  if( (digit >= 3) && (digit <= 6) )
  {
    switch(digit)
    {
      case 3:
        digit = 36;
        break;
      case 4:
        digit = 32;
        break;
      case 5:
        digit = 28;
        break;
      case 6:
        digit = 24;
        break;
    }
    //Clears Segments N E F H
    mytempreg = *(pLCDPIN5+LCDPIN_NEFH_OPP+digit) & ~MASK_NEFH;
    mytempreg |= lcd_char_table[table_pos].NEFH;
    //Writes Segments N E F H
    *(pLCDPIN5+LCDPIN_NEFH_OPP+digit) = mytempreg;
    
    //Clears Segments M G J A
    mytempreg = *(pLCDPIN5+LCDPIN_MGJA_OPP+digit) & ~MASK_MGJA;
    mytempreg |= lcd_char_table[table_pos].MGJA;
    //Writes Segments M G J A
    *(pLCDPIN5+LCDPIN_MGJA_OPP+digit) = mytempreg;
    
    //Clears Segments D L K
    mytempreg = *(pLCDPIN5+LCDPIN_DLK_OPP+digit) & ~MASK_DLK;
    mytempreg |= lcd_char_table[table_pos].DLK;
    //Writes Segments D L K
    *(pLCDPIN5+LCDPIN_DLK_OPP+digit) = mytempreg;
    
    //Clears Segments C B 
    mytempreg = *(pLCDPIN5+LCDPIN_CB_OPP+digit) & ~MASK_CB;
    mytempreg |= lcd_char_table[table_pos].CB;
    //Writes Segments C B
    *(pLCDPIN5+LCDPIN_CB_OPP+digit) = mytempreg;
  }
  else
  {
    switch(digit)
    {
      case 1:
        digit = 0;
        break;
      case 2:
        digit = 4;
        break;
      case 7:
        digit = 8;
        break;
      case 8:
        digit = 12;
        break;
      case 9:
        digit = 16;
        break;
    }
    //Clears Segments N E F H
    mytempreg = *(pLCDPIN5+LCDPIN_NEFH+digit) & ~MASK_NEFH;
    mytempreg |= lcd_char_table[table_pos].NEFH;
    //Writes Segments N E F H
    *(pLCDPIN5+LCDPIN_NEFH+digit) = mytempreg;
    
    //Clears Segments M G J A
    mytempreg = *(pLCDPIN5+LCDPIN_MGJA+digit) & ~MASK_MGJA;
    mytempreg |= lcd_char_table[table_pos].MGJA;
    //Writes Segments M G J A
    *(pLCDPIN5+LCDPIN_MGJA+digit) = mytempreg;
    
    //Clears Segments D L K
    mytempreg = *(pLCDPIN5+LCDPIN_DLK+digit) & ~MASK_DLK;
    mytempreg |= lcd_char_table[table_pos].DLK;
    //Writes Segments D L K
    *(pLCDPIN5+LCDPIN_DLK+digit) = mytempreg;
    
    //Clears Segments C B 
    mytempreg = *(pLCDPIN5+LCDPIN_CB+digit) & ~MASK_CB;
    mytempreg |= lcd_char_table[table_pos].CB;
    //Writes Segments C B
    *(pLCDPIN5+LCDPIN_CB+digit) = mytempreg;
  }
  
  #else
  
  //Clears Segments J K B C L M D 
    mytempreg = *(pLCDPIN9+(digit-1)) & ~MASK_JKBCLMD;
    mytempreg |= lcd_char_table[table_pos].JKBCLMD;
    //Writes Segments J K B C L M D
    *(pLCDPIN9+(digit-1)) = mytempreg;
    
    //Clears Segments A H F G E N
    mytempreg = *(pLCDPIN28-(digit-1)) & ~MASK_AHFGEN;
    mytempreg |= lcd_char_table[table_pos].AHFGEN;
    //Writes Segments A H F G E N
    *(pLCDPIN28 -(digit-1)) = mytempreg;
  
  #endif
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDWriteSegment
// --------------------------------------------------------------------------------------
// Enables Segment in the LCDRAM Register
/////////////////////////////////////////////////////////////////////////////////////////
void LCDWriteSegment(enum symbol s, unsigned char num)
{

  #if LCD_PIN_44
  
  switch(s)
  {
  
    case TIME:
      LCDPIN7 |= LCD_SEG_SIGN;
      break;
    case DT:
      LCDPIN8 |= LCD_SEG_M2;
      break;
    case DATE:
      LCDPIN11 |= LCD_SEG_SIGN;
      break;
    case FREESCALE:
      //LCDPIN42 |= LCD_SEG_SIGN;
      break;
    case FIRE:
      //LCDPIN41 |= LCD_SEG_SIGN;
      break;
    case TEMP:
      //LCDPIN38 |= LCD_SEG_SIGN;
      break;  
    case SNOWFLAKE:
      //LCDPIN37 |= LCD_SEG_M2;
      break;
    case VOLUME:
      //LCDPIN34 |= LCD_SEG_SIGN;
      break;
    case ALARM:
      LCDPIN30 |= LCD_SEG_SIGN;
      break;
    case T:
      LCDPIN29 |= LCD_SEG_M2;
      break;
    case P:
      LCDPIN15 |= LCD_SEG_SIGN;
      break;
    case CONTRAST:
      LCDPIN19 |= LCD_SEG_M2;
      break;
    case MODE:
      LCDPIN20 |= LCD_SEG_M2;
      break;
    case AM:
      LCDPIN25 |= LCD_SEG_M4;
      break;
    case PM:
      LCDPIN25 |= LCD_SEG_M5;
      break;
    case OM:
      LCDPIN25 |= LCD_SEG_M6;
      break;
    case KWH:
      LCDPIN26 |= LCD_SEG_M3;
      break;
    case AMPS:
      LCDPIN26 |= LCD_SEG_M4;
      break;
    case VOLTS:
      LCDPIN26 |= LCD_SEG_M5;
      break;
    case PROGRAM:
      LCDPIN26 |= LCD_SEG_M6;
      break;
    case VOL:
      LCDPIN27 |= LCD_SEG_M3;
      break;
      
    case DTx:
      switch(num)
      {
        case 1:
          LCDPIN8 |= LCD_SEG_M1;
          break;
        case 2:
          LCDPIN12 |= LCD_SEG_M2;
          break;        
        case 3:
          //LCDPIN41 |= LCD_SEG_M1;
          break;         
        case 4:
          //LCDPIN37 |= LCD_SEG_M1;
          break;        
        case 5:
          //LCDPIN33 |= LCD_SEG_M2;
          break;        
        case 6:
          LCDPIN29 |= LCD_SEG_M1;
          break;        
        case 7:
          LCDPIN16 |= LCD_SEG_M2;
          break;        
        case 8:
          LCDPIN20 |= LCD_SEG_M1;
          break;        
        case 9:
          LCDPIN24 |= LCD_SEG_M1;
          break;        
      }
      break;
    case COLx:
      switch(num)
      {
        case 2:
          LCDPIN12 |= LCD_SEG_M1;
          break;
        case 5:
          //LCDPIN33 |= LCD_SEG_M1;
          break;        
        case 7:
          LCDPIN16 |= LCD_SEG_M1;
          break;         
        case 8: //Degrees
          LCDPIN23 |= LCD_SEG_SIGN;
          break;               
      }
      break;
    case Vx:
      switch(num)
      {
        case 1:
          LCDPIN24 |= LCD_SEG_M2;
          break;
        case 2:
          LCDPIN25 |= LCD_SEG_M3;
          break;        
        case 3:
          LCDPIN27 |= LCD_SEG_M4;
          break;         
        case 4:
          LCDPIN27 |= LCD_SEG_M5;
          break;
        case 5:
          LCDPIN27 |= LCD_SEG_M6;
          break;                          
      }
      break;
    case Tx:
      switch(num)
      {
        case 1:
          LCDPIN28 |= LCD_SEG_M6;
          break;
        case 2:
          LCDPIN28 |= LCD_SEG_M5;
          break;        
        case 3:
          LCDPIN28 |= LCD_SEG_M4;
          break;         
        case 4:
          LCDPIN28 |= LCD_SEG_M3;
          break;                        
      }
      break;
  }
 
 
 #else
 
 switch(s)
  {
  
    case TIME:
      LCDPIN9  |= LCD_SEG_SP1;
      break;
    case DATE:
      LCDPIN10 |= LCD_SEG_SP1;
      break;
    case TEMP:
      LCDPIN11 |= LCD_SEG_SP1;
      break;
    case VOLUME:
      LCDPIN12 |= LCD_SEG_SP1;
      break;
    case CONTRAST:
      LCDPIN13 |= LCD_SEG_SP1;
      break;
    case MODE:
      LCDPIN14 |= LCD_SEG_SP1;
      break;
    case PROGRAM:
      LCDPIN15 |= LCD_SEG_SP1;
      break;
    case OM:
      LCDPIN16 |= LCD_SEG_SP1;
      break;
    case AM:
      LCDPIN18 |= LCD_SEG_M3;
      break;
    case PM:
      LCDPIN18 |= LCD_SEG_M2;
      break;
    case KWH:
      LCDPIN18 |= LCD_SEG_M7;
      break;
    case AMPS:
      LCDPIN18 |= LCD_SEG_M1;
      break;
    case VOLTS:
      LCDPIN18 |= LCD_SEG_M8;
      break;
    case VOL:
      LCDPIN19 |= LCD_SEG_M6;
      break;
    case T:
      LCDPIN19 |= LCD_SEG_M3;
      break;
    case P:
      LCDPIN22 |= LCD_SEG_SP3;
      break;    
    case FIRE:
      //LCDPIN27 |= LCD_SEG_SP3;
      break;    
    case FREESCALE:
      LCDPIN28 |= LCD_SEG_SP3;
      break;
    case SNOWFLAKE:
      LCDPIN25 |= LCD_SEG_SP3;
      break;
    case ALARM:
      LCDPIN24 |= LCD_SEG_SP3;
      break;
    
    case DTx:
      switch(num)
      {
        case 1:
          LCDPIN27 |= LCD_SEG_SP2;
          break;
        case 2:
          LCDPIN26 |= LCD_SEG_SP2;
          break;        
        case 3:
          LCDPIN25 |= LCD_SEG_SP2;
          break;         
        case 4:
          LCDPIN24 |= LCD_SEG_SP2;
          break;        
        case 5:
          LCDPIN23 |= LCD_SEG_SP2;
          break;        
        case 6:
          LCDPIN22 |= LCD_SEG_SP2;
          break;        
        case 7:
          LCDPIN21 |= LCD_SEG_SP2;
          break;        
        case 8:
          LCDPIN20 |= LCD_SEG_SP2;
          break;        
        case 9:
          LCDPIN17 |= LCD_SEG_SP1;
          break;        
      }
      break;
    case COLx:
      switch(num)
      {
        case 2:
          LCDPIN26 |= LCD_SEG_SP3;
          break;
        case 5:
          LCDPIN23 |= LCD_SEG_SP3;
          break;        
        case 7:
          LCDPIN21 |= LCD_SEG_SP3;
          break;         
        case 8: //Degrees
          LCDPIN22 |= LCD_SEG_SP3;
          break;               
      }
      break;
    case Vx:
      switch(num)
      {
        case 1:
          LCDPIN19 |= LCD_SEG_M4;
          break;
        case 2:
          LCDPIN19 |= LCD_SEG_M5;
          break;        
        case 3:
          LCDPIN18 |= LCD_SEG_M5;
          break;         
        case 4:
          LCDPIN18 |= LCD_SEG_M4;
          break;
        case 5:
          LCDPIN18 |= LCD_SEG_M6;
          break;                          
      }
      break;
    case Tx:
      switch(num)
      {
        case 1:
          LCDPIN19 |= LCD_SEG_M8;
          break;
        case 2:
          LCDPIN19 |= LCD_SEG_M1;
          break;        
        case 3:
          LCDPIN19 |= LCD_SEG_M2;
          break;         
        case 4:
          LCDPIN19 |= LCD_SEG_M7;
          break;                        
      }
      break;
  }  
  
 
 #endif
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDClearSegment
// --------------------------------------------------------------------------------------
// Disables Segment in the LCDRAM Register
/////////////////////////////////////////////////////////////////////////////////////////
void LCDClearSegment(enum symbol s, unsigned char num)
{
 #if LCD_PIN_44
 
  switch(s)
  {
  
    case TIME:
      LCDPIN7 &= ~LCD_SEG_SIGN;
      break;
    case DT:
      LCDPIN8 &= ~LCD_SEG_M2;
      break;
    case DATE:
      LCDPIN11 &= ~LCD_SEG_SIGN;
      break;
    case FREESCALE:
      //LCDPIN42 &= ~LCD_SEG_SIGN;
      break;
    case FIRE:
      //LCDPIN41 &= ~LCD_SEG_SIGN;
      break;
    case TEMP:
      //LCDPIN38 &= ~LCD_SEG_SIGN;
      break;  
    case SNOWFLAKE:
      //LCDPIN37 &= ~LCD_SEG_M2;
      break;
    case VOLUME:
      //LCDPIN34 &= ~LCD_SEG_SIGN;
      break;
    case ALARM:
      LCDPIN30 &= ~LCD_SEG_SIGN;
      break;
    case T:
      LCDPIN29 &= ~LCD_SEG_M2;
      break;
    case P:
      LCDPIN15 &= ~LCD_SEG_SIGN;
      break;
    case CONTRAST:
      LCDPIN19 &= ~LCD_SEG_M2;
      break;
    case MODE:
      LCDPIN20 &= ~LCD_SEG_M2;
      break;
    case AM:
      LCDPIN25 &= ~LCD_SEG_M4;
      break;
    case PM:
      LCDPIN25 &= ~LCD_SEG_M5;
      break;
    case OM:
      LCDPIN25 &= ~LCD_SEG_M6;
      break;
    case KWH:
      LCDPIN26 &= ~LCD_SEG_M3;
      break;
    case AMPS:
      LCDPIN26 &= ~LCD_SEG_M4;
      break;
    case VOLTS:
      LCDPIN26 &= ~LCD_SEG_M5;
      break;
    case PROGRAM:
      LCDPIN26 &= ~LCD_SEG_M6;
      break;
    case VOL:
      LCDPIN27 &= ~LCD_SEG_M3;
      break;
      
    case DTx:
      switch(num)
      {
        case 1:
          LCDPIN8 &= ~LCD_SEG_M1;
          break;
        case 2:
          LCDPIN12 &= ~LCD_SEG_M2;
          break;        
        case 3:
          //LCDPIN41 &= ~LCD_SEG_M1;
          break;         
        case 4:
          //LCDPIN37 &= ~LCD_SEG_M1;
          break;        
        case 5:
          //LCDPIN33 &= ~LCD_SEG_M2;
          break;        
        case 6:
          LCDPIN29 &= ~LCD_SEG_M1;
          break;        
        case 7:
          LCDPIN16 &= ~LCD_SEG_M2;
          break;        
        case 8:
          LCDPIN20 &= ~LCD_SEG_M1;
          break;        
        case 9:
          LCDPIN24 &= ~LCD_SEG_M1;
          break;        
      }
      break;
    case COLx:
      switch(num)
      {
        case 2:
          LCDPIN12 &= ~LCD_SEG_M1;
          break;
        case 5:
          //LCDPIN33 &= ~LCD_SEG_M1;
          break;        
        case 7:
          LCDPIN16 &= ~LCD_SEG_M1;
          break;         
        case 8: //Degrees
          LCDPIN23 &= ~LCD_SEG_SIGN;
          break;               
      }
      break;
    case Vx:
      switch(num)
      {
        case 1:
          LCDPIN24 &= ~LCD_SEG_M2;
          break;
        case 2:
          LCDPIN25 &= ~LCD_SEG_M3;
          break;        
        case 3:
          LCDPIN27 &= ~LCD_SEG_M4;
          break;         
        case 4:
          LCDPIN27 &= ~LCD_SEG_M5;
          break;
        case 5:
          LCDPIN27 &= ~LCD_SEG_M6;
          break;                          
      }
      break;
    case Tx:
      switch(num)
      {
        case 1:
          LCDPIN28 &= ~LCD_SEG_M6;
          break;
        case 2:
          LCDPIN28 &= ~LCD_SEG_M5;
          break;        
        case 3:
          LCDPIN28 &= ~LCD_SEG_M4;
          break;         
        case 4:
          LCDPIN28 &= ~LCD_SEG_M3;
          break;                        
      }
      break;
  }

#else
switch(s)
  {
  
    case TIME:
      LCDPIN9  &= ~LCD_SEG_SP1;
      break;
    case DATE:
      LCDPIN10 &= ~LCD_SEG_SP1;
      break;
    case TEMP:
      LCDPIN11 &= ~LCD_SEG_SP1;
      break;
    case VOLUME:
      LCDPIN12 &= ~LCD_SEG_SP1;
      break;
    case CONTRAST:
      LCDPIN13 &= ~LCD_SEG_SP1;
      break;
    case MODE:
      LCDPIN14 &= ~LCD_SEG_SP1;
      break;
    case PROGRAM:
      LCDPIN15 &= ~LCD_SEG_SP1;
      break;
    case OM:
      LCDPIN16 &= ~LCD_SEG_SP1;
      break;
    case AM:
      LCDPIN18 &= ~LCD_SEG_M3;
      break;
    case PM:
      LCDPIN18 &= ~LCD_SEG_M2;
      break;
    case KWH:
      LCDPIN18 &= ~LCD_SEG_M7;
      break;
    case AMPS:
      LCDPIN18 &= ~LCD_SEG_M1;
      break;
    case VOLTS:
      LCDPIN18 &= ~LCD_SEG_M8;
      break;
    case VOL:
      LCDPIN19 &= ~LCD_SEG_M6;
      break;
    case T:
      LCDPIN19 &= ~LCD_SEG_M3;
      break;
    case P:
      LCDPIN22 &= ~LCD_SEG_SP3;
      break;    
    case FIRE:
      LCDPIN27 &= ~LCD_SEG_SP3;
      break;    
    case FREESCALE:
      LCDPIN28 &= ~LCD_SEG_SP3;
      break;
    case SNOWFLAKE:
      LCDPIN25 &= ~LCD_SEG_SP3;
      break;
    case ALARM:
      LCDPIN24 &= ~LCD_SEG_SP3;
      break;
    
    case DTx:
      switch(num)
      {
        case 1:
          LCDPIN27 &= ~LCD_SEG_SP2;
          break;
        case 2:
          LCDPIN26 &= ~LCD_SEG_SP2;
          break;        
        case 3:
          LCDPIN25 &= ~LCD_SEG_SP2;
          break;         
        case 4:
          LCDPIN24 &= ~LCD_SEG_SP2;
          break;        
        case 5:
          LCDPIN23 &= ~LCD_SEG_SP2;
          break;        
        case 6:
          LCDPIN22 &= ~LCD_SEG_SP2;
          break;        
        case 7:
          LCDPIN21 &= ~LCD_SEG_SP2;
          break;        
        case 8:
          LCDPIN20 &= ~LCD_SEG_SP2;
          break;        
        case 9:
          LCDPIN17 &= ~LCD_SEG_SP1;
          break;        
      }
      break;
    case COLx:
      switch(num)
      {
        case 2:
          LCDPIN26 &= ~LCD_SEG_SP3;
          break;
        case 5:
          LCDPIN23 &= ~LCD_SEG_SP3;
          break;        
        case 7:
          LCDPIN21 &= ~LCD_SEG_SP3;
          break;         
        case 8: //Degrees
          LCDPIN22 &= ~LCD_SEG_SP3;
          break;               
      }
      break;
    case Vx:
      switch(num)
      {
        case 1:
          LCDPIN19 &= ~LCD_SEG_M4;
          break;
        case 2:
          LCDPIN19 &= ~LCD_SEG_M5;
          break;        
        case 3:
          LCDPIN18 &= ~LCD_SEG_M5;
          break;         
        case 4:
          LCDPIN18 &= ~LCD_SEG_M4;
          break;
        case 5:
          LCDPIN18 &= ~LCD_SEG_M6;
          break;                          
      }
      break;
    case Tx:
      switch(num)
      {
        case 1:
          LCDPIN19 &= ~LCD_SEG_M8;
          break;
        case 2:
          LCDPIN19 &= ~LCD_SEG_M1;
          break;        
        case 3:
          LCDPIN19 &= ~LCD_SEG_M2;
          break;         
        case 4:
          LCDPIN19 &= ~LCD_SEG_M7;
          break;                        
      }
      break;
  }  
  
 
 #endif
 
}

/////////////////////////////////////////////////////////////////////////////////////////
// LCDTest
// --------------------------------------------------------------------------------------
// Enable all segments in the LCDRAM Register
/////////////////////////////////////////////////////////////////////////////////////////
void LCDTest (void)
{
  
  char i;
  
  LCDWriteSegment(TIME,0);
  LCDWriteSegment(DT,0);
  LCDWriteSegment(DATE,0);
  LCDWriteSegment(FREESCALE,0);
  LCDWriteSegment(FIRE,0);
  LCDWriteSegment(TEMP,0);
  LCDWriteSegment(SNOWFLAKE,0);
  LCDWriteSegment(VOLUME,0);
  LCDWriteSegment(ALARM,0);
  LCDWriteSegment(T,0);
  LCDWriteSegment(P,0);
  LCDWriteSegment(CONTRAST,0);
  LCDWriteSegment(MODE,0);
  LCDWriteSegment(AM,0);
  LCDWriteSegment(PM,0);
  LCDWriteSegment(OM,0);
  LCDWriteSegment(KWH,0);
  LCDWriteSegment(AMPS,0);
  LCDWriteSegment(VOLTS,0); 
  LCDWriteSegment(PROGRAM,0);
  LCDWriteSegment(VOL,0);
  for (i=1; i<10; i++) 
  { 
    LCDPutChar('?', i);
    LCDWriteSegment(DTx,i);
    if (i<6) 
    {
      LCDWriteSegment(Vx,i);
      if (i<5) 
      {
        LCDWriteSegment(COLx,i);
        LCDWriteSegment(Tx,i);
      }
    }
  }  
       
}

void LCDClearSegs(void)
{
  
  char i;
  
  LCDWriteSegment(TIME,0);
  Delay(1);
  LCDWriteSegment(DT,0);
  Delay(1);
  LCDWriteSegment(DATE,0);
  Delay(1);
  LCDWriteSegment(FREESCALE,0);
  Delay(1);
  LCDWriteSegment(FIRE,0);
  Delay(1);
  LCDWriteSegment(TEMP,0);
  Delay(1);
  LCDWriteSegment(SNOWFLAKE,0);
  Delay(1);
  LCDWriteSegment(VOLUME,0);
  Delay(1);
  LCDWriteSegment(ALARM,0);
  Delay(1);
  LCDWriteSegment(T,0);
  Delay(1);
  LCDWriteSegment(P,0);
  Delay(1);
  LCDWriteSegment(CONTRAST,0);
  Delay(1);
  LCDWriteSegment(MODE,0);
  Delay(1);
  LCDWriteSegment(AM,0);
  Delay(1);
  LCDWriteSegment(PM,0);
  Delay(1);
  LCDWriteSegment(OM,0);
  Delay(1);
  LCDWriteSegment(KWH,0);
  Delay(1);
  LCDWriteSegment(AMPS,0);
  Delay(1);
  LCDWriteSegment(VOLTS,0); 
  Delay(1);
  LCDWriteSegment(PROGRAM,0);
  Delay(1);
  LCDWriteSegment(VOL,0);
  Delay(1);
  for (i=1; i<10; i++) 
  {
    LCDWriteSegment(DTx,i);
    Delay(1);
    if (i<6) 
    {
      LCDWriteSegment(Vx,i);
      Delay(1);
      if (i<5) 
      {
        LCDWriteSegment(COLx,i);
        Delay(1);
        LCDWriteSegment(Tx,i);
        Delay(1);
      }
    }
  }       
}

void LCDClearDigit(unsigned char digit)
{
  unsigned char *pLCDPIN9 = &LCDPIN8;
  unsigned char *pLCDPIN28= &LCDPIN29;

  #if LCD_PIN_44
  
  switch(digit)
  {
    case 1:
          LCDPIN5 = 0;
          LCDPIN6 = 0;
          LCDPIN7 &= ~MASK_DLK;
          LCDPIN8 &= ~MASK_CB;
          break;
          
    case 2:
          LCDPIN9 = 0;
          LCDPIN10 = 0;
          LCDPIN11 &= ~MASK_DLK;
          LCDPIN12 &= ~MASK_CB;
          break;
          
    /*case 3:
          LCDPIN44 = 0;
          LCDPIN43 = 0;
          LCDPIN42 &= ~MASK_DLK;
          LCDPIN41 &= ~MASK_CB;
          break;*/
          
    /*case 4:
          LCDPIN40 = 0;
          LCDPIN39 = 0;
          LCDPIN38 &= ~MASK_DLK;
          LCDPIN37 &= ~MASK_CB;
          break;*/
          
    /*case 5:
          LCDPIN36 = 0;
          LCDPIN35 = 0;
          LCDPIN34 &= ~MASK_DLK;
          LCDPIN33 &= ~MASK_CB;
          break;*/
          
    case 6:
          LCDPIN32 = 0;
          LCDPIN31 = 0;
          LCDPIN30 &= ~MASK_DLK;
          LCDPIN29 &= ~MASK_CB;
          break;
          
    case 7:
          LCDPIN13 = 0;
          LCDPIN14 = 0;
          LCDPIN15 &= ~MASK_DLK;
          LCDPIN16 &= ~MASK_CB;
          break;
          
    case 8:
          LCDPIN17 = 0;
          LCDPIN18 = 0;
          LCDPIN19 &= ~MASK_DLK;
          LCDPIN20 &= ~MASK_CB;
          break;
          
    case 9:
          LCDPIN21 = 0;
          LCDPIN22 = 0;
          LCDPIN23 &= ~MASK_DLK;
          LCDPIN24 &= ~MASK_CB;
          break;
  }
  
 #else
 
 *(pLCDPIN9+(digit-1)) &= ~MASK_JKBCLMD;
 *(pLCDPIN28-(digit-1)) &= ~MASK_AHFGEN;
 
 #endif 
}

void LCDScrollNumbersAndAlphabetbets(unsigned char digit)
{
  char c= 'A';
  unsigned char i;

  for(i='0';i<='9';i++)
  {
    LCDClearDigit(digit);
    LCDPutChar(i, digit);
    Delay(1);
  }
  
  for(;c<='Z';c++)
  {
    LCDClearDigit(digit);
    LCDPutChar(c, digit);
    Delay(1);
  }
}

void LCDKey1Func()
{
  unsigned char digit = 1;
  
  LCD_ALLFP_OFF
  
  while(digit<=9)
  {
    LCDScrollNumbersAndAlphabetbets(digit);
    digit++;
  }
}

void LCDDiagnosticFunc(void)
{
 
  
  LCDTest();     // Special segments only
  
  Delay(10);
  
  LCDKey1Func();     //All segment glow with special segments
}

void LCDVolFunc(void)
{
       
  char i;   

  LCDWriteSegment(VOL,0);
  Delay(1);
  for (i=1; i<10; i++) 
  {
    LCDWriteSegment(Vx,i);
    Delay(1);
  }
}

void LCDPowerFunc(void)
{
       
  char i;   
   
  LCDWriteSegment(P,0);
  LCDWriteSegment(T,0);
  Delay(1);
  for (i=1; i<10; i++) 
  { 
    LCDWriteSegment(Tx,i);
    Delay(1);
  }
}