/*
 * File:		QSPI.c
 * Purpose:		QSPI transfer using wraparound mode
 *
 * Notes:		
 *  
 */

#include "common.h"
#include "qspi.h"

/*
 * QSPIinit: starts QSPI controller
 *
 * Parameters: none
 *
 * Return : none.
 */
void QSPIinit()
{
	/* Enable QSPI Pins */
	MCF_GPIO_PQSPAR = (0
					| MCF_GPIO_PQSPAR_PQSPAR0(1)//dout
					| MCF_GPIO_PQSPAR_PQSPAR1(1)//din
					| MCF_GPIO_PQSPAR_PQSPAR2(1)//sck
					| MCF_GPIO_PQSPAR_PQSPAR3(1)//cs0
					);
					
	#ifdef QSPI_STANDAR_CONFIGURATION
		/* CPHA = CPOL = OFF; */
		MCF_QSPI_QMR &= ~(0
					 | MCF_QSPI_QMR_CPHA 
					 | MCF_QSPI_QMR_CPOL
					 );

	#else
		/* CPHA = CPOL = ON; */
		MCF_QSPI_QMR |= (0
				     | MCF_QSPI_QMR_CPHA 
					 | MCF_QSPI_QMR_CPOL
					 );
	#endif
	
	/* Set SPI baud rate */
  	/* 20MHz / QMR[BAUD] = baudrate; 1 < QMR[BAUD] < 256 @40MHz bus clock */
	MCF_QSPI_QMR |= (0
				 | MCF_QSPI_QMR_MSTR
				 | MCF_QSPI_QMR_BAUD(2)//10Mbps
				 );
	
	/* Set QCD */
	MCF_QSPI_QDLYR = MCF_QSPI_QDLYR_QCD(QSPI_QCD_DELAY);
	
	/* Set DTL */
	MCF_QSPI_QDLYR = MCF_QSPI_QDLYR_DTL(QSPI_DTL_DELAY);
	
	/* Make sure SPIF is cleared */
	MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
	
	return;
}

/*
 * QSPIsendReceive: send and receive through SPI bus
 *
 * Parameters: qspi_state: struct containing spi state
 *
 * Return : error flag.
 *
 * Non-reentrant nor interruptible
 */
uint8 QSPIsendReceive(qspi_state_t * qspi_state)
{
	uint32 i,j,temp_write,temp_read;
	uint8 complete_buffers,remain_entries;
	uint8 remain_entries_last_buffer,remain_entries_first_buffer;
	/* used to avoid getting stuck on code */
	vuint32 u32Rescue;

	u32Rescue = 0;

	/* setting transfer size */
	MCF_QSPI_QMR &= ~(MCF_QSPI_QMR_BITS(0xF));	
	MCF_QSPI_QMR |= MCF_QSPI_QMR_BITS(qspi_state->bits_per_data);

	/* less than a queue buffer */
	if(qspi_state->size < QSPI_BUFFER_CAPACITY+1)
	{
		/* sets up wrap register for a 16-bit transfers */
		MCF_QSPI_QWR = 0
					 | MCF_QSPI_QWR_CSIV 
					 | MCF_QSPI_QWR_ENDQP(qspi_state->size - 1)
					 ;
		/* Make sure SPIF is cleared */
		MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
		/* sets queue pointer to the first command RAM entry */		
		MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
		
		/* command entry for transfer to QSPI_CS0 */
		for(i=0;i<qspi_state->size;i++)
		{
			MCF_QSPI_QDR = 0
						 | MCF_QSPI_QCR_CS(0)
						 | MCF_QSPI_QCR_BITSE
						 | MCF_QSPI_QCR_CONT
						 ;
		}

		/* sets queue pointer to the first data RAM entry */
		MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
		
		/* data RAM entry for transfer to QSPI_CS0 */
		for(i=0;i<qspi_state->size;i++)
		{
			MCF_QSPI_QDR = qspi_state->data[i];
		}
		
		/* starts the transfer */
		MCF_QSPI_QDLYR |= MCF_QSPI_QDLYR_SPE;
		
		while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF))
		{
	        if (u32Rescue++ == RESCUE_VALUE)
	            return ERROR;
        } /* wait until transfer is over */
		
		if(qspi_state->receive_flag)
		{
			/* Transfer data from recieve ram to data pointer */
			MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
			
			/* data RAM entry from receive data */
			for(i=0;i<qspi_state->size;i++)
			{
				qspi_state->data[i] = MCF_QSPI_QDR;
			}
		}
	}
	/* needs wraparound */
	else
	{
		/************** taking index of transfers *****************/
	
		/* calculating number of complete buffers */
		complete_buffers = qspi_state->size / QSPI_BUFFER_CAPACITY;
		/* calculating number of reminders, if there are */
		remain_entries = qspi_state->size % QSPI_BUFFER_CAPACITY;
		
		/************** (I) special first case *****************/
		
		MCF_QSPI_QWR = 0
					 | MCF_QSPI_QWR_CSIV
					 | MCF_QSPI_QWR_WREN
					 | MCF_QSPI_QWR_ENDQP(QSPI_BUFFER_CAPACITY-1)
					 ;
		
		/* Make sure SPIF is cleared */
		MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
		/* set queue pointer to the first command RAM entry */		
		MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
		
		for(i=0;i<QSPI_BUFFER_CAPACITY;i++)
		{
			MCF_QSPI_QDR = 0
						 | MCF_QSPI_QCR_CS(0)
						 | MCF_QSPI_QCR_BITSE
						 | MCF_QSPI_QCR_CONT
						 ;
		}
		
		/* set queue pointer to the first data RAM entry */
		MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
		
		for(i=0;i<QSPI_BUFFER_CAPACITY;i++)
		{
			MCF_QSPI_QDR = qspi_state->data[i];
		}
		
		/* start the transfer */
		MCF_QSPI_QDLYR |= MCF_QSPI_QDLYR_SPE;		
		
		complete_buffers--;
		
		/* waiting until queue pointer points to zero to start */
		while (GET_MCF_QSPI_QWR_CPTQP)
		{
			if (u32Rescue++ == RESCUE_VALUE)
	        	return ERROR;	
		}		
		
		/**************** (II) complete next buffers ******/
		
		for(i=0;i<complete_buffers;i++)
		{	
			/* calculating array pointers */
			temp_read = i*QSPI_BUFFER_CAPACITY;
			temp_write = QSPI_BUFFER_CAPACITY+temp_read;			
		
			/* waiting until queue pointer is in the second half of buffer */
			while (GET_MCF_QSPI_QWR_CPTQP < (QSPI_BUFFER_CAPACITY/2))
			{
				if (u32Rescue++ == RESCUE_VALUE)
	            	return ERROR;	
			}
			
			/* reading received data if needed from buffer's first half */
			if(qspi_state->receive_flag)
			{
				/* Transfer data from recieve ram to data pointer */
				MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
				
				/* data RAM entry for receive data */
				for(j=0;j<(QSPI_BUFFER_CAPACITY/2);j++)
				{
					qspi_state->data[temp_read+j] = MCF_QSPI_QDR;
				}
			}
			
			/* writing to buffer's first half */
			MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
			
			for(j=0;j<(QSPI_BUFFER_CAPACITY/2);j++)
			{	
				MCF_QSPI_QDR = qspi_state->data[temp_write+j];
			}
			
			/* waiting until queue pointer has finished with the buffer */
			/* it will return to first element of buffer due to WREN */
			while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF))
			{
		        if (u32Rescue++ == RESCUE_VALUE)
		            return ERROR;
			}
			
			/******** CASE: elements finished, no remain_entries **********/
			/* checking if it's neccesary to turn off wraparound mode */
			/* if it's true, turn it off and wait for SPIF flag on next
			 * buffer complete due to lack of more elements to send */
			if( (i == (complete_buffers-1) ) && ( remain_entries == 0 ) )
			{
				MCF_QSPI_QWR = 0
					 		 | MCF_QSPI_QWR_CSIV
					 		 | MCF_QSPI_QWR_ENDQP(QSPI_BUFFER_CAPACITY-1)
					 		 ;
			}
			
			/* reading received data if needed from buffer's second half */
			if(qspi_state->receive_flag)
			{
				/* Transfer data from recieve ram */
				MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS+(QSPI_BUFFER_CAPACITY/2);
				
				/* data RAM entry for receive data */
				for(j=(QSPI_BUFFER_CAPACITY/2);j<QSPI_BUFFER_CAPACITY;j++)
				{
					qspi_state->data[temp_read+j] = MCF_QSPI_QDR;
				}
			}
			
			/* writing to buffer's second half */
			MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS+(QSPI_BUFFER_CAPACITY/2);
			
			for(j=(QSPI_BUFFER_CAPACITY/2);j<QSPI_BUFFER_CAPACITY;j++)
			{
				MCF_QSPI_QDR = qspi_state->data[temp_write+j];
			}
			
			/* Clear SPIF */
			MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
		}
		
		/************** (III) next: possible remain entries **********/
		
		if(remain_entries)
		{	
			/* calculating array index to tx/rx */
			if(remain_entries > (QSPI_BUFFER_CAPACITY/2))
			{
				remain_entries_last_buffer = remain_entries-(QSPI_BUFFER_CAPACITY/2);
				remain_entries_first_buffer = (QSPI_BUFFER_CAPACITY/2);
			}
			else
			{
				remain_entries_last_buffer = 0;
				remain_entries_first_buffer = remain_entries;
			}
		
			/* calculating array pointers */
			temp_read = QSPI_BUFFER_CAPACITY*complete_buffers;
			temp_write = QSPI_BUFFER_CAPACITY+temp_read;	
	        
			/* waiting until queue pointer is in the second half of buffer */
			while (GET_MCF_QSPI_QWR_CPTQP < (QSPI_BUFFER_CAPACITY/2))
			{
				if (u32Rescue++ == RESCUE_VALUE)
	            	return ERROR;	
			}
	        
	        /* reading received data if needed from buffer's first half */			
			if(qspi_state->receive_flag)
			{
				/* Transfer data from recieve ram */
				MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
				
				/* data RAM entry for receive data */
				for(j=0;j<(QSPI_BUFFER_CAPACITY/2);j++)
				{
					qspi_state->data[temp_read+j] = MCF_QSPI_QDR;
				}
			}
			
			/* writing to buffer's first half */
			MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
			
			for(j=0;j<remain_entries_first_buffer;j++)
			{
				MCF_QSPI_QDR = qspi_state->data[temp_write+j];
			}
			
			/* waiting until queue pointer is in the second half of buffer */
			while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF))
			{
				if (u32Rescue++ == RESCUE_VALUE)
	            	return ERROR;	
			}			
			
			/* Clear SPIF */
			MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;			
			
			/* turning off wraparound */    
			MCF_QSPI_QWR = 0
					 	 | MCF_QSPI_QWR_CSIV
					 	 | MCF_QSPI_QWR_ENDQP(remain_entries-1)
					 	 ;
			
			/* reading received data if needed from buffer's second half */
			if(qspi_state->receive_flag)
			{
				/* Transfer data from recieve ram */
				MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS + (QSPI_BUFFER_CAPACITY/2);
				
				/* data RAM entry for receive data */
				for(j=(QSPI_BUFFER_CAPACITY/2);j<(QSPI_BUFFER_CAPACITY);j++)
				{
					qspi_state->data[temp_read+j] = MCF_QSPI_QDR;
				}
			}
			
			/* check if there is something to write to buffer's second half */
			if(remain_entries_last_buffer)
			{
				/* writing to buffer's second half */
				MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS + (QSPI_BUFFER_CAPACITY/2);
				
				for(j=0;j<remain_entries_last_buffer ;j++)
				{
					MCF_QSPI_QDR = qspi_state->data[temp_write+j+(QSPI_BUFFER_CAPACITY/2)];	
				}
			}
			
			while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF))
			{
		        if (u32Rescue++ == RESCUE_VALUE)
		            return ERROR;
	        } /* wait until transfer is over */
			
			/* reading all the remainders */
			if(qspi_state->receive_flag)
			{
				/* Transfer data from recieve ram */
				MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
				
				/* data RAM entry for receive data */
				for(j=0;j<remain_entries;j++)
				{
					qspi_state->data[temp_write+j] = MCF_QSPI_QDR;
				}
			}
		}
		else
		{
			/******** CASE: elements finished, no remain_entries **********/
			while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF))
			{
		        if (u32Rescue++ == RESCUE_VALUE)
		            return ERROR;
	        } /* wait until transfer is over */
		}
	}

	return NO_ERROR;
}