/************************************************************************************
* MC1322x flash routines
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#include "EmbeddedTypes.h"
#include "Platform.h"
#include "NVM.h"

#define mNvmCmd_SST_ReadId_c                 0x90

#define mNvmCmdReadData_c                    0x03
#define mNvmCmdReadStatus_c                  0x05
#define mNvmCmd_SST_EWSR_c                   0x50
#define mNvmCmd_WSR_c                        0x01         //Write Status Register command
#define mNvmCmd_WEN_c                        0x06         //Write ENable command
#define mNvmCmd_SST_SER_c                    0x20         //Sector ERase command (4KB)
#define mNvmCmd_PROG_c                       0x02         //Byte program command

#define mNvmId_SST_c                         0xBF

#define mNvmMaxAddress_c                     0x1ffff
#define mInternal_SST_RestrictedArea_c       0x1f000
#define mInternal_SST_RestrictedSector_c     0x80000000

#define mNvmSectorSize_SST_c                 4096
#define mNvmSectorsNumber_SST_c              32
#define mNvmSectorSize_ST_ATM_c              32768
#define mNvmSectorsNumber_ST_ATM_c           4

#define mNvmStatus_Busy_c                    (1<<0)
#define mNvmStatus_WE_c                      (1<<1)
#define mNvmStatus_WP_c                      ((1<<2)|(1<<3))
#define mNvmLastQuarterProtected_c           (1<<2)
#define mNvmLastHalfProtected_c              (1<<3)
#define mNvmAllProtected_c                   ((1<<2)|(1<<3))
#define mNvmLastQuarterFirstAddress_c        0x018000
#define mNvmLastHalfFirstAddress_c           0x010000

#define Nvm_ClockNo_DataBitsNo(ClockNo , DataBitsNo)  ( (((ClockNo)-1)<<8) | (1<<7) | (DataBitsNo) )

#define NvmClrStatus(pSpi, bits)    NvmSetClearStatus(pSpi,mNvmClearStatusBits_c, bits)
#define NvmSetStatus(pSpi, bits)    NvmSetClearStatus(pSpi,mNvmSetStatusBits_c, bits)
#define NvmEnableStatusWrite(pSpi)  NvmSendCmd(pSpi,(uint32_t)mNvmCmd_SST_EWSR_c << 24,Nvm_ClockNo_DataBitsNo(8,8))

#define NvmUnlockAll(pSpi)          NvmClrStatus(pSpi,1<<7 | 1<<5 | 1<<4 | 1<<3 | 1<<2 )

/************************************************************************************
*************************************************************************************
* Private type definitions

*************************************************************************************
************************************************************************************/
typedef enum
{
  mNvmClearStatusBits_c,
  mNvmSetStatusBits_c
} mNvmStatusBitsOp_t;

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
static uint8_t NvmReadStatus(SpiRegs_t *pSpi);
static void NvmSetClearStatus(SpiRegs_t *pSpi,mNvmStatusBitsOp_t mNvmStatusBitsOp,uint8_t bits);
static nvmErr_t NvmWrite(SpiRegs_t *pSpi, void *pSrc , uint32_t address, uint32_t numBytes);
static void NvmSendCmd(SpiRegs_t *pSpi ,uint32_t txData, uint32_t clkCtrl );

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
void NVM_EraseSector(SpiRegs_t *pSpi,uint32_t addr)
{
  register  uint32_t cmd;
  //Set SPI_SS_OUT in Auto/Active Low mode
  pSpi->Setup = 1;
  //Unlock all the FLASH memory for writing
  NvmUnlockAll(pSpi);
  //Send the Write ENable command
  NvmSendCmd(pSpi,(uint32_t)mNvmCmd_WEN_c << 24 , Nvm_ClockNo_DataBitsNo(8,8));
  //Prepare the sector erase command
  cmd = ((unsigned int)mNvmCmd_SST_SER_c << 24) | (addr & 0x00FFFFFF);
  //Send the sector erase command
  NvmSendCmd(pSpi,cmd,Nvm_ClockNo_DataBitsNo(32,32));  
  //Wait untill the sector is erased
  while (NvmReadStatus(pSpi) & 0x01);
  //We are done, lock all the FLASH
  NvmSetStatus(pSpi,1<<3 | 1<<2);
}

/***********************************************************************************/
nvmErr_t NVM_Write(SpiRegs_t *pSpi,void *pSrc, uint32_t address, uint32_t numBytes)
{
  nvmErr_t nvmErr ;

  //Set SPI_SS_OUT in Auto/Active Low mode    
  pSpi->Setup = 1;
  //Unlock all the FLASH memory for writing
  NvmUnlockAll(pSpi);
  //Write the data to the specified address
  nvmErr = NvmWrite(pSpi, pSrc ,  address, numBytes);
  //We are done, lock all the FLASH
  NvmSetStatus(pSpi,1<<3 | 1<<2);
  if(gNvmErrNoError_c != nvmErr) 
    return nvmErr;
  //Werify the written data
  return NVM_ReadOp(mNvmOpVerify_c , pSpi , (uint8_t*)pSrc , address, numBytes,(uint16_t*)NULL);
}

/***********************************************************************************/
nvmErr_t NVM_ReadOp(mNvmReadOpType_t mNvmReadOpType ,SpiRegs_t *pSpi, uint8_t* pUint8 , uint32_t address, uint32_t numBytes , uint16_t* pCrc)
{
  nvmErr_t nvmErr ;
  uint8_t rx;
  nvmErr = gNvmErrNoError_c;
  
  // Setup SPI. Use SPI_CLK_POL = 0, RESYNC = 0, SPI0_SETUP = 2 (CE low). Other bits 0.
  if(pCrc)
    *pCrc = 0;
  pSpi->Setup = 2;
  // Write "read" cmd + address first
  // Write 0x03 0xaddress, 0xaddress, 0xaddress
  NvmSendCmd(pSpi,((uint32_t)mNvmCmdReadData_c << 24) | (address & 0x00FFFFFF),Nvm_ClockNo_DataBitsNo(32,32));
  //Start execution the requested operation, byte by byte
  while (numBytes--) 
  {
    NvmSendCmd(pSpi,((uint32_t)0x0),Nvm_ClockNo_DataBitsNo(8,0));  
    rx = pSpi->RxData;
    //Update the CRC
    if(pCrc)
      *pCrc +=rx;
    //Plain read
    if(mNvmOpRead_c == mNvmReadOpType)       
    {
      *pUint8++ = rx;
    }
    else if(mNvmOpVerify_c == mNvmReadOpType) //Verify the data from the FLASH
    {
      if(*pUint8++ != rx)      
        {
          nvmErr = gNvmErrVerifyError_c;
          break;
        }
    }
    else  //execute blank check
    {
      if( 0xff != rx)
        {
          nvmErr = gNvmErrBlankCheckError_c;
          break;
        }  
    } 
  }
  //Read operation finished, set SPI_SS_OUT high
  pSpi->Setup = 3;
  //Restore SPI to default. Use SPI_CLK_POL = 0, RESYNC = 0, SPI0_SETUP = 1. Other bits 0.
  pSpi->Setup = 1;
  return nvmErr;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
static void NvmSendCmd(SpiRegs_t *pSpi ,uint32_t txData, uint32_t clkCtrl )
{
  pSpi->TxData = txData;
  pSpi->ClkCtrl = clkCtrl; // 48 cycle sequence, "GO", 32 bit output (rest, 16 bit, are input)
  while (!(pSpi->Status & 0x1));
}

/***********************************************************************************/
static uint8_t NvmReadStatus(SpiRegs_t *pSpi)
{
  // Read status register (0x05)
  // Write 0x05, then read 8 bit.
  NvmSendCmd(pSpi,(uint32_t)mNvmCmdReadStatus_c << 24,Nvm_ClockNo_DataBitsNo(16,8));    
  return (pSpi->RxData & 0xFF);
}

/***********************************************************************************/
static void NvmSetClearStatus(SpiRegs_t *pSpi,mNvmStatusBitsOp_t mNvmStatusBitsOp,uint8_t bits)
{
  uint8_t tmpStatus;
  tmpStatus = NvmReadStatus(pSpi);
  
  NvmEnableStatusWrite(pSpi);
  // Write to status register (0x01), with tmpStatus | bits 
  // Write 0x01, newBits
  if(mNvmStatusBitsOp == mNvmSetStatusBits_c )
  tmpStatus |= bits;
  else
  tmpStatus &= (~bits);  
  NvmSendCmd(pSpi,((uint32_t)mNvmCmd_WSR_c << 24) | ((tmpStatus) << 16) , Nvm_ClockNo_DataBitsNo(16,16));
 }

/***********************************************************************************/
static nvmErr_t NvmWrite(SpiRegs_t *pSpi, void *pSrc , uint32_t address, uint32_t numBytes)
{
  uint8_t fStatus;
  uint8_t* pSrc_b;
  pSrc_b =  (uint8_t*)pSrc;
  while(numBytes--)
  {
    while ((fStatus=NvmReadStatus(pSpi)) & mNvmStatus_Busy_c); // Wait while busy is set!
    switch(fStatus & mNvmStatus_WP_c)
    {
    case mNvmAllProtected_c:
      return gNvmErrWriteProtect_c;
    case mNvmLastHalfProtected_c:
      if(address >= mNvmLastHalfFirstAddress_c)
        return gNvmErrWriteProtect_c;
      break;
    case mNvmLastQuarterProtected_c:
      if(address >= mNvmLastQuarterFirstAddress_c)
        return gNvmErrWriteProtect_c;
      break;
    }
    if((fStatus&0x2)==0)
    {
      // Nvm Write Enable
      NvmSendCmd(pSpi,(uint32_t)mNvmCmd_WEN_c << 24 , Nvm_ClockNo_DataBitsNo(8,8));
    }
    pSpi->Setup =  2;
    // Write "WRITE" cmd + address first
    // Write 0x02 0xaddress, 0xaddress, 0xaddress
    NvmSendCmd(pSpi,((uint32_t)mNvmCmd_PROG_c << 24) | (address++ & 0x00FFFFFF),Nvm_ClockNo_DataBitsNo(32,32));
    NvmSendCmd(pSpi,(uint32_t)*pSrc_b++ << 24,Nvm_ClockNo_DataBitsNo(8,8));
    pSpi->Setup =  3;
    pSpi->Setup =  1;
  }  
  while (NvmReadStatus(pSpi) & 0x01); // Wait while busy is set!	
  return gNvmErrNoError_c;
}
