/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#include "EmbeddedTypes.h"
#include "Platform.h"
#include "Engine.h"
#include "UART.h"
#include "NVM.h"

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#define SECTOR_SIZE         0x1000
#define FLASH_SIZE          0x20000
#define RAM_SIZE            0x18000
#define INTERNAL_FLASH      FLASH_REGS_P
#define EXTERNAL_FLASH      SPI_REGS_P
#define FLASH_HEADER_SIZE   8

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
void            SendConf(uint8_t status);
ENGCmdStatus_t  ExecuteErase(uint32_t address);
ENGCmdStatus_t  ExecuteWrite(uint32_t startAddress, uint16_t length, uint8_t* pBuffer);
ENGCmdStatus_t  ExecuteRead(uint32_t startAddress, uint16_t length, uint8_t* pBuffer);
ENGCmdStatus_t  ExecuteCommit(uint32_t length, ENGSecureOption_t secured);

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
const uint8_t securedString[] = "SECU";
const uint8_t unsecuredString[] = "OKOK";

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
ENGCommand_t command;

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
void ENG_Process(void)
{
  uint16_t length;
  uint16_t commandLength;
  ENGCmdStatus_t status;
  
  //Read command, test if the CRC validated
  if (FALSE == UART_ReceiveCmd((uint8_t*)&command, (uint16_t*)&commandLength)) {
    SendConf(gEngCRCError_c);
    return ;
  }
  
  //CRC was valid, move to execute the command
  switch(command.commandId){
    case engEraseReq_c: //Erase a sector or all the FLASH
      status = ExecuteErase(command.fields.eraseReq.address);
      SendConf(status);
    break;
    case engWriteReq_c: //Write to FLASH
      status = ExecuteWrite(command.fields.writeReq.startAddress, 
                                command.fields.writeReq.length,
                                command.fields.writeReq.buffer);
      SendConf(status);
    break;
    case engReadReq_c:  //Read from FLASH, in-place over the command, and send back as a ReadResp
      //Save the length, as it will be corrupted by the in-place write of the buffer in ReadResp 
      length = command.fields.readReq.length;
      status = ExecuteRead(command.fields.readReq.startAddress,
                              command.fields.readReq.length,
                              command.fields.readResp.buffer);
      if(status == gEngSuccessOp_c)
      {
        //Fill in the response command fields
        command.commandId = engReadResp_c;
        command.fields.readResp.cmdStatus = gEngSuccessOp_c;
        command.fields.readResp.length = length;
        //Send back the read response
        UART_SendCmd((uint8_t*)&command, sizeof(ENGCommand_t) - ENG_BUFFER_SIZE + length);
      }
      else
      {
        SendConf(gEngReadError_c);
      }
    break;
    case engCommitReq_c:
      status = ExecuteCommit(command.fields.commitReq.binLength,     
                            (ENGSecureOption_t)command.fields.commitReq.secured);
      SendConf(status);
    break;
    default: //If we ended up here it we received an invalid request
      SendConf(gEngInvalidReq_c);
  }
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Send a confirm command, with the specified status. A static array is used, with
* command id fixed, for speed.
*
*  Input parameters:
*  - status: status for the confirm to be sent.
*  Return:
*  - Nothing.
************************************************************************************/
void SendConf(uint8_t status)
{
  //Confirm message placeholder. Helps us avoid allocating a full packet.
  static uint8_t cnfArray[] = {engCmdCnf_c, gEngSuccessOp_c};

  //Copy the status code to the confirm message placeholder
  cnfArray[1] = status;
  UART_SendCmd((uint8_t*)cnfArray, 2);
}

/************************************************************************************
*  Erase a sector or all sectors of the MC1322x. Note that the last sector of the 
* internal FLASH of the MC1322x hold production data and will not be erased.
*
*  Input parameters:
*  - address: address in the FLASH sector to be erased. The LSB's of the address will
*      be ignored, only the bits specifying the sector are relevant. If the address 
*      is 0xFFFFFFFF, all the FLASH is erased.
*  Return:
*  - ENGCmdStatus_t: status of the erase operation.
************************************************************************************/
ENGCmdStatus_t ExecuteErase(uint32_t address)
{
  ENGCmdStatus_t status = gEngSuccessOp_c;
  uint16_t localAddress;
  //Test if we are required to erase only one sector
  if(address != 0xFFFFFFFF)
  {
    //Address validity: below 128k, can not erase the last sector (it holds production data)
    if(address < 0x0001F000)
    {
      //Request the erase of the specified sector
      NVM_EraseSector(INTERNAL_FLASH, address);
    }
    else
    {
      status = gEngInvalidReq_c;
    }
  }
  else
  {
    //Erase the whole FLASH (except the last sector), sector by sector
    for(localAddress = 0; localAddress < 31; localAddress++)
    {
      NVM_EraseSector(INTERNAL_FLASH, localAddress << 12);
    }
  }
  return status;
}

/************************************************************************************
*  Write data into the internal FLASH of the MC1322x. All of the written bytes in 
* FLASH are read back and verified.
*
*  Input parameters:
*  - startAddress: address in the FLASH to start writing from. It can not be in the
*       last sector, as the last sector of the internal FLASH holds production data.
*  - length: nuber of bytes to be written from the buffer. 
*  - pBuffer: pointer to the buffer containing the data to be written to the FLASH.
*  Return:
*  - ENGCmdStatus_t: status of the write operation.
************************************************************************************/
ENGCmdStatus_t ExecuteWrite(uint32_t startAddress, uint16_t length, uint8_t* pBuffer)
{
  ENGCmdStatus_t status = gEngSuccessOp_c;

  //Test address validity: last byte written needs to be below 124K
  if((startAddress + length) > 0x0001F000)
  {
    status = gEngInvalidReq_c;
  }
  else
  {
    //Address is OK, move to the actual write operation
    if(NVM_Write(INTERNAL_FLASH, pBuffer, startAddress, length) != gNvmErrNoError_c)
    {
      status = gEngWriteError_c;
    }
  }
  return status;
}

/************************************************************************************
*  Read data from the internal FLASH of the MC1322x.
*
*  Input parameters:
*  - startAddress: address in the FLASH to start reading from. It can not be in the last
*      sector, as the last sector of the internal FLASH holds production data.
*  - length: nuber of bytes to be read from the FLASH. Needs to be smaller than
*      the maximum payload of a read command (ENG_BUFFER_SIZE).
*  - pBuffer: pointer to the buffer ready to receive the data read from the FLASH.
*  Return:
*  - ENGCmdStatus_t: status of the read operation.
************************************************************************************/
ENGCmdStatus_t ExecuteRead(uint32_t startAddress, uint16_t length, uint8_t* pBuffer)
{
  ENGCmdStatus_t status = gEngSuccessOp_c;

  //Test address validity: last byte read needs to be below 124K
  if((startAddress + length) > 0x0001F000)
  {
    status = gEngInvalidReq_c;
  }
  else
  {
    if(NVM_Read(INTERNAL_FLASH, pBuffer, startAddress, length) != gNvmErrNoError_c)
    {
      status = gEngReadError_c;
    }
  }
  return status;   
}

/************************************************************************************
*  Commit the image written the internal FLASH of the MC1322x by writing the FLASH 
* header with the image descriptor.
*
*  Input parameters:
*  - length: the length of the image from the FLASH. This is the number of bytes that
*      the bootstrap will load from FLASH to RAM, starting from the first location
*      after the FLASH header.
*  - secured: secure or unsecure load of the image by the bootloader from FLASH
*      to RAM. Secure means that the ETM module shall not be enabled, thus
*      preventing the image to be copied.
*  Return:
*  - ENGCmdStatus_t: status of the commit operation.
************************************************************************************/
ENGCmdStatus_t ExecuteCommit(uint32_t length, ENGSecureOption_t secured)
{
  ENGCmdStatus_t status = gEngSuccessOp_c;
  uint8_t* securedOption;
  //Test the validity of the parameters
  if((length > RAM_SIZE - FLASH_HEADER_SIZE) || ((secured != engSecured_c)&&(secured != engUnsecured_c)))
  {
    status = gEngInvalidReq_c;  
  }
  else
  { 
    if(secured == engSecured_c) //Select the string to be written in FLASH based on the host option
    {
      securedOption = (uint8_t*)securedString;
    }
    else
    {
      securedOption = (uint8_t*)unsecuredString;
    }
    //Write the secure option to FLASH
    if(NVM_Write(INTERNAL_FLASH, securedOption, 0, 4) == gNvmErrNoError_c)
    {
      //Write the length of the image: 4 bytes from address 4 in FLASH
      if(NVM_Write(INTERNAL_FLASH, (uint8_t*)&length, 4, 4) != gNvmErrNoError_c)
      {
        status = gEngWriteError_c;
      }
    }
    else
    {
      status = gEngWriteError_c;
    }
  }
  
  return status;
}