/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#include "stdafx.h"
#include <stdio.h>
#include "UART.h"

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
static void printErrorDescription();

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
HANDLE UART_OpenCom(char* comPort, int baudRate)
{
  DCB configInfo;
  HANDLE hCom = INVALID_HANDLE_VALUE;
  COMMTIMEOUTS timeouts;

  //Prepare DCB structure
  memset(&configInfo, 0x00, sizeof(DCB));
  configInfo.DCBlength = sizeof(DCB);
  
  //Get a handle to COM
  hCom = CreateFile(comPort, GENERIC_READ|GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
  if(hCom == INVALID_HANDLE_VALUE)
  {
    printf("Can't open COM!\n");
#if VERBOSE_MODE
    printErrorDescription();
#endif
    return INVALID_HANDLE_VALUE;
  }
  //Get the config structure for the COM port
  if(!GetCommState(hCom, &configInfo))
  {
    //If there is an error, print a message. No need to exit here, we can still try to configure it.
    printf("Can't get COM config!\n");
#if VERBOSE_MODE
    printErrorDescription();
#endif
  }
  
  //Modify it to fit our needs
  configInfo.BaudRate = baudRate;
  configInfo.ByteSize = 8;
  configInfo.fParity = NOPARITY;
  configInfo.StopBits = ONESTOPBIT;
  configInfo.fRtsControl = RTS_CONTROL_ENABLE;

  //Configure COM
  if(!SetCommState(hCom, &configInfo))
  {
    printf("Can't config COM!\n");
#if VERBOSE_MODE
    printErrorDescription();
#endif
    //Clean up
    CloseHandle(hCom);
    return INVALID_HANDLE_VALUE;
  }
  //Set timeouts
  timeouts.ReadIntervalTimeout = 20;
  timeouts.ReadTotalTimeoutMultiplier = 5;
  timeouts.ReadTotalTimeoutConstant = 30;
  timeouts.WriteTotalTimeoutConstant = 20;
  timeouts.WriteTotalTimeoutMultiplier = 2;
  if(!SetCommTimeouts(hCom, &timeouts))
  {
    printf("Can't set time-out on COM!\n");
#if VERBOSE_MODE
    printErrorDescription();
#endif
  }
  //Return the handle we have  
  return hCom;
}

/***********************************************************************************/
UARTStatus_t UART_SendCmd(HANDLE hCom, char* pBuffer, unsigned short bytesToSend)
{
  DWORD bytesOps = 0;
  int index;
  char computedCRC = 0;
  char data;

  //Compute the CRC
  for(index = 0; index < bytesToSend; index++)
  {
    computedCRC+= *(pBuffer + index);        
  }

  //Send start of frame
  data = UART_SOF;
  WriteFile(hCom, &data, 1, &bytesOps, NULL);
  //Send the length of the packet
  WriteFile(hCom, &bytesToSend, 2, &bytesOps, NULL);
  //Send the buffer
  WriteFile(hCom, pBuffer, bytesToSend, &bytesOps, NULL);
  //Send the CRC
  WriteFile(hCom, &computedCRC, 1, &bytesOps, NULL);

  //We are done
  return gUARTSuccess_c;
}

/***********************************************************************************/
UARTStatus_t UART_ReceiveCmd(HANDLE hCom, DWORD timeOut, char* pBuffer, unsigned short* pBytesReceived)
{
  DWORD bytesOps = 0;
  DWORD startTime;  
  int index;
  char data;
  char computedCRC = 0;
  char receivedCRC = 0;
  unsigned short receivedLength = 0;
  bool lookingForSOF;

  //Get the start time
  startTime = timeGetTime();

  //Look for the start of the packet
  lookingForSOF = TRUE;
  do
  {
    if(!ReadFile(hCom, &data, 1, &bytesOps, NULL))
    {
      return gUARTReadFailure_c;
    }
    //Test first is the time expired
    if(((timeGetTime() - startTime) > timeOut))
    {
      return gUARTTimeOut_c;
    }
    //Verify if we received the SOF
    if((bytesOps == 1)&&(data == UART_SOF))
    {
      lookingForSOF = FALSE;
    }
  }while(lookingForSOF);
  
  //SOF detected, move to read the packet
  //Get the length of the packet
  if(!ReadFile(hCom, (char*)&receivedLength, 2, &bytesOps, NULL)) return gUARTReadFailure_c;
  //We should have received 2 bytes
  if(bytesOps != 2) return gUARTReadFailure_c;

  //Get the actual packet
  if(!ReadFile(hCom, pBuffer, receivedLength, &bytesOps, NULL)) return gUARTReadFailure_c;
  //We should have received *pBytesReceived bytes
  if(bytesOps != receivedLength) return gUARTReadFailure_c;

  //Get the CRC
  if(!ReadFile(hCom, (char*)&receivedCRC, 1, &bytesOps, NULL)) return gUARTReadFailure_c;
  //We should have received 1 byte
  if(bytesOps != 1) return gUARTReadFailure_c;
  //Copy the received length to the user - only now it is valid
  *pBytesReceived = receivedLength;

  //Compute the CRC on the received data and compare it with the received CRC
  computedCRC = 0;
  for(index = 0; index < receivedLength; index++)
  {
    computedCRC+= *(pBuffer + index);        
  }

  //Test the computed CRC against the received CRC
  if(computedCRC == receivedCRC)
  {
    return gUARTSuccess_c;
  }
  else
  {
    return gUARTCrcError_c;
  }
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Gets and prints information about the last error recorded by the OS.
*
*  Input parameters:
*  - None.
*  Return:
*  - Nothing.
************************************************************************************/
static void printErrorDescription()
{
  LPVOID msgBuf;
  DWORD dw = GetLastError();

  FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER | 
        FORMAT_MESSAGE_FROM_SYSTEM,
        NULL,
        dw,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
        (LPTSTR) &msgBuf,
        0, NULL );

  printf("\nError %d: %s", dw, msgBuf); 
  LocalFree(msgBuf);
}