/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* (c) Copyright 2001-2004 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      adc.c
*
* @brief     Emulated ADC convertor
*
* @author    r79251
* 
* @version   1.1.0.0
* 
* @date      Sep-20-2007
*
******************************************************************************/

#include "derivative.h"
#include "adc_cfg.h"

#include "adc.h"
#include "sci.h"

#define __WAIT()    __asm  WAIT
#define __NOP()     __asm  NOP

const UBYTE ADC_temperatureTable[] = {
  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,
  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  72,  73,  74,  75,  76,  77,  77,  78,  79,  80,  81,  82,  83,  84,  85,
  86,  87,  88,  89,  90,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99, 100,
 100, 101, 102, 103, 104, 105, 106, 107, 108, 108, 109, 110, 111, 112, 113, 114,
 115, 115, 116, 117, 118, 119, 120, 121, 121, 122, 123, 124, 125, 126, 126, 127,
 128, 129, 130, 131, 132, 132, 133, 134, 135, 136, 136, 137, 138, 139, 140, 141,
 141, 142, 143, 144, 145, 145, 146, 147, 148, 149, 149, 150, 151, 152, 153, 153,
 154, 155, 156, 156, 157, 158, 159, 159, 160, 161, 162, 163, 163, 164, 165, 166,
 166, 167, 168, 169, 169, 170, 171, 172, 172, 173, 174, 174, 175, 176, 177, 177,
 178, 179, 180, 180, 181, 182, 182, 183, 184, 185, 185, 186, 187, 187, 188, 189,
 189, 190, 191, 192, 192, 193, 194, 194, 195, 196, 196, 197, 198, 198, 199, 200,
 200, 201, 202, 202, 203, 204, 204, 205, 206, 206, 207, 208, 208, 209, 209, 210,
 211, 211, 212, 213, 213, 214, 215, 215, 216, 216, 217, 218, 218, 219, 219, 220,
 221, 221, 222, 223, 223, 224, 224, 225, 226, 226, 227, 227, 228, 228, 229, 230,
};


#if (USED_TABLE == 0) 

  const UBYTE ADC_potenciometerTable[] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,  70,  71,  72,  73,  74,  75,  76,  77,
   78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,
   94,  95,  96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109,
  110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 111, 112, 113,
  114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
  130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145,
  146, 147, 148, 149, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150,
  150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165,
  168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183,
  184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199,
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 201, 202, 203,
  204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219,
  220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220,
  220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220,
  220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220
  }; 
#endif

#if (USED_TABLE == 2) 
  const UBYTE ADC_potenciometerTable[] = {
    0,   0,   0,   0,  80,  82,  84,  86,  88,  90,  92,  94,  96,  98, 100, 102,
  104, 106, 108, 110, 110, 110, 110, 112, 114, 116, 118, 120, 122, 124, 126, 128,
  130, 132, 134, 136, 138, 140, 142, 144, 146, 148, 150, 150, 150, 150, 152, 155,
  160, 165, 170, 175, 180, 185, 190, 195, 200, 200, 200, 200, 205, 210, 215, 220,
  };
#endif

#define  ADC_DELAY    170        /* delay of start time measuring - the shorties time is 140 tics of timer  */ 
                                 /* so the time is measured after 130 tics. The 10 tics is reserve. */

/***************************************************************************//*!
*
* @brief  The function emulates ADC convertor with a usage of a internal analog comparator 
*
* @param ADC_CHANNEL_TYPE               /selection of measurement channel
*                 - CHAN_POTENTIOMETER  /for measurement of desired temperature
*                 - CHAN_THERMISTOR     /for measurement of actual  temperature
*
* @return ADC_output - a value corresponding to required or measured temperature
*
******************************************************************************
*
* Detailed description of the function:
* or an analog  measuring is used an internal analog comparator (ACMP) with an internal
* reference voltage (1.218 V). The ACMP compares voltage on a capacitor that is charged through
* potenciometer (temperature potentiometer) or through temperature sensor which resistance
* depends on temperature.
* It means that there are two measurement channels with selection - by high on PTA 4 (thermosensor)
* or on PTA 5 (potenciometer). From start (high on one of those pins)  is measured out time
* ADC_DELAY that is shorter then the shorties possible time from both channels. The delay time
* is saved in MTIMMOD register, then is the register reset and further time is measured 
* by timer until the comparator event occures (as comparator threshold is internal voltage
* reference - 1.218 V). After a executed measuring the capacitor is discharge by all
* connected pins. The measured time (saved in ADC_time) is used as index from a reading a corresponding
* value of temparature from appropriate look-up table.
* The function returns just the read value.
*
**********************
*
* Bus = 8 MHz
* Timer prescalar=1 -> Timer clock~8 MHz, Timer period ~ 32 us 
* Timer resolution = 125ns
*
******************************************************************************/ 


UBYTE ADC_Read (ADC_CHANNEL_TYPE eADC_channel)
{
    UBYTE  ADC_output;                                        /* measured voltage level */
    UBYTE  ADC_time = 0;                                      /* measured charging time; initialiation to zero */
       
    PTADD_PTADD1 = 0;                                         /* preparing  - PIN for measurement */
    
    ACMPSC = ACMPSC_ACME_MASK|ACMPSC_ACIE_MASK|ACMPSC_ACBGS_MASK|ACMPSC_ACMOD1_MASK|ACMPSC_ACF_MASK;   /* Enable ACMP */


                                                     
    if (eADC_channel == CHAN_THERMISTOR)                      /* set measurment channel  -  ChannelPot x ChannelTherm */
    {
        PTAD_PTAD4   = 1;                                     /* start RC rise - 5 V through sensor - set */
        PTADD_PTADD4 = 1;                                     /* start RC rise - 5 V through sensor - start*/
    }
    else 
    {
        PTAD_PTAD5   = 1;                                     /* start RC rise - 5 V through Potenciometer - set */
        PTADD_PTADD5 = 1;                                     /* start RC rise - 5 V through Potenciometer - start */
    }

    MTIMMOD = ADC_DELAY;                                      /* delay of measurment  */
    MTIMSC = MTIMSC_TRST_MASK|MTIMSC_TOIE_MASK;               /* Reset and enable overflow interupt from timer  */
 
    

    __WAIT();                                                 /*  wait for interupt - overflow of Timer modulo */
    if (MTIMSC_TOF == 1) MTIMSC_TOF = 0;                      /*  test if the event is really from timer */
    MTIMMOD = 0;                                              /*  write zero to modulo, also erase timer flag */  
    __WAIT();                                                 /*  wait for interupt from comparator - compare level was reached */
    while (ACMPSC_ACF == 0) __NOP();                          /*  test if the event is really from comparator */
   
    ADC_time = MTIMCNT;                                       /*  storing of actual timer value when the event occured */
    
    
    PTAD  &= ~(PTAD_PTAD1_MASK|PTAD_PTAD4_MASK|PTAD_PTAD5_MASK);  /* discharging of capacitor by pin 1 and pin 4 and 5 */
    PTADD = (PTADD & ~(PTADD_PTADD4_MASK|PTADD_PTADD5_MASK)) | PTADD_PTADD1_MASK; /* turn to input */
     
    ACMPSC_ACF = 1;                                           /* Clear ACMP Flag */
    ACMPSC = 0;                                               /* disable ACMP */
    MTIMSC_TOF = 0;                                           /* Clear  MTIM pending interrupt and    */
    MTIMSC_TOIE = 0;                                          /* disable overflow interupt from timer */
                                                                                                                           
   
    // SCI_putc(ADC_time);      //for debug
   
    
     /* a reading of corresponding values from appropriate look-up tables */
    if (eADC_channel == CHAN_THERMISTOR) ADC_output = ADC_temperatureTable[ADC_time];       
         
    else ADC_output = ADC_potenciometerTable[ADC_time >> USED_TABLE];

    return (UBYTE)ADC_output;                                 /* return back the measured value */
}
