
// ========================================================
//  Copyright (c)2006 Freescale Semiconductor Inc
//  All Rights Reserved
//
//  The code is the property of Freescale Inc.
//
//  The copyright notice above does not evidence any
//  actual or intended publication of such source code.
//
//  This software is provided by Freescale "AS IS" and any
//  expressed or implied warrenties, including, but not 
//  limited to, the implied warrenties of merchantability
//  and fitness for a particular purpose are disclaimed.
//  In no event shall Freescale or its contributors be 
//  liable for any direct, indirect, incidental, special,
//  examplary, or consequential damages (including, but not 
//  limited to, procurement of substitue goods or services;
//  loss of use, data, or profits; or business interruption)
//  However caused and on any theory of liability, whether 
//  in contract, strict liability, or tort (including 
//  negligence or otherwise) arising in any way out of the 
//  use of this software, even if advised of the possibility
//  of such damage.
// 
//
//
//  Filename:     $Source$
//  Author:       $Author$
//  Locker:       $Locker$
//  State:        $State$
//  Revision:     2.0
//
//  Functions:    
//
//  History:      Use the CVS command log to display 
//                revision history information.
//
//  Description:  
//
//
//
//  Notes:        
//        Rev 1.2   Update added for brake & turn signal monitoring
//        Rev 2.0   Update added for flasher support & Rev 2.0 HW
//        Rev 2.01  Updated for Left, Right, and Brake
//        Rev 2.11  Updated by WB & MR for Publication
//
// ========================================================

// --------------------------------------------------------
// Includes 
// --------------------------------------------------------
#include <stdlib.h>
#include "Types.h"
#include "Hicross.h"    // compiler specific
#include "ChipSets.h"   // hardware specific  
#include "Main.h"        // application specific

#include "l_api.h"      // lin api 
#include "l_diag.h"     // lin debugger

#include "TimerA.h"
#include "TimerB.h"
#include "Timer.h"
#include "Flasher.h"    // Routines to check L, R, Brake lights
#include "Filter.h"     // low pass filter for g-sensor values

// --------------------------------------------------------
// Macros
// --------------------------------------------------------

#define EVER   (;;)
#define UP      0
#define DOWN    1
#define OFF     0
#define ON      1

// --------------------------------------------------------
// redefine these based on final HW - mbr
#define UNUSED_PORT_A_PINS   (bPTA1|bPTA0)
#define UNUSED_PORT_B_PINS   (bPTB7|bPTB6|bPTB5|bPTB4|bPTB3|bPTB2|bPTB1)
#define UNUSED_PORT_C_PINS   (bPTC4|bPTC3|bPTC2)
#define UNUSED_PORT_D_PINS   (0)


// --------------------------------------------------------
//  Accelerometer Calibration Macros
//
//  These are the cutoff threshholds for ADC readings on G-sensor
//  outputs.  These will vary depending on sensor orientation,
//  sensitivity, and use of 8 or 10 bit ADC measurements.
//
//  Also - Vrefh will be 5V scale, but max sensor output is 3.3V
//
// --------------------------------------------------------

// Z-axis - 8 bit values, sensitivity at 1.5g based on ZSTAR boards
// At 0-g reads 120/255 = 47.1%  
// At 1-g reads 190/255 = 74.5%

// 3.3V on 5V scale is 66%    == 676 / 1024

// So translating to 5V ADC readings, should get
// 0-g reads 47.1% * 66% = 31.086% ~= 318/1024
// 1-g reads 74.5% * 66% = 49.170% ~= 503/1024

// So working range for all accelerations is about 318 to 676
// Deceleration goes in opposite direction, so full scale is 0 on ADC
// 
// Accelerometer mounted on other side of board for final HW - 
// Therefore 318 downto 0 will give deceleration

// Static definitions of deceleration levels

#define DECEL_LEVEL_0   400 // no deceleration - Must be calibrated based on orientation of board
#define DECEL_LEVEL_100   0 // full (100%) deceleration
    
    // Full deceleration and zero deceleration levels must be defined

#define DECEL_LEVEL_10   (UINT16)(DECEL_LEVEL_0*0.9)     // 10% of Maximum Deceleration 
#define DECEL_LEVEL_20   (UINT16)(DECEL_LEVEL_0*0.8)     // 20% of Maximum Deceleration 
#define DECEL_LEVEL_30   (UINT16)(DECEL_LEVEL_0*0.7)     // 30% of Maximum Deceleration 
#define DECEL_LEVEL_40   (UINT16)(DECEL_LEVEL_0*0.6)     // 40% of Maximum Deceleration 
#define DECEL_LEVEL_50   (UINT16)(DECEL_LEVEL_0*0.5)     // 50% of Maximum Deceleration 
#define DECEL_LEVEL_60   (UINT16)(DECEL_LEVEL_0*0.4)     // 60% of Maximum Deceleration 
#define DECEL_LEVEL_70   (UINT16)(DECEL_LEVEL_0*0.3)     // 70% of Maximum Deceleration 
#define DECEL_LEVEL_80   (UINT16)(DECEL_LEVEL_0*0.2)     // 80% of Maximum Deceleration 
#define DECEL_LEVEL_90   (UINT16)(DECEL_LEVEL_0*0.1)     // 90% of Maximum Deceleration 

// should be stored in an const array in FLASH for calibration
const UINT16 DECEL_LEVEL[] = {
  DECEL_LEVEL_0,
  DECEL_LEVEL_10,
  DECEL_LEVEL_20,
  DECEL_LEVEL_30,
  DECEL_LEVEL_40,
  DECEL_LEVEL_50,
  DECEL_LEVEL_60,
  DECEL_LEVEL_70,
  DECEL_LEVEL_80,
  DECEL_LEVEL_90,
  DECEL_LEVEL_100,
};


// --------------------------------------------------------
// HBLED State Macros
// --------------------------------------------------------

#define LEDON_oooo_oooo   0                           // No LEDs lit    (- - - -   - - - -)
#define LEDON_oooX_Xooo   bHB4_L                      // Center pair    (- - - *   * - - -)
#define LEDON_ooXX_XXoo   bHB4_L|bHB3_L               // Center 2 pair  (- - * *   * * - -)
#define LEDON_oXXX_XXXo   bHB4_L|bHB3_L|bHB2_L        // Center 3 pair  (- * * *   * * * -)
#define LEDON_XXXX_XXXX   bHB4_L|bHB3_L|bHB2_L|bHB1_L //    All 4 pair  (* * * *   * * * *)

#define LEDON_XXoo_ooXX   bHB2_L|bHB1_L               // Outside 2 pair for STOPPEDALERT (* * - -  - - * *)

#define X_AXIS_INPUT    ADC_CHANNEL_PTB5    // X-axis connection from MMA7260Q
#define Y_AXIS_INPUT    ADC_CHANNEL_PTB3    // Y-axis connection from MMA7260Q
#define Z_AXIS_INPUT    ADC_CHANNEL_PTB1    // Z-axis connection from MMA7260Q

// Macros to switch on/off photo relay bypasses for individual HBLEDs           LEFT   RIGHT
//                                                                           4 3 2 1   1 2 3 4
//                                                                           -------   -------
#define LED_L4(v)      if(v)  rPORTA &= ~bPTA0; else   rPORTA |=  bPTA0  //  4 * * *   * * * *
#define LED_L3(v)      if(v)  rPORTC &= ~bPTC4; else   rPORTC |=  bPTC4  //  * 3 * *   * * * *
#define LED_L2(v)      if(v)  rPORTC &= ~bPTC3; else   rPORTC |=  bPTC3  //  * * 2 *   * * * *
#define LED_L1(v)      if(v)  rPORTC &= ~bPTC2; else   rPORTC |=  bPTC2  //  * * * 1   * * * *                

#define LED_R1(v)      if(v)  rPORTA &= ~bPTA4; else   rPORTA |=  bPTA4  //  * * * *   1 * * *
#define LED_R2(v)      if(v)  rPORTA &= ~bPTA3; else   rPORTA |=  bPTA3  //  * * * *   * 2 * *
#define LED_R3(v)      if(v)  rPORTA &= ~bPTA2; else   rPORTA |=  bPTA2  //  * * * *   * * 3 *
#define LED_R4(v)      if(v)  rPORTA &= ~bPTA1; else   rPORTA |=  bPTA1  //  * * * *   * * * 4 

#define SwitchInit()      (SPI_Write(erHACTL, bH3EN|bH2EN|bH1EN))     // enable hall inputs
#define BRAKESwitch()     (SPI_RegValue(erHASTAT)&bH1F)               // H1 hall input
#define TURNRSwitch()     (SPI_RegValue(erHASTAT)&bH2F)               // H2 hall input
#define TURNLSwitch()     (SPI_RegValue(erHASTAT)&bH3F)               // H3 hall input


//________________G-Sensor Brake Lamp - Defined SW States______________________

#define STATE_MASK              0x3F        // --11 1111 - Masks off state bits
#define STATE_RESET             0           // --00 0000 - RESET state
#define STATE_CALIBRATE         1           // --00 0001 - Sensor Calibration State
#define STATE_STOPPEDALERT      2           // --00 0010 - Zero deceleration, but brake lever pulled
#define STATE_BRAKE_NO          5           // --00 0101 - Zero deceleration, no brake lever pulled
#define STATE_BRAKE_LOW         6           // --00 0110 - Level 1 deceleration (- - - *   * - - -)
#define STATE_BRAKE_MEDIUM      7           // --00 0111 - Level 2 deceleration (- - * *   * * - -)
#define STATE_BRAKE_HIGH        8           // --00 1000 - Level 3 deceleration (- * * *   * * * -)
#define STATE_BRAKE_EXTREME     9           // --00 1001 - Level 4 deceleration (* * * *   * * * *)
                                            //             Level 4 is considered an emergency braking situation

#define STATE_LEFT_MASK         0x40        // 0100 0000 - Mask for left side turn signal on
#define STATE_RIGHT_MASK        0x80        // 1000 0000 - Mask for right side turn signal on
                                            //           - Both masks can be set in case of flashing hazard lights
// --------------------------------------------------------
// timings

#define STATE_CHANGE_DELAY            1  // Delay (in ms) between state changes
#define EMERGENCY_STATE_HOLD_DELAY  500  // Delay (in ms) to hold in emergency brake state
//#define EMERGENCY_STATE_HOLD_DELAY  10  // Alternate shorter delay - useful for debugging

#define LIGHTSAMPLEINTERVAL         50      // every second?? 
//__________end___G-Sensor Brake Lamp - Defined SW States______________________



// --------------------------------------------------------
// Prototypes
// --------------------------------------------------------
void Init_IO(void);

void LEDInit(void);
void LEDEnable(UINT8 mask);


// --------------------------------------------------------
// Globals
// --------------------------------------------------------
UINT8  LedOn;

// --------------------------------------------------------
// -- MAIN --
// --------------------------------------------------------
void main(void)  {

  static UINT16 LedIntensity;      // static for debugging

  static UINT16 Decel_Value;              // static for debugging
  static UINT8  Current_State;            // static for debugging
  static UINT8  Next_State;               // static for debugging

  static INT16 State_Hold_Count;          // static for debugging - used to hold
  
//  int i;  //temp loop counter

  // -----  Config registers (write once) 
#if defined(MON08)  
  rCONFIG2 = bEXTCLKEN|bSSBPUENB|bESCIBDSRC|bTMBCLKSEL;     // allow ext. clock, disable default SS pull ups, ESCI clock = bus clock, TBM clock /128
  while(rICGCR!= (bCS|bECGON|bECGS))
    {
    rICGCR = (bCS|bECGON|bECGS);                            // turn to external clock source for debugging  
    }
#else
  rCONFIG2 = bSSBPUENB|bESCIBDSRC|bTMBCLKSEL;               // disable default SS pull ups, ESCI clock = bus clock, TBM clock /128
#endif

  rCONFIG1 = bLVI5OR3|bSTOP|bCOPD;                          // 5V operation, allow stop, disable Watchdog
  
    // Trimming MCU  
  if(trimICGTR!=0xFF)  {rICGTR = trimICGTR;}                // trim to f_base 307,2kHz
  rICGMR = 64;                                              // f_bus = f_base * 64 / 4 => 4.9152MHz internal Bus

  SPI_Init();                                               // Changed SS pin in "ChipSets.h"
  ADC_Init10();
  Init_IO();                                                // Set up port pins

  SPI_Read(erSYSCTL);
  SPI_Write(erSYSCTL, SPI_RegValue(erSYSCTL)|bPSON);        // set PSON Bit

  // Init current limitation value
  SPI_Write(erHBCTL, bOFC_EN|bCLS2|      bCLS0);            // 370mA, Offset chopping enabled

  SPI_Read(erPOUT);
  SPI_Write(erPOUT , SPI_RegValue(erPOUT)|bHVDDON|bHS_ON|bCSEN1|bCSSEL1|bCSSEL0);   
                                                            // HVDD on for 3.3V supply
                                                            // HS on for license LEDs
                                                            // CSEN1 - Current Source ON - PhotoResistor
                                                            // CSSEL1:0 = 11  670uA
  
  (void)l_sys_init();  l_ifc_init_sci08();  (void)l_ifc_connect_sci08();
  
  TimerAInit();                                             // FGEN 25kHz 50%duty   
  TimerAEnable();  

  TimerBInit();                                             // LED Intensity control

  TimerInit();                                              // General SW Timer 
  TimerEnable();

  cli();                                                    // enable interrupt


  FilterInit(*ADC_Value10(Z_AXIS_INPUT));                   // Initialize LP filter variable

      

  Current_State     = STATE_RESET;
  Next_State        = STATE_RESET;                       
  FlasherInit();                                            // Initialize Flasher State
  
  State_Hold_Count  = 0;                                    // Number of times to stay in emergency state
  
  
// --------------------------------------------------------
// -- MAIN LOOP --
// --------------------------------------------------------
  for EVER  {      
 
    // -----------------------------
    // LIN debug interface
    // -----------------------------
    if(l_u8_tst_SysCmd())  {
      l_u8_clr_SysCmd();
      if(SysMasterRequest_Sleep())  {
        //EnterStopMode();
        //EnterSleepMode();
      }else{
        ldd_Handler();
      }
    }

    // -----------------------------
    // Ambient light compensation
    // -----------------------------
    if(!(TimerRunning(TIMER4)))  {
      TimerValue[TIMER4IDX] = (UINT16)(LIGHTSAMPLEINTERVAL/TIMERBASEINTERVALLMS); 
      TimerStart(TIMER4);                                          

      SPI_Write(erADOUT, bSS2|bSS1|bSS0);                     // Send SPI command to display PA1 on ADC terminal
                                                              // 5V is low light, 0V is for bright light (inversely proportional)
      LedIntensity = ((1054-(*ADC_Value10(ADC_CHANNEL_PTB0)))/10) ; // Read ADC value off that pin
                                                                  // (1054-reading) ensures minimum of 3% brightness
      sei();  TimerBSetDuty((UINT8)LedIntensity);  cli();            // not atomic!!!
    }

    // ----------------------------------------------
    // - Determine deceleration rate & set correct HBLEDs

    // -------------------------------
    // take sample and low pass filter 
    // -------------------------------
    if(!(TimerRunning(TIMER1)))  {
      TimerValue[TIMER1IDX] = SENSORSAMPLEINTERVAL;            
      TimerStart(TIMER1);
      Decel_Value = Filter(*ADC_Value10(Z_AXIS_INPUT));     // new reading, calc new value
    }

    // -------------------------------
    // state machine for brake level
    // -------------------------------
    if(!(TimerRunning(TIMER2)))  {
      TimerValue[TIMER2IDX] = (UINT16)(STATE_CHANGE_DELAY/TIMERBASEINTERVALLMS); 
      TimerStart(TIMER2);                                          
      
      SPI_Read(erHASTAT);               // Read LEFT, BRAKE, RIGHT inputs
             
      // --------------------- Decide next state ------------------------
      //
      // different up/down value => hysteresis -  depending on actual Current_State
      // statemachine changes are only incremental e.g. no -> low -> medium -> high
      // not no -> high 

      switch(Current_State&STATE_MASK)  {                   // only states without left/right

        case STATE_RESET:
          Next_State = STATE_BRAKE_LOW;  
          break;

        case STATE_CALIBRATE:                               // Calibration of G-Sensor
          break;

        case STATE_STOPPEDALERT:
          if(!BRAKESwitch())                    Next_State = STATE_BRAKE_LOW;  
          break;

        case STATE_BRAKE_NO:
          if(Decel_Value<DECEL_LEVEL_20)        Next_State = STATE_BRAKE_LOW;
          if(BRAKESwitch())                     Next_State = STATE_STOPPEDALERT;  // If Brake light on - go to stopped alert
          break;

        case STATE_BRAKE_LOW:                   
          if(Decel_Value<DECEL_LEVEL_40)        Next_State = STATE_BRAKE_MEDIUM;                   
          else if(Decel_Value>DECEL_LEVEL_10)   Next_State = STATE_BRAKE_NO;
          break;

        case STATE_BRAKE_MEDIUM:
          if(Decel_Value<DECEL_LEVEL_60)        Next_State = STATE_BRAKE_HIGH;                   
          else if(Decel_Value>DECEL_LEVEL_30)   Next_State = STATE_BRAKE_LOW;
          break;

        case STATE_BRAKE_HIGH:
          if(Decel_Value<DECEL_LEVEL_80)        Next_State = STATE_BRAKE_EXTREME;                   
          else if(Decel_Value>DECEL_LEVEL_50)   Next_State = STATE_BRAKE_MEDIUM;
          break;

        case STATE_BRAKE_EXTREME:
         if(State_Hold_Count<EMERGENCY_STATE_HOLD_DELAY)  { // Latch in STATE_BRAKE_EXTREME
           State_Hold_Count++;                              // This will hold this brake level for defined time
         }else{
           if(Decel_Value>DECEL_LEVEL_70)   {
             Next_State = STATE_BRAKE_HIGH;
             State_Hold_Count=0;
           }
         }
         break;
      }

      // now check sitches and add to into state (Bitmask)
      if(TURNLSwitch())                         Next_State |= STATE_LEFT_MASK;
      else if(TURNRSwitch())                    Next_State |= STATE_RIGHT_MASK;
      else                                      Next_State &= STATE_MASK;

      // --------------------- Process current state --------------------
      // - Sets the HBLED output based upon the current state
      //


      if(Current_State==Next_State)  {
        // only act if state change
      }else{
  
        Current_State = Next_State;           // Assign Next State


        switch(Current_State&STATE_MASK)  {                   // Decision only based on states without left/right
                                                              // LEFT/RIGHT Bit masks handled in each state processing

          case STATE_CALIBRATE:                               // Calibration of G-Sensor
              // Assume 1-g gravity and calculate the new Z-axis
              // It will be a function of Z and X axes if mounted horizontally (1-g if mounted perfectly vertically)
              // It will be a function of Z and Y axes if mounted vertically
              
              // Once determined, the new orientation should be saved in NVM
              //    Note - Will have to load into RAM and write into Flash

            break;
          
          case STATE_STOPPEDALERT:
            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xF3);                                    // Enable HBLEDs (* * * *   - - * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0xCF);                                    // Enable HBLEDs (* * - -   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else{
              LEDEnable(0xC3);                                    // Enable HBLEDs (* * - -   - - * *)
              LedOn = LEDON_XXoo_ooXX;                            // Switch on HBLED channels 
            }
            break;

          case STATE_BRAKE_NO:

            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xF0);                                    // Enable HBLEDs (* * * *   - - - -)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0x0F);                                    // Enable HBLEDs (- - - -   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels  
            }else{
              LEDEnable(0x00);                                    // Enable HBLEDs (- - - -   - - - -)
              LedOn = LEDON_oooo_oooo;                            // Switch on HBLED channels 
            }
            break;

          case STATE_BRAKE_LOW:
            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xF8);                                    // Enable HBLEDs (* * * *   * - - -)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0x1F);                                    // Enable HBLEDs (- - - *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels  
            }else{
              LEDEnable(0x18);                                    // Enable HBLEDs (- - - *   * - - -)
              LedOn = LEDON_oooX_Xooo;                            // Switch on HBLED channels 
            }
            break;

          case STATE_BRAKE_MEDIUM:
            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xFC);                                    // Enable HBLEDs (* * * *   * * - -)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels // Switch on all HBLEDs 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0x3F);                                    // Enable HBLEDs (- - * *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels // Switch on all HBLEDs 
            }else{
              LEDEnable(0x3C);                                    // Enable HBLEDs (- - * *   * * - -)
              LedOn = LEDON_ooXX_XXoo;                            // Switch on HBLED channels // Switch off HBLEDs
            }
            break;

          case STATE_BRAKE_HIGH:
            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xFE);                                    // Enable HBLEDs (* * * *   * * * -)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0x7F);                                    // Enable HBLEDs (- * * *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else{
              LEDEnable(0x7E);                                    // Enable HBLEDs (- * * *   * * * -)
              LedOn = LEDON_oXXX_XXXo;                            // Switch on HBLED channels 
            }
            break;

          case STATE_BRAKE_EXTREME:
            if(Current_State&STATE_LEFT_MASK)  {
              LEDEnable(0xFF);                                    // Enable HBLEDs (* * * *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else if(Current_State&STATE_RIGHT_MASK)  {
              LEDEnable(0xFF);                                    // Enable HBLEDs (* * * *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }else{
              LEDEnable(0xFF);                                    // Enable HBLEDs (* * * *   * * * *)
              LedOn = LEDON_XXXX_XXXX;                            // Switch on HBLED channels 
            }
   
    /*        if(FlasherStatus()==FLASHEROFF)   // 
              {
              if(!(TimerRunning(TIMER3)))  
                {                            // OFF phase of flasher
                TimerValue[TIMER3IDX] = (UINT16)(FLASHER_COUNT/TIMERBASEINTERVALLMS);  // ms
                TimerStart(TIMER3);
                FlasherBrake();
                LedOn = BRAKE_LED_OUT_4;      // All on
                }
              }
              else
              {
              if(!(TimerRunning(TIMER3)))  
                {
                TimerValue[TIMER3IDX] = (UINT16)(FLASHER_COUNT/TIMERBASEINTERVALLMS);  // ms
                TimerStart(TIMER3);
                FlasherTurnOff();           
                LedOn = BRAKE_LED_OUT_0;            // Switch off HBLEDs
                }
              }
    */          

            break;
        }
      }
    }     // Timer2 - State Change Timer
  }  // for(EVER);
}


// --------------------------------------------------------
void Init_IO(void)  {

  // Init Photo Relay Outputs - quick test
  // Set to outputs - low (disables HBLED bypassing)
  LEDInit();
  
  // Init Photo Relay Outputs 
  // Init License Plate LEDs
  // Init G-Sensor Sensitivity Selection Outputs
  // Init PhotoResistor Output

  SwitchInit();  // Init BRAKE, LEFT, RIGHT signal monitor inputs (Hall effect inputs)
}

  
  
// --------------------------------------------------------
// als LED bypass GPIO are initialized as outputs OFF
// Init Photo Relay Outputs - quick test
// Set to outputs - low (disables HBLED bypassing)
void LEDInit(void)  {

  LEDEnable(0x00);                                          // all off
  rDDRA  |=  0x1F;                                          // 1 -> output
  rDDRC  |=  0x1C;                                          // 1 -> output
  rDDRC  &= ~bMCLKEN;
}

// --------------------------------------------------------
// LED mask specifies the LEDs from left to right 
// 
// bit   7  6  5  4       3  2  1  0
//       L4 L3 L2 L1      R1 R2 R3 R4
void LEDEnable(UINT8 mask)  {
UINT16 i;

  LED_L4(mask&0x80);
  LED_L3(mask&0x40);
  LED_L2(mask&0x20);
  LED_L1(mask&0x10);
  LED_R1(mask&0x08);
  LED_R2(mask&0x04);
  LED_R3(mask&0x02);
  LED_R4(mask&0x01);
  
  for(i=0;i<1000;i++){}                               // Delay to allow photo relays to kick in
  
}


// --------------------------------------------------------
// Interrupt Service Routines
// --------------------------------------------------------
#pragma TRAP_PROC
void isrHWIRQ(void)  {                                      // HWIRQ, for Hall and WakeUP

}


// --------------------------------------------------------
#pragma TRAP_PROC
void isrESCIReceive(void) {                                 // SCI Rx interrupt handler
  
  l_ifc_rx_sci08();
}
// --------------------------------------------------------
#pragma TRAP_PROC
void isrESCIError(void)  {                                  // SCI Error interrupt handler
  
  l_ifc_rx_sci08();
}
// --------------------------------------------------------
l_irqmask l_sys_irq_disable(void)  {                        // Disable interrupts
  
  asm tpa;
  asm sei;
}
// --------------------------------------------------------
void l_sys_irq_restore(l_irqmask previous)  {               // Restore interrupts

  asm tap;
}






