#define LINPROT_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved                             
*                                                       
* Filename:     linapi.c                
*                                                       
* Revision:     v1.1                                    
*                                                       
* Functions:    LIN API realization
*                                                       
* Description:  Some Freescale and LIN API functions realization
*                                                       
* Notes:        Realization of another API functions -- in another modules
*                                                       
******************************************************************************/

#include <linbase.h>

#if defined(LINAPI_1_0)

/***************************************************************************
 * Function :   l_ifc_ioctl_sci0
 *
 * Description: Protocol-specific functionality.
 * 
 * Returns:     depends on the operation
 *
 * Notes:       Operations:
 *  l_op_getrxerr   -- Provide the Receive Errors counter of the LIN Driver 
 *  l_op_gettxerr   -- Provide the Transmit Errors counter of the LIN Driver 
 *  l_op_clrrxerr   -- Clear the Receive Errors counter of the LIN Driver 
 *  l_op_clrtxerr   -- Clear the Transmit Errors counter of the LIN Driver 
 *  l_op_wakeup     -- Issues the LIN bus wakeup frame transmission 
 *  l_op_getidle    -- Check No-Bus-Activity condition 
 *  l_op_idleclock  -- Update No-Bus-Activity condition counter by 1 
 *  l_op_getsyncpd  -- Provide the latest synchronized prescaler divisor 
 *
 **************************************************************************/
void l_ifc_ioctl_sci0(i_ioctl_op op, void *pv)
{
    LIN_BYTE        intMask;

    LIN_DBG_SET_PORT_7;

    intMask = LIN_DisableInt();         /* Disable interrupts */

    /* OPTIM:
        switch  -- for COSMIC08 (5 bytes less);
        if      -- for CW08 */

    if (op == l_op_getrxerr)
    {
        *( (LINErrCounterType*) pv) = LIN_ErrRxCounter;
    }

    else if (op == l_op_gettxerr)
    {
        *( (LINErrCounterType*) pv) = LIN_ErrTxCounter;
    }

    else if (op == l_op_clrrxerr)
    {
        LIN_ErrRxCounter = 0;
    }

    else if (op == l_op_clrtxerr)
    {
        LIN_ErrTxCounter = 0;
    }

    else if (op == l_op_wakeup)
    {
        /* possible only if we are connected and doing nothing */
        if ( ( (LIN_StateFlags & LIN_FLAG_DISCONNECT)  == 0 ) && 
             ( (LIN_StateFlags & LIN_FLAG_IGNORE) != 0 ) )
        {
            LIN_StateFlags = LIN_FLAG_WAKEUP_TIMEOUT;

            /* send Wakeup message */    
            LIN_SCISendWakeup();

            /* Slave -- wait for Rx interrupt */

            /* timeout is not required */
            
            *( (l_bool*) pv) = 0;    /* success */
        }
        else
        {
            *( (l_bool*) pv) = ~0;   /* pending or SCI disconnected*/
        }
    }

    else if (op == l_op_getidle)
    {
        if ( LIN_IdleTimeout == 0 )
        {
            *( (l_bool*) pv) = ~0;   /* IDLE */
        }
        else
        {
            *( (l_bool*) pv) = 0;    /* no IDLE */
        }
    }

    else if (op == l_op_idleclock)
    {
        if ( LIN_IdleTimeout != 0  )
        {
            --LIN_IdleTimeout;
        }
    }

    else if (op == l_op_getsyncpd)
    {
        *( (LINSyncPDType*) pv) = LIN_SynchPrescalerDivisor;
    }

    LIN_EnableInt(intMask);             /* Enable interrupts */    

    LIN_DBG_CLR_PORT_7;
}


#else /* defined(LINAPI_1_0) */


/***************************************************************************
 * Function :   LIN_Wakeup
 *
 * Description: Send wakeup frame
 * 
 * Returns:     LIN_OK                No error, service call has succeeded.                   
 *              LIN_REQ_PENDING       Another LIN bus frame is currently 
 *                                    processed. Or wakeup process is running
 *              LIN_INVALID_MODE      The driver is in the sleep mode during 
 *                                    message request. 
 *
 * Notes:       
 *
 **************************************************************************/
LINStatusType          LIN_Wakeup( void )
{
    LIN_BYTE        intMask;

    if (LIN_StateFlags & LIN_FLAG_SLEEP )
    {
        /* we don't need interrupt disabling becouse in SLEEP mode driver doing nothing */
        return LIN_INVALID_MODE;                
    }
    else if (LIN_StateFlags & LIN_FLAG_IGNORE)
    {
        LIN_DBG_SET_PORT_7;
        intMask = LIN_DisableInt();         /* Disable interrupts */
    
        LIN_StateFlags = LIN_FLAG_WAKEUP_TIMEOUT;
    
        /* send Wakeup message */    
        LIN_SCISendWakeup();

        LIN_EnableInt(intMask);             /* Enable interrupts */    
        LIN_DBG_CLR_PORT_7;

        return LIN_OK;
    }
    else
    {
        return LIN_REQ_PENDING;
    }
}

/***************************************************************************
 * Function :   LIN_GotoRun
 *
 * Description: Change the current driver state from SLEEP to RUN
 * 
 * Returns:     none
 *
 * Notes:       1. API Service call
 *              3. SLAVE:  Set Idle timeout and go to IGNORE state
 *              4. If in RUN already -> no any actions
 *              5. DisableInterrupt don't needed becouse in SLEEP mode
 *                 driver do nothing
 *
 **************************************************************************/
void LIN_GotoRun( void )
{
    if ( LIN_StateFlags & LIN_FLAG_SLEEP )                    
    {
        /* we don't need interrupt disabling becouse in SLEEP mode driver doing nothing */
        
        LIN_SetIdleTimeout();                   /* Set idle timeout again */

        LIN_StateFlags = LIN_FLAG_IGNORE;

        LIN_SCPSC = LIN_CfgConst.LIN_SciPreScaler;
    }
}

/***************************************************************************
 * Function :   LIN_DriverStatus
 *
 * Description: Return actual status of the LIN Driver.
 * 
 * Returns:     LIN_STATUS_RUN            Driver state is run 
 *                                        or sleep                
 *              LIN_STATUS_IDLE           No-Bus_activity timeout 
 *                                        is expired  
 *              LIN_STATUS_PENDING        LIN bus frame is pending 
 *                                        only for MASTER
 *
 * Notes:       Freescale API Service call
 *              MASTER and SLAVE
 *              NB: !!!  LIN_STATUS_RUN     = ~LIN_FLAG_SLEEP     
 *                       LIN_STATUS_PENDING = ~LIN_FLAG_IGNORE
 *                       LIN_STATUS_IDLE    =  (LIN_IdleTimeout == 0)
 *
 *              Interrupts are not disabled here, although LIN_IdleTimeout
 *              is 16-bit variable.
 *              
 **************************************************************************/
LINDriverStatusType    LIN_DriverStatus( void )
{
    LINDriverStatusType retVal;

#if defined(SLAVE)
    retVal = ( LIN_StateFlags ^ LIN_FLAG_SLEEP ) & ( LIN_FLAG_SLEEP );
#endif /* defined(SLAVE) */

    if ( LIN_IdleTimeout == 0 )
    {
        retVal |=  LIN_STATUS_IDLE;
    } 

    return retVal;
}

/***************************************************************************
 * Function :   LIN_GetSyncPD
 *
 * Description: Return latest synchronized prescaler divisor
 *
 * Returns:     LIN_SynchPrescalerDivisor
 *
 * Notes:       API call 
 **************************************************************************/
LINSyncPDType LIN_GetSyncPD( void )
{
    return LIN_SynchPrescalerDivisor;       
}

#endif /* defined(LINAPI_1_0) */
