//////////////////////////////////////////////////////////////////
// This code demonstrates the minimal amount of configurations necessary to successfully generate SOFs.
// This example uses an MPC8272 mounted on an MPC8272ADS board.
// Board Configuration:
// 		SW5: 10100100
//		SW2: 0010
//		Input Clock:  100MHz
//		JP8:  Host
// 10/5/05 Randy Ryder
//////////////////////////////////////////////////////////////////


#include <stdio.h>
#include "mpc8272.h"

uword GetIMMR(void);
ushort reverse8(ushort input, int len);
uchar crc5(ushort frame);
ushort sofgen (ushort frame);

ushort reverse8(ushort input, int len)        //Reverse bit order
{
    int i;
    ushort the_reverse = 0;
    ushort the_bit;

    for(i=0;i<len;i++){
        the_bit = ((input & (1<<i)) != 0);
        the_reverse |= the_bit << ((len-1) - i);
    }

    return the_reverse;
}

uchar crc5(ushort frame)                        //CRC5 routine
{
    ushort  hold = 0x1f;
    ushort tmpFrameNum, frameMask = 0x0400;
    int i;
    tmpFrameNum = reverse8(frame, 11);

    for (i = 0; i < 11 ; i++)
    {
        hold = (ushort)(hold << 1);
        if (!!(tmpFrameNum & frameMask) != !!(hold & 0x20))
            hold ^= 0x5;

        frameMask = (ushort)(frameMask >> 1);
    }

    return (uchar)reverse8((ushort)(hold ^ 0x1f), 5);
}


ushort sofgen (ushort frame)          //Generates SOF frame & CRC5
{
    return (ushort)((frame & 0x07ff) | (ushort)(crc5(frame) << 11));
}


void main()
{
	typedef struct 
	{
		ushort status;
		ushort length;
		uword  point;
	} bd;
	
	uword  *bcsr3;
	t_endpt_Pram *eppram;
	bd *txbd;
	ushort frame;
	int i=0;
	
/*------------------------------------------*/
/* Declare pointer to Internal Memory Map	*/
/*------------------------------------------*/
	
	EPPC *immr = (EPPC *) (GetIMMR() & 0xffff0000);
	t_USB_Pram *usbpram = (t_USB_Pram *) immr->pram.standard.usb;
	bcsr3 = (uword *) ((immr->memc_regs[1].br & 0xFFFF0000) + 0xC);


/*************************************/
	/*  USB PARAMETER RAM INIT */

	usbpram->ep1ptr  = 0x8b40;
	usbpram->rstate  = 0x00000000;
	usbpram->frame_n = 0x0000;
	
/*************************************/
	
/*  ENDPOINT 0 PARAMETER BLOCK INIT */

	eppram = (t_endpt_Pram *) ((uword)usbpram->ep1ptr + (uword)immr);
	eppram->tbase  = 0x8b90;
	eppram->rfcr   = 0x18;
	eppram->tfcr   = 0x18;
	eppram->tstate = 0x00000000;
	eppram->himmr  = 0x0470;			// IMM is 0x0470 for this example

/**************************************/
		/*  TxBD Init  */
   
   	txbd = (bd *) ((uword)immr + (uword)eppram->tbase);
    txbd->status = 0x2a00;				// This is ex. of minimum BD init for EP0
    txbd->length = 0x0000;
    txbd->point  = 0x00000000;
    
/**************************************/

	/*  PARALLEL PORT CONFIG */
	immr->io_regs[2].ppar |= 0x00300880;
	immr->io_regs[2].pdir |= 0x00000800;
	immr->io_regs[2].psor |= 0x00000000;
	immr->io_regs[3].pdir |= 0x00000180;
	immr->io_regs[3].psor |= 0x00000000;
	immr->io_regs[3].ppar |= 0x000001c0;
	
/***************************************/

	/*  BOARD/DEVICE CONFIGURATION */
	
	*bcsr3  &= 0x01111111;				// put ADS in host mode, Vbus off, Full-speed

	immr->cpm_mux_cmxscr |= 0x00000700;
	immr->cpm_rccr 		 |= 0x00080040;
	immr->usep1			  = 0x0020;		// EP0 acts as the host    */


 /***************************************/  
  	
	immr->usmod |= 0x0b;                                

	while (1) 
	{

		for (frame = 0x0000; frame < 0x801;) // Loop between min & max frame count
		{
			if (immr->usber & 0x0008)		// Trigger on SOF transmission event to create frame & crc
			{
				usbpram->frame_n = sofgen(frame);
				frame++;
				immr->usber |= 0x0008;		// Clear event register
			}
		}
 	}
}

/*-----------------------------------------------------------------------------
*
* FUNCTION NAME: GetIMMR
*
* DESCRIPTION:
*
*     Returns the current value in IMMR register. The IMMR register must
*     be accessed in this manner because it is a special purpose register.
*
* EXTERNAL EFFECTS: None
*
* PARAMETERS: none
*
* RETURNS: IMMR value 
*-----------------------------------------------------------------------------*/
uword GetIMMR(void)
{
   register unsigned long gpr = 0;
   asm( mfspr gpr,311);
   return gpr;
}