/**************************************************************************/
/* FILE NAME: tpu_comm.c                      COPYRIGHT (c) MOTOROLA 2002 */
/*                                                All Rights Reserved     */
/*                                                                        */
/* DESCRIPTION: This file contains the TPU COMM interface functions.      */
/* These functions allow the user to control control TPU channels running */
/* the COMM function.                                                     */
/* The functions included are as follows:                                 */
/*                                                                        */
/* tpu_comm_init_sensored                                                 */
/* tpu_comm_force_state                                                   */
/* tpu_comm_init_sensorless_match                                         */
/* tpu_comm_write_upper                                                   */
/* tpu_comm_write_lower                                                   */
/* tpu_comm_write_offset                                                  */
/* tpu_comm_get_state_no                                                  */
/* tpu_comm_start_update                                                  */
/*                                                                        */
/*========================================================================*/
/* HISTORY           ORIGINAL AUTHOR: Ken Terry                           */
/* REV      AUTHOR      DATE       DESCRIPTION OF CHANGE                  */
/* ---   -----------  ---------    ---------------------                  */
/* 1.0   K Terry      28/Aug/02    Initial version of function.           */
/**************************************************************************/
#include "tpu_comm.h"
#include "mpc500_util.h"

/*******************************************************************************
FUNCTION      : tpu_comm_init_sensored
PURPOSE       : Initialises the TPU to run the COMM function in sensored mode
INPUTS NOTES  : This function has 4 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                           	channel.

                 no_of_pins 	- This parameter determines the number of TPU channels
                 	    	required to generate the commutation signal. The maximum
                 	    	number of channels that can be used is 8.
                 	   
                 update_period 	- This parameter is used to schedule an update of all channels
                 	    	when a link is received by the master channel so that the
                 	    	new pin states occur simultaneously on all channels.
                 	   
                 comm_state[] 	- This is an array of values which are written to the parameter
                	    	RAM. Each comm_state value defines an output state in the
                	    	commutation sequence. No checking is done to determine if the
                	    	parameter RAM of channels beyond those used by the COMM function
                	    	are written. The user must ensure that the state_table fits into
                	    	the available channel parameter RAM.
                	    
                 no_of_states 	- This parameter defines the number of comm_state[] elements
                	    	to be written tot he parameter RAM 

RETURNS NOTES : none

*******************************************************************************/
void tpu_comm_init_sensored (struct TPU3_tag *tpu, UINT8 channel, INT16 no_of_pins,   
                    INT16 update_period, INT16 comm_states[], INT16 no_of_states)
{

	UINT8	chan_index;
	UINT8	act_channel;
	UINT8	state_index;
	INT16	no_pins_pram;

/* Disable all channels required for the COMM function */
	for (chan_index = 0; chan_index < no_of_pins; chan_index++)
	{
		act_channel = (channel+chan_index) & 0xf;
		tpu_disable(tpu, act_channel);
	}

/* Select COMM function on master COMM channel */
	tpu_func( tpu, channel, TPU_FUNCTION_COMM);


/* Select Sensored Mode for Host Sequence Register */	
	tpu_hsq(tpu, channel, COMM_SENSORED);

/* Write NO_OF_PINS and UPDATE_PERIOD to parameter RAM */
	no_pins_pram = (VUINT16)(no_of_pins<<8 & 0x0f00);
	tpu->PARM.R[channel][COMM_NO_OF_PINS] = no_pins_pram;
	tpu->PARM.R[channel][COMM_UPDATE_PERIOD] = update_period;
	
/* Write state values to the Parameter RAM */
	
	state_index = 0;	
	while(state_index < no_of_states)
	{
		tpu->PARM.R[(channel + 1 + state_index/8)][state_index - (state_index/8)*8] = comm_states[state_index];
		state_index++;
	}

}


/*******************************************************************************
FUNCTION      : tpu_comm_force_state
PURPOSE       : This function updates the STATE_NO parameter in the master COMM channel
		and then issues an hsr request to force the COMM channels to the required
		state
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.

                 state_no 	- This parameter selects the required STATEx value to be output
                 		on the COMM channel pins
                 	   
RETURNS NOTES : none

*******************************************************************************/


void tpu_comm_force_state (struct TPU3_tag *tpu, UINT8 channel, INT16 state_no)
{

INT16	no_of_states_pram;


	no_of_states_pram = tpu->PARM.R[channel][COMM_NO_OF_STATES];
	no_of_states_pram = (no_of_states_pram & 0xff00) | (state_no & 0x00ff);
	tpu->PARM.R[channel][COMM_STATE_NO] = no_of_states_pram;
    	tpu_hsr(tpu, channel, COMM_FORCE_STATE);
}


/*******************************************************************************
FUNCTION      : tpu_comm_init_sensorless_match
PURPOSE       : Initialises the TPU to run the COMM function in sensorless match update mode
INPUTS NOTES  : This function has 5 parameters:

                *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                channel 	- This is the channel number of the master COMM
                           	channel.

                no_of_pins 	- This parameter determines the number of TPU channels
                 	    	required to generate the commutation signal. The maximum
                 	    	number of channels that can be used is 8.

		counter-addr	- This parameter contains the address in parameter RAM of
				of the angualr position counter - POSITION_COUNT - used for
				state tests on sensorless mode.                 	   

                update_period 	- This parameter is used to schedule an update of all channels
                 	    	when a link is received by the master channel so that the
                 	    	new pin states occur simultaneously on all channels.

                comm_state[] 	- This is an array of values which are written to the parameter
                	    	RAM. Each comm_state value defines an output state in the
                	    	commutation sequence. No checking is done to determine if the
                	    	parameter RAM of channels beyond those used by the COMM function
                	    	are written. The user must ensure that the state_table fits into
                	    	the available channel parameter RAM.
                	    
                no_of_states 	- This parameter defines the number of comm_state[] elements
                	    	to be written tot he parameter RAM 

RETURNS NOTES : none

*******************************************************************************/
void tpu_comm_init_sensorless_match (struct TPU3_tag *tpu, UINT8 channel, INT16 no_of_pins,   
                    INT16 counter_addr, INT16 update_period, INT16 comm_states[],
                    INT16 no_of_states)
{

	UINT8	chan_index;
	UINT8	act_channel;
	UINT8	state_index;
	INT16	no_pins_pram;
	

/* Disable all channels required for the COMM function */
	for (chan_index = 0; chan_index < no_of_pins; chan_index++)
	{
		act_channel = (channel+chan_index) & 0xf;
		tpu_disable(tpu, act_channel);
	}

/* Select COMM function on master COMM channel */
	tpu_func( tpu, channel, TPU_FUNCTION_COMM);


/* Select Sensorless Match Mode for Host Sequence Register */	
	tpu_hsq(tpu, channel, COMM_SENSORLESS_MATCH);

/* Write NO_OF_PINS and COUNTER_ADDR to parameter RAM */
	no_pins_pram = (VUINT16)((no_of_pins<<8 & 0x0f00)|(counter_addr & 0x00ff));
	tpu->PARM.R[channel][COMM_NO_OF_PINS] = no_pins_pram;
	
/* Write UPDATE_PERIOD to parameter RAM */
	tpu->PARM.R[channel][COMM_UPDATE_PERIOD] = update_period;
	
/* Write state values to the parameter RAM */
	
	state_index = 0;	
	while(state_index < no_of_states)
	{
		tpu->PARM.R[(channel + 1 + state_index/8)][state_index - (state_index/8)*8] = comm_states[state_index];
		state_index++;
	}

/* Write NO_OF_STATES to parameter RAM */
	tpu->PARM.R[channel][COMM_NO_OF_STATES] = (no_of_states<<8) & 0xff00;

}



/*******************************************************************************
FUNCTION      : tpu_comm_write_upper
PURPOSE       : This function writes the upper angular boundary of the current commutation
		state to the parameter RAM.
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.

                 upper 	- 	This value is the upper angular boundary of the current
                 		commutation state
                 	   
RETURNS NOTES : none

*******************************************************************************/

void tpu_comm_write_upper (struct TPU3_tag *tpu, UINT8 channel, INT16 upper)
{
	/* Write UPPER to parameter RAM */	
	tpu->PARM.R[channel][COMM_UPPER] = upper;
}



/*******************************************************************************
FUNCTION      : tpu_comm_write_lower
PURPOSE       : This function writes the lower angular boundary of the current commutation
		state to the parameter RAM.
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.

                 lower 	- 	This value is the lower angular boundary of the current
                 		commutation state
                 	   
RETURNS NOTES : none

*******************************************************************************/

void tpu_comm_write_lower (struct TPU3_tag *tpu, UINT8 channel, INT16 lower)
{
	/* Write LOWER to parameter RAM */
	tpu->PARM.R[channel][COMM_LOWER] = lower;
}



/*******************************************************************************
FUNCTION      : tpu_comm_write_offset
PURPOSE       : This function writes the offset value to the parameter RAM to advance or
		retard the state switching angle when COMM is used in sensorless mode.
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.

                 offset 	- This value advances or retards the the state switching
                 		angles when COMM is used in sensorless mode.
                 	   
RETURNS NOTES : none

*******************************************************************************/

void tpu_comm_write_offset (struct TPU3_tag *tpu, UINT8 channel, INT16 offset)
{
	/* Write OFFSET to parameter RAM */	
	tpu->PARM.R[channel][COMM_OFFSET] = offset;
}



/*******************************************************************************
FUNCTION      : tpu_comm_get_state_no
PURPOSE       : This function returns the value of the current state_no held in the 
		parameter RAM of COMM function master channel
INPUTS NOTES  : This function has 2 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.


RETURNS NOTES : none

*******************************************************************************/

INT16	 tpu_comm_get_state_no (struct TPU3_tag *tpu, UINT8 channel)
{
	INT16 state_no;
	state_no = (tpu->PARM.R[channel][COMM_STATE_NO]) & 0x00ff;
	return(state_no);	
}

/*******************************************************************************
FUNCTION      : tpu_comm_start_update
PURPOSE       : This function issues an HSR to the COMM master channel to start
		the periodic update matches.
INPUTS NOTES  : This function has 2 parameters:

                 *tpu    	- This is a pointer to the TPU3 module to use. It is of
                           	type TPU3_tag which is defined in m_tpu3.h

                 channel 	- This is the channel number of the master COMM
                          	channel.


RETURNS NOTES : none

*******************************************************************************/

void tpu_comm_start_update (struct TPU3_tag *tpu, UINT8 channel)
{
	/* issue HSR %11 to start periodic state update matches*/	

	tpu_hsr(tpu, channel, COMM_START_UPDATE);
}



/*********************************************************************
 *
 * Copyright:
 *	MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS.
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support
 *  of this software
 ********************************************************************/