/*
 * Copyright 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "memory.h"

static flash_config_t flashInstance;
static lpspi_nor_flash_handle_t spiFlashInstance;

int memory_init(void)
{
    status_t ret;
    lpspi_nor_flash_config_t flashConfig = {
        .lpspiBase  = SPI_FLASH_LPSPI,
        .dmaBase    = SPI_FLASH_DMA,
        .dmamuxBase = SPI_FLASH_DMAMUX,
        .baudRate   = SPI_FLASH_BAUDRATE,
        .txChannel  = SPI_FLASH_DMA_TX_CHANNEL,
        .rxChannel  = SPI_FLASH_DMA_RX_CHANNEL,
        .txSource   = SPI_FLASH_DMA_TX_SOURCE,
        .rxSource   = SPI_FLASH_DMA_RX_SOURCE
    };

	memset(&spiFlashInstance, 0, sizeof(lpspi_nor_flash_handle_t));
    LPSPI_Nor_Flash_Init(&spiFlashInstance, &flashConfig, SPI_FLASH_CLK_FREQ);

	memset(&flashInstance, 0, sizeof(flash_config_t));
    ret = FLASH_Init(&flashInstance);

    if (ret == kStatus_Success)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

int memory_erase(uint32_t addr, uint32_t len)
{
    status_t ret;
    if(addr >= SPI_FLASH_ADDRESS_START && addr + len <= SPI_FLASH_ADDRESS_END)
    {
        ret = LPSPI_Nor_Flash_Erase(&spiFlashInstance, addr - SPI_FLASH_ADDRESS_START, len);
    }
    else
    {
        ret = FLASH_Erase(&flashInstance, addr, len, kFLASH_ApiEraseKey);
    }

    if (ret == kStatus_Success)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

int memory_write(uint32_t start_addr, const uint8_t *buf, uint32_t len)
{
    status_t ret;
    if(start_addr >= SPI_FLASH_ADDRESS_START && start_addr + len <= SPI_FLASH_ADDRESS_END)
    {
        ret = LPSPI_Nor_Flash_Program(&spiFlashInstance, start_addr - SPI_FLASH_ADDRESS_START, buf, len);
    }
    else
    {
        ret = FLASH_Program(&flashInstance, start_addr, (uint8_t *)buf, len);
    }

    if (ret == kStatus_Success)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}


int memory_read(uint32_t addr, uint8_t *buf, uint32_t len)
{
    if(addr >= SPI_FLASH_ADDRESS_START && addr + len <= SPI_FLASH_ADDRESS_END)
    {
        LPSPI_Nor_Flash_Read(&spiFlashInstance, addr - SPI_FLASH_ADDRESS_START, buf, len);
    }
    else
    {
        memcpy(buf, (uint8_t *)addr, len);
    }

    return 0;
}
