/*
 * Copyright 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_lpspi_nor_flash.h"

static void masterCallback(LPSPI_Type *base, lpspi_master_edma_handle_t *handle, status_t status, void *userData)
{
    lpspi_nor_flash_handle_t *nor_flash_handle = (lpspi_nor_flash_handle_t *)userData;
    nor_flash_handle->masterFinished = true;
}

static void spi_update_frame_size(lpspi_nor_flash_handle_t *handle, uint32_t dataSize)
{
    uint32_t frameSize;

    if (dataSize % 4 == 0)
    {
        frameSize = 32;
    }
    else if (dataSize % 2 == 0)
    {
        frameSize = 16;
    }
    else
    {
        frameSize = 8;
    }

    if (frameSize != handle->frameSize)
    {
        LPSPI_SetFrameSize(handle->lpspiBase, frameSize);
        LPSPI_MasterTransferPrepareEDMALite(handle->lpspiBase, handle->handle, kLPSPI_MasterPcs0 | kLPSPI_MasterByteSwap | kLPSPI_MasterPcsContinuous);
        handle->frameSize = frameSize;
    }
}

static void spi_blocking(lpspi_nor_flash_handle_t *handle, lpspi_transfer_t *xfer)
{
    spi_update_frame_size(handle, xfer->dataSize);
    handle->masterFinished = false;
    LPSPI_MasterTransferEDMALite(handle->lpspiBase, handle->handle, xfer);
    while (handle->masterFinished != true);
}

static void write_enable(lpspi_nor_flash_handle_t *handle, uint32_t enable)
{
    write_enable_buffer_t writeEnableTxBuf;
    lpspi_transfer_t xfer = {0};

    memset(&writeEnableTxBuf, 0, sizeof(writeEnableTxBuf));
    if(enable)
    {
        writeEnableTxBuf.cmd = SPI_FLASH_CMD_WRITE_ENABLE;
    }
    else
    {
        writeEnableTxBuf.cmd = SPI_FLASH_CMD_WRITE_DISABLE;
    }
    xfer.txData   = (const uint8_t *)&writeEnableTxBuf;
    xfer.rxData   = NULL;
    xfer.dataSize = sizeof(write_enable_buffer_t);
    spi_blocking(handle, &xfer);
}

static void wait_bus_busy(lpspi_nor_flash_handle_t *handle)
{
    /* Wait status ready. */
    bool isBusy;
    uint32_t readValue;
    read_status_buffer_t readStatusTxBuf;
    read_status_buffer_t readStatusRxBuf;
    lpspi_transfer_t xfer = {0};

    memset(&readStatusTxBuf, 0, sizeof(readStatusTxBuf));
    readStatusTxBuf.cmd = SPI_FLASH_CMD_READ_STATUS;
    xfer.txData   = (const uint8_t *)&readStatusTxBuf;
    xfer.rxData   = (uint8_t *)&readStatusRxBuf;
    xfer.dataSize = sizeof(read_status_buffer_t);

    do
    {
        spi_blocking(handle, &xfer);
        readValue = readStatusRxBuf.data;
        if (readValue & (1U << FLASH_BUSY_STATUS_OFFSET))
        {
            isBusy = true;
        }
        else
        {
            isBusy = false;
        }

    } while (isBusy);

}

void LPSPI_Nor_Flash_Init(lpspi_nor_flash_handle_t *handle, lpspi_nor_flash_config_t *config, uint32_t srcClock_Hz)
{
    static lpspi_master_edma_handle_t lpspiHandle;
    static edma_handle_t txHandle;
    static edma_handle_t rxHandle;

    handle->lpspiBase = config->lpspiBase;
    handle->handle = &lpspiHandle;
    handle->frameSize = 0;
    handle->masterFinished = false;

    DMAMUX_Init(config->dmamuxBase);
    DMAMUX_SetSource(config->dmamuxBase, config->txChannel, config->txSource);
    DMAMUX_SetSource(config->dmamuxBase, config->rxChannel, config->rxSource);
    DMAMUX_EnableChannel(config->dmamuxBase, config->txChannel);
    DMAMUX_EnableChannel(config->dmamuxBase, config->rxChannel);

    /* Init the DMA module */
    edma_config_t edmaConfig = {0};
    EDMA_GetDefaultConfig(&edmaConfig);
    EDMA_Init(config->dmaBase, &edmaConfig);
    EDMA_CreateHandle(&txHandle, config->dmaBase, config->txChannel);
    EDMA_CreateHandle(&rxHandle, config->dmaBase, config->rxChannel);
    
    lpspi_master_config_t userConfig;
    LPSPI_MasterGetDefaultConfig(&userConfig);
    userConfig.baudRate = config->baudRate;
    LPSPI_MasterInit(handle->lpspiBase, &userConfig, srcClock_Hz);
    LPSPI_MasterTransferCreateHandleEDMA(handle->lpspiBase, handle->handle, masterCallback, handle, &rxHandle, &txHandle);
}

void LPSPI_Nor_Flash_Read(lpspi_nor_flash_handle_t *handle, uint32_t addr, uint8_t *buf, uint32_t len)
{
    read_buffer_t readTxBuf;
    read_buffer_t readRxBuf;
    lpspi_transfer_t xfer = {0};

    memset(&readTxBuf, 0, sizeof(readTxBuf));
    while(len > 0)
    {
        uint32_t min_len;
        min_len = MIN(len, READ_BUFFER_SIZE);
        readTxBuf.cmd = __REV(SPI_FLASH_CMD_READ << 24 | addr & 0xFFFFFF);
        xfer.txData   = (const uint8_t *)&readTxBuf;
        xfer.rxData   = (uint8_t *)&readRxBuf;
        xfer.dataSize = 4 + min_len;
        spi_blocking(handle, &xfer);
        memcpy(buf, readRxBuf.data, min_len);
        len -= min_len;
        buf += min_len;
        addr += min_len;
    }
}

status_t LPSPI_Nor_Flash_Program(lpspi_nor_flash_handle_t *handle, uint32_t addr, const uint8_t *buf, uint32_t len)
{
    page_program_buffer_t pageProgramTxBuf;
    lpspi_transfer_t xfer = {0};

    memset(&pageProgramTxBuf, 0, sizeof(pageProgramTxBuf));
    while(len > 0)
    {
        uint32_t min_len;
        min_len = MIN(len, WRITE_BUFFER_SIZE);
        min_len = MIN(min_len, FLASH_PAGE(addr + SPI_FLASH_PAGE) - addr); /* Can't program in multiple pages */
        memcpy(pageProgramTxBuf.data, buf, min_len);
        pageProgramTxBuf.cmd = __REV(SPI_FLASH_CMD_PAGE_PROGRAM << 24 | addr & 0xFFFFFF);;
        xfer.txData   = (const uint8_t *)&pageProgramTxBuf;
        xfer.rxData   = NULL;
        xfer.dataSize = 4 + min_len;
        write_enable(handle, true);
        spi_blocking(handle, &xfer);
        wait_bus_busy(handle);
        len -= min_len;
        buf += min_len;
        addr += min_len;
    }

    return kStatus_Success;
}

status_t LPSPI_Nor_Flash_Erase(lpspi_nor_flash_handle_t *handle, uint32_t addr, uint32_t len)
{
    erase_buffer_t eraseTxBuf;
    lpspi_transfer_t xfer = {0};
    
    if(addr % SPI_FLASH_SECTOR || len % SPI_FLASH_SECTOR)
    {
        return kStatus_InvalidArgument;
    }

    memset(&eraseTxBuf, 0, sizeof(eraseTxBuf));
    while(len > 0)
    {
        eraseTxBuf.cmd = __REV(SPI_FLASH_CMD_ERASE << 24 | addr & 0xFFFFFF);
        xfer.txData   = (const uint8_t *)&eraseTxBuf;
        xfer.rxData   = NULL;
        xfer.dataSize = sizeof(erase_buffer_t);
        write_enable(handle, true);
        spi_blocking(handle, &xfer);
        wait_bus_busy(handle);
        len -= SPI_FLASH_SECTOR;
        addr += SPI_FLASH_SECTOR;
    }

    return kStatus_Success;
}
