/*
 * Copyright 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _FSL_LPSPI_NOR_FLASH_H_
#define _FSL_LPSPI_NOR_FLASH_H_

#include "fsl_lpspi_edma.h"
#include "fsl_dmamux.h"

#define SPI_FLASH_SIZE              (0x100000 * 32) /* 32MB */
#define SPI_FLASH_SECTOR            0x10000         /* 64KB */
#define SPI_FLASH_PAGE              0x100           /* 256B */

#define SPI_FLASH_CMD_PAGE_PROGRAM  0x02
#define SPI_FLASH_CMD_READ          0x03
#define SPI_FLASH_CMD_READ_STATUS   0x05
#define SPI_FLASH_CMD_ERASE         0xD8
#define SPI_FLASH_CMD_WRITE_ENABLE  0x06
#define SPI_FLASH_CMD_WRITE_DISABLE 0x04

#define FLASH_BUSY_STATUS_OFFSET 0

#define READ_BUFFER_SIZE  256
#define WRITE_BUFFER_SIZE SPI_FLASH_PAGE

#if !defined(ALIGN_DOWN)
#define ALIGN_DOWN(x, a) (((uint32_t)(x)) & ~((uint32_t)(a)-1u))
#endif

#define FLASH_PAGE(addr) (ALIGN_DOWN((addr), SPI_FLASH_PAGE))

typedef struct _page_program_buffer_t 
{
    uint32_t cmd;
    uint8_t data[WRITE_BUFFER_SIZE];
} page_program_buffer_t;

typedef struct _read_buffer_t 
{
    uint32_t cmd;
    uint8_t data[READ_BUFFER_SIZE];
} read_buffer_t;

typedef struct _read_status_buffer_t 
{
    uint8_t cmd;
    uint8_t data;
} read_status_buffer_t;

typedef struct _write_enable_buffer_t 
{
    uint8_t cmd;
} write_enable_buffer_t;

typedef struct _erase_buffer_t 
{
    uint32_t cmd;
} erase_buffer_t;


typedef struct _lpspi_nor_flash_handle
{
    LPSPI_Type *lpspiBase;
    lpspi_master_edma_handle_t *handle;
    uint32_t frameSize;
    volatile uint32_t masterFinished;
} lpspi_nor_flash_handle_t;

typedef struct _lpspi_nor_flash_config
{
    LPSPI_Type *lpspiBase;
    DMA_Type *dmaBase;
    DMAMUX_Type *dmamuxBase;
    uint32_t baudRate;
    uint32_t txChannel;
    uint32_t rxChannel;
    dma_request_source_t txSource;
    dma_request_source_t rxSource;
} lpspi_nor_flash_config_t;

void LPSPI_Nor_Flash_Init(lpspi_nor_flash_handle_t *handle, lpspi_nor_flash_config_t *config, uint32_t srcClock_Hz);
void LPSPI_Nor_Flash_Read(lpspi_nor_flash_handle_t *handle, uint32_t addr, uint8_t *buf, uint32_t len);
status_t LPSPI_Nor_Flash_Program(lpspi_nor_flash_handle_t *handle, uint32_t addr, const uint8_t *buf, uint32_t len);
status_t LPSPI_Nor_Flash_Erase(lpspi_nor_flash_handle_t *handle, uint32_t addr, uint32_t len);

#endif /* _FSL_LPSPI_NOR_FLASH_H_ */
