/*
 * Copyright  2017-2019 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "pin_mux.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_pxp.h"
#include "lcdifv2_support.h"
#include "fsl_dc_fb.h"

#include "fsl_soc_src.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_PXP PXP
#define APP_IMG_WIDTH  640
#define APP_IMG_HEIGHT 480
#define APP_BPP 2U // Byte per Pixel



/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void APP_InitInputBuffer(void);
static void APP_InitPxp(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/

/* PXP Output buffer config. */
static pxp_output_buffer_config_t outputBufferConfig;

/*
 * When new frame buffer sent to display, it might not be shown immediately.
 * Application could use callback to get new frame shown notification, at the
 * same time, when this flag is set, application could write to the older
 * frame buffer.
 */
static volatile bool s_newFrameShown = false;


/*******************************************************************************
 * Code
 ******************************************************************************/
#define APP_PXP_PS_FORMAT  kPXP_PsPixelFormatUYVY1P422  // Need to combine with swap to support YUYV
#define APP_PXP_OUT_FORMAT kPXP_PsPixelFormatUYVY1P422

/* PS configure. */
static pxp_ps_buffer_config_t psBufferConfig = {
    .pixelFormat = APP_PXP_PS_FORMAT,
    .swapByte    = false,
    .bufferAddr  = (uint32_t)0,
    .bufferAddrU = 0U,
    .bufferAddrV = 0U,
    .pitchBytes  = APP_IMG_WIDTH * APP_BPP,
};

static void APP_InitPxp(void)
{
    PXP_Init(APP_PXP);

#if defined(FSL_FEATURE_PXP_V3) && FSL_FEATURE_PXP_V3
    PXP_SetProcessSurfaceBackGroundColor(APP_PXP, 0U, 0U);
#else
    PXP_SetProcessSurfaceBackGroundColor(APP_PXP, 0U);
#endif

    /* Disable AS. */
    PXP_SetAlphaSurfacePosition(APP_PXP, 0xFFFFU, 0xFFFFU, 0U, 0U);

    /* Output config. */
    outputBufferConfig.pixelFormat    = APP_PXP_OUT_FORMAT;
    outputBufferConfig.interlacedMode = kPXP_OutputProgressive;
    outputBufferConfig.buffer1Addr    = 0U;
    outputBufferConfig.pitchBytes     = 720 * APP_BPP;
    outputBufferConfig.width          = 720;
    outputBufferConfig.height         = 1280;

    /* Disable CSC1, it is enabled by default. */
    PXP_SetCsc1Mode(APP_PXP, kPXP_Csc1YCbCr2RGB);
    PXP_EnableCsc1(APP_PXP, false);
}

int pxp_main(void)
{
    APP_InitPxp();
}

//#define PXP_BY_PASS
void APP_Scale_Rotate(char * lcd_buf, char * usb_pic)
{

	// ROTATE
	psBufferConfig.bufferAddr =  (uint32_t)usb_pic;
	psBufferConfig.pitchBytes = 640 * APP_BPP;
	psBufferConfig.swapByte   = true; //true;
	PXP_SetProcessSurfaceBufferConfig(APP_PXP, &psBufferConfig);


	// scale at first
	/*
		------------               ------------------------
		-          - 480           -                      -
		-          -       -->     -                      -       -->  Rotate
		------------               -                      -720
		    640                    -                      -
                                   -                      -
                                   -                      -
								   ------------------------
								             1280
	 */

	// Config scale factor
#ifdef PXP_BY_PASS
	PXP_SetProcessSurfaceScaler(APP_PXP, 640, 480, 640, 480);
#else
	PXP_SetProcessSurfaceScaler(APP_PXP, 640, 480, 1280, 720);
#endif

	// out PS position
	PXP_SetProcessSurfacePosition(APP_PXP,
			0,
			0,
			0 + 1280 - 1U,
			0 + 1280 - 1U);

	// rotate
#ifdef PXP_BY_PASS
	PXP_SetRotateConfig(APP_PXP, kPXP_RotateOutputBuffer, kPXP_Rotate0, kPXP_FlipDisable);
#else
	PXP_SetRotateConfig(APP_PXP, kPXP_RotateOutputBuffer, kPXP_Rotate270, kPXP_FlipDisable);
#endif
	outputBufferConfig.pixelFormat    = APP_PXP_OUT_FORMAT;
	outputBufferConfig.interlacedMode = kPXP_OutputProgressive;
#ifdef PXP_BY_PASS
	outputBufferConfig.pitchBytes     = 640 * APP_BPP;
	outputBufferConfig.width          = 640;
	outputBufferConfig.height         = 480;
#else
	outputBufferConfig.pitchBytes     = 720 * APP_BPP;
	outputBufferConfig.width          = 1280;
	outputBufferConfig.height         = 720;
#endif
	outputBufferConfig.buffer0Addr    = (uint32_t)lcd_buf;
	PXP_SetOutputBufferConfig(APP_PXP, &outputBufferConfig);
	PXP_Start(APP_PXP);

	/* Wait for process complete. */
	while (!(kPXP_CompleteFlag & PXP_GetStatusFlags(APP_PXP)))
	{
	}

	PXP_ClearStatusFlags(APP_PXP, kPXP_CompleteFlag);
}



