/* ---------------------------------------------------------------------------- */
/* Copyright 2020 NXP.                                                          */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */
/*
 * The Clear BSD License
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 * that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef VIRTUAL_COM_H_
#define VIRTUAL_COM_H_

/*!
 * @defgroup KL_MODULES_USB_VCOM VCOM API
 * @ingroup KL_MODULES_USB
 * @{
 */

#include "app_defines.h"
#include "usb_device_descriptor.h"
#include "usb_device_config.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/
#if defined(USB_DEVICE_CONFIG_EHCI) && (USB_DEVICE_CONFIG_EHCI > 0)
    #define CONTROLLER_ID   kUSB_ControllerEhci0
    #define DATA_BUFF_SIZE  HS_CDC_VCOM_BULK_OUT_PACKET_SIZE
#endif
#if defined(USB_DEVICE_CONFIG_KHCI) && (USB_DEVICE_CONFIG_KHCI > 0)
    #define CONTROLLER_ID   kUSB_ControllerKhci0
    #define DATA_BUFF_SIZE  FS_CDC_VCOM_BULK_OUT_PACKET_SIZE
#endif
#if defined(USB_DEVICE_CONFIG_LPCIP3511FS) && (USB_DEVICE_CONFIG_LPCIP3511FS > 0U)
    #define CONTROLLER_ID   kUSB_ControllerLpcIp3511Fs0
    #define DATA_BUFF_SIZE  FS_CDC_VCOM_BULK_OUT_PACKET_SIZE
#endif

#if defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U)
    #define CONTROLLER_ID   kUSB_ControllerLpcIp3511Hs0
    #define DATA_BUFF_SIZE  HS_CDC_VCOM_BULK_OUT_PACKET_SIZE
#endif

#define USB_DEVICE_INTERRUPT_PRIORITY  (3U) /**< interrupt priority of VCOM */

/* Currently configured line coding */
#define LINE_CODING_SIZE        (0x07)   /**< Size of a get/set line coding request */
#define LINE_CODING_DTERATE     (460800) /**< Data terminal rate, in bits per second */
#define LINE_CODING_CHARFORMAT  (0x00)   /**< Number of stop bits of the line coding */
#define LINE_CODING_PARITYTYPE  (0x00)   /**< Parity type of the line coding */
#define LINE_CODING_DATABITS    (0x08)   /**< Number of data bits of the line coding */

/* Communications feature */
#define COMM_FEATURE_DATA_SIZE  (0x02)   /**< Data size of a communication feature of cdc device */
#define STATUS_ABSTRACT_STATE   (0x0000) /**< Abstract state of cdc device */
#define COUNTRY_SETTING         (0x0000) /**< Country setting of cdc device */

/* Notification of serial state */
#define NOTIF_PACKET_SIZE   (0x08) /**< Packet size of a cdc notification */
#define UART_BITMAP_SIZE    (0x02) /**< Value of field 'wLength' in a cdc notification */
#define NOTIF_REQUEST_TYPE  (0xA1) /**< Request type of a cdc notification */


/**
 * Callback prototype for data received over VCOM
 *
 * @param data pointer to data received
 * @param length length of data received in bytes
 */
typedef void (*vcom_data_rcvd_cb)(uint8_t *data, uint32_t length);

/**
 * Callback to the user of the vcom service to notify that the transfer is complete
 */
typedef void (*vcom_transfer_complete_cb_t)(void);

/**
 * Callback to the user of the vcom service to notify that the presence of the VCOM
 * host changed. It either became present (connection opened) or absent (connection closed).
 * @param present   true if host is present; false if host is absent
 */
typedef void (*vcom_host_presence_cb)(bool present);

#if USB_DEVICE_CONFIG_CDC_ACM
/**
 * @brief USB VCOM Instance, an instance is a COM-port in Windows or a /dev/ttyACM? in Linux
 */
typedef enum _USBVcomInstance {
    #if USB_HCI_VCOM_ENABLE
    USB_VCOM_INSTANCE_HCI = 0,          /**< HCI command interface. */
    #endif
    #if SERIAL_PORT_TYPE_USBCDC
    USB_VCOM_INSTANCE_LOGGING,          /**< general-purpose logging interface. */
    #endif
} usbVcomInstance_t;
#endif

/**
 * Possible states as part of VCOM transfers.
 */
typedef enum _usb_vcom_state {
    kUSB_VcomStateIdle = 0x0,           /* Idle state, not receiving or sending */
    kUSB_VcomStateReady,                /* Ready to receive/send data */
    kUSB_VcomStateReceiving,            /* Processing received data */
    kUSB_VcomStateSending,              /* Sending data */
} usb_vcom_state_t;

/**
 * Define the infomation relates to abstract control model
 */
typedef struct _usb_cdc_acm_info {
    /** Serial state buffer of the CDC device to notify the serial state to host. */
    uint8_t serialStateBuf[NOTIF_PACKET_SIZE + UART_BITMAP_SIZE];
    bool dtePresent;                                             /**< A flag to indicate whether DTE is present.         */
    uint16_t breakDuration;                                      /**< Length of time in milliseconds of the break signal */
    uint8_t dteStatus;                                           /**< Status of data terminal equipment                  */
    uint8_t currentInterface;                                    /**< Current interface index.                           */
    uint16_t uartState;                                          /**< UART state of the CDC device.                      */
} usb_cdc_acm_info_t;

/**
 * Define the types for application
 */
typedef struct _usb_cdc_vcom_struct {
    usb_device_handle deviceHandle;          /**< USB device handle. */
    class_handle_t cdcAcmHandle;             /**< USB CDC ACM class handle. */
    uint8_t *lineCoding;                     /**< Line codinig of cdc device */
    uint8_t *abstractState;                  /**< Abstract state of cdc device */
    uint8_t *countryCode;                    /**< Country code of cdc device */
    usb_cdc_acm_info_t *usbCdcAcmInfo;       /**< CDC ACM information */
    uint8_t *currRecvBuf;                    /**< receive buffer*/
    volatile uint32_t recvSize;              /**< the data length received from host*/
    uint16_t bulkOutEndpointMaxPacketSize;   /**< bulk out endpoint maxpacket size */
    uint16_t bulkInEndpointMaxPacketSize;    /**< bulk in endpoint maxpacket size */
    uint16_t interruptEndpointMaxPacketSize; /**< interrupt  endpoint maxpacket size */
    uint8_t attach;                          /**< A flag to indicate whether a usb device is attached. 1: attached, 0: not attached */
    uint8_t speed;                           /**< Speed of USB device. USB_SPEED_FULL/USB_SPEED_LOW/USB_SPEED_HIGH.                 */
    uint8_t startTransactions;               /**< A flag to indicate whether a CDC device is ready to transmit and receive data.    */
    uint8_t currentConfiguration;            /**< Current configuration value. */
    uint8_t currentInterfaceAlternateSetting
    [USB_CDC_VCOM_INTERFACE_COUNT];    /**< Current alternate setting value for each interface. */
    uint8_t bulkInEndpoint;            /**< bulk in endpoint number*/
    uint8_t bulkOutEndpoint;           /**< bulk out endpoint number*/
    uint8_t interruptEndpoint;         /**< interrupt endpoint number*/
} usb_cdc_vcom_struct_t;

/**
 * VCOM Layer context
 */
typedef struct _vcom_ctrl_ctxt_t {
    volatile usb_cdc_vcom_struct_t *vcomInstance;  /**< Vcom CDC Instance, map to COM port in Windows or /dev/ttyACM? in *nix */
    //framework_dpc_srp_t dataRecvSrp;               /**< srp to use when data is received */
    //framework_dpc_srp_t sendCmplSrp;               /**< srp to use when data send-transfer is completed */
    //framework_dpc_srp_t hostPresenceSrp;           /**< srp to use when host presence is changed */
    vcom_transfer_complete_cb_t cb;                /**< transfer complete callback */
    vcom_data_rcvd_cb rcvdCb;                      /**< data received callback */
    vcom_host_presence_cb hostCb;                  /**< host presence changed callback */

    const uint8_t *data;                           /**< transmit buffer */
    const volatile uint8_t *dataIndex;             /**< Index in transmit buffer */

    volatile uint32_t toWrite;                     /**< number of bytes still to write */
    volatile usb_vcom_state_t state;               /**< Current state of the VCOM */
} vcom_ctrl_ctxt_t;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

#if USB_DEVICE_CONFIG_CDC_ACM
/**
 * Send data to PC host using the USB VCOM
 *
 * This function is NOT reentrant, so the user has to make sure the callback (@p cb) is called
 * before scheduling a new request.
 *
 * IMPORTANT: When sending over the logging instance, the callback will be called in ISR!
 *            This can trigger artifacts in the audio stream and also, any blocking tasks should
 *            not be done in this callback (schedule a DPC if any work needs to be done).
 *
 *
 * @param instance VCOM instance to use when sending the data
 * @param data     Data to send
 * @param len      Length of data to send in bytes
 * @param cb       Transfer complete callback which will be called when the data was send.
 * @return errorcode. kERROR_Ok if transmit request is started,
 *                    kERROR_Fail otherwise (VCOM is busy, no VCOM host connected, ...)
 */
error_t UsbVcom_Send(usbVcomInstance_t instance, const uint8_t *data, const uint32_t len, const vcom_transfer_complete_cb_t cb);
#endif

#if USB_HCI_VCOM_ENABLE
/**
 * Send data to PC host using the USB HCI VCOM
 * This is a wrapper around @ref UsbVcom_Send using the HCI instance.
 *
 * This function is NOT reentrant, so the user has to make sure the callback (@p cb) is called
 * before scheduling a new request.
 *
 * @param data     Data to send
 * @param len      Length of data to send in bytes
 * @param cb       Transfer complete callback which will be called when the data was send.
 * @return errorcode. kERROR_Ok if transmit request is started,
 *                    kERROR_Fail otherwise (VCOM is busy, no VCOM host connected, ...)
 */
error_t UsbVcom_SendHci(const uint8_t *data, const uint32_t len, const vcom_transfer_complete_cb_t cb);
#endif

/**
 * Reset VCOM state
 */
void UsbVcom_Reset(void);

#if SERIAL_PORT_TYPE_USBCDC

    #include "serial_manager.h"

/*!
 * @brief Logging VCOM initialization function.
 *
 * This function initializes the logging VCOM.
 *
 * @param serialHandle to USB serial handle
 * @param config pointer to config structure, we don't use this parameter
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcInit(serial_handle_t serialHandle, void *config);

/*!
 * @brief Logging VCOM deinit function.
 *
 * This function deinits the logging VCOM.
 *
 * @param serialHandle to USB serial handle
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcDeinit(serial_handle_t serialHandle);

/*!
 * @brief USB send data to host using a blocking method.
 *
 * This function sends data to host by usb cdc protocol
 * @param serialHandle to USB serial handle
 * @param buffer pointer to the data.
 * @param length size of the transfer in bytes.
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcWrite(serial_handle_t serialHandle, uint8_t *buffer, uint32_t length);

/*!
 * @brief Cancel current USB write operation
 *
 * @param serialHandle pointer to USB serial handle
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcCancelWrite(serial_handle_t serialHandle);

/*!
 * @brief Register USB data-transmit-done callback
 *
 * @param serialHandle pointer to USB serial handle
 * @param callback Callback which will be called (in ISR!) when data has been sent over the USB VCOM
 * @param callbackParam Parameter passed to @p callback
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcInstallTxCallback(serial_handle_t serialHandle,
                                                       serial_manager_callback_t callback,
                                                       void *callbackParam);

/*!
 * @brief Register USB data-receive callback
 *
 * @param serialHandle pointer to USB serial handle
 * @param callback Callback which will be called (in ISR!) when data is received over the USB VCOM
 * @param callbackParam Parameter passed to @p callback
 *
 * @return serial manager status code
 */
serial_manager_status_t Serial_UsbCdcInstallRxCallback(serial_handle_t serialHandle,
                                                       serial_manager_callback_t callback,
                                                       void *callbackParam);

/*!
 * @brief USB isr function
 *
 * @param serialHandle pointer to USB serial handle
 */
void Serial_UsbCdcIsrFunction(serial_handle_t serialHandle);

/*!
 *
 * Callback for USB data received
 *
 * @param buff Buffer which holds the received data
 * @param length length of the data in @p buff in bytes
 */
void Serial_DataRecv(uint8_t *buff, uint32_t length);

#endif  /* SERIAL_PORT_TYPE_USBCDC */

/** @} */

#endif /* VIRTUAL_COM_H_ */
