/*
 * Copyright 2019-2020, 2024 NXP
 * SPDX-License-Identifier: BSD-3-Clause
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */


#ifndef __USB_DEVICE_DESCRIPTOR_STRUCTS_H__
#define __USB_DEVICE_DESCRIPTOR_STRUCTS_H__

#include <stdint.h>
#include "usb_spec.h"
#include "usb_device_config.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define HEADSET_AIC_COUNT     (2U)/* Nr of Audio Interfaces in the Collection for a headset device.*/
#define HEADPHONES_AIC_COUNT  (1U)/* Nr of Audio Interfaces in the Collection for a headphones device.*/

typedef struct _usb_descriptor_hid {
    uint8_t bLength;              /* Total size of the HID descriptor*/
    uint8_t bDescriptorType;      /* Constant name specifying type of HID descriptor*/
    uint8_t bcdHID[2];            /* Numeric expression identifying the HID Class Specification release*/
    uint8_t bCountryCode;         /* Numeric expression identifying country code of the localized hardware*/
    uint8_t bNumDescriptors;      /* Numeric expression specifying the number of class descriptors*/
    uint8_t bHidDescriptorType;   /* Constant name identifying type of class descriptor*/
    uint8_t wDescriptorLength[2]; /* Numeric expression that is the total size of the Report descriptor*/
} usb_descriptor_hid_t;

typedef struct _usb_descriptor_interface_association {
    uint8_t bLength;              /* Size of this descriptor in bytes*/
    uint8_t bDescriptorType;      /* Constant name specifying INTERFACE ASSOCIATION Descriptor*/
    uint8_t bFirstInterface;      /* Interface number of the first interface that is associated with this function.*/
    uint8_t bInterfaceCount;      /* Number of contiguous interfaces that are associated with this function.*/
    uint8_t bFunctionClass;       /* Class code (assigned by USB-IF). A value of zero is not allowed in this descriptor.*/
    uint8_t bFunctionSubClass;    /* Subclass code (assigned by USB-IF).*/
    uint8_t bFunctionProtocol;    /* Protocol code (assigned by USB-IF).*/
    uint8_t iFunction;            /* Index of string descriptor describing this function.*/
} usb_descriptor_interface_association_t;


#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
#define GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(__NAME)                                                               \
    typedef struct _usb_descriptor_class_specific_ac_interface_##__NAME                                                                         \
    {                                                                                                                                           \
        /* The class-specific Audio Control (AC) interface descriptor is a concatenation of all the descriptors that are used to fully
        describe the audio function, i.e. all Unit Descriptors (UDs) and Terminal Descriptors (TDs). */                                         \
        uint8_t bLength;                  /* Size of this descriptor in bytes (8+n).*/                                                          \
        uint8_t bDescriptorType;          /* Constant name specifying CS_INTERFACE descriptor type.*/                                           \
        uint8_t bDescriptorSubType;       /* Constant name specifying HEADER descriptor subtype.*/                                              \
        uint8_t bcdADC[2];                /* Audio Device Class Specification Release Number in Binary-Coded Decimal.*/                         \
        uint8_t bCategory;                /* Total number of bytes returned for the class-specific AudioControl interface descriptor.*/         \
        uint8_t wTotalLength[2];          /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                             Includes the combined length of this descriptor header and all Unit and Terminal descriptors.*/    \
        uint8_t bmControls;               /* Array of interface numbers of the AudioStreaming or MIDIStreaming interface in the Collection.*/   \
    } usb_descriptor_class_specific_ac_interface_##__NAME##_t;
GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(headset)
GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(headphones)

typedef struct _usb_descriptor_audio_clock_source {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 12.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* INPUT_TERMINAL descriptor subtype.*/
    uint8_t bClockID;           /* Constant uniquely identifying the Terminal within the audio function.
                                   This value is used in all requests to address this Terminal.*/
    uint8_t bmAttrButes;
    uint8_t bmControls;
    uint8_t bAssocTerminal;     /* ID of the Output Terminal to which this Input Terminal is associated.*/
    uint8_t iClockSource;          /* Index of a string descriptor, describing the Input Terminal.*/
} usb_descriptor_audio_clock_source_t;

typedef struct _usb_descriptor_audio_input_terminal {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 12.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* INPUT_TERMINAL descriptor subtype.*/
    uint8_t bTerminalId;        /* Constant uniquely identifying the Terminal within the audio function.
                                   This value is used in all requests to address this Terminal.*/
    uint8_t wTerminalType[2];   /* Constant characterizing the type of Terminal. See USB Audio Terminal Types.*/
    uint8_t bAssocTerminal;     /* ID of the Output Terminal to which this Input Terminal is associated.*/
    uint8_t bCSourceID;
    uint8_t bNrChannels;        /* Number of logical output channels in the Terminals output audio channel cluster.*/
    uint8_t bmChannelConfig[4];  /* Describes the spatial location of the logical channels.*/
    uint8_t iChannelNames;      /* Index of a string descriptor, describing the name of the first logical channel.*/
    uint8_t bmControls[2];
    uint8_t iTerminal;          /* Index of a string descriptor, describing the Input Terminal.*/
} usb_descriptor_audio_input_terminal_t;

typedef struct _usb_descriptor_audio_feature_unit {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* FEATURE_UNIT descriptor subtype.*/
    uint8_t bUnitId;            /* Constant uniquely identifying the Unit within the audio function.
                                   This value is used in all requests to address this Unit.*/
    uint8_t bSourceID;          /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
    uint8_t bmaControls0[4];     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                   D0: Mute, D1: Volume, ...*/
    uint8_t bmaControls1[4];     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                   D0: Mute, D1: Volume, ...*/
    uint8_t bmaControls2[4];     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                   D0: Mute, D1: Volume, ...*/
    uint8_t iFeature;           /* Index of a string descriptor, describing this Feature Unit.*/

} usb_descriptor_audio_feature_unit_t;

typedef struct _usb_descriptor_audio_output_terminal {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 9.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* OUTPUT_TERMINAL descriptor subtype.*/
    uint8_t bTerminalId;        /* Constant uniquely identifying the Terminal within the audio function.
                                   This value is used in all requests to address this Terminal.*/
    uint8_t wTerminalType[2];   /* Constant characterizing the type of Terminal. See USB Audio Terminal Types.*/
    uint8_t bAssocTerminal;     /* ID of the Input Terminal to which this Ouput Terminal is associated.*/
    uint8_t bSourceID;          /* ID of the Unit or Terminal to which this Terminal is connected.*/
    uint8_t bCSourceID;
    uint8_t bmControls[2];
    uint8_t iTerminal;          /* Index of a string descriptor, describing the Output Terminal.*/
} usb_descriptor_audio_output_terminal_t;

typedef struct _usb_descriptor_ac_interrupt_endpoint {
    uint8_t bLength;           /* Size of this descriptor, in bytes: 9 */
    uint8_t bDescriptorType;   /* ENDPOINT descriptor type */
    uint8_t bEndpointAddress;  /* The address of the endpoint on the USB device described by this descriptor.
                                  The address is encoded as follows
                                  D7: Direction. 1 = IN endpoint
                                  D6..4: Reserved, reset to zero
                                  D3..0: The endpoint number, determined by the designer*/
    uint8_t bmAttributes;      /* D3..2: Synchronization type 00 = None, D1..0: Transfer type 11 = Interrupt*/
    uint8_t wMaxPacketSize[2]; /* Maximum packet size this endpoint is capable of sending or receiving when this
                                  configuration is selected. Used here to pass 2-byte status information.
                                  Set to 2 if not shared, set to the appropriate value if shared.*/
    uint8_t bInterval;         /* Left to the designers discretion. A value of 10 ms or more seems sufficient.*/
} usb_descriptor_ac_interrupt_endpoint_t;

#else
#define GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(__NAME, __CNT)                                                               \
    typedef struct _usb_descriptor_class_specific_ac_interface_##__NAME                                                                         \
    {                                                                                                                                           \
        /* The class-specific Audio Control (AC) interface descriptor is a concatenation of all the descriptors that are used to fully
        describe the audio function, i.e. all Unit Descriptors (UDs) and Terminal Descriptors (TDs). */                                         \
        uint8_t bLength;                  /* Size of this descriptor in bytes (8+n).*/                                                          \
        uint8_t bDescriptorType;          /* Constant name specifying CS_INTERFACE descriptor type.*/                                           \
        uint8_t bDescriptorSubType;       /* Constant name specifying HEADER descriptor subtype.*/                                              \
        uint8_t bcdADC[2];                /* Audio Device Class Specification Release Number in Binary-Coded Decimal.*/                         \
        uint8_t wTotalLength[2];          /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                             Includes the combined length of this descriptor header and all Unit and Terminal descriptors.*/    \
        uint8_t bInCollection;            /* The number of AudioStreaming and MIDIStreaming interfaces in the Audio Interface Collection
                                             to which this AudioControl interface belongs: n.*/                                                 \
        uint8_t baInterfaceNr[__CNT];     /* Array of interface numbers of the AudioStreaming or MIDIStreaming interface in the Collection.*/   \
    } usb_descriptor_class_specific_ac_interface_##__NAME##_t;
GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(headset, HEADSET_AIC_COUNT)
GENERATE_USB_DESCRIPTOR_CLASS_SPECIFIC_AC_INTERFACE_STRUCT(headphones, HEADPHONES_AIC_COUNT)

typedef struct _usb_descriptor_audio_input_terminal {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 12.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* INPUT_TERMINAL descriptor subtype.*/
    uint8_t bTerminalId;        /* Constant uniquely identifying the Terminal within the audio function.
                                   This value is used in all requests to address this Terminal.*/
    uint8_t wTerminalType[2];   /* Constant characterizing the type of Terminal. See USB Audio Terminal Types.*/
    uint8_t bAssocTerminal;     /* ID of the Output Terminal to which this Input Terminal is associated.*/
    uint8_t bNrChannels;        /* Number of logical output channels in the Terminals output audio channel cluster.*/
    uint8_t wChannelConfig[2];  /* Describes the spatial location of the logical channels.*/
    uint8_t iChannelNames;      /* Index of a string descriptor, describing the name of the first logical channel.*/
    uint8_t iTerminal;          /* Index of a string descriptor, describing the Input Terminal.*/
} usb_descriptor_audio_input_terminal_t;

typedef struct _usb_descriptor_audio_feature_unit {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* FEATURE_UNIT descriptor subtype.*/
    uint8_t bUnitId;            /* Constant uniquely identifying the Unit within the audio function.
                                   This value is used in all requests to address this Unit.*/
    uint8_t bSourceID;          /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
    uint8_t bControlSize;       /* Size in bytes of an element of the bmaControls() array: n.*/
    uint8_t bmaControls[3];     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                   D0: Mute, D1: Volume, ...*/
    uint8_t iFeature;           /* Index of a string descriptor, describing this Feature Unit.*/

} usb_descriptor_audio_feature_unit_t;

typedef struct _usb_descriptor_audio_output_terminal {
    uint8_t bLength;            /* Size of this descriptor, in bytes: 9.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* OUTPUT_TERMINAL descriptor subtype.*/
    uint8_t bTerminalId;        /* Constant uniquely identifying the Terminal within the audio function.
                                   This value is used in all requests to address this Terminal.*/
    uint8_t wTerminalType[2];   /* Constant characterizing the type of Terminal. See USB Audio Terminal Types.*/
    uint8_t bAssocTerminal;     /* ID of the Input Terminal to which this Ouput Terminal is associated.*/
    uint8_t bSourceID;          /* ID of the Unit or Terminal to which this Terminal is connected.*/
    uint8_t iTerminal;          /* Index of a string descriptor, describing the Output Terminal.*/
} usb_descriptor_audio_output_terminal_t;

typedef struct _usb_descriptor_ac_interrupt_endpoint {
    uint8_t bLength;           /* Size of this descriptor, in bytes: 9 */
    uint8_t bDescriptorType;   /* ENDPOINT descriptor type */
    uint8_t bEndpointAddress;  /* The address of the endpoint on the USB device described by this descriptor.
                                  The address is encoded as follows
                                  D7: Direction. 1 = IN endpoint
                                  D6..4: Reserved, reset to zero
                                  D3..0: The endpoint number, determined by the designer*/
    uint8_t bmAttributes;      /* D3..2: Synchronization type 00 = None, D1..0: Transfer type 11 = Interrupt*/
    uint8_t wMaxPacketSize[2]; /* Maximum packet size this endpoint is capable of sending or receiving when this
                                  configuration is selected. Used here to pass 2-byte status information.
                                  Set to 2 if not shared, set to the appropriate value if shared.*/
    uint8_t bInterval;         /* Left to the designers discretion. A value of 10 ms or more seems sufficient.*/
    uint8_t bRefresh;          /* Reset to 0.*/
    uint8_t bSynchAddress;     /* Reset to 0.*/
} usb_descriptor_ac_interrupt_endpoint_t;

#endif



typedef struct _usb_descriptor_as_iso_sync_endpoint {
    uint8_t bLength;           /* Size of this descriptor, in bytes: 9 */
    uint8_t bDescriptorType;   /* ENDPOINT descriptor type */
    uint8_t bEndpointAddress;  /* The address of the endpoint on the USB device described by this descriptor.
                                  The address is encoded as follows
                                  D7: Direction. 0 = OUT endpoint for sources, 1 = IN endpoint for sinks
                                  D6..4: Reserved, reset to zero
                                  D3..0: The endpoint number, determined by the designer*/
    uint8_t bmAttributes;      /* D3..2: Synchronization type 00 = None, D1..0: Transfer type 01 = Isochronous*/
    uint8_t wMaxPacketSize[2]; /* Maximum packet size this endpoint is capable of sending or receiving when this
                                  configuration is selected.*/
    uint8_t bInterval;         /* Interval for polling endpoint for data transfers expressed in milliseconds.
                                  Must be set to 1.*/
    uint8_t bRefresh;          /* This field indicates the rate at which an isochronous synchronization pipe
                                  provides new synchronization feedback data. This rate must be a power of 2,
                                  therefore only the power is reported back and the range of this field is from 1
                                  (2 ms) to 9 (512 ms).*/
    uint8_t bSynchAddress;     /* Must be reset to zero.*/
} usb_descriptor_as_iso_sync_endpoint_t;

#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
typedef struct _usb_descriptor_class_specific_as_interface {
    /* The class-specific Audio Stream (AS) interface descriptor. */
    uint8_t bLength;            /* Size of this descriptor in bytes: 7.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* AS_GENERAL descriptor subtype.*/
    uint8_t bTerminalLink;      /* The Terminal ID of the Terminal to which the endpoint of this interface is connected.*/
    uint8_t bmControls;
    uint8_t bFormatType;
    uint8_t bmFormats[4];
    uint8_t bNrChannels;
    uint8_t bmChannelConfig[4];
    uint8_t iChannelNames;
} usb_descriptor_class_specific_as_interface_t;

typedef struct _usb_descriptor_audio_stream_format_type {
    /* Format type descriptor */
    uint8_t bLength;            /* Total size of the descriptor*/
    uint8_t bDescriptorType;    /* Descriptor type of the descriptor*/
    uint8_t bDescriptorSubType; /* Subtype of the descriptor*/
    uint8_t bFormatType;        /* Constant identifying the Format Type the AudioStreaming interface is using*/
    uint8_t bSubslotSize;
    uint8_t bBitResolution;
} usb_descriptor_audio_stream_format_type_t;

typedef struct _usb_descriptor_as_specific_iso_endpoint {
    /*Audio as isochronous audio data endpoint descriptor structure */
    uint8_t bLength;            /* Total size of the descriptor*/
    uint8_t bDescriptorType;    /* Descriptor type of the descriptor*/
    uint8_t bDescriptorSubType; /* Subtype of the descriptor*/
    uint8_t bmAttributes;       /* A bit in the range D6..0 set to 1 indicates that the mentioned Control is supported by
                                   this endpoint*/
    uint8_t bmControls;
    uint8_t bLockDelayUnits;    /* Indicates the units used for the wLockDelay field*/
    uint8_t wLockDelay[2];      /* Indicates the time it takes this endpoint to reliably lock its internal clock recovery
                                   circuitry. Units used depend on the value of the bLockDelayUnits field.*/
} usb_descriptor_as_specific_iso_endpoint_t;

#else
typedef struct _usb_descriptor_class_specific_as_interface {
    /* The class-specific Audio Stream (AS) interface descriptor. */
    uint8_t bLength;            /* Size of this descriptor in bytes: 7.*/
    uint8_t bDescriptorType;    /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubType; /* AS_GENERAL descriptor subtype.*/
    uint8_t bTerminalLink;      /* The Terminal ID of the Terminal to which the endpoint of this interface is connected.*/
    uint8_t bDelay;             /* Delay (d) introduced by the data path (see Section 3.4, Inter Channel
                                   Synchronization). Expressed in number of frames.*/
    uint8_t wFormatTag[2];      /* The Audio Data Format that has to be used to communicate with this interface.*/
} usb_descriptor_class_specific_as_interface_t;

typedef struct _usb_descriptor_audio_stream_format_type {
    /* Format type descriptor */
    uint8_t bLength;            /* Total size of the descriptor*/
    uint8_t bDescriptorType;    /* Descriptor type of the descriptor*/
    uint8_t bDescriptorSubType; /* Subtype of the descriptor*/
    uint8_t bFormatType;        /* Constant identifying the Format Type the AudioStreaming interface is using*/
    uint8_t bNrChannels;        /* Number of channels of device*/
    uint8_t bSubFrameSize;      /* Bytes per audio subframe*/
    uint8_t bBitResolution;     /* Bits per sample*/
    uint8_t bSamFreqType;       /* Frequency supported*/
    uint8_t tSamFreq[1][3];     /* Sample frequency*/
} usb_descriptor_audio_stream_format_type_t;

typedef struct _usb_descriptor_as_specific_iso_endpoint {
    /*Audio as isochronous audio data endpoint descriptor structure */
    uint8_t bLength;            /* Total size of the descriptor*/
    uint8_t bDescriptorType;    /* Descriptor type of the descriptor*/
    uint8_t bDescriptorSubType; /* Subtype of the descriptor*/
    uint8_t bmAttributes;       /* A bit in the range D6..0 set to 1 indicates that the mentioned Control is supported by
                                   this endpoint*/
    uint8_t bLockDelayUnits;    /* Indicates the units used for the wLockDelay field*/
    uint8_t wLockDelay[2];      /* Indicates the time it takes this endpoint to reliably lock its internal clock recovery
                                   circuitry. Units used depend on the value of the bLockDelayUnits field.*/
} usb_descriptor_as_specific_iso_endpoint_t;
#endif




typedef struct _usb_descriptor_cdc_header_functional {
    uint8_t bFunctionLength;    /* Total size of the descriptor*/
    uint8_t bdescriptortype;    /* Descriptor type of the descriptor*/
    uint8_t bdescriptorsubtype; /* Subtype of the descriptor*/
    uint8_t bcdUSB[2];          /* USB Class Definitions for Communications Devices
                                   Specification release number in binary-coded decimal. */
} usb_descriptor_cdc_header_functional_t;

typedef struct _usb_descriptor_cdc_functional_call_mgmt {
    uint8_t bFunctionLength;    /* Total size of the descriptor*/
    uint8_t bdescriptortype;    /* CS_INTERFACE, as defined in Table 12.*/
    uint8_t bdescriptorsubtype; /* Identifier (ID) of functional descriptor */
    uint8_t bmCapabilities;     /* Bit 0: Whether device handle call management itself 1,
                                   Bit 1: Whether device can send/receive call management info over a Data Class Interface 0 */
    uint8_t bDataInterface;     /* Indicates that multiplexed commands are handled via data interface */
} usb_descriptor_cdc_functional_call_mgmt_t;

typedef struct _usb_descriptor_cdc_functional_abstract_control {
    uint8_t bFunctionLength;    /* Total size of the descriptor*/
    uint8_t bdescriptortype;    /* CS_INTERFACE, as defined in Table 12.*/
    uint8_t bdescriptorsubtype; /* Identifier (ID) of functional descriptor */
    uint8_t bmCapabilities;     /* Bit 0: Whether device supports the request combination of Set_Comm_Feature, Clear_Comm_Feature, and
                                   Get_Comm_Feature 0, Bit 1: Whether device supports the request combination of Set_Line_Coding,
                                   Set_Control_Line_State, Get_Line_Coding, and the notification Serial_State 1, Bit ...  */
} usb_descriptor_cdc_functional_abstract_control_t;

typedef struct _usb_descriptor_cdc_union_interface {
    /* CDC Header Functional Descriptor structure according to the 5.2.3 in CDC specification */
    uint8_t bFunctionLength;       /* Size of this descriptor in bytes.*/
    uint8_t bDescriptorType;       /* CS_INTERFACE descriptor type.*/
    uint8_t bDescriptorSubtype;    /* Union Functional Descriptor SubType.*/
    uint8_t bControlInterface;     /* USB Class Definitions for Communications Devices Specification release number in
                                      binary-coded decimal.*/
    uint8_t bSubordinateInterface; /* Interface number of first subordinate interface in the union. */
} usb_descriptor_cdc_union_interface_t;

typedef struct _usb_class_audio_control_io_descriptor_group {
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
    usb_descriptor_audio_clock_source_t clockSource;
#endif
    usb_descriptor_audio_input_terminal_t inputTerminal;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
    usb_descriptor_audio_feature_unit_t featureUnit;
#endif    
    usb_descriptor_audio_output_terminal_t outputTerminal;
} usb_class_audio_control_io_descriptor_group_t;

typedef struct _usb_class_audio_control_ifo_descriptor_group {
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
    usb_descriptor_audio_clock_source_t clockSource;
#endif
    usb_descriptor_audio_input_terminal_t inputTerminal;
    usb_descriptor_audio_feature_unit_t featureUnit;
    usb_descriptor_audio_output_terminal_t outputTerminal;
} usb_class_audio_control_ifo_descriptor_group_t;

typedef struct _usb_class_audio_stream_descriptor_group {
    usb_descriptor_interface_t altSet0;
    usb_descriptor_interface_t altSet1;
    usb_descriptor_class_specific_as_interface_t asInterface;
    usb_descriptor_audio_stream_format_type_t audioFormat;
    usb_descriptor_ac_interrupt_endpoint_t isoEndpoint;
    usb_descriptor_as_specific_iso_endpoint_t specificIsoEndpoint;
} usb_class_audio_stream_descriptor_group_t;

typedef struct _usb_class_hid_descriptor {
    usb_descriptor_interface_t consumerControlInterface;
    usb_descriptor_hid_t hidReport;
    usb_descriptor_endpoint_t interruptEndpoint;
} usb_class_hid_descriptor_t;

typedef struct _usb_class_cdc_vcom_descriptor {
    usb_descriptor_interface_association_t iad;
    usb_descriptor_interface_t cic;
    usb_descriptor_cdc_header_functional_t functionalHeader;
    usb_descriptor_cdc_functional_call_mgmt_t callMgmt;
    usb_descriptor_cdc_functional_abstract_control_t abstractControl;
    usb_descriptor_cdc_union_interface_t unionInterface;
    //usb_descriptor_endpoint_t interruptEndpoint;
    usb_descriptor_interface_t dic;
    usb_descriptor_endpoint_t bulkIn;
    usb_descriptor_endpoint_t bulkOut;
} usb_class_cdc_vcom_descriptor_t;

#endif /* __USB_DEVICE_DESCRIPTOR_STRUCTS_H__ */
