/*
 * Copyright (c) 2015 - 2016, Freescale Semiconductor, Inc.
 * Copyright 2016,2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <stdio.h>
#include <stdlib.h>
/*${standard_header_anchor}*/
#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"

#include "usb_device_ch9.h"
#include "usb_device_descriptor.h"

#include "composite.h"
#include "app_defines.h"
#include "hid_consumer_control.h"

#include "fsl_device_registers.h"
#include "clock_config.h"
#include "fsl_debug_console.h"
#include "board.h"
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#include "fsl_ctimer.h"
#endif

#include "audio_mixer.h"
#include "audio_latency_debug.h"
#include "fsl_i2s.h"
#include "fsl_i2s_dma.h"
#include "fsl_adapter_audio.h"
#include "composite.h"
#include "audio_rx.h"
#include "audio.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* audio 2.0 and high speed, use low latency, but IP3511HS controller do not have micro frame count */
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
#if (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
volatile static uint8_t s_microFrameCountIp3511HS = 0;
#endif
#endif

#define USB_AUDIO_ENTER_CRITICAL() \
                                   \
    OSA_SR_ALLOC();                \
                                   \
    OSA_ENTER_CRITICAL()

#define USB_AUDIO_EXIT_CRITICAL() OSA_EXIT_CRITICAL()
      
#define USB_AUDIO_SYNC_STAGE_0         0
#define USB_AUDIO_SYNC_STAGE_1         1
#define USB_AUDIO_SYNC_STAGE_2         2     
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
usb_status_t USB_DeviceAudioCallback(class_handle_t handle, uint32_t event, void *param);
usb_status_t USB_DeviceCallback(usb_device_handle handle, uint32_t event, void *param);

extern uint32_t DMA_GetRemainingBytes(DMA_Type *base, uint32_t channel);
extern void BOARD_SetCodecMuteUnmute(bool);
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
extern void CTIMER_CaptureInit(void);
#if ((defined(USB_DEVICE_CONFIG_LPCIP3511FS) && (USB_DEVICE_CONFIG_LPCIP3511FS > 0U)) || \
     (defined(USB_DEVICE_CONFIG_KHCI) && (USB_DEVICE_CONFIG_KHCI > 0U)))
extern void audio_fro_trim_up(void);
extern void audio_fro_trim_down(void);
#endif
extern void USB_AudioPllChange(void);
#endif

extern void AUDIO_DMA_EDMA_Start();
extern uint8_t DMA_GetValidTransfer(i2s_dma_handle_t *handle);
extern uint32_t AUDIO_GetTxDMATransferSize(void);
/*******************************************************************************
 * Variables
 ******************************************************************************/
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
#if USB_CHAT_SPEAKER_ENABLE
uint8_t audioChatPlayPacket[(AUDIO_OUT_TRANSFER_LENGTH_ONE_FRAME + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_TX_FORMAT_MAX)];
#endif
#if USB_GAME_SPEAKER_ENABLE
uint8_t audioGamePlayPacket[(AUDIO_OUT_TRANSFER_LENGTH_ONE_FRAME + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_TX_FORMAT_MAX)];
#endif

uint8_t audioPlayDataBuff[AUDIO_SPEAKER_DATA_WHOLE_BUFFER_COUNT_NORMAL * AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME];


USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t audioRecDataBuff[AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE];
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t audioRecPacket[(FS_ISO_IN_ENDP_PACKET_SIZE)];
#else
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t audioRecPacket[(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE)];
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t usbAudioFeedBackBuffer[4];
USB_RAM_ADDRESS_ALIGNMENT(4) uint8_t audioFeedBackBuffer[4];
volatile uint8_t feedbackValueUpdating;
#endif

usb_device_composite_struct_t *g_deviceAudioComposite;
volatile bool g_CodecSpeakerMuteUnmute    = false;
volatile bool g_CodecMicrophoneMuteUnmute = false;
extern usb_device_composite_struct_t g_composite;
extern uint8_t g_I2sTxStarted;
extern uint32_t g_mixerRingBufferMinalFilling;
extern uint32_t g_mixerRingBufferFilling[2];
extern uint32_t g_ringbufferThreshold;
extern HAL_AUDIO_HANDLE_DEFINE(audioTxHandle);

uint32_t g_dmaRemainingBytes = 0;
uint32_t g_dmaRemainingBytesPrev = 0;

/* used to count the number of USB audio packet between two SOF Toggle signal, SOF toggle freq is 500Hz
   For UAC1.0 device, usbIrqNumPerSOFToggle should be 4 when two audio speaker enabled and 2 when only one audio speaker enabled.
   usbIrqNumPerSOFToggle is cleared in Ctimer IRQ and accumulated in USB audio packet IRQ */
uint8_t usbIrqNumPerSOFToggle = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Audio class specific request function.
 *
 * This function handles the Audio class specific requests.
 *
 * @param handle           The USB device handle.
 * @param event            The USB device event type.
 * @param param            The parameter of the device specific request.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceAudioRequest(class_handle_t handle, uint32_t event, void *param)
{
    usb_device_control_request_struct_t *request = (usb_device_control_request_struct_t *)param;
    usb_status_t error                           = kStatus_USB_Success;
    uint8_t entity_id                            = (uint8_t)(request->setup->wIndex >> 0x08);
#if (!USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    uint16_t volume;
#endif

    switch (event)
    {
        case USB_DEVICE_AUDIO_FU_GET_CUR_MUTE_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerMute20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute20);
            }
            else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerMute20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute20);
            }
            else if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curMicrophoneMute20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneMute20);
            }
            else
            {

            }
#else

           if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curMicrophoneMute;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneMute);
            }
            else
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerMute;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute);
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_VOLUME_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerVolume20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume20);
            }
            else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerVolume20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume20);
            }
            else if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curMicrophoneVolume20;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneVolume20);
            }
            else
            {

            }
#else
            if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curMicrophoneVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneVolume);
            }
            else
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curSpeakerVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume);
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_AUTOMATIC_GAIN_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curAutomaticGain;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curAutomaticGain);
            break;
        case USB_DEVICE_AUDIO_FU_GET_CUR_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.curDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.minTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MIN_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.maxTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_MAX_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxDelay);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_VOLUME_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resVolume;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resVolume);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_BASS_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resBass;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resBass);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_MID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resMid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resMid);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_TREBLE_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.resTreble;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resTreble);
            break;
        case USB_DEVICE_AUDIO_FU_GET_RES_DELAY_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resDelay;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resDelay);
            break;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
        case USB_DEVICE_AUDIO_CS_GET_CUR_SAMPLING_FREQ_CONTROL:
            if (entity_id == USB_AUDIO_CONTROL_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
            }
            else if (entity_id == USB_AUDIO_CONTROL_CHAT_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
            }
            else if (entity_id == USB_AUDIO_CONTROL_GAME_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
            }
            else if (entity_id == USB_AUDIO_CONTROL_RECORDER_CLOCK_SOURCE_ENTITY_ID)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curRecorderSampleFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curRecorderSampleFrequency);
            }
            else
            {
                /* no action */
            }
            break;
        case USB_DEVICE_AUDIO_CS_SET_CUR_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                if (entity_id == USB_AUDIO_CONTROL_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
                {
                    request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
                }
                else if (entity_id == USB_AUDIO_CONTROL_CHAT_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
                {
                    request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
                }
                else if (entity_id == USB_AUDIO_CONTROL_GAME_SPEAKER_CLOCK_SOURCE_ENTITY_ID)
                {
                    request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency);
                }
                else if (entity_id == USB_AUDIO_CONTROL_RECORDER_CLOCK_SOURCE_ENTITY_ID)
                {
                    request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.curRecorderSampleFrequency;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curRecorderSampleFrequency);
                }
                else
                {
                    /* no action */
                }
            }
            break;
        case USB_DEVICE_AUDIO_CS_GET_CUR_CLOCK_VALID_CONTROL:
            request->buffer = &g_deviceAudioComposite->audioUnified.curClockValid;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curClockValid);
            break;
        case USB_DEVICE_AUDIO_CS_SET_CUR_CLOCK_VALID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curClockValid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curClockValid);
            }
            break;
        case USB_DEVICE_AUDIO_CS_GET_RANGE_SAMPLING_FREQ_CONTROL:
            if (USB_AUDIO_CONTROL_SPEAKER_CLOCK_SOURCE_ENTITY_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.speakerFreqControlRange;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.speakerFreqControlRange);
            }
            else if (USB_AUDIO_CONTROL_CHAT_SPEAKER_CLOCK_SOURCE_ENTITY_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.speakerFreqControlRange;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.speakerFreqControlRange);
            }
            else if (USB_AUDIO_CONTROL_GAME_SPEAKER_CLOCK_SOURCE_ENTITY_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.speakerFreqControlRange;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.speakerFreqControlRange);
            }
            else if (USB_AUDIO_CONTROL_RECORDER_CLOCK_SOURCE_ENTITY_ID == entity_id)
            {
                request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.recorderFreqControlRange;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.recorderFreqControlRange);
            }
            else
            {
                /* no action */
            }
            break;
        case USB_DEVICE_AUDIO_FU_GET_RANGE_VOLUME_CONTROL:
            request->buffer = (uint8_t *)&g_deviceAudioComposite->audioUnified.volumeControlRange;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.volumeControlRange);
            break;
#else
        case USB_DEVICE_AUDIO_EP_GET_CUR_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.curSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.curSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_MIN_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.minSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.minSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_MAX_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.maxSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.maxSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_GET_RES_SAMPLING_FREQ_CONTROL:
            request->buffer = g_deviceAudioComposite->audioUnified.resSamplingFrequency;
            request->length = sizeof(g_deviceAudioComposite->audioUnified.resSamplingFrequency);
            break;
        case USB_DEVICE_AUDIO_EP_SET_CUR_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_RES_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_MAX_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxSamplingFrequency);
            }
            break;
        case USB_DEVICE_AUDIO_EP_SET_MIN_SAMPLING_FREQ_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minSamplingFrequency;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minSamplingFrequency);
            }
            break;
#endif
        case USB_DEVICE_AUDIO_FU_SET_CUR_VOLUME_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (request->isSetup == 1U)
            {
                if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = g_deviceAudioComposite->audioUnified.curSpeakerVolume20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume20);
                }
                else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = g_deviceAudioComposite->audioUnified.curSpeakerVolume20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume20);
                }
                else if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = g_deviceAudioComposite->audioUnified.curMicrophoneVolume20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneVolume20);
                }
                else
                {
                }
            }
            else
            {
                if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    g_deviceAudioComposite->audioUnified.codecSpeakerTask |= VOLUME_CHANGE_TASK;
                }
                else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    g_deviceAudioComposite->audioUnified.codecSpeakerTask |= VOLUME_CHANGE_TASK;
                }
                else if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= VOLUME_CHANGE_TASK;
                }
                else
                {

                }
            }
#else
            if (request->isSetup == 1U)
            {
                if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = g_deviceAudioComposite->audioUnified.curMicrophoneVolume;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneVolume);
                }
                else
                {
                    request->buffer = g_deviceAudioComposite->audioUnified.curSpeakerVolume;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerVolume);
                }
            }
            else
            {
                if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    volume = (uint16_t)((uint16_t)g_deviceAudioComposite->audioUnified.curMicrophoneVolume[1] << 8U);
                    volume |= (uint8_t)(g_deviceAudioComposite->audioUnified.curMicrophoneVolume[0]);
                    g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= VOLUME_CHANGE_TASK;
                }
                else
                {
                    volume = (uint16_t)((uint16_t)g_deviceAudioComposite->audioUnified.curSpeakerVolume[1] << 8U);
                    volume |= (uint8_t)(g_deviceAudioComposite->audioUnified.curSpeakerVolume[0]);
                    g_deviceAudioComposite->audioUnified.codecSpeakerTask |= VOLUME_CHANGE_TASK;
                }
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_MUTE_CONTROL:
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
            if (request->isSetup == 1U)
            {
                if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = &g_deviceAudioComposite->audioUnified.curSpeakerMute20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute20);
                }
                else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = &g_deviceAudioComposite->audioUnified.curSpeakerMute20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute20);
                }
                if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = &g_deviceAudioComposite->audioUnified.curMicrophoneMute20;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneMute20);
                }
                else
                {
                }
            }
            else
            {
                /* trigger task, only adjust speaker mute/unmute practically */
                if (USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    if (g_deviceAudioComposite->audioUnified.curSpeakerMute20)
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= MUTE_CODEC_TASK;
                    }
                    else
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= UNMUTE_CODEC_TASK;
                    }
                }
                else if (USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    if (g_deviceAudioComposite->audioUnified.curSpeakerMute20)
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= MUTE_CODEC_TASK;
                    }
                    else
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= UNMUTE_CODEC_TASK;
                    }
                }
                else if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    if (g_deviceAudioComposite->audioUnified.curSpeakerMute20)
                    {
                        g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= MUTE_CODEC_TASK;
                    }
                    else
                    {
                        g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= UNMUTE_CODEC_TASK;
                    }
                }
                else
                {
                }
            }
#else
            if (request->isSetup == 1U)
            {
                if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    request->buffer = &g_deviceAudioComposite->audioUnified.curMicrophoneMute;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curMicrophoneMute);
                }
                else
                {
                    request->buffer = &g_deviceAudioComposite->audioUnified.curSpeakerMute;
                    request->length = sizeof(g_deviceAudioComposite->audioUnified.curSpeakerMute);
                }
            }
            else
            {
                if (USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID == entity_id)
                {
                    if (g_deviceAudioComposite->audioUnified.curMicrophoneMute)
                    {
                        g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= MUTE_CODEC_TASK;
                    }
                    else
                    {
                        g_deviceAudioComposite->audioUnified.codecMicrophoneTask |= UNMUTE_CODEC_TASK;
                    }
                }
                else
                {
                    if (g_deviceAudioComposite->audioUnified.curSpeakerMute)
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= MUTE_CODEC_TASK;
                    }
                    else
                    {
                        g_deviceAudioComposite->audioUnified.codecSpeakerTask |= UNMUTE_CODEC_TASK;
                    }
                }
            }
#endif
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_AUTOMATIC_GAIN_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.curAutomaticGain;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curAutomaticGain);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_CUR_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.curDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.curDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.minTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MIN_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.minDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.minDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.maxTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_MAX_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.maxDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.maxDelay);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_VOLUME_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resVolume;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resVolume);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_BASS_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resBass;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resBass);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_MID_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resMid;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resMid);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_TREBLE_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = &g_deviceAudioComposite->audioUnified.resTreble;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resTreble);
            }
            break;
        case USB_DEVICE_AUDIO_FU_SET_RES_DELAY_CONTROL:
            if (request->isSetup == 1U)
            {
                request->buffer = g_deviceAudioComposite->audioUnified.resDelay;
                request->length = sizeof(g_deviceAudioComposite->audioUnified.resDelay);
            }
            break;
        default:
            error = kStatus_USB_InvalidRequest;
            break;
    }
    return error;
}

/* The USB_AudioSpeakerBufferSpaceUsed() function gets the used speaker ringbuffer size */
uint32_t USB_AudioSpeakerBufferSpaceUsed(void)
{
    uint64_t write_count = 0U;
    uint64_t read_count  = 0U;

    write_count = (uint64_t)((((uint64_t)g_deviceAudioComposite->audioUnified.audioSpeakerWriteDataCount[1]) << 32U) |
                             g_deviceAudioComposite->audioUnified.audioSpeakerWriteDataCount[0]);
    read_count  = (uint64_t)((((uint64_t)g_deviceAudioComposite->audioUnified.audioSpeakerReadDataCount[1]) << 32U) |
                            g_deviceAudioComposite->audioUnified.audioSpeakerReadDataCount[0]);

    if (write_count >= read_count)
    {
        return (uint32_t)(write_count - read_count);
    }
    else
    {
        return 0;
    }
}

/* The USB_AudioRecorderBufferSpaceUsed() function gets the reserved recorder ringbuffer size */
uint32_t USB_AudioRecorderBufferSpaceUsed(void)
{
    if (g_deviceAudioComposite->audioUnified.tdWriteNumberRec > g_deviceAudioComposite->audioUnified.tdReadNumberRec)
    {
        g_deviceAudioComposite->audioUnified.recorderReservedSpace =
            g_deviceAudioComposite->audioUnified.tdWriteNumberRec -
            g_deviceAudioComposite->audioUnified.tdReadNumberRec;
    }
    else
    {
        g_deviceAudioComposite->audioUnified.recorderReservedSpace =
            g_deviceAudioComposite->audioUnified.tdWriteNumberRec +
            AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE -
            g_deviceAudioComposite->audioUnified.tdReadNumberRec;
    }
    return g_deviceAudioComposite->audioUnified.recorderReservedSpace;
}

/* The USB_AudioRecorderGetBuffer() function gets audioRecPacket from the audioRecDataBuff in every callback*/
void USB_AudioRecorderGetBuffer(uint8_t *buffer, uint32_t size)
{
    while (size)
    {
        *buffer = audioRecDataBuff[g_deviceAudioComposite->audioUnified.tdReadNumberRec];
        g_deviceAudioComposite->audioUnified.tdReadNumberRec++;
        buffer++;
        size--;

        if (g_deviceAudioComposite->audioUnified.tdReadNumberRec >=
            AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE)
        {
            g_deviceAudioComposite->audioUnified.tdReadNumberRec = 0;
        }
    }
}


#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#else
void USB_DeviceCalculateFeedback(void)
{
    volatile static uint64_t totalFrameValue = 0U;
    volatile static uint32_t frameDistance   = 0U;
    volatile static uint32_t feedbackValue   = 0U;

    uint32_t audioSpeakerUsedSpace = 0U;

    /* feedback interval is AUDIO_CALCULATE_Ff_INTERVAL */
    if (USB_SPEED_HIGH == g_deviceAudioComposite->speed)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0) /* high speed, feedback interval is AUDIO_CALCULATE_Ff_INTERVAL ms */
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount !=
            AUDIO_CALCULATE_Ff_INTERVAL *
                (AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME / g_deviceAudioComposite->audioUnified.audioPlayTransferSize))
#else
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount != AUDIO_CALCULATE_Ff_INTERVAL)
#endif
        {
            g_deviceAudioComposite->audioUnified.speakerIntervalCount++;
            return;
        }
    }
    else /* full speed, feedback interval is AUDIO_CALCULATE_Ff_INTERVAL ms */
    {
        if (g_deviceAudioComposite->audioUnified.speakerIntervalCount != AUDIO_CALCULATE_Ff_INTERVAL)
        {
            g_deviceAudioComposite->audioUnified.speakerIntervalCount++;
            return;
        }
    }

    if (0U == g_deviceAudioComposite->audioUnified.firstCalculateFeedback)
    {
        g_deviceAudioComposite->audioUnified.speakerIntervalCount = 0;
        g_deviceAudioComposite->audioUnified.currentFrameCount    = 0;
        g_deviceAudioComposite->audioUnified.audioSendCount[0]    = 0;
        g_deviceAudioComposite->audioUnified.audioSendCount[1]    = 0;
        totalFrameValue                                           = 0;
        frameDistance                                             = 0;
        feedbackValue                                             = 0;
        USB_DeviceClassGetCurrentFrameCount(CONTROLLER_ID,
                                            (uint32_t *)&g_deviceAudioComposite->audioUnified.lastFrameCount);
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
#if (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
        if (USB_SPEED_HIGH == g_deviceAudioComposite->speed)
        {
            g_deviceAudioComposite->audioUnified.lastFrameCount += s_microFrameCountIp3511HS;
        }
#endif
#endif
        g_deviceAudioComposite->audioUnified.firstCalculateFeedback = 1U;
        return;
    }
    g_deviceAudioComposite->audioUnified.speakerIntervalCount = 0;
    USB_DeviceClassGetCurrentFrameCount(CONTROLLER_ID,
                                        (uint32_t *)&g_deviceAudioComposite->audioUnified.currentFrameCount);
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
#if (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
    if (USB_SPEED_HIGH == g_deviceAudioComposite->speed)
    {
        g_deviceAudioComposite->audioUnified.currentFrameCount += s_microFrameCountIp3511HS;
    }
#endif
#endif
    frameDistance = ((g_deviceAudioComposite->audioUnified.currentFrameCount + USB_DEVICE_MAX_FRAME_COUNT + 1U -
                      g_deviceAudioComposite->audioUnified.lastFrameCount) &
                     USB_DEVICE_MAX_FRAME_COUNT);
    g_deviceAudioComposite->audioUnified.lastFrameCount = g_deviceAudioComposite->audioUnified.currentFrameCount;

    totalFrameValue += frameDistance;

    if (1U == g_deviceAudioComposite->audioUnified.stopFeedbackUpdate)
    {
        return;
    }

    if (1U == g_deviceAudioComposite->audioUnified.feedbackDiscardFlag)
    {
        if (0 != g_deviceAudioComposite->audioUnified.feedbackDiscardTimes)
        {
            g_deviceAudioComposite->audioUnified.feedbackDiscardTimes--;
            if (0 != g_deviceAudioComposite->audioUnified.lastFeedbackValue)
            {
                AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, g_deviceAudioComposite->audioUnified.lastFeedbackValue);
            }
            return;
        }
        else
        {
            g_deviceAudioComposite->audioUnified.feedbackDiscardFlag = 0;
        }
    }

    if (USB_SPEED_HIGH == g_deviceAudioComposite->speed)
    {
        feedbackValue = (uint32_t)((((((uint64_t)g_deviceAudioComposite->audioUnified.audioSendCount[1]) << 32U) |
                                     g_deviceAudioComposite->audioUnified.audioSendCount[0])) *
                                   1024UL * 8UL / totalFrameValue /
                                   ((uint64_t)(AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)));
    }
    else
    {
        feedbackValue =
            (uint32_t)((((((uint64_t)g_deviceAudioComposite->audioUnified.audioSendCount[1]) << 32U) |
                         g_deviceAudioComposite->audioUnified.audioSendCount[0])) *
                       1024UL / totalFrameValue / ((uint64_t)(AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)));
    }

    audioSpeakerUsedSpace = USB_AudioSpeakerBufferSpaceUsed();
    if (audioSpeakerUsedSpace <= (g_deviceAudioComposite->audioUnified.audioPlayTransferSize *
                                  USB_AUDIO_PLAY_BUFFER_FEEDBACK_TOLERANCE_THRESHOLD))
    {
        feedbackValue += AUDIO_ADJUST_MIN_STEP;
    }

    if ((audioSpeakerUsedSpace + (g_deviceAudioComposite->audioUnified.audioPlayTransferSize *
                                  USB_AUDIO_PLAY_BUFFER_FEEDBACK_TOLERANCE_THRESHOLD)) >=
        g_deviceAudioComposite->audioUnified.audioPlayBufferSize)
    {
        feedbackValue -= AUDIO_ADJUST_MIN_STEP;
    }

    g_deviceAudioComposite->audioUnified.lastFeedbackValue = feedbackValue;
    AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, feedbackValue);
}

/* The USB_RecorderDataMatch() function increase/decrease the adjusted packet interval according to the reserved
 * ringbuffer size */
uint32_t USB_RecorderDataMatch(uint32_t reservedspace)
{
    uint32_t epPacketSize = 0;
    if (reservedspace >=
        AUDIO_BUFFER_UPPER_LIMIT(AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE))
    {
        epPacketSize = FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_SIZE * AUDIO_IN_FORMAT_CHANNELS;
    }
    else if ((reservedspace >=
              AUDIO_BUFFER_LOWER_LIMIT(AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE)) &&
             (reservedspace <
              AUDIO_BUFFER_UPPER_LIMIT(AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE)))
    {
        epPacketSize = FS_ISO_IN_ENDP_PACKET_SIZE;
    }
    else if (reservedspace <
             AUDIO_BUFFER_LOWER_LIMIT(AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE))
    {
        epPacketSize = FS_ISO_IN_ENDP_PACKET_SIZE - AUDIO_IN_FORMAT_SIZE * AUDIO_IN_FORMAT_CHANNELS;
    }
    else
    {
    }
    return epPacketSize;
}
#endif

/*!
 * @brief device Audio callback function.
 *
 * This function handle the Audio class specified event.
 * @param handle          The USB class  handle.
 * @param event           The USB device event type.
 * @param param           The parameter of the class specific event.
 * @return kStatus_USB_Success or error.
 */
usb_status_t USB_DeviceAudioCompositeCallback(class_handle_t handle, uint32_t event, void *param)
{
    usb_status_t error = kStatus_USB_InvalidRequest;
    usb_device_endpoint_callback_message_struct_t *ep_cb_param;
    ep_cb_param           = (usb_device_endpoint_callback_message_struct_t *)param;
    uint32_t epPacketSize = FS_ISO_IN_ENDP_PACKET_SIZE;
    
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    UsbFeedback_t *usbFeedback = NULL;
    uint32_t epPacketSize = FS_ISO_IN_ENDP_PACKET_SIZE;
    uint32_t fbPacketSize = (USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE : FS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
    uint8_t feedbackEp = 0;
#endif

    usb_speaker_instance_t speakerInstance;
    uint8_t *audioPlayPacket = NULL;
    uint8_t streamEp = 0;

#if USB_CHAT_SPEAKER_ENABLE
    if (handle == g_deviceAudioComposite->audioUnified.audioChatSpeakerHandle) {
        speakerInstance = kUSB_SPEAKER_INSTANCE_Chat;
        audioPlayPacket = (uint8_t *)audioChatPlayPacket;
        streamEp = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
        usbFeedback = &g_UsbChatSpeaker_FeedbackSettings;
        feedbackEp = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT;
#endif
    }
#endif
    
#if USB_GAME_SPEAKER_ENABLE
    if (handle == g_deviceAudioComposite->audioUnified.audioGameSpeakerHandle) {
        speakerInstance = kUSB_SPEAKER_INSTANCE_Game;
        audioPlayPacket = (uint8_t *)audioGamePlayPacket;
        streamEp = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
        usbFeedback = &g_UsbGameSpeaker_FeedbackSettings;
        feedbackEp = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT;
#endif
    }
#endif    
    
    switch (event)
    {
        case kUSB_DeviceAudioEventStreamSendResponse:
            /* endpoint callback length is USB_CANCELLED_TRANSFER_LENGTH (0xFFFFFFFFU) when transfer is canceled */
            if ((g_deviceAudioComposite->audioUnified.attach) &&
                (ep_cb_param->length != (USB_CANCELLED_TRANSFER_LENGTH)))
            {
                USB_LatencyDebug_SendingAudioToHost(true);
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
                if (0)
                {
                }
#else
                if (ep_cb_param->length == ((USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE :
                                                                                    FS_ISO_FEEDBACK_ENDP_PACKET_SIZE))
                {
                    if (!feedbackValueUpdating)
                    {
                        *((uint32_t *)&usbAudioFeedBackBuffer[0]) = *((uint32_t *)&audioFeedBackBuffer[0]);
                    }
                    error =
                        USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
                                            USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT, usbAudioFeedBackBuffer,
                                            (USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE :
                                                                                    FS_ISO_FEEDBACK_ENDP_PACKET_SIZE);
                }
#endif
                else
                {
                    if (g_deviceAudioComposite->audioUnified.startRec == 0)
                    {
                        g_deviceAudioComposite->audioUnified.startRec = 1;
                    }
                    if ((g_deviceAudioComposite->audioUnified.tdWriteNumberRec >=
                         AUDIO_RECORDER_DATA_WHOLE_BUFFER_COUNT_NORMAL * FS_ISO_IN_ENDP_PACKET_SIZE / 2) &&
                        (g_deviceAudioComposite->audioUnified.startRecHalfFull == 0))
                    {
                        g_deviceAudioComposite->audioUnified.startRecHalfFull = 1;
                    }
                    

                    audio_ReadBuffer(audioRecPacket, epPacketSize / AUDIO_RX_FORMAT_SIZE);
                    error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
                                                    USB_AUDIO_RECORDER_STREAM_ENDPOINT, &audioRecPacket[0],
                                                    FS_ISO_IN_ENDP_PACKET_SIZE);

                }
                USB_LatencyDebug_SendingAudioToHost(false);
            }
            break;
        case kUSB_DeviceAudioEventStreamRecvResponse:
            /* endpoint callback length is USB_CANCELLED_TRANSFER_LENGTH (0xFFFFFFFFU) when transfer is canceled */
            if ((g_deviceAudioComposite->audioUnified.attach) &&
                (ep_cb_param->length != (USB_CANCELLED_TRANSFER_LENGTH)))
            {
                USB_LatencyDebug_ReceivingAudioFromHost(true);
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
#if (defined(USB_DEVICE_CONFIG_EHCI) && (USB_DEVICE_CONFIG_EHCI > 0U)) || \
    (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
                if (USB_SPEED_HIGH ==
                    g_deviceAudioComposite->speed) /* high speed and audio 2.0, use low latency solution  */
                {
                    if (g_deviceAudioComposite->audioUnified.tdWriteNumberPlay >=
                        (g_deviceAudioComposite->audioUnified.audioPlayTransferSize *
                         AUDIO_CLASS_2_0_HS_LOW_LATENCY_TRANSFER_COUNT))
                    {
                        g_deviceAudioComposite->audioUnified.startPlayFlag = 1;
                    }
#if (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
#if (1U == HS_ISO_OUT_ENDP_INTERVAL)
                    if (s_microFrameCountIp3511HS < 7U)
                    {
                        s_microFrameCountIp3511HS++;
                    }
                    else
                    {
                        s_microFrameCountIp3511HS = 0U;
                    }
#elif (2U == HS_ISO_OUT_ENDP_INTERVAL)
                    if (s_microFrameCountIp3511HS < 6U)
                    {
                        s_microFrameCountIp3511HS += 2U;
                    }
                    else
                    {
                        s_microFrameCountIp3511HS = 0U;
                    }
#elif (3U == HS_ISO_OUT_ENDP_INTERVAL)
                    if (s_microFrameCountIp3511HS < 4U)
                    {
                        s_microFrameCountIp3511HS += 4U;
                    }
                    else
                    {
                        s_microFrameCountIp3511HS = 0U;
                    }
#else
                    s_microFrameCountIp3511HS = 0;
#endif
#endif
                }
                else
                {
                    if ((g_deviceAudioComposite->audioUnified.tdWriteNumberPlay >=
                         (g_deviceAudioComposite->audioUnified.audioPlayBufferSize / 2U)) &&
                        (g_deviceAudioComposite->audioUnified.startPlayFlag == 0))
                    {
                        g_deviceAudioComposite->audioUnified.startPlayFlag = 1;
                    }
                }
#else
                if ((g_deviceAudioComposite->audioUnified.tdWriteNumberPlay >=
                     (g_deviceAudioComposite->audioUnified.audioPlayBufferSize / 2U)) &&
                    (g_deviceAudioComposite->audioUnified.startPlayFlag == 0))
                {
                    g_deviceAudioComposite->audioUnified.startPlayFlag = 1;
                }
#endif /* USB_DEVICE_CONFIG_EHCI, USB_DEVICE_CONFIG_LPCIP3511HS */
#else
                if ((g_deviceAudioComposite->audioUnified.tdWriteNumberPlay >=
                     (g_deviceAudioComposite->audioUnified.audioPlayBufferSize / 2U)) &&
                    (g_deviceAudioComposite->audioUnified.startPlayFlag == 0))
                {
                    g_deviceAudioComposite->audioUnified.startPlayFlag = 1;
                }
#endif /* USB_DEVICE_CONFIG_AUDIO_CLASS_2_0 */
                //USB_AudioSpeakerPutBuffer(audioPlayPacket, ep_cb_param->length);
                AUDIO_MIXER_WriteSamples(speakerInstance, audioPlayPacket,
                                             (ep_cb_param->length / AUDIO_TX_FORMAT_BYTES),
                                             AUDIO_TX_FORMAT_BYTES);
    
                g_deviceAudioComposite->audioUnified.usbRecvCount += ep_cb_param->length;
                g_deviceAudioComposite->audioUnified.usbRecvTimes++;
                error = USB_DeviceAudioRecv(handle, streamEp, &audioPlayPacket[0],
                                            g_deviceAudioComposite->audioUnified.currentStreamOutMaxPacketSize);
                USB_LatencyDebug_ReceivingAudioFromHost(false);
                usbIrqNumPerSOFToggle ++;
            }
            break;

        default:
            if (param && (event > 0xFF))
            {
                error = USB_DeviceAudioRequest(handle, event, param);
            }
            break;
    }

    return error;
}

#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#if ((defined(USB_DEVICE_CONFIG_LPCIP3511FS) && (USB_DEVICE_CONFIG_LPCIP3511FS > 0U)) || \
     (defined(USB_DEVICE_CONFIG_KHCI) && (USB_DEVICE_CONFIG_KHCI > 0U)))
void CTIMER_SOF_TOGGLE_HANDLER_FRO(uint32_t i)
{
    uint32_t currentCtCap = 0, pllCountPeriod = 0;
    //uint32_t usedSpace      = 0;
    static int32_t pllCount = 0, pllDiff = 0;
    static int32_t err, abs_err;
    uint32_t up_change                 = 0;
    uint32_t down_change               = 0;
    //static uint32_t delay_adj_up       = 0;
    //static uint32_t delay_adj_down     = 0;
    //static uint32_t FroPreUsbRecvCount = 0U;

    if (CTIMER_GetStatusFlags(CTIMER1) & (1 << 4U))
    {
        /* Clear interrupt flag.*/
        CTIMER_ClearStatusFlags(CTIMER1, (1 << 4U));
    }

    if (g_composite.audioUnified.froTrimIntervalCount != AUDIO_FRO_ADJUST_INTERVAL)
    {
        g_composite.audioUnified.froTrimIntervalCount++;
        return;
    }

    g_composite.audioUnified.froTrimIntervalCount = 1;
    currentCtCap                                  = CTIMER1->CR[0];
    pllCountPeriod                                = currentCtCap - g_composite.audioUnified.usbFroTicksPrev;
    g_composite.audioUnified.usbFroTicksPrev      = currentCtCap;
    pllCount                                      = pllCountPeriod;

    if (g_composite.audioUnified.attach)
    {
        if (abs(pllCount - AUDIO_FRO_USB_SOF_INTERVAL_TICK_COUNT) < AUDIO_FRO_USB_SOF_INTERVAL_VALID_DEVIATION)
        {
            pllDiff = pllCount - g_composite.audioUnified.usbFroTicksEma;
            g_composite.audioUnified.usbFroTickEmaFrac += (pllDiff % 8);
            g_composite.audioUnified.usbFroTicksEma += (pllDiff / 8) + g_composite.audioUnified.usbFroTickEmaFrac / 8;
            g_composite.audioUnified.usbFroTickEmaFrac = (g_composite.audioUnified.usbFroTickEmaFrac % 8);

            err     = g_composite.audioUnified.usbFroTicksEma - AUDIO_FRO_USB_SOF_INTERVAL_TICK_COUNT;
            abs_err = abs(err);
            if (abs_err >= g_composite.audioUnified.usbFroTickBasedPrecision)
            {
                if (err > 0)
                {
                    down_change = 1;
                }
                else
                {
                    up_change = 1;
                }
            }
        }

        if (down_change)
        {
            audio_fro_trim_down();
        }
        if (up_change)
        {
            audio_fro_trim_up();
        }
    }
}
#endif /* USB_DEVICE_CONFIG_LPCIP3511FS  USB_DEVICE_CONFIG_KHCI*/
void CTIMER_SOF_TOGGLE_HANDLER_PLL(uint32_t i)
{
    uint32_t currentCtCap = 0, pllCountPeriod = 0, pll_change = 0;
    static int32_t pllCount = 0, pllDiff = 0;
    static int32_t err, abs_err;
    uint32_t initialRemaining = 0;
    
    usbIrqNumPerSOFToggle = 0;
    
    if (CTIMER_GetStatusFlags(CTIMER0) & (1 << 4U))
    {
        /* Clear interrupt flag.*/
        CTIMER_ClearStatusFlags(CTIMER0, (1 << 4U));
    }

    if (g_composite.audioUnified.speakerIntervalCount != AUDIO_PLL_ADJUST_INTERVAL)
    {
        g_composite.audioUnified.speakerIntervalCount++;
        return;
    }

    g_composite.audioUnified.speakerIntervalCount = 1;
    currentCtCap                                  = CTIMER0->CR[0];
    pllCountPeriod                                = currentCtCap - g_composite.audioUnified.audioPllTicksPrev;
    g_composite.audioUnified.audioPllTicksPrev    = currentCtCap;
    pllCount                                      = pllCountPeriod;

    AUDIO_LatencyDebug_ReceivingAudioFromI2s(true);
    if (g_composite.audioUnified.attach)
    {
        /* The tick deviation within 2ms is less than 384 (15.625us) */
        if (abs(pllCount - AUDIO_PLL_USB_SOF_INTERVAL_TICK_COUNT) < AUDIO_PLL_USB_SOF_INTERVAL_VALID_DEVIATION)
        {
            /* the initial value of audioPllTicksEma is AUDIO_PLL_USB_SOF_INTERVAL_TICK_COUNT */
            pllDiff = pllCount - g_composite.audioUnified.audioPllTicksEma;
            g_composite.audioUnified.audioPllTickEmaFrac += (pllDiff % 8);
            g_composite.audioUnified.audioPllTicksEma +=
                (pllDiff / 8) + g_composite.audioUnified.audioPllTickEmaFrac / 8;
            g_composite.audioUnified.audioPllTickEmaFrac = (g_composite.audioUnified.audioPllTickEmaFrac % 8);          

            err     = g_composite.audioUnified.audioPllTicksEma - AUDIO_PLL_USB_SOF_INTERVAL_TICK_COUNT;
            abs_err = abs(err);
            if (abs_err >= g_composite.audioUnified.audioPllTickBasedPrecision)
            {
                if (err > 0)
                {
                    g_composite.audioUnified.curAudioPllFrac -=
                        abs_err * 100 / g_composite.audioUnified.audioPllTickBasedPrecision;
                }
                else
                {
                    g_composite.audioUnified.curAudioPllFrac +=
                        abs_err * 100 / g_composite.audioUnified.audioPllTickBasedPrecision;
                }
                pll_change = 1;
                
            }
            
           /* adjust ringbuffer filling level to initial ringbuffer remaining +/- 0.42ms, 
              the initial ringbuffer remaining is the ringbuffer remaining value when the first Ctimer interrupt occurs after starting DMA transfer.       
              Before starting DMA transfer, there are two data copy processes. One is to copy the USB audio packet to
              the corresponding speaker's ringbuffer, and the other is to mix the audio data in the speaker interface
              ringbuffer and copy it to the source address of the DMA transfer. These two process may take 0.5ms.
              so the initial ringbuffer remaining value should be around (g_ringbufferThreshold - 0.5ms) */
            
            /* Assume initial ringbuffer remaining = g_ringbufferThreshold - 0.5ms,
               0.5ms corresponds to 48 samples, because they are two channel speakers. */
            initialRemaining = g_ringbufferThreshold - 48 * I2S_TX_WORD_LENGTH_BYTES;
            
            uint8_t minimalIndex = 0;
            uint8_t validTransferNum = 0;
            
            if(g_I2sTxStarted)
            {
                validTransferNum = DMA_GetValidTransfer((i2s_dma_handle_t *)((uint8_t *)&audioTxHandle[0] + 8));
                if(validTransferNum > 0)
                {
                    g_dmaRemainingBytes = DMA_GetRemainingBytes(DMA0, DEMO_I2S_TX_CHANNEL) * I2S_TX_WORD_LENGTH_BYTES + (validTransferNum -1) * AUDIO_GetTxDMATransferSize(); 
                }
                else
                {
                    g_dmaRemainingBytes = DMA_GetRemainingBytes(DMA0, DEMO_I2S_TX_CHANNEL) * I2S_TX_WORD_LENGTH_BYTES;
                }
                
                /* The ringbuffer remaining changes slowly and does not change suddenly. 
                If it changes suddenly and greatly, it means that the DMA completion interrupt
                may be occurring. The calculation of the remaining DMA bytes differs
                by a DMA transfer length and needs to be compensated manually. */
                if(g_dmaRemainingBytesPrev != 0)
                {
                    if((g_dmaRemainingBytes > g_dmaRemainingBytesPrev) &&  (g_dmaRemainingBytes - g_dmaRemainingBytesPrev> 100 ))
                    {
                        g_dmaRemainingBytes = g_dmaRemainingBytes - AUDIO_GetTxDMATransferSize();
                    }
                    else if((g_dmaRemainingBytes < g_dmaRemainingBytesPrev) && (g_dmaRemainingBytesPrev - g_dmaRemainingBytes > 100))
                    {
                        g_dmaRemainingBytes = g_dmaRemainingBytes + AUDIO_GetTxDMATransferSize();
                    }
                }
                else
                {
                    g_dmaRemainingBytesPrev = g_dmaRemainingBytes;
                }
            }
            else
            {
                g_dmaRemainingBytesPrev = 0;
            }
                     
            uint32_t minimalFilling = AUDIO_MIXER_GetMininalFillingLevel(&minimalIndex);           
            g_mixerRingBufferMinalFilling = minimalFilling + g_dmaRemainingBytes;              
          
            g_mixerRingBufferFilling[0] = AUDIO_MIXER_GetBufferFilling(0) + g_dmaRemainingBytes;
            g_mixerRingBufferFilling[1] = AUDIO_MIXER_GetBufferFilling(1) + g_dmaRemainingBytes;
                
                
            if(AUDIO_MIXER_GetActiveSpeakerCount() == USB_AUDIO_SYNC_STAGE_1)
            {               
               /* keep the ringbuffer remaining  > (initial ringbuffer remaining - 0.42ms), 40/96 = 0.42ms */
                if(g_mixerRingBufferMinalFilling < initialRemaining - (40 * I2S_TX_WORD_LENGTH_BYTES))
                {
                    g_composite.audioUnified.curAudioPllFrac -= 10000;
                }
                /* keep the ringbuffer remaining  < (initial ringbuffer remaining + 0.42ms) */
                if(g_mixerRingBufferMinalFilling > initialRemaining + (40 * I2S_TX_WORD_LENGTH_BYTES))
                {
                    g_composite.audioUnified.curAudioPllFrac += 10000;
                }
                
            }
            else if(AUDIO_MIXER_GetActiveSpeakerCount() == USB_AUDIO_SYNC_STAGE_2)
            {
                /* two speakers, adjust the minil ringbuffer remaining  > (initial ringbuffer remaining - 0.42ms) */
                if(g_mixerRingBufferMinalFilling < initialRemaining - (40 * I2S_TX_WORD_LENGTH_BYTES))
                {
                    g_composite.audioUnified.curAudioPllFrac -= 10000;
                }
                /* adjust the minil ringbuffer remaining  < (initial ringbuffer remaining + 0.42ms)*/
                if(g_mixerRingBufferMinalFilling > initialRemaining + (40 * I2S_TX_WORD_LENGTH_BYTES))
                {
                    g_composite.audioUnified.curAudioPllFrac += 10000;
                }
              
            }
            else
            {
                g_mixerRingBufferMinalFilling = 0;
                g_mixerRingBufferFilling[0] = 0;
                g_mixerRingBufferFilling[1] = 0;
                g_dmaRemainingBytesPrev = 0;
            }
        }

        if (pll_change)
        {
            USB_AudioPllChange();
        }
    }
    AUDIO_LatencyDebug_ReceivingAudioFromI2s(false);
    
}
#endif /* USB_DEVICE_AUDIO_USE_SYNC_MODE */

/* The USB_DeviceAudioRecorderStatusReset() function resets the audio recorder status to the initialized status */
void USB_DeviceAudioRecorderStatusReset(void)
{
    g_deviceAudioComposite->audioUnified.startRec              = 0;
    g_deviceAudioComposite->audioUnified.startRecHalfFull      = 0;
    g_deviceAudioComposite->audioUnified.audioRecvCount        = 0;
    g_deviceAudioComposite->audioUnified.usbSendTimes          = 0;
    g_deviceAudioComposite->audioUnified.tdWriteNumberRec      = 0;
    g_deviceAudioComposite->audioUnified.tdReadNumberRec       = 0;
    g_deviceAudioComposite->audioUnified.recorderReservedSpace = 0;
}

/* The USB_DeviceAudioSpeakerStatusReset() function resets the audio speaker status to the initialized status */
void USB_DeviceAudioSpeakerStatusReset(void)
{
    g_deviceAudioComposite->audioUnified.startPlayFlag                 = 0;
    g_deviceAudioComposite->audioUnified.startPlayHalfFull             = 0;
    g_deviceAudioComposite->audioUnified.tdReadNumberPlay              = 0;
    g_deviceAudioComposite->audioUnified.tdWriteNumberPlay             = 0;
    g_deviceAudioComposite->audioUnified.audioSpeakerReadDataCount[0]  = 0;
    g_deviceAudioComposite->audioUnified.audioSpeakerReadDataCount[1]  = 0;
    g_deviceAudioComposite->audioUnified.audioSpeakerWriteDataCount[0] = 0;
    g_deviceAudioComposite->audioUnified.audioSpeakerWriteDataCount[1] = 0;
    g_deviceAudioComposite->audioUnified.audioSendCount[0]             = 0;
    g_deviceAudioComposite->audioUnified.audioSendCount[1]             = 0;
    g_deviceAudioComposite->audioUnified.usbRecvCount                  = 0;
    g_deviceAudioComposite->audioUnified.lastAudioSendCount            = 0;
    g_deviceAudioComposite->audioUnified.audioSendTimes                = 0;
    g_deviceAudioComposite->audioUnified.usbRecvTimes                  = 0;
    g_deviceAudioComposite->audioUnified.speakerIntervalCount          = 0;
    g_deviceAudioComposite->audioUnified.speakerReservedSpace          = 0;
    g_deviceAudioComposite->audioUnified.timesFeedbackCalculate        = 0;
    g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput        = 0;
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
    g_deviceAudioComposite->audioUnified.audioPllTicksPrev = 0U;
#if ((defined(USB_DEVICE_CONFIG_LPCIP3511FS) && (USB_DEVICE_CONFIG_LPCIP3511FS > 0U)) || \
     (defined(USB_DEVICE_CONFIG_KHCI) && (USB_DEVICE_CONFIG_KHCI > 0U)))
    g_deviceAudioComposite->audioUnified.usbFroTicksPrev          = 0U;
    g_deviceAudioComposite->audioUnified.usbFroTicksEma           = AUDIO_FRO_USB_SOF_INTERVAL_TICK_COUNT;
    g_deviceAudioComposite->audioUnified.usbFroTickEmaFrac        = 0U;
    g_deviceAudioComposite->audioUnified.usbFroTickBasedPrecision = AUDIO_USB_FRO_TRIM_TICK_BASED_PRECISION;
#endif
    g_deviceAudioComposite->audioUnified.audioPllTicksEma           = AUDIO_PLL_USB_SOF_INTERVAL_TICK_COUNT;
    g_deviceAudioComposite->audioUnified.audioPllTickEmaFrac        = 0U;
    g_deviceAudioComposite->audioUnified.audioPllTickBasedPrecision = AUDIO_PLL_FRACTION_TICK_BASED_PRECISION;
    g_deviceAudioComposite->audioUnified.stopDataLengthAudioAdjust  = 0U;
#else
    g_deviceAudioComposite->audioUnified.firstCalculateFeedback = 0;
    g_deviceAudioComposite->audioUnified.lastFrameCount         = 0;
    g_deviceAudioComposite->audioUnified.currentFrameCount      = 0;
    g_deviceAudioComposite->audioUnified.feedbackDiscardFlag    = 0U;
    g_deviceAudioComposite->audioUnified.feedbackDiscardTimes   = AUDIO_SPEAKER_FEEDBACK_DISCARD_COUNT;

    /* use the last saved feedback value */
    if (g_deviceAudioComposite->audioUnified.lastFeedbackValue)
    {
        AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, g_deviceAudioComposite->audioUnified.lastFeedbackValue);
    }
#endif
}

/*!
 * @brief Audio set configuration function.
 *
 * This function sets configuration for msc class.
 *
 * @param handle The Audio class handle.
 * @param configure The Audio class configure index.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceAudioCompositeSetConfigure(class_handle_t handle, uint8_t configure)
{
    if (USB_COMPOSITE_CONFIGURE_INDEX == configure)
    {
        g_deviceAudioComposite->audioUnified.attach = 1U;
    }
    return kStatus_USB_Success;
}

usb_status_t USB_DeviceAudioRecorderSetInterface(class_handle_t handle, uint8_t interface, uint8_t alternateSetting)
{
    usb_status_t error = kStatus_USB_Success;

    if (alternateSetting == USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_1)
    {
        USB_LatencyDebug_SendingAudioToHost(true);
        USB_DeviceAudioRecorderStatusReset();
        error =
            USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioRecorderHandle,
                                USB_AUDIO_RECORDER_STREAM_ENDPOINT, &audioRecDataBuff[0], FS_ISO_IN_ENDP_PACKET_SIZE);
        USB_LatencyDebug_SendingAudioToHost(false);
    }
    return error;
}

usb_status_t USB_DeviceAudioSpeakerSetInterface(class_handle_t handle, uint8_t interface, uint8_t alternateSetting)
{
    usb_status_t error = kStatus_USB_Success;
    
    //usb_speaker_instance_t speakerInstance;
    uint8_t *audioPlayPacket = NULL;
    uint8_t streamEp = 0;
    
#if USB_CHAT_SPEAKER_ENABLE
    if (handle == g_deviceAudioComposite->audioUnified.audioChatSpeakerHandle) {
        //speakerInstance = kUSB_SPEAKER_INSTANCE_Chat;
        audioPlayPacket = (uint8_t *)audioChatPlayPacket;
        streamEp = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
        feedbackEp = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT;
#endif
    }
#endif
#if USB_GAME_SPEAKER_ENABLE
    if (handle == g_deviceAudioComposite->audioUnified.audioGameSpeakerHandle) {
        //speakerInstance = kUSB_SPEAKER_INSTANCE_Game;
        audioPlayPacket = (uint8_t *)audioGamePlayPacket;
        streamEp = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
        feedbackEp = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT;
#endif
    }
#endif

    if (USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_1 == alternateSetting)
    {
        //USB_LatencyDebug_ReceivingAudioFromHost(true);

        memset(audioPlayPacket, 0, HS_ISO_OUT_ENDP_PACKET_SIZE);
        error = USB_DeviceAudioRecv(handle, streamEp, &audioPlayPacket[0], HS_ISO_OUT_ENDP_PACKET_SIZE);   
        
        if (error != kStatus_USB_Success)
        {
            return error;
        }
        else
        {
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#else
            if (!feedbackValueUpdating)
            {
                *((uint32_t *)&usbAudioFeedBackBuffer[0]) = *((uint32_t *)&audioFeedBackBuffer[0]);
            }
            error = USB_DeviceAudioSend(g_deviceAudioComposite->audioUnified.audioSpeakerHandle,
                                        USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT, usbAudioFeedBackBuffer,
                                        (USB_SPEED_HIGH == g_composite.speed) ? HS_ISO_FEEDBACK_ENDP_PACKET_SIZE :
                                                                                FS_ISO_FEEDBACK_ENDP_PACKET_SIZE);
#endif
        }
        //USB_LatencyDebug_ReceivingAudioFromHost(false);
    }
    return error;
}
/*!
 * @brief Audio init function.
 *
 * This function initializes the device with the composite device class information.
 *
 * @param device_composite          The pointer to the composite device structure.
 * @return kStatus_USB_Success .
 */
usb_status_t USB_DeviceAudioCompositeInit(usb_device_composite_struct_t *device_composite)
{
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
    CTIMER_CaptureInit();
#else
    AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_10_14);
#endif
    g_deviceAudioComposite                                       = device_composite;
    g_deviceAudioComposite->audioUnified.copyProtect             = 0x01U;
    g_deviceAudioComposite->audioUnified.curSpeakerMute          = 0x00U;
    g_deviceAudioComposite->audioUnified.curMicrophoneMute       = 0x00U;
    g_deviceAudioComposite->audioUnified.curSpeakerVolume[0]     = 0x00U;
    g_deviceAudioComposite->audioUnified.curSpeakerVolume[1]     = 0x1fU;
    g_deviceAudioComposite->audioUnified.curMicrophoneVolume[0]  = 0x00U;
    g_deviceAudioComposite->audioUnified.curMicrophoneVolume[1]  = 0x1fU;
    g_deviceAudioComposite->audioUnified.minVolume[0]            = 0x00U;
    g_deviceAudioComposite->audioUnified.minVolume[1]            = 0x00U;
    g_deviceAudioComposite->audioUnified.maxVolume[0]            = 0x00U;
    g_deviceAudioComposite->audioUnified.maxVolume[1]            = 0X43U;
    g_deviceAudioComposite->audioUnified.resVolume[0]            = 0x01U;
    g_deviceAudioComposite->audioUnified.resVolume[1]            = 0x00U;
    g_deviceAudioComposite->audioUnified.curBass                 = 0x00U;
    g_deviceAudioComposite->audioUnified.curBass                 = 0x00U;
    g_deviceAudioComposite->audioUnified.minBass                 = 0x80U;
    g_deviceAudioComposite->audioUnified.maxBass                 = 0x7FU;
    g_deviceAudioComposite->audioUnified.resBass                 = 0x01U;
    g_deviceAudioComposite->audioUnified.curMid                  = 0x00U;
    g_deviceAudioComposite->audioUnified.minMid                  = 0x80U;
    g_deviceAudioComposite->audioUnified.maxMid                  = 0x7FU;
    g_deviceAudioComposite->audioUnified.resMid                  = 0x01U;
    g_deviceAudioComposite->audioUnified.curTreble               = 0x01U;
    g_deviceAudioComposite->audioUnified.minTreble               = 0x80U;
    g_deviceAudioComposite->audioUnified.maxTreble               = 0x7FU;
    g_deviceAudioComposite->audioUnified.resTreble               = 0x01U;
    g_deviceAudioComposite->audioUnified.curAutomaticGain        = 0x01U;
    g_deviceAudioComposite->audioUnified.curDelay[0]             = 0x00U;
    g_deviceAudioComposite->audioUnified.curDelay[1]             = 0x40U;
    g_deviceAudioComposite->audioUnified.minDelay[0]             = 0x00U;
    g_deviceAudioComposite->audioUnified.minDelay[1]             = 0x00U;
    g_deviceAudioComposite->audioUnified.maxDelay[0]             = 0xFFU;
    g_deviceAudioComposite->audioUnified.maxDelay[1]             = 0xFFU;
    g_deviceAudioComposite->audioUnified.resDelay[0]             = 0x00U;
    g_deviceAudioComposite->audioUnified.resDelay[1]             = 0x01U;
    g_deviceAudioComposite->audioUnified.curLoudness             = 0x01U;
    g_deviceAudioComposite->audioUnified.curSamplingFrequency[0] = 0x00U;
    g_deviceAudioComposite->audioUnified.curSamplingFrequency[1] = 0x00U;
    g_deviceAudioComposite->audioUnified.curSamplingFrequency[2] = 0x01U;
    g_deviceAudioComposite->audioUnified.minSamplingFrequency[0] = 0x00U;
    g_deviceAudioComposite->audioUnified.minSamplingFrequency[1] = 0x00U;
    g_deviceAudioComposite->audioUnified.minSamplingFrequency[2] = 0x01U;
    g_deviceAudioComposite->audioUnified.maxSamplingFrequency[0] = 0x00U;
    g_deviceAudioComposite->audioUnified.maxSamplingFrequency[1] = 0x00U;
    g_deviceAudioComposite->audioUnified.maxSamplingFrequency[2] = 0x01U;
    g_deviceAudioComposite->audioUnified.resSamplingFrequency[0] = 0x00U;
    g_deviceAudioComposite->audioUnified.resSamplingFrequency[1] = 0x00U;
    g_deviceAudioComposite->audioUnified.resSamplingFrequency[2] = 0x01U;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
    g_deviceAudioComposite->audioUnified.curSpeakerMute20         = 0U;
    g_deviceAudioComposite->audioUnified.curMicrophoneMute20      = 0U;
    g_deviceAudioComposite->audioUnified.curClockValid            = 1U;
    g_deviceAudioComposite->audioUnified.curSpeakerVolume20[0]    = 0x00U;
    g_deviceAudioComposite->audioUnified.curSpeakerVolume20[1]    = 0x1FU;
    g_deviceAudioComposite->audioUnified.curMicrophoneVolume20[0] = 0x00U;
    g_deviceAudioComposite->audioUnified.curMicrophoneVolume20[1] = 0x1FU;

    g_deviceAudioComposite->audioUnified.curSpeakerSampleFrequency             = AUDIO_OUT_SAMPLING_RATE;
    g_deviceAudioComposite->audioUnified.speakerFreqControlRange.wNumSubRanges = 1U;
    g_deviceAudioComposite->audioUnified.speakerFreqControlRange.wMIN          = 48000U;
    g_deviceAudioComposite->audioUnified.speakerFreqControlRange.wMAX          = 48000U;
    g_deviceAudioComposite->audioUnified.speakerFreqControlRange.wRES          = 0U;

    g_deviceAudioComposite->audioUnified.curRecorderSampleFrequency             = AUDIO_IN_SAMPLING_RATE;
    g_deviceAudioComposite->audioUnified.recorderFreqControlRange.wNumSubRanges = 1U;
    g_deviceAudioComposite->audioUnified.recorderFreqControlRange.wMIN          = 48000U;
    g_deviceAudioComposite->audioUnified.recorderFreqControlRange.wMAX          = 48000U;
    g_deviceAudioComposite->audioUnified.recorderFreqControlRange.wRES          = 0U;

    g_deviceAudioComposite->audioUnified.volumeControlRange.wNumSubRanges = 1U;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wMIN          = 0x8001U;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wMAX          = 0x7FFFU;
    g_deviceAudioComposite->audioUnified.volumeControlRange.wRES          = 1U;

#endif
    g_deviceAudioComposite->audioUnified.tdReadNumberPlay       = 0;
    g_deviceAudioComposite->audioUnified.tdWriteNumberPlay      = 0;
    g_deviceAudioComposite->audioUnified.tdWriteNumberRec       = 0;
    g_deviceAudioComposite->audioUnified.tdReadNumberRec        = 0;
    g_deviceAudioComposite->audioUnified.audioSendCount[0]      = 0;
    g_deviceAudioComposite->audioUnified.audioSendCount[1]      = 0;
    g_deviceAudioComposite->audioUnified.lastAudioSendCount     = 0;
    g_deviceAudioComposite->audioUnified.usbRecvCount           = 0;
    g_deviceAudioComposite->audioUnified.audioSendTimes         = 0;
    g_deviceAudioComposite->audioUnified.usbRecvTimes           = 0;
    g_deviceAudioComposite->audioUnified.audioRecvCount         = 0;
    g_deviceAudioComposite->audioUnified.usbSendTimes           = 0;
    g_deviceAudioComposite->audioUnified.startPlayFlag          = 0;
    g_deviceAudioComposite->audioUnified.startPlayHalfFull      = 0;
    g_deviceAudioComposite->audioUnified.startRec               = 0;
    g_deviceAudioComposite->audioUnified.startRecHalfFull       = 0;
    g_deviceAudioComposite->audioUnified.speakerIntervalCount   = 0;
    g_deviceAudioComposite->audioUnified.speakerReservedSpace   = 0;
    g_deviceAudioComposite->audioUnified.recorderReservedSpace  = 0;
    g_deviceAudioComposite->audioUnified.timesFeedbackCalculate = 0;
    g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput = 0;
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
#if ((defined(USB_DEVICE_CONFIG_LPCIP3511FS) && (USB_DEVICE_CONFIG_LPCIP3511FS > 0U)) || \
     (defined(USB_DEVICE_CONFIG_KHCI) && (USB_DEVICE_CONFIG_KHCI > 0U)))
    g_deviceAudioComposite->audioUnified.froTrimIntervalCount     = 0;
    g_deviceAudioComposite->audioUnified.usbFroTicksPrev          = 0;
    g_deviceAudioComposite->audioUnified.usbFroTicksEma           = AUDIO_FRO_USB_SOF_INTERVAL_TICK_COUNT;
    g_deviceAudioComposite->audioUnified.usbFroTickEmaFrac        = 0;
    g_deviceAudioComposite->audioUnified.usbFroTickBasedPrecision = AUDIO_USB_FRO_TRIM_TICK_BASED_PRECISION;
#endif
    g_deviceAudioComposite->audioUnified.curAudioPllFrac            = AUDIO_PLL_FRACTIONAL_DIVIDER;
    g_deviceAudioComposite->audioUnified.audioPllTicksPrev          = 0;
    g_deviceAudioComposite->audioUnified.audioPllTicksEma           = AUDIO_PLL_USB_SOF_INTERVAL_TICK_COUNT;
    g_deviceAudioComposite->audioUnified.audioPllTickEmaFrac        = 0;
    g_deviceAudioComposite->audioUnified.audioPllTickBasedPrecision = AUDIO_PLL_FRACTION_TICK_BASED_PRECISION;
    g_deviceAudioComposite->audioUnified.stopDataLengthAudioAdjust  = 0U;
#endif
    return kStatus_USB_Success;
}

void USB_AudioCodecTask(void)
{
    if (g_deviceAudioComposite->audioUnified.codecSpeakerTask & MUTE_CODEC_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Speaker Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curSpeakerMute20);
#else
        usb_echo("Set Speaker Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curSpeakerMute);
#endif
        BOARD_SetCodecMuteUnmute(true);
        g_deviceAudioComposite->audioUnified.codecSpeakerTask &= ~MUTE_CODEC_TASK;
        g_CodecSpeakerMuteUnmute = true;
    }
    if (g_deviceAudioComposite->audioUnified.codecMicrophoneTask & MUTE_CODEC_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Microphone Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curMicrophoneMute20);
#else
        usb_echo("Set Microphone Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curMicrophoneMute);
#endif
        /* here add code to set mute practically */
        g_deviceAudioComposite->audioUnified.codecMicrophoneTask &= ~MUTE_CODEC_TASK;
        g_CodecMicrophoneMuteUnmute = true;
    }
    if (g_deviceAudioComposite->audioUnified.codecSpeakerTask & UNMUTE_CODEC_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Speaker Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curSpeakerMute20);
#else
        usb_echo("Set Speaker Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curSpeakerMute);
#endif
        BOARD_SetCodecMuteUnmute(false);
        g_deviceAudioComposite->audioUnified.codecSpeakerTask &= ~UNMUTE_CODEC_TASK;
        g_CodecSpeakerMuteUnmute = false;
    }
    if (g_deviceAudioComposite->audioUnified.codecMicrophoneTask & UNMUTE_CODEC_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Microphone Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curMicrophoneMute20);
#else
        usb_echo("Set Microphone Cur Mute : %x\r\n", g_deviceAudioComposite->audioUnified.curMicrophoneMute);
#endif
        /* here add code to set unmute practically */
        g_deviceAudioComposite->audioUnified.codecMicrophoneTask &= ~UNMUTE_CODEC_TASK;
        g_CodecMicrophoneMuteUnmute = false;
    }
    if (g_deviceAudioComposite->audioUnified.codecSpeakerTask & VOLUME_CHANGE_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Speaker Cur Volume : %x\r\n",
                 (uint16_t)(g_deviceAudioComposite->audioUnified.curSpeakerVolume20[1] << 8U) |
                     g_deviceAudioComposite->audioUnified.curSpeakerVolume20[0]);
#else
        usb_echo("Set Speaker Cur Volume : %x\r\n",
                 (uint16_t)(g_deviceAudioComposite->audioUnified.curSpeakerVolume[1] << 8U) |
                     g_deviceAudioComposite->audioUnified.curSpeakerVolume[0]);
#endif
        g_deviceAudioComposite->audioUnified.codecSpeakerTask &= ~VOLUME_CHANGE_TASK;
    }
    if (g_deviceAudioComposite->audioUnified.codecMicrophoneTask & VOLUME_CHANGE_TASK)
    {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
        usb_echo("Set Microphone Cur Volume : %x\r\n",
                 (uint16_t)(g_deviceAudioComposite->audioUnified.curMicrophoneVolume20[1] << 8U) |
                     g_deviceAudioComposite->audioUnified.curMicrophoneVolume20[0]);
#else
        usb_echo("Set Microphone Cur Volume : %x\r\n",
                 (uint16_t)(g_deviceAudioComposite->audioUnified.curMicrophoneVolume[1] << 8U) |
                     g_deviceAudioComposite->audioUnified.curMicrophoneVolume[0]);
#endif
        g_deviceAudioComposite->audioUnified.codecMicrophoneTask &= ~VOLUME_CHANGE_TASK;
    }
}

void USB_AudioSpeakerResetTask(void)
{
    if (g_deviceAudioComposite->audioUnified.speakerDetachOrNoInput)
    {
        USB_DeviceAudioSpeakerStatusReset();
    }
}
