/*
 * Copyright 2019-2020, 2024 NXP
 * SPDX-License-Identifier: BSD-3-Clause
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms.  By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms.  If you do not agree to
 * be bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#ifndef AUDIO_MIXER_H_
#define AUDIO_MIXER_H_

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

/* ---------------------------------------------------------------------------- */
/* Configuration                                                                */
/* ---------------------------------------------------------------------------- */
#define AUDIO_SPEAKER_CNT  (2U)

#ifdef AUDIO_SPEAKER_CNT
/**
 * The number of input interfaces of the mixer.
 */
    #define MIXER_INPUT_INTERFACE_CNT  AUDIO_SPEAKER_CNT    /* defined in audio.h */
#endif

/* ---------------------------------------------------------------------------- */
/* Defines                                                                      */
/* ---------------------------------------------------------------------------- */
/**
 * Generic error codes
 */
typedef enum _error {
    kERROR_Ok                  = 0,      /**< No error */
    kERROR_Fail                = 1,      /**< Error, not specified */
    kERROR_Busy                = 2,      /**< Busy error, previous operation not finished */
    kERROR_EndOfFile           = 3,      /**< End-of-file reached */
    kERROR_CommandNotProcessed = 4,      /**< Command was received but not processed due to an error */
    kERROR_InvalidRequest      = 5,      /**< Invalid request received */
    kERROR_Count,
} error_t;

/**
 * Set the nr of TX audio channels
 */
#define TX_AUDIO_CHANNELS  (2)
/**
 * Set the nr of RX audio channels
 */
#define RX_AUDIO_CHANNELS  (1)

/**
 * audio interface word length 2 bytes
 */
#define AUDIO_IF_WORD_LENGTH_2_BYTES  (2)

/**
 * audio interface word length 3 bytes
 */
#define AUDIO_IF_WORD_LENGTH_3_BYTES  (3)

/**
 * audio interface word length 4 bytes
 */
#define AUDIO_IF_WORD_LENGTH_4_BYTES  (4)

/**
 * audio interface word length 8 bits
 */
#define AUDIO_IF_WORD_LENGTH_8_BITS  (8)

/**
 * audio interface word length 16 bits
 */
#define AUDIO_IF_WORD_LENGTH_16_BITS  (AUDIO_IF_WORD_LENGTH_8_BITS * AUDIO_IF_WORD_LENGTH_2_BYTES)

/**
 * audio interface word length 32 bits
 */
#define AUDIO_IF_WORD_LENGTH_32_BITS  (AUDIO_IF_WORD_LENGTH_8_BITS * AUDIO_IF_WORD_LENGTH_4_BYTES)

/**
 *  Set the sampling rate to 48kHz
 */
#define AUDIO_SERVICE_SAMPLING_RATE_48  (48)

/**
 * Set the sampling rate
 */
#ifndef USE_SAMPLERATE_32KHZ
/* Setting 48kHz sample rate on I2S_FS with result in (2 * 32bits/word * 48000 Hz = 3.07 MHz I2S_BCLK */
    #define AUDIO_SERVICE_SYSTEM_SAMPLING_RATE  (AUDIO_SERVICE_SAMPLING_RATE_48)
#endif

/** TX DMA transfer size based on audio_GetTxSampleSizeBytes() */
#define TX_AUDIO_DMA_TRANSFER_SIZE(x)  (AUDIO_SERVICE_SYSTEM_SAMPLING_RATE * x * TX_AUDIO_CHANNELS)
/** RX DMA transfer size*/
//#define RX_AUDIO_DMA_TRANSFER_SIZE  (AUDIO_SERVICE_SYSTEM_SAMPLING_RATE * AUDIO_RX_SAMPLE_SIZE_BYTES * RX_AUDIO_CHANNELS)


/**
 * Audio transmission complete callback prototype.
 * This will be called by the module when the DMA transfer of AudioTX is completed.
 *
 * @param audioTxTransferSize size of the data transfered via DMA.
 * @param status returns kERROR_Ok if audio tx write was success, kERROR_Fail if otherwise.
 */
typedef void (*audio_tx_write_complete_cb_t)(uint16_t audioTxTransferSize, error_t status);

/**
 * @brief Required to control the current status of the Audio Service
 */
typedef enum _audioStatus {
    AUDIO_SERVICE_TX_COMPLETE,
    AUDIO_SERVICE_RX_COMPLETE,
    AUDIO_SERVICE_TX_STARTED,
    AUDIO_SERVICE_RX_STARTED
}audioStatus_t;

/**
 * @brief Required to configure the current status variable that control the Audio Service.
 */
typedef enum _audioStatusMasks {
    AUDIO_SERVICE_TX_COMPLETE_MASK = (1 << AUDIO_SERVICE_TX_COMPLETE),
    AUDIO_SERVICE_RX_COMPLETE_MASK = (1 << AUDIO_SERVICE_RX_COMPLETE),
    AUDIO_SERVICE_TX_STARTED_MASK  = (1 << AUDIO_SERVICE_TX_STARTED),
    AUDIO_SERVICE_RX_STARTED_MASK  = (1 << AUDIO_SERVICE_RX_STARTED)
}audioStatusMasks_t;



/**
 * The resolution of an audio amplification factor. It determines the valid range of values for
 * an amplification factor, being 0 to 2^MIXER_AMPLIFIER_RESOLUTION.
 */
#define MIXER_AMPLIFIER_RESOLUTION  (5)  /* Resolution of an amplification factor -> range is [0; 2^5] */

/**
 * Minimum value of an amplification factor
 */
#define MIXER_MIN_AMP  (0)

/**
 * Maximum value for an amplification factor
 */
#define MIXER_MAX_AMP  (1 << MIXER_AMPLIFIER_RESOLUTION)

/**
 * Number of (stereo) samples to target for buffering
 * 128 samples buffer target at 48kHz introduces a delay of 20.833us * 128 samples = 2.667ms
 */
#define NR_OF_SAMPLES_TO_BUFFER  (128 * TX_AUDIO_CHANNELS)

/**
 * The maximum expected audio sample size coming into the mixer (from USB) in bytes
 */
#define MAX_AUDIO_SAMPLE_SIZE_IN      (3)
/**
 * Size of the internal ring buffer for an input interface.
 * The buffer should be at least twice the target level.
 */
#define MIXER_INPUT_RING_BUFFER_SIZE  (3072)
#if ((NR_OF_SAMPLES_TO_BUFFER * MAX_AUDIO_SAMPLE_SIZE_IN) > MIXER_INPUT_RING_BUFFER_SIZE / 2)
    #error buffer too small
#endif

/**
 * Audio mixer target buffer filling level.
 * Target for NR_OF_SAMPLES_TO_BUFFER samples
 */
#define MIXER_TARGET_BUFFER_FILLING(x)  (NR_OF_SAMPLES_TO_BUFFER * x)

/**
 * The possible states of the audio mixer.
 */
typedef enum _audio_mixer_state {
    /** Not ready to output mixed samples (i.e. none of the input interfaces has reached @ref MIXER_TARGET_BUFFER_FILLING level) */
    kAUDIO_MIXER_STATE_NotReady,
    /** Ready to output mixed samples (i.e. at least one input interface has reached @ref MIXER_TARGET_BUFFER_FILLING level) */
    kAUDIO_MIXER_STATE_Ready,
    /** Used as counter for number of possible states */
    kAUDIO_MIXER_STATE_Count,
} audio_mixer_state_t;

/**
 * Callback type for Audio Mixer's ready-state change events.
 *
 * @param state The current state of the mixer (@ref audio_mixer_state_t).
 */
typedef void (*mixer_state_changed_cb)(audio_mixer_state_t state);

/* -------------------------------------------------------------------------
 * Public functions
 * ------------------------------------------------------------------------- */
uint8_t audio_GetTxSampleSizeBytes(void);

/**
 * Initialize the entire mixer component.
 */
void AUDIO_MIXER_Init(void);

/**
 * De-initialize the entire mixer component.
 */
void AUDIO_MIXER_Deinit(void);

/**
 * Get the current amplification factors of the input interfaces of the mixer.
 *
 * @param[out] amps User-provided array of size MIXER_INPUT_INTERFACE_CNT in which the amplification
 *             factor of each input interface will be stored.
 */
void AUDIO_MIXER_GetMixingFactors(uint8_t *amps);

/**
 * Set the amplification factors for the input interfaces of the mixer.
 * Each amplification factor needs to be valid (a value between MIXER_MIN_AMP and MIXER_MAX_AMP),
 * and their sum must be equal to MIXER_MAX_AMP (to avoid changing the amplitude of the output signal).
 *
 * @param[in] amps Array of size MIXER_INPUT_INTERFACE_CNT specifying the amplification factor for each input interface.
 * @return A result code:
 *         - kERROR_Ok: indicates success.
 *         - kERROR_Fail: indicates invalid amplification factor(s) were chosen.
 *         .
 */
error_t AUDIO_MIXER_SetMixingFactors(uint8_t *amps);

/**
 * Configure the callback to be used when the ready-state of the mixer changes; mixer did (not)
 * become ready to output mixed samples.
 *
 * @param cb The callback function to be used.
 */
void AUDIO_MIXER_RegisterStateCb(mixer_state_changed_cb cb);

/**
 * Writes data into the buffer of an input interface of the mixer.
 * @note The mixer implements buffering; it will only start outputting audio samples
 * as soon as one of the input interfaces has received plenty of samples.
 *
 * @param interfaceIdx Index of input interface that is providing the samples.
 * @param pBufferInput Buffer that is going to be written.
 * @param nbSamples Number of audio samples being written.
 * @param bufferInputFormat Data format of pBufferInput buffer.
 */
void AUDIO_MIXER_WriteSamples(uint8_t interfaceIdx, uint8_t *pBufferInput, uint16_t nbSamples, uint8_t bufferInputFormat);

/**
 * Write a certain number of output samples (i.e. mixed samples) to a buffer.
 * @note Only input interfaces that have enough samples buffered, will be mixed according to the mixing
 * factors. If the mixer was not ready to output mixed samples (@ref kAUDIO_MIXER_STATE_NotReady), the
 * output samples will be all zeroes.
 *
 * @param[out] outBuffer Pointer to buffer to which the mixed samples will be written.
 * @param[in]  nbSamples Number of audio samples to be mixed.
 */
void AUDIO_MIXER_GetMixedSamples(void *outBuffer, uint16_t nbSamples);

/**
 * Write a certain number of empty samples (i.e. all zeroes) to a buffer.
 *
 * @param[out] outBuffer Pointer to buffer to which the empty samples will be written.
 * @param[in]  nbSamples Number of audio samples to be generated.
 */
void AUDIO_MIXER_GetEmptySamples(void *outBuffer, uint16_t nbSamples);

/**
 * Stop an input interface of the mixer.
 * @note The mixer will stop outputting audio samples as soon as all input interfaces have been stopped.
 *
 * @param interfaceIdx Index of input interface to be stopped.
 */
void AUDIO_MIXER_StopInterface(uint8_t interfaceIdx);

/**
 * Stop all input interfaces of the mixer.
 */
void AUDIO_MIXER_StopAllInterfaces(void);

/**
 * Returns the buffer size of an input interface of the mixer.
 *
 * @param interfaceIdx Index of the input interface.
 * @return The buffer size of the input interface.
 */
uint32_t AUDIO_MIXER_GetBufferSize(uint8_t interfaceIdx);

/**
 * Returns the current filling level (space in use) of an input interface's buffer.
 *
 * @param interfaceIdx Index of the input interface.
 * @return The current filling level (in bytes) of the input interface.
 */
uint32_t AUDIO_MIXER_GetBufferFilling(uint8_t interfaceIdx);

uint32_t AUDIO_MIXER_GetMininalFillingLevel(uint8_t* idx);

/**
 * Get the maximum filling of one of the active interfaces
 *
 * @return the filling level in bytes
 */
uint32_t AUDIO_MIXER_GetBiggestFilling(uint8_t* idx);

uint8_t AUDIO_MIXER_GetActiveSpeakerCount(void);

bool AUDIO_MIXER_IsInterfaceActive(uint8_t interfaceIdx);

void AUDIO_MIXER_AdjustSecondReadPointer(void);

/** @} */

#endif /* AUDIO_MIXER_H_ */
