#! /bin/bash
# Copyright 2020-2024 NXP
# SPDX-License-Identifier: BSD-3-Clause

# The script for writing the image into MCX A14x/A15x


# Environment variables - absolute paths simplifying re-use of the script on another machine
#  - Absolute path to SPT workspace (it is recommended to copy/move all user files in this workspace)
if [ -z "${SPT_WORKSPACE}" ]; then
  export "SPT_WORKSPACE=$(cd "$(dirname "$0")"; pwd -P)"
fi
if ! [ -d "$SPT_WORKSPACE" ]; then
  echo "FAILURE: Directory not found: $SPT_WORKSPACE"
  exit 2
fi
#  - Absolute path to SPT installation directory
if [ -z "${SPT_INSTALL_BIN}" ]; then
  export "SPT_INSTALL_BIN=C:/nxp/MCUX_Provi_v9/bin/_internal"
fi
if ! [ -d "$SPT_INSTALL_BIN" ]; then
  echo "FAILURE: Directory not found: $SPT_INSTALL_BIN"
  exit 2
fi

# Use parameter: "blhost_connect <connection_param>" to use custom connection parameters for blhost
#     <connection_param> should be in format "-p COMx[,baud]" or "-u VID,PID" and must be enclosed in quotes
# Use parameter: "manufacturing_task_no index" to set number/index of the manufacturing task (to avoid same filename created from two tasks)
# Use parameter: "erase_all" to perform an erase of the entire flash memory instead erasing only regions that will be written

# Default connection parameters
export "blhost_connect=-u 0x1FC9,0x0155"
export "manufacturing_task_no="

echo "### Parse input arguments ###"
i=1
while (( i<=$#)); do
    param=${!i}
    case $param in
        "blhost_connect")
            i=$((i+1))
            export "blhost_connect=${!i}"
            ;;
        "manufacturing_task_no")
            i=$((i+1))
            export "manufacturing_task_no=${!i}"
            ;;
        "erase_all")
            export "erase_all=1"
            ;;
        *)
            echo "ERROR: Unsupported argument ${!i}"
            exit 2
    esac
    i=$((i+1))
done

# Used command line utilities
export "blhost=${SPT_INSTALL_BIN}/tools_scripts/blhost_spsdk_mac_wrapper.sh"

if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed before any other command is executed
    "${SPT_WORKSPACE}/hooks/write_mac.sh" started
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi

# Ping the device to establish communication; Result is ignored as the communication sometime fails for the first time
"$blhost" $blhost_connect -j -- get-property 1 0 > /dev/null 2> /dev/null

echo "### Check connection ###"
"$blhost" $blhost_connect -j -- get-property 1 0
if [ $? -ge 2 ]; then
    exit 2
fi

echo "### Erase flash ###"
if [ "$erase_all" = "1" ]; then
    "$blhost" $blhost_connect -j -- flash-erase-all 0
    if [ $? -ge 1 ]; then
      exit 2
    fi
else
    "$blhost" $blhost_connect -j -- flash-erase-region 0x00000000 24576 0
    if [ $? -ge 1 ]; then
      exit 2
    fi

    # Erase memory for additional image: ${SPT_WORKSPACE}/bootable_images/frdmmcxa153_ota_mcuboot_basic_signed.bin
    "$blhost" $blhost_connect -j -- flash-erase-region 0x6000 49152 0
    if [ $? -ge 2 ]; then
        exit 2
    fi
fi
if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed after erase of the memory is done
    "${SPT_WORKSPACE}/hooks/write_mac.sh" erase_done
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi

# CMPA must be erased before write, to avoid memory cumulative write
echo "### Erase CMPA ###"
"$blhost" $blhost_connect -j -- flash-erase-region 0x01000000 512
if [ $? -ge 2 ]; then
    exit 2
fi
echo "### Write Customer Manufacturing/Factory Programmable Area [CMPA] ###"
"$blhost" $blhost_connect -j -- write-memory 0x01000000 "${SPT_WORKSPACE}/configs/cmpa.bin"
if [ $? -ge 2 ]; then
    exit 2
fi
if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed after write of the CMPA is done
    "${SPT_WORKSPACE}/hooks/write_mac.sh" cmpa_written
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi

echo "### Write image 0 ###"
"$blhost" $blhost_connect -j -- write-memory 0x00000000 "${SPT_WORKSPACE}/bootable_images/frdmmcxa153_mcuboot_opensource.bin"
if [ $? -ge 1 ]; then
    exit 2
fi
if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed after write of the image is done
    "${SPT_WORKSPACE}/hooks/write_mac.sh" image_written
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi

echo "### Write additional image: ${SPT_WORKSPACE}/bootable_images/frdmmcxa153_ota_mcuboot_basic_signed.bin ###"
"$blhost" $blhost_connect -j -- write-memory 0x6000 "${SPT_WORKSPACE}/bootable_images/frdmmcxa153_ota_mcuboot_basic_signed.bin" 0
if [ $? -ge 2 ]; then
    exit 2
fi
if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed after write of additional images is done
    "${SPT_WORKSPACE}/hooks/write_mac.sh" additional_images_written
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi
if [ -x "${SPT_WORKSPACE}/hooks/write_mac.sh" ]; then
    # hook executed after all steps of the script were executed
    "${SPT_WORKSPACE}/hooks/write_mac.sh" finished
    if [ $? -ge 1 ]; then
        exit 2
    fi
fi
