/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017, 2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "fsl_lpspi_edma.h"
#include "fsl_flexio_qspi_edma.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#if defined(FSL_FEATURE_SOC_DMAMUX_COUNT) && FSL_FEATURE_SOC_DMAMUX_COUNT
#include "fsl_dmamux.h"
#endif

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BOARD_FLEXIO_BASE   FLEXIO0
#define FLEXIO_QSPI_MOSI_PIN 28U
#define FLEXIO_QSPI_SCK_PIN  26U
#define FLEXIO_QSPI_PCS0_PIN 27U

#define BOARD_SPI_SLAVE_BASE LPSPI1
#define SLAVE_LPSPI_PCS_FOR_INIT     (kLPSPI_Pcs0)
#define SLAVE_LPSPI_PCS_FOR_TRANSFER (kLPSPI_SlavePcs0)

#define EXAMPLE_DMA            DMA0
#define EXAMPLE_QSPI_MASTER_TX_CHANNEL 9
#define EXAMPLE_SPI_SLAVE_RX_CHANNEL  10
#define EXAMPLE_SPI_SLAVE_TX_CHANNEL  11
#define EXAMPLE_QSPI_MASTER_TX_DMA_SOURCE     kDmaRequestMuxFlexIO0ShiftRegister0Request
#define EXAMPLE_SPI_SLAVE_TX_DMA_SOURCE       kDmaRequestMuxLpFlexcomm1Tx
#define EXAMPLE_SPI_SLAVE_RX_DMA_SOURCE       kDmaRequestMuxLpFlexcomm1Rx

/* Select flexio clock source */
#define FLEXIO_CLOCK_FREQUENCY CLOCK_GetFlexioClkFreq()
#define TRANSFER_BAUDRATE 25000000U /*! Transfer baudrate */

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* SPI user callback */
void FLEXIO_QSPI_MasterUserCallback(FLEXIO_QSPI_Type *base,
                                    flexio_qspi_master_edma_handle_t *handle,
                                    status_t status,
                                    void *userData);
void LPSPI_SlaveUserCallback(LPSPI_Type *base, lpspi_slave_edma_handle_t *handle, status_t status, void *userData);
/*******************************************************************************
 * Variables
 ******************************************************************************/
#define DISPLAY_WIDTH  64 // because Flexcomm SPI slave doesn't support 640x480
#define DISPLAY_HEIGHT 48 // because Flexcomm SPI slave doesn't support 640x480

#define COMMAND_CYCLE 8
#define ADDRESS_CYCLE 24
#define DUMMY_CYCLE   8
#define FOOTER_CYCLE  8

AT_NONCACHEABLE_SECTION_ALIGN(uint8_t header[(COMMAND_CYCLE+ADDRESS_CYCLE+DUMMY_CYCLE)/2], 4);
AT_NONCACHEABLE_SECTION_ALIGN(uint8_t body[DISPLAY_WIDTH*DISPLAY_HEIGHT/2], 4);
AT_NONCACHEABLE_SECTION_ALIGN(uint8_t footer[FOOTER_CYCLE/2], 4);

#define TRANSFER_SIZE ((sizeof(header) + sizeof(body) + sizeof(footer)) / 4)    /*! Transfer dataSize */
uint8_t masterTxData[TRANSFER_SIZE]  = {0U};
uint8_t slaveRxData[TRANSFER_SIZE]  = {0U};

flexio_qspi_master_edma_handle_t g_m_handle;
lpspi_slave_edma_handle_t g_s_handle;

edma_handle_t masterQspiTxHandle;
edma_handle_t slaveRxHandle;
edma_handle_t slaveTxHandle;

volatile bool isTransferCompleted = false;

/*******************************************************************************
 * Code
 ******************************************************************************/
static inline void setNibbleByteSwappedBit(uint8_t* buf, uint8_t data, uint8_t data_line, uint8_t nbit)
{
	#define BITSET(byte, bit, nbit)   ((byte) = ((byte) & ~(1<<(nbit))) | ((bit)<<(nbit)))
	BITSET(buf[3 - (nbit/2)], (data>>nbit) & 0x1 , data_line + 4*(nbit%2));
}

static void setData(uint8_t* buf, uint8_t data, uint8_t data_line)
{
	setNibbleByteSwappedBit(buf, data, data_line, 0);
	setNibbleByteSwappedBit(buf, data, data_line, 1);
	setNibbleByteSwappedBit(buf, data, data_line, 2);
	setNibbleByteSwappedBit(buf, data, data_line, 3);
	setNibbleByteSwappedBit(buf, data, data_line, 4);
	setNibbleByteSwappedBit(buf, data, data_line, 5);
	setNibbleByteSwappedBit(buf, data, data_line, 6);
	setNibbleByteSwappedBit(buf, data, data_line, 7);
}

static void setCmd(uint8_t *header, uint8_t cmd)
{
    setData(&header[0], cmd, 0);
}

static void setAddress(uint8_t *header, uint32_t address)
{
    setData(&header[4] , address >> 16 & 0xff, 0);
    setData(&header[8] , address >> 8  & 0xff, 0);
    setData(&header[12], address       & 0xff, 0);
}

void FLEXIO_QSPI_MasterUserCallback(FLEXIO_QSPI_Type *base,
                                    flexio_qspi_master_edma_handle_t *handle,
                                    status_t status,
                                    void *userData)
{
    if (status == kStatus_Success)
    {
        __NOP();
    }
    PRINTF("This is QSPI Master call back.\r\n");
}

void LPSPI_SlaveUserCallback(LPSPI_Type *base, lpspi_slave_edma_handle_t *handle, status_t status, void *userData)
{
    if (status == kStatus_Success)
    {
        __NOP();
    }

    if (status == kStatus_LPSPI_Error)
    {
        __NOP();
    }

    isTransferCompleted = true;

    PRINTF("This is LPSPI slave call back.\r\n");
}

int main(void)
{
	uint8_t tmpData;
    BOARD_InitPins();
    BOARD_PowerMode_OD();
    BOARD_InitBootClocks();
    BOARD_InitDebugConsole();

    /* attach PLL0 to PLLCLKDIV */
    CLOCK_SetClkDiv(kCLOCK_DivPllClk, 1u);
    CLOCK_AttachClk(kPLL0_to_PLLCLKDIV);

    /* attach FRO 12M to FLEXCOMM4 (debug console) */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom4Clk, 1u);
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    /* attach PLL0 to FLEXIO */
    CLOCK_SetClkDiv(kCLOCK_DivFlexioClk, 1u);
    CLOCK_AttachClk(kPLL0_to_FLEXIO);

    /* attach PLL_DIV to FLEXCOMM1 */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom1Clk, 2u);
    CLOCK_AttachClk(kPLL_DIV_to_FLEXCOMM1);

    PRINTF("FLEXIO Master edma - LPSPI Slave edma example start.\r\n");
    PRINTF("This example use one flexio spi as master and one lpspi instance as slave on one board.\r\n");
    PRINTF("Master and slave both use edma way.\r\n");
    PRINTF("Please make sure you make the correct line connection. Basically, the connection is:\r\n");
    PRINTF("FLEXIO_QSPI_master -- LPSPI_slave   \r\n");
    PRINTF("   SCK      --    SCK  \r\n");
    PRINTF("   PCS0     --    PCS0 \r\n");
    PRINTF("   MOSI     --    MOSI \r\n");

    uint32_t errorCount;
    uint32_t i;
    flexio_spi_master_config_t masterConfig;
    lpspi_slave_config_t slaveConfig;
    flexio_qspi_transfer_t masterXferQspi;
    lpspi_transfer_t slaveXfer;
    edma_config_t config;

    FLEXIO_QSPI_Type qspiDev = {
        .flexioBase      = BOARD_FLEXIO_BASE,
        .SDOPinIndex     = FLEXIO_QSPI_MOSI_PIN,
        .SCKPinIndex     = FLEXIO_QSPI_SCK_PIN,
        .CSnPinIndex     = FLEXIO_QSPI_PCS0_PIN
    };

    /* Slave config */
    LPSPI_SlaveGetDefaultConfig(&slaveConfig);
    LPSPI_SlaveInit(BOARD_SPI_SLAVE_BASE, &slaveConfig);

    /* Set up the transfer data */
    setCmd(header, 0xde);
    setAddress(header, 0xadbeef);
    for (i = 0U; i < sizeof(body); i++)
    {
        body[i] = 2*i << 4 | (2*i+1) & 0xf;
    }

    for (i = 0U; i < TRANSFER_SIZE; i++)
    {
        masterTxData[i] = i;
    }
    masterTxData[0] = 0xde;
    masterTxData[1] = 0xad;
    masterTxData[2] = 0xbe;
    masterTxData[3] = 0xef;
    masterTxData[4] = 0x00;
    
    /* DMA init */
    EDMA_GetDefaultConfig(&config);
    EDMA_Init(EXAMPLE_DMA, &config);
    /* configure channel/priority and create handle for TX and RX. */
    EDMA_SetChannelMux(EXAMPLE_DMA, EXAMPLE_QSPI_MASTER_TX_CHANNEL, EXAMPLE_QSPI_MASTER_TX_DMA_SOURCE);
    EDMA_SetChannelMux(EXAMPLE_DMA, EXAMPLE_SPI_SLAVE_TX_CHANNEL, EXAMPLE_SPI_SLAVE_TX_DMA_SOURCE);
    EDMA_SetChannelMux(EXAMPLE_DMA, EXAMPLE_SPI_SLAVE_RX_CHANNEL, EXAMPLE_SPI_SLAVE_RX_DMA_SOURCE);
    EDMA_CreateHandle(&masterQspiTxHandle, EXAMPLE_DMA, EXAMPLE_QSPI_MASTER_TX_CHANNEL);
    EDMA_CreateHandle(&slaveTxHandle, EXAMPLE_DMA, EXAMPLE_SPI_SLAVE_TX_CHANNEL);
    EDMA_CreateHandle(&slaveRxHandle, EXAMPLE_DMA, EXAMPLE_SPI_SLAVE_RX_CHANNEL);
    /* Set up slave first */
    LPSPI_SlaveTransferCreateHandleEDMA(BOARD_SPI_SLAVE_BASE, &g_s_handle, LPSPI_SlaveUserCallback, NULL, &slaveRxHandle,
                                     &slaveTxHandle);
    /*Set slave transfer ready to receive/send data*/
    slaveXfer.txData   = NULL;
    slaveXfer.rxData   = slaveRxData;
    slaveXfer.dataSize = TRANSFER_SIZE;

    /*Start master transfer*/
    masterXferQspi.bufs[0].txData   = header;
    masterXferQspi.bufs[0].dataSize = sizeof(header)/4;
    masterXferQspi.bufs[1].txData   = body;
    masterXferQspi.bufs[1].dataSize = sizeof(body)/4;
    masterXferQspi.bufs[2].txData   = footer;
    masterXferQspi.bufs[2].dataSize = sizeof(footer)/4;
    masterXferQspi.flags    = kFLEXIO_SPI_csContinuous;

    /* Set up master transfer */
    FLEXIO_QSPI_MasterGetDefaultConfig(&masterConfig);
    masterConfig.baudRate_Bps = TRANSFER_BAUDRATE;
    FLEXIO_QSPI_MasterInit(&qspiDev, &masterConfig, FLEXIO_CLOCK_FREQUENCY);
    FLEXIO_QSPI_MasterTransferCreateHandleEDMA(&qspiDev, &g_m_handle, FLEXIO_QSPI_MasterUserCallback, NULL, &masterQspiTxHandle);        /* Set up master transfer */

    while (1)
    {
        /* Set up the transfer data */
        for (i = 0U; i < TRANSFER_SIZE; i++)
        {
            slaveRxData[i]  = 0U;
        }
        isTransferCompleted = false;
        LPSPI_SlaveTransferEDMA(BOARD_SPI_SLAVE_BASE, &g_s_handle, &slaveXfer);
        FLEXIO_QSPI_MasterTransferEDMA(&qspiDev, &g_m_handle, &masterXferQspi);

        /* Wait slave received all data. */
        while (!isTransferCompleted)
        {
        }

        uint8_t *masterTxDataQspi = header;
        for (i = 0; i < 4; i++)
        {
            errorCount = 0U;
            for (uint32_t j = 0U; j < TRANSFER_SIZE; j++)
            {
                tmpData = (masterTxDataQspi[4*j+3] >> i    ) & 0x1 |
                         ((masterTxDataQspi[4*j+3] >> (i+4)) & 0x1) << 1 |
                         ((masterTxDataQspi[4*j+2] >> i    ) & 0x1) << 2 |
                         ((masterTxDataQspi[4*j+2] >> (i+4)) & 0x1) << 3 |
                         ((masterTxDataQspi[4*j+1] >> i    ) & 0x1) << 4 |
                         ((masterTxDataQspi[4*j+1] >> (i+4)) & 0x1) << 5 |
                         ((masterTxDataQspi[4*j]   >> i    ) & 0x1) << 6 |
                         ((masterTxDataQspi[4*j]   >> (i+4)) & 0x1) << 7 ;
                if(tmpData != slaveRxData[j])
                {
                    errorCount++;
                }
            }
            if(errorCount == 0U)
            {
                break;
            }
        }
        if (errorCount == 0U)
        {
            PRINTF("FLEXIO QSPI master[%d] <-> SPI slave transfer all data matched!\r\n\r\n", i);
        }
        else
        {
            PRINTF("Error occurred in FLEXIO QSPI master <-> SPI slave transfer!\r\n\r\n");
        }

        /* Wait for press any key */
        PRINTF("\r\n Press any key to run again\r\n\r\n");
        GETCHAR();
    }

    LPSPI_Deinit(BOARD_SPI_SLAVE_BASE);

}
