/*
 * Copyright 2019-2021, 2023, 2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_flexio_qspi_smartdma.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Component ID definition, used by tools. */
#ifndef FSL_COMPONENT_ID
#define FSL_COMPONENT_ID "platform.drivers.flexio_qspi_smartdma"
#endif

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*!
 * @brief Callback function registered to SMARTDMA driver.
 */
static void FLEXIO_QSPI_SMARTDMA_Callback(void *param);

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * brief Initializes the FLEXO QSPI master SMARTDMA handle.
 *
 * This function initializes the FLEXO QSPI master SMARTDMA handle which can be
 * used for other FLEXO QSPI transactional APIs. For a specified FLEXO QSPI
 * instance, call this API once to get the initialized handle.
 *
 * param base Pointer to FLEXIO_QSPI_Type structure.
 * param handle Pointer to flexio_qspi_smartdma_handle_t structure to store the
 * transfer state.
 * param config Pointer to the configuration.
 * param callback QSPI transfer complete callback, NULL means no callback.
 * param userData callback function parameter.
 * retval kStatus_Success Successfully create the handle.
 */
status_t FLEXIO_QSPI_TransferCreateHandleSMARTDMA(FLEXIO_QSPI_Type *base,
                                                    flexio_qspi_smartdma_handle_t *handle,
                                                    flexio_qspi_smartdma_transfer_callback_t callback,
                                                    void *userData)
{
    AT_NONCACHEABLE_SECTION_ALIGN(static smartdma_flexio_qspi_param_t smartdmaParam, 4);
    assert(handle != NULL);

    IRQn_Type flexio_irqs[] = FLEXIO_IRQS;

    /* Zero the handle. */
    (void)memset(handle, 0, sizeof(*handle));

    handle->smartdmaApi = kSMARTDMA_FLEXIO_QSPI_DMA_NIBBLE_BYTE_SWAP;
    handle->smartdmaParam = &smartdmaParam;

    /* Initialize the state. */
    handle->state = (uint32_t)kFLEXIO_QSPI_Idle;

    /* Register callback and userData. */
    handle->callback = callback;
    handle->userData = userData;
    handle->base     = base;

    SMARTDMA_InstallFirmware(SMARTDMA_DISPLAY_MEM_ADDR, s_smartdmaDisplayFirmware, SMARTDMA_DISPLAY_FIRMWARE_SIZE);

    SMARTDMA_InstallCallback(FLEXIO_QSPI_SMARTDMA_Callback, handle);

    /* Clear pending NVIC IRQ before enable NVIC IRQ. */
    NVIC_ClearPendingIRQ(flexio_irqs[FLEXIO_QSPI_GetInstance(base)]);
    /* Disable interrupt in NVIC. */
    (void)DisableIRQ(flexio_irqs[FLEXIO_QSPI_GetInstance(base)]);

    return kStatus_Success;
}

/*!
 * brief Performs a non-blocking FlexIO QSPI transfer using SMARTDMA.
 *
 * This function returns immediately after transfer initiates. Use the callback
 * function to check whether the transfer is completed.
 *
 * param base pointer to FLEXIO_QSPI_Type structure.
 * param handle pointer to flexio_qspi_smartdma_handle_t structure to store the
 * transfer state.
 * param xfer Pointer to FlexIO QSPI transfer structure.
 * retval kStatus_Success Successfully start a transfer.
 * retval kStatus_InvalidArgument Input argument is invalid.
 * retval kStatus_FLEXIO_SPI_Busy FlexIO QSPI is not idle, it is running another
 * transfer.
 */
status_t FLEXIO_QSPI_TransferSMARTDMA(FLEXIO_QSPI_Type *base,
                                        flexio_qspi_smartdma_handle_t *handle,
                                        flexio_qspi_transfer_t *xfer)
{
    assert(handle != NULL);
    assert(xfer != NULL);

    /* Check if SPI is busy. */
    if (handle->state == (uint32_t)kFLEXIO_QSPI_Busy)
    {
        return kStatus_FLEXIO_SPI_Busy;
    }

    for(uint32_t i = 0; i < FLEXIO_QSPI_BUF_MAX; i++)
    {
        if ((uint32_t)xfer->bufs[i].txData & 0x3U || (uint32_t)xfer->bufs[i].rxData & 0x3U)
        {
            return kStatus_InvalidArgument;
        }
        if (xfer->bufs[i].txData != NULL && xfer->bufs[i].rxData != NULL)
        {
            return kStatus_InvalidArgument;
        }
        if (xfer->bufs[i].txData == NULL && xfer->bufs[i].rxData == NULL)
        {
            return kStatus_InvalidArgument;
        }
    }

    handle->state            = (uint32_t)kFLEXIO_QSPI_Busy;
    handle->current_buffer = 0;
    
    for(uint32_t i = 0; i < FLEXIO_QSPI_BUF_MAX; i++)
    {
        handle->bufs[i] = xfer->bufs[i];
    }

    handle->smartdmaParam->txRemainingBytes = handle->bufs[0].dataSize;
    handle->smartdmaParam->rxRemainingBytes = handle->bufs[0].dataSize;
    handle->smartdmaParam->txData           = handle->bufs[0].txData;
    handle->smartdmaParam->rxData           = handle->bufs[0].rxData;
    if(handle->smartdmaParam->txData != NULL)
    {
        FLEXIO_QSPI_SetDirection(base, kFLEXIO_QSPI_Transmit);
    }
    else
    {
        FLEXIO_QSPI_SetDirection(base, kFLEXIO_QSPI_Receive);
    }
    
    SMARTDMA_Reset();
    SMARTDMA_Boot(handle->smartdmaApi, handle->smartdmaParam, 0);

    FLEXIO_QSPI_ChipSelect(base, 0);
    /* The shifter interrupt is used by the SMARTDMA. */
    FLEXIO_QSPI_EnableInterrupts(base, (uint32_t)kFLEXIO_SPI_RxFullInterruptEnable);
    FLEXIO_QSPI_EnableInterrupts(base, (uint32_t)kFLEXIO_SPI_TxEmptyInterruptEnable);

    return kStatus_Success;
}

/*!
 * brief Aborts a FlexIO QSPI transfer using SMARTDMA.
 *
 * param base pointer to FLEXIO_QSPI_Type structure.
 * param handle FlexIO QSPI SMARTDMA handle pointer.
 */
void FLEXIO_QSPI_TransferAbortSMARTDMA(FLEXIO_QSPI_Type *base, flexio_qspi_smartdma_handle_t *handle)
{
    assert(handle != NULL);

    FLEXIO_QSPI_DisableInterrupts(base, (uint32_t)kFLEXIO_SPI_TxEmptyInterruptEnable);
    FLEXIO_QSPI_DisableInterrupts(base, (uint32_t)kFLEXIO_SPI_RxFullInterruptEnable);
    FLEXIO_QSPI_ChipSelect(base, 1);
    
    SMARTDMA_Reset();

    /* Set the handle state. */
    handle->state     = (uint32_t)kFLEXIO_QSPI_Idle;
    handle->current_buffer = 0;
    handle->smartdmaParam->txRemainingBytes = 0;
    handle->smartdmaParam->rxRemainingBytes = 0;
}

static void FLEXIO_QSPI_SMARTDMA_Callback(void *param)
{
    flexio_qspi_smartdma_handle_t *handle = (flexio_qspi_smartdma_handle_t *)param;
    FLEXIO_QSPI_Type *base = handle->base;

    if(handle->current_buffer == (FLEXIO_QSPI_BUF_MAX-1) || handle->bufs[handle->current_buffer+1].dataSize == 0)
    {
        FLEXIO_QSPI_TransferAbortSMARTDMA(base, handle);
    
        if (handle->callback != NULL)
        {
            (handle->callback)(base, handle, kStatus_FLEXIO_SPI_Idle, handle->userData);
        }
    }
    else
    {
        handle->current_buffer += 1;
        handle->smartdmaParam->txRemainingBytes = handle->bufs[handle->current_buffer].dataSize;
        handle->smartdmaParam->rxRemainingBytes = handle->bufs[handle->current_buffer].dataSize;
        handle->smartdmaParam->txData           = handle->bufs[handle->current_buffer].txData;
        handle->smartdmaParam->rxData           = handle->bufs[handle->current_buffer].rxData;
        if(handle->smartdmaParam->txData != NULL)
        {
            FLEXIO_QSPI_SetDirection(base, kFLEXIO_QSPI_Transmit);
        }
        else
        {
            FLEXIO_QSPI_SetDirection(base, kFLEXIO_QSPI_Receive);
        }

        FLEXIO_QSPI_EnableInterrupts(base, (uint32_t)kFLEXIO_SPI_TxEmptyInterruptEnable);
    }
}
