/*
 * Copyright 2019, 2021-2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <setjmp.h>

#include "fsl_cmc.h"
#include "fsl_spc.h"
#include "fsl_clock.h"
#include "fsl_debug_console.h"
#include "power_mode_switch.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "peripherals.h"
#include "board.h"
#include "fsl_lpuart.h"
#include "fsl_crc.h"
#include "fsl_vbat.h"
#include "fsl_ccm32k.h"
#include "fsl_port.h"

#include "fsl_rtc.h"     //include

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define APP_MODE_NAME_ARRAY                                                                                     \
    {                                                                                                           \
        "Active", "Sleep", "DeepSleep","PowerDown", "DeepPowerDown", "PowerSwitchOff"                    \
    }

#define APP_MODE_DESC_ARRAY                                                                                            \
    {                                                                                                                  \
        "Active: DCDC in normal mode(voltage 1.5V); core voltage is 1.1V; core clock is 96MHz; power consumption is "  \
        "about 7.5 mA",                                                                                                \
            "Sleep: Core, platform and peripheral clocks are gated; CORE_MAIN, CORE_WAKE and Radio domain enter "     \
            "into sleep mode; DCDC voltage is 1.25V; core voltage is 1.0V; all CM33 RAM retained; power consumption "  \
            "is about 53.5uA.",                                                                                        \
            "Deep Sleep: Core, platform and peripheral clocks are gated; CORE_MAIN, CORE_WAKE and Radio domain "      \
            "enter into deep sleep mode; DCDC voltage is 1.25V; core voltages is 1.0V; all RAMs retained; OSC32K "     \
            "enabled; power consumption is about 2.8uA.",                                                              \
            "Power Down: Core, platform and peripheral clocks are gated; CORE_MAIN, CORE_WAKE and Radio domain "      \
            "enter into power down mode; DCDC voltage is 1.25V; core voltages is 1.0V; 32KB of CM33 RAM retained; "    \
            "all radio RAMs retained; OSC32K enabled; power consumption is about 1.8uA.",                              \
            "Deep Power Down: Core, platform and peripheral clocks are gated; CORE_MAIN, CORE_WAKE and Radio domain " \
            "enter into deep power down mode; DCDC voltage is 1.25V; core voltages is 1.0V; 8KB of CM33 RAM "          \
            "retained; no radio RAM retained; OSC32K enabled; power consumption is about 1.1uA.",                      \
            "Switch off: All power domain except VBAT power domain are powered off; power consumption is about 0.2 "   \
            "uA."                                                                                                      \
    }
#define APP_CMC   CMC0
#define APP_LPTMR DEMO_LPTMR_PERIPHERAL
#define APP_WUU   DEMO_WUU_PERIPHERAL
#define APP_SPC   SPC0
#define APP_VBAT  VBAT0

#define WUU_WAKEUP_VBAT_IDX  4U

#define WUU_WAKEUP_BUTTON_NAME "SW3"

#define APP_VBAT_IRQ_HANDLER  VBAT_IRQHandler


#define APP_SRAM_ARRAYS_TO_POWER_OFF                                                                       \
    (kCMC_SRAMBank0 | kCMC_SRAMBank1 | kCMC_SRAMBank2 | kCMC_SRAMBank3 | kCMC_SRAMBank4 | kCMC_SRAMBank5 | \
     kCMC_SRAMBank6 | kCMC_SRAMBank7)

#define APP_POWERDOWN1_SRAM_POWER_MODE (0x3CU) // 32KB of RAM retained: STCM5(8K), STCM4(8K),CTCM1(8K),CTCM0(8K).

/* Debug console RX pin: PORTC2 MUX: 3 */
#define DEBUG_CONSOLE_RX_PORT   PORTC
#define DEBUG_CONSOLE_RX_GPIO   GPIOC
#define DEBUG_CONSOLE_RX_PIN    2U
#define DEBUG_CONSOLE_RX_PINMUX kPORT_MuxAlt3
/* Debug console TX pin: PORTC3 MUX: 3 */
#define DEBUG_CONSOLE_TX_PORT   PORTC
#define DEBUG_CONSOLE_TX_GPIO   GPIOC
#define DEBUG_CONSOLE_TX_PIN    3U
#define DEBUG_CONSOLE_TX_PINMUX kPORT_MuxAlt3

#define RTC_IRQn       RTC_Alarm_IRQn
#define RTC_IRQHandler RTC_Alarm_IRQHandler
#define EXAMPLE_OSC_WAIT_TIME_MS 1000UL
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void APP_InitDebugConsole(void);
void APP_DeinitDebugConsole(void);
void APP_DeinitWakeupButton(void);
static void APP_SetSPCConfiguration(void);
static void APP_SetCMCConfiguration(void);
static void APP_DeInitVbat(void);

//static void APP_PowerPreSwitchHook(void);
static void APP_PowerPostSwitchHook(void);

static void APP_ShowPowerMode(cmc_low_power_mode_t powermode);
static void APP_EnterSleep1Mode(void);
static void APP_EnterDeepSleep1Mode(void);
static void APP_EnterPowerDown1Mode(void);
static void APP_EnterDeepPowerDown1Mode(void);
static void APP_PowerModeSwitch(app_power_mode_t targetPowerMode);
static app_power_mode_t APP_GetTargetPowerMode(void);

static void APP_SaveRuntimeContext(void);
static void APP_RestoreRuntimeContext(void);

static void APP_WakeupFunction(void);
static void APP_SetPowerDownModeWakeupConfig(void);

void config_RTC(void);
void set_time_RTC(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/
volatile app_wakeup_source_t g_selectedWakeupSource;

AT_ALWAYS_ON_DATA(uint32_t g_scbVtor);
AT_ALWAYS_ON_DATA(uint32_t g_scbIcsr);
AT_ALWAYS_ON_DATA(uint32_t g_scbAircr);
AT_ALWAYS_ON_DATA(uint32_t g_scbCCR);
AT_ALWAYS_ON_DATA(uint8_t g_scbShp[12]);
AT_ALWAYS_ON_DATA(uint32_t g_scbShcsr);
AT_ALWAYS_ON_DATA(uint32_t g_nvicIser[16U]);
AT_ALWAYS_ON_DATA(uint8_t g_nvicIp[496U]);
AT_ALWAYS_ON_DATA(uint32_t g_cpuControl);
AT_ALWAYS_ON_DATA(uint32_t g_wakeupEntry[12]);
AT_ALWAYS_ON_DATA(jmp_buf g_coreContext);
AT_ALWAYS_ON_DATA(app_power_mode_t g_targetPowerMode);

AT_ALWAYS_ON_DATA_INIT(char *g_modeNameArray[]) = APP_MODE_NAME_ARRAY;
AT_ALWAYS_ON_DATA_INIT(char *g_modeDescArray[]) = APP_MODE_DESC_ARRAY;

extern uint32_t m_warmboot_stack_end;


volatile bool busyWait;
rtc_datetime_t date;
/*******************************************************************************
 * Code
 ******************************************************************************/
void APP_InitDebugConsole(void)
{
    /*
     * Debug console RX pin is set to disable for current leakage, need to re-configure pinmux.
     * Debug console TX pin: Don't need to change.
     */
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
}

void APP_DeinitDebugConsole(void)
{
    DbgConsole_Deinit();
    /*
     * Set pin for current leakage.
     * Debug console RX pin: Set to pinmux to disable.
     * Debug console TX pin: Don't need to change.
     */
    PORT_SetPinMux(DEBUG_CONSOLE_RX_PORT, DEBUG_CONSOLE_RX_PIN, kPORT_PinDisabledOrAnalog);
    PORT_SetPinMux(DEBUG_CONSOLE_TX_PORT, DEBUG_CONSOLE_TX_PIN, kPORT_PinDisabledOrAnalog);
}


void APP_LPTMR_IRQ_HANDLER(void)
{
    if (kLPTMR_TimerInterruptEnable & LPTMR_GetEnabledInterrupts(APP_LPTMR))
    {
        LPTMR_DisableInterrupts(APP_LPTMR, kLPTMR_TimerInterruptEnable);
        LPTMR_ClearStatusFlags(APP_LPTMR, kLPTMR_TimerCompareFlag);
        LPTMR_StopTimer(APP_LPTMR);
    }
}

void APP_VBAT_IRQ_HANDLER(void)
{
    if (VBAT_GetStatusFlags(APP_VBAT) & kVBAT_StatusFlagWakeupPin)
    {
        VBAT_ClearStatusFlags(APP_VBAT, kVBAT_StatusFlagWakeupPin);
    }
}

void WUU0_IRQHandler(void)
{
    if ((WUU_GetExternalWakeUpPinsFlag(APP_WUU) & 0x800UL) != 0UL)
    {
        WUU_ClearExternalWakeUpPinsFlag(APP_WUU, 0x800UL);
    }
}

void RTC_IRQHandler(void)
{
    uint32_t status = RTC_GetStatusFlags(RTC);

    if (status & kRTC_AlarmFlag)
    {
        busyWait = false;

        /* Clear alarm flag */
        RTC_ClearStatusFlags(RTC, kRTC_AlarmInterruptEnable);
    }
    else if (status & kRTC_TimeInvalidFlag)
    {
        /* Clear timer invalid flag */
        RTC_ClearStatusFlags(RTC, kRTC_TimeInvalidFlag);
    }
    else
    {
    }
    SDK_ISR_EXIT_BARRIER;
}

void main(void)
{
    uint32_t freq;
    cmc_low_power_mode_t curmode;
    bool needSetWakeup = false;

    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    BOARD_InitBootPeripherals();

    CLOCK_DeinitSysOsc();
    CLOCK_DeinitSirc();

    APP_SetSPCConfiguration();
    config_RTC(); 						//RTC initialization

    PRINTF("\r\n----------------------- Normal Boot -----------------------\r\n");

    while (1)
    {
        PRINTF("\r\nSticky Reset Reason %x.\r\n", CMC_GetStickySystemResetStatus(APP_CMC));
        CMC_ClearStickySystemResetStatus(CMC0, CMC_GetStickySystemResetStatus(APP_CMC));
        APP_SetCMCConfiguration();
        APP_DeInitVbat();

        curmode = CMC_GetMAINPowerMode(APP_CMC);
        PRINTF("\r\n###########################    Power Mode Switch Demo    ###########################\r\n");

        APP_ShowPowerMode(curmode);
        g_targetPowerMode = APP_GetTargetPowerMode();

        PRINTF("%s\r\n", g_modeDescArray[(uint8_t)(g_targetPowerMode - kAPP_PowerModeActive)]);

            //APP_PowerPreSwitchHook();
            APP_PowerModeSwitch(g_targetPowerMode);
            APP_PowerPostSwitchHook();
            PRINTF("\r\n************* Wake Up From %s Mode *************\r\n",
                   g_modeNameArray[(uint8_t)(g_targetPowerMode - kAPP_PowerModeActive)]);

        PRINTF("\r\n-----------------------Next loop -----------------------\r\n");
    }
}

/*!
 * @brief In active mode, all HVDs/LVDs are disabled, DCDC regulated to 1.5V, Core LDO regulated to 1.1V;
 * In low power modes, all HVDs/LVDs are disabled, bandgap is disabled, DCDC regulated to 1.25V, Core LDO regulated to
 * 1.0V.
 */
static void APP_SetSPCConfiguration(void)
{
    status_t status;

    spc_active_mode_regulators_config_t activeModeRegulatorOption;

    /* Disable LVDs and HVDs in Active mode. */
    SPC_EnableActiveModeCoreHighVoltageDetect(APP_SPC, false);
    SPC_EnableActiveModeCoreLowVoltageDetect(APP_SPC, false);
    SPC_EnableActiveModeSystemHighVoltageDetect(APP_SPC, false);
    SPC_EnableActiveModeSystemLowVoltageDetect(APP_SPC, false);
    SPC_EnableActiveModeIOHighVoltageDetect(APP_SPC, false);
    SPC_EnableActiveModeIOLowVoltageDetect(APP_SPC, false);

    activeModeRegulatorOption.bandgapMode                      = kSPC_BandgapEnabledBufferDisabled;
    activeModeRegulatorOption.lpBuff                           = false;
    activeModeRegulatorOption.DCDCOption.DCDCVoltage           = kSPC_DCDC_NormalVoltage; /* DCDC regulate to 1.5V. */
    activeModeRegulatorOption.DCDCOption.DCDCDriveStrength     = kSPC_DCDC_NormalDriveStrength;
    activeModeRegulatorOption.SysLDOOption.SysLDOVoltage       = kSPC_SysLDO_NormalVoltage;
    activeModeRegulatorOption.SysLDOOption.SysLDODriveStrength = kSPC_SysLDO_NormalDriveStrength;
    activeModeRegulatorOption.CoreLDOOption.CoreLDOVoltage =
        kSPC_CoreLDO_NormalVoltage; /* Core LDO regulate to 1.1V. */
#if defined(FSL_FEATURE_SPC_HAS_CORELDO_VDD_DS) && FSL_FEATURE_SPC_HAS_CORELDO_VDD_DS
    activeModeRegulatorOption.CoreLDOOption.CoreLDODriveStrength = kSPC_CoreLDO_NormalDriveStrength;
#endif /* FSL_FEATURE_SPC_HAS_CORELDO_VDD_DS */

    status = SPC_SetActiveModeRegulatorsConfig(APP_SPC, &activeModeRegulatorOption);
    if (status != kStatus_Success)
    {
        PRINTF("Fail to set regulators in Active mode.");
        return;
    }

    spc_lowpower_mode_regulators_config_t lowPowerRegulatorOption;

    lowPowerRegulatorOption.lpIREF                             = false;
    lowPowerRegulatorOption.bandgapMode                        = kSPC_BandgapDisabled;
    lowPowerRegulatorOption.lpBuff                             = false;
    lowPowerRegulatorOption.CoreIVS                            = false;
    lowPowerRegulatorOption.DCDCOption.DCDCVoltage             = kSPC_DCDC_LowUnderVoltage;
    lowPowerRegulatorOption.DCDCOption.DCDCDriveStrength       = kSPC_DCDC_LowDriveStrength;
    lowPowerRegulatorOption.SysLDOOption.SysLDODriveStrength   = kSPC_SysLDO_LowDriveStrength;
    lowPowerRegulatorOption.CoreLDOOption.CoreLDOVoltage       = kSPC_CoreLDO_MidDriveVoltage;
    lowPowerRegulatorOption.CoreLDOOption.CoreLDODriveStrength = kSPC_CoreLDO_LowDriveStrength;

    status = SPC_SetLowPowerModeRegulatorsConfig(APP_SPC, &lowPowerRegulatorOption);
    if (status != kStatus_Success)
    {
        PRINTF("Fail to set regulators in Low Power Mode.");
        return;
    }

    SPC_SetLowPowerWakeUpDelay(APP_SPC, 0xFFFFU);
}

/*!
 * @brief 1. Set power mode protection; 2. Disable low power mode debug; 3. Enable Flash Doze mode.
 */
static void APP_SetCMCConfiguration(void)
{
    CMC_SetPowerModeProtection(APP_CMC, kCMC_AllowAllLowPowerModes);
    CMC_LockPowerModeProtectionSetting(APP_CMC);
    CMC_EnableDebugOperation(APP_CMC, false);
    CMC_ConfigFlashMode(APP_CMC, false, true, false);
}

/*!
 * @brief Disable Backup SRAM regulator, FRO16K and Bandgap which locates in VBAT power domain for most of power modes.
 *
 */
static void APP_DeInitVbat(void)
{
    VBAT_EnableBackupSRAMRegulator(APP_VBAT, false);
    VBAT_EnableFRO16k(APP_VBAT, false);
    while (VBAT_CheckFRO16kEnabled(APP_VBAT))
        ;
    VBAT_EnableBandgap(APP_VBAT, false);
    while (VBAT_CheckBandgapEnabled(APP_VBAT))
        ;
}


//static void APP_PowerPreSwitchHook(void)
//{
//    /* Wait for debug console output finished. */
//    while (!(kLPUART_TransmissionCompleteFlag & LPUART_GetStatusFlags((LPUART_Type *)BOARD_DEBUG_UART_BASEADDR)))
//    {
//    }
//    APP_DeinitDebugConsole();
//}

static void APP_PowerPostSwitchHook(void)
{
    if (SPC_CheckPowerDomainLowPowerRequest(APP_SPC, kSPC_PowerDomain0))
    {
        SPC_ClearPowerDomainLowPowerRequestFlag(APP_SPC, kSPC_PowerDomain0);
    }
    if (SPC_CheckPowerDomainLowPowerRequest(APP_SPC, kSPC_PowerDomain1))
    {
        SPC_ClearPowerDomainLowPowerRequestFlag(APP_SPC, kSPC_PowerDomain1);
    }
    if (SPC_CheckPowerDomainLowPowerRequest(APP_SPC, kSPC_PowerDomain2))
    {
        RFMC->RF2P4GHZ_CTRL = (RFMC->RF2P4GHZ_CTRL & (~RFMC_RF2P4GHZ_CTRL_LP_MODE_MASK));
        RFMC->RF2P4GHZ_CTRL &= ~RFMC_RF2P4GHZ_CTRL_LP_ENTER_MASK;
        SPC_ClearPowerDomainLowPowerRequestFlag(APP_SPC, kSPC_PowerDomain2);
    }
    SPC_ClearLowPowerRequest(APP_SPC);

    CLOCK_EnableClock(kCLOCK_Lpuart0);
    APP_InitDebugConsole();
}

static void APP_ShowPowerMode(cmc_low_power_mode_t powermode)
{
    if (powermode == kCMC_ActiveMode)
    {
        PRINTF("    Power mode: Active\r\n");
    }
}

static app_power_mode_t APP_GetTargetPowerMode(void)
{
    uint8_t ch;
    app_power_mode_t inputPowerMode;

    while (1)
    {
        PRINTF("\r\nPlease Select the desired power mode: \n\r\n");

        for (app_power_mode_t powerModeIndex = kAPP_PowerModeActive; powerModeIndex < kAPP_PowerModeMax;
             powerModeIndex++)
        {
            PRINTF("\tPress %c to enter %s mode!\r\n", powerModeIndex,
                   g_modeNameArray[(uint8_t)(powerModeIndex - kAPP_PowerModeActive)]);
        }
        PRINTF("\r\n\tWaiting for power mode select...\r\n\r\n");

        ch = GETCHAR();

        if ((ch >= 'a') && (ch <= 'z'))
        {
            ch -= 'a' - 'A';
        }
        PUTCHAR(ch);
        PUTCHAR('\n');

        inputPowerMode = (app_power_mode_t)ch;
        if ((inputPowerMode > kAPP_PowerModeMin) && (inputPowerMode < kAPP_PowerModeMax))
        {
            return inputPowerMode;
        }
        PRINTF("Wrong input, please re-try!\r\n");
    }
}

static void APP_PowerModeSwitch(app_power_mode_t targetPowerMode)
{
    if (targetPowerMode != kAPP_PowerModeActive)
    {
        switch (targetPowerMode)
        {
            case kAPP_PowerModeSleep1:
            	set_time_RTC();
                APP_EnterSleep1Mode();
                break;

            case kAPP_PowerModeDeepSleep1:
            	set_time_RTC();
                APP_EnterDeepSleep1Mode();
                break;

            case kAPP_PowerModePowerDown1:
            	set_time_RTC();
                APP_EnterPowerDown1Mode();
                break;

            case kAPP_PowerModeDeepPowerDown1:
            	set_time_RTC();
                APP_EnterDeepPowerDown1Mode();
                break;

            case kAPP_PowerSwitchOff:
                SPC_PowerModeControlPowerSwitch(APP_SPC);
                APP_EnterDeepPowerDown1Mode();
                break;
            default:
                assert(false);
                break;
        }
    }
}

static void APP_EnterSleep1Mode(void)
{
    cmc_power_domain_config_t config;

    /* Set NBU into Sleep Mode */
    RFMC->RF2P4GHZ_CTRL = (RFMC->RF2P4GHZ_CTRL & (~RFMC_RF2P4GHZ_CTRL_LP_MODE_MASK)) | RFMC_RF2P4GHZ_CTRL_LP_MODE(0x1);
    RFMC->RF2P4GHZ_CTRL |= RFMC_RF2P4GHZ_CTRL_LP_ENTER_MASK;

    /* Set MAIN_CORE and MAIN_WAKE power domain into sleep mode. */
    config.clock_mode  = kCMC_GateAllSystemClocksEnterLowPowerMode;
    config.main_domain = kCMC_SleepMode;
    config.wake_domain = kCMC_SleepMode;
    CMC_EnterLowPowerMode(APP_CMC, &config);
}

static void APP_EnterDeepSleep1Mode(void)
{
    cmc_power_domain_config_t config;

    /* Enable CORE VDD Voltage scaling. */
    SPC_EnableLowPowerModeCoreVDDInternalVoltageScaling(APP_SPC, true);

    /* Set NBU into Deep Sleep Mode */
    RFMC->RF2P4GHZ_CTRL = (RFMC->RF2P4GHZ_CTRL & (~RFMC_RF2P4GHZ_CTRL_LP_MODE_MASK)) | RFMC_RF2P4GHZ_CTRL_LP_MODE(0x3);
    RFMC->RF2P4GHZ_CTRL |= RFMC_RF2P4GHZ_CTRL_LP_ENTER_MASK;

    /* Set MAIN_CORE and MAIN_WAKE power domain into Deep Sleep Mode. */
    config.clock_mode  = kCMC_GateAllSystemClocksEnterLowPowerMode;
    config.main_domain = kCMC_DeepSleepMode;
    config.wake_domain = kCMC_DeepSleepMode;

    CMC_EnterLowPowerMode(APP_CMC, &config);
}

static void APP_EnterPowerDown1Mode(void)
{
    cmc_power_domain_config_t config;

    /* Set NBU into Power Down mode. */
    RFMC->RF2P4GHZ_CTRL = (RFMC->RF2P4GHZ_CTRL & (~RFMC_RF2P4GHZ_CTRL_LP_MODE_MASK)) | RFMC_RF2P4GHZ_CTRL_LP_MODE(0x7);
    RFMC->RF2P4GHZ_CTRL |= RFMC_RF2P4GHZ_CTRL_LP_ENTER_MASK;

    config.clock_mode  = kCMC_GateAllSystemClocksEnterLowPowerMode;
    config.main_domain = kCMC_PowerDownMode;
    config.wake_domain = kCMC_DeepSleepMode;

    CMC_PowerOffSRAMLowPowerOnly(APP_CMC, APP_POWERDOWN1_SRAM_POWER_MODE);
    SPC_EnableLowPowerModeCoreVDDInternalVoltageScaling(APP_SPC, true);
    APP_SetPowerDownModeWakeupConfig();

    APP_SaveRuntimeContext();

    CLOCK_DisableClock(kCLOCK_Secsubsys);
    memset(&g_coreContext, 0, sizeof(g_coreContext));
    if (setjmp(g_coreContext) == 0)
    {
        CMC_EnterLowPowerMode(APP_CMC, &config);
    }
    /* Wake up from power down1 mode. */
    APP_RestoreRuntimeContext();
    __enable_irq();
}

static void APP_EnterDeepPowerDown1Mode(void)
{
    cmc_power_domain_config_t config;

    /* Power off all radio RAMs. */
    RF_CMC1->RAM_PWR = 0x000004FF;
    /* Set NBU into Deep Power Down Mode. */
    RFMC->RF2P4GHZ_CTRL = (RFMC->RF2P4GHZ_CTRL & (~RFMC_RF2P4GHZ_CTRL_LP_MODE_MASK)) | RFMC_RF2P4GHZ_CTRL_LP_MODE(0xF);
    RFMC->RF2P4GHZ_CTRL |= RFMC_RF2P4GHZ_CTRL_LP_ENTER_MASK;

    /* Set CORE_MAIN, CORE_WAKE into deep power down mode. */
    config.clock_mode  = kCMC_GateAllSystemClocksEnterLowPowerMode;
    config.main_domain = kCMC_DeepPowerDown;
    config.wake_domain = kCMC_DeepPowerDown;

    SPC_EnableLowPowerModeCoreVDDInternalVoltageScaling(APP_SPC, true);
    APP_SetPowerDownModeWakeupConfig();

    /* Enable VBAT backup SRAM regulator to supply STCM5, because CORE_LDO will be disabled in deep power down mode. */
    vbat_fro16k_config_t vbatFro16kConfig = {
        .enableFRO16k       = true,
        .enableFRO16kOutput = true,
    };
    VBAT_ConfigFRO16k(VBAT0, &vbatFro16kConfig);
    VBAT_EnableBandgap(VBAT0, true);
    VBAT_EnableBandgapRefreshMode(VBAT0, true);
    VBAT_EnableBackupSRAMRegulator(VBAT0, true);

    APP_SaveRuntimeContext();
    memset(&g_coreContext, 0, sizeof(g_coreContext));
    if (setjmp(g_coreContext) == 0)
    {
        CMC_EnterLowPowerMode(APP_CMC, &config);
    }
    /* Wake up from Deep power down mode. */
    APP_RestoreRuntimeContext();
    __enable_irq();
}

static void APP_SaveRuntimeContext(void)
{
    uint32_t i = 0UL;

    g_scbVtor  = SCB->VTOR;
    g_scbIcsr  = SCB->ICSR;
    g_scbAircr = ((SCB->AIRCR) & ~SCB_AIRCR_VECTKEY_Msk) | (0x5FA << SCB_AIRCR_VECTKEY_Pos);
    g_scbCCR   = SCB->CCR;
    for (i = 0UL; i < 12UL; i++)
    {
        g_scbShp[i] = SCB->SHPR[i];
    }
    g_scbShcsr = SCB->SHCSR;
    for (i = 0UL; i < 16U; i++)
    {
        g_nvicIser[i] = NVIC->ISER[i];
    }
    for (i = 0UL; i < 496UL; i++)
    {
        g_nvicIp[i] = NVIC->IPR[i];
    }
    g_cpuControl = __get_CONTROL();
}

static void APP_RestoreRuntimeContext(void)
{
    uint32_t i;

    SCB->VTOR  = g_scbVtor;
    SCB->ICSR  = g_scbIcsr;
    SCB->AIRCR = g_scbAircr;
    SCB->CCR   = g_scbCCR;
    for (i = 0UL; i < 12UL; i++)
    {
        SCB->SHPR[i] = g_scbShp[i];
    }
    SCB->SHCSR = g_scbShcsr;
    for (i = 0UL; i < 16UL; i++)
    {
        NVIC->ISER[i] = g_nvicIser[i];
    }
    for (i = 0UL; i < 496UL; i++)
    {
        NVIC->IPR[i] = g_nvicIp[i];
    }
}

static void APP_SetPowerDownModeWakeupConfig(void)
{
    /* The first word from wakeup address must be SP. */
    g_wakeupEntry[0] = ((uint32_t)&m_warmboot_stack_end);
    /* The second word from wakeup address must be PC. */
    g_wakeupEntry[1] = (uint32_t)APP_WakeupFunction;

    for (uint32_t i = 2UL; i < 12UL; i++)
    {
        g_wakeupEntry[i] = i;
    }
    /* Set wake up address. */
    SPC_SetWakeUpValue(APP_SPC, (uint32_t)g_wakeupEntry);

    uint32_t crcResult;
    crc_config_t config;
    config.polynomial         = 0x04C11DB7U;
    config.seed               = 0xFFFFFFFFU;
    config.reflectIn          = false;
    config.reflectOut         = false;
    config.complementChecksum = false;
    config.crcBits            = kCrcBits32;
    config.crcResult          = kCrcFinalChecksum;

    CLOCK_DisableClock(kCLOCK_Crc0);
    CRC_Init(CRC0, &config);
    CRC_WriteData(CRC0, (const uint8_t *)((uint32_t)g_wakeupEntry), 48);
    crcResult        = CRC_Get32bitResult(CRC0);
    REGFILE1->REG[0] = crcResult;
}

static void APP_WakeupFunction(void)
{
    uint32_t ramAddress = 0x4000000UL;
    /* Please note that RAM ECC is enabled in KW45B41Z A1.
     * After waking up from low power modes that RAM blocks are powered off,
     * corresponding RAM blocks should be re-initialized.
     */
    while (ramAddress < 0x4004000UL)
    {
        memset((void *)(uint32_t *)ramAddress, 0UL, sizeof(uint32_t));
        ramAddress = ramAddress + 4UL;
    }

    ramAddress = 0x20004000UL;
    while (ramAddress < 0x20010000UL)
    {
        memset((void *)(uint32_t *)ramAddress, 0UL, sizeof(uint32_t));
        ramAddress = ramAddress + 4UL;
    }

    WDOG0->CS    = (uint32_t)((WDOG0->CS) & ~WDOG_CS_EN_MASK);
    WDOG0->TOVAL = 0xFFFF;
    WDOG0->CS    = (uint32_t)(((WDOG0->CS) & ~WDOG_CS_EN_MASK) | WDOG_CS_UPDATE_MASK);
    WDOG0->TOVAL = 0xFFFF;
    WDOG0->CNT   = 0xD928C520U;

#if ((__FPU_PRESENT == 1) && (__FPU_USED == 1))
    SCB->CPACR |= ((3UL << 10 * 2) | (3UL << 11 * 2)); /* set CP10, CP11 Full Access in Secure mode */
#if defined(__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U)
    SCB_NS->CPACR |= ((3UL << 10 * 2) | (3UL << 11 * 2)); /* set CP10, CP11 Full Access in Non-secure mode */
#endif                                                    /* (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U) */
#endif                                                    /* ((__FPU_PRESENT == 1) && (__FPU_USED == 1)) */

    /* Restore the CONTROL register. */
    __set_CONTROL(g_cpuControl);
    longjmp(g_coreContext, true);
}

void config_RTC(void)
{

  rtc_config_t rtcConfig;

  PRINTF("RTC Init\r\n");
  BOARD_InitPinsRTC();

    ccm32k_osc_config_t osc32kConfig = {
        .enableInternalCapBank = true,
        .xtalCap               = kCCM32K_OscXtal0pFCap,
        .extalCap              = kCCM32K_OscExtal16pFCap,
        .coarseAdjustment      = kCCM32K_OscCoarseAdjustmentRange0,
    };
    CCM32K_Set32kOscConfig(CCM32K, kCCM32K_Enable32kHzCrystalOsc, &osc32kConfig);
    CCM32K_SelectClockSource(CCM32K, kCCM32K_ClockSourceSelectOsc32k);

    RTC_GetDefaultConfig(&rtcConfig);
    RTC_Init(RTC, &rtcConfig);


    RTC->CR |= RTC_CR_CPE(0x01);
    RTC->CR |= RTC_CR_CPS(0x1);

    /* Set a start date time and start RT */
    date.year   = 2014U;
    date.month  = 12U;
    date.day    = 25U;
    date.hour   = 19U;
    date.minute = 0;
    date.second = 0;

    /* RTC time counter has to be stopped before setting the date & time in the TSR register */
    RTC_StopTimer(RTC);

    /* Set RTC time to default */
    RTC_SetDatetime(RTC, &date);

    /* Enable RTC alarm interrupt */
    RTC_EnableInterrupts(RTC, kRTC_AlarmInterruptEnable);

    /* Enable at the NVIC */
    WUU_SetInternalWakeUpModulesConfig(APP_WUU, 0x6, kWUU_InternalModuleInterrupt);
    EnableIRQ(RTC_IRQn);

    /* Start the RTC time counter */
    RTC_StartTimer(RTC);

}

void set_time_RTC(void)
{

      uint32_t sec;
      uint32_t currSeconds;
      uint8_t index;

      rtc_datetime_t date;

        busyWait = true;
        index    = 0;
        sec      = 0;
        /* Get date time */
        RTC_GetDatetime(RTC, &date);

        /* Get alarm time from user */
        PRINTF("\n\nPlease input the number of second to wait for alarm \r\n");
        PRINTF("The second must be positive value\r\n");
        while (index != 0x0D)
        {
            index = GETCHAR();
            if ((index >= '0') && (index <= '9'))
            {
                PUTCHAR(index);
                sec = sec * 10 + (index - 0x30U);
            }
        }
        PRINTF("\r\n");

        /* Read the RTC seconds register to get current time in seconds */
        currSeconds = RTC->TSR;

        /* Add alarm seconds to current time, because RTC alarm will happen when RTC->TAR = RTC->TSR and RTC->TSR
        increments, thus there's possible 1 second maximum delay here. */
        currSeconds += sec;

        /* Set alarm time in seconds */
        RTC->TAR = currSeconds;

        /* Get alarm time */
        RTC_GetAlarm(RTC, &date);

        RTC->IER = RTC_IER_TAIE(0x01);
}

