/*
 * Copyright 2016-2022 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of NXP Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_nboot.h"

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*! @brief Interface for the NBOOT API driver.*/
typedef struct
{
romapi_status_t (*romapi_rng_generate_random)(uint8_t *output, size_t outputByteLen);
nboot_status_t (*nboot_context_init)(nboot_context_t *context);
nboot_status_t (*nboot_context_deinit)(nboot_context_t *context);
nboot_status_protected_t (*nboot_sb3_load_manifest)(nboot_context_t *context,
uint32_t *manifest,
nboot_sb3_load_manifest_parms_t *parms);
nboot_status_protected_t (*nboot_sb3_load_block)(nboot_context_t *context,
uint32_t *block);
nboot_status_protected_t (*nboot_img_authenticate_ecdsa)(nboot_context_t *context,
uint8_t imageStartAddress[],
nboot_bool_t *isSignatureVerified,
nboot_img_auth_ecdsa_parms_t *parms);
nboot_status_protected_t (*nboot_img_authenticate_cmac)(nboot_context_t *context,
uint8_t imageStartAddress[],
nboot_bool_t *isSignatureVerified,
nboot_img_authenticate_cmac_parms_t *parms);
} nboot_interface_t;

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* Locating the random number generation API using the ROM API structure */
#define ROM_API_TREE *((uint32_t *)(0x0302fc00)+10)
#define NBOOT_API_TREE ((nboot_interface_t *) ROM_API_TREE)

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    char ch;
    uint8_t rndBuffer[32];
    static nboot_context_t contextt;

    /* Init board hardware. */
    /* attach main clock divide to FLEXCOMM0 (debug console) */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 0u, false);
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom0Clk, 1u, true);
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    BOARD_InitPins();
    BOARD_BootClockPLL150M();
    BOARD_InitDebugConsole();

    /* Initialization of nboot context data structure */
    nboot_status_t status_nboot = NBOOT_API_TREE->nboot_context_init(&contextt);

    for(int x=0;x<11;x++){

    	/* Generate random number with specified length */
    	romapi_status_t status_romapi = NBOOT_API_TREE->romapi_rng_generate_random(&rndBuffer[0], 32);

    	/* If operation is successful, print the random number generated */
    	if(status_romapi == kStatus_NBOOT_Success){
    		for(int i=0;i<32;i++)
    			PRINTF("%x", rndBuffer[i]);
    		PRINTF("\r\n");
    	}

    	/* Print error message if invalid arguments */
    	if(status_romapi == kStatus_NBOOT_InvalidArgument){
    		PRINTF("Invalid Arguments\n");
    	}

    	/* Print error message if operation failed */
    	if(status_romapi == kStatus_NBOOT_Fail){
    		PRINTF("Random Number Generation Failed!\n");
    	}

    	PRINTF("\r\n");
    }

    while (1)
    {
        ch = GETCHAR();
        PUTCHAR(ch);
    }
}
