/*
 * Copyright 2020 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_inputmux.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_dsp.h"
#include "fsl_dmic.h"
#include "vit_launcher.h"
#include "dsp_support.h"
#include "fsl_power.h"
#include "fsl_dmic_dma.h"
#include "audio_capture_render.h"
#include "dualcore_ringbuffer_of.h"
#include "fsl_mu.h"
#include "fsl_ctimer.h"
#include "fsl_pca9420.h"
#include "pmic_support.h"
#include "fsl_i2s.h"
#include "fsl_sema42.h"



#define FIFO_DEPTH 15
volatile uint32_t dsp_started = 0;
#define BOOT_FLAG 0x01U

//#define CHECK_DATA

#ifdef CHECK_DATA
    #define AUDIO_DUMP_START_0 0x202C0000
    #define AUDIO_DUMP_MAX     0x00020000
    #define AUDIO_DUMP_DONE    0x20340000
    static uint32_t audio_dump_pos = 0;
    uint8_t *p_audio_dump_addr = NULL;
#endif
 
/* Match Configuration for Channel 0 */
static ctimer_match_config_t matchConfig0;
void ctimer_match0_callback(uint32_t flags);
/* Array of function pointers for callback for each channel */
ctimer_callback_t ctimer_callback_table[] = {
    ctimer_match0_callback, NULL, NULL, NULL, NULL, NULL, NULL, NULL};
    
/*DMIC DMA Transfer*/
static dmic_channel_config_t dmic_channel_cfg;
static dma_handle_t s_dmicRxDmaHandle;
static dmic_dma_handle_t s_dmicDmaHandle;
static int16_t audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE * AUDIO_CAPTURE_BUFFER_COUNT] = { 0 };
static int32_t audio_capture_dmic_buffer_index = 0;
static uint32_t audio_capture_dmic_started = 0;
volatile int16_t *p_dmic_last_address = NULL;
volatile uint32_t dmic_last_size;
static dcringbuf_t *p_dcringbuf_context = NULL;

SDK_ALIGN(dma_descriptor_t s_dmaDescriptorPingpong[2], 16);
void DMIC_PDM_24M(void);
void DMIC_PDM_500k(void);
void InitCTimer(void);
void StartDMIC_DMA_Transfer(void);
static dmic_transfer_t s_receiveXfer[2U] = {
    /* transfer configurations for channel0 */
    {
        .data                   = audio_capture_dmic_DMABuff,
        .dataWidth              = sizeof(int16_t),
        .dataSize               = AUDIO_CAPTURE_TRANSFER_SIZE * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave1xWidth,
        .linkTransfer           = &s_receiveXfer[1],
    },

    {
        .data                   = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE],
        .dataWidth              = sizeof(int16_t),
        .dataSize               = AUDIO_CAPTURE_TRANSFER_SIZE * sizeof(int16_t),
        .dataAddrInterleaveSize = kDMA_AddressInterleave1xWidth,
        .linkTransfer           = &s_receiveXfer[0],
    },
};


void BOARD_ConfigPMICModes(pca9420_modecfg_t *cfg, uint32_t num)
{
    assert(cfg);

    /* Configuration PMIC mode to align with power lib like below:
     *  0b00    run mode, no special.
     *  0b01    deep sleep mode, vddcore 0.6V.
     *  0b10    deep powerdown mode, vddcore off.
     *  0b11    full deep powerdown mode vdd1v8 and vddcore off. */

    /* Mode 1: VDDCORE 0.6V. */
    cfg[1].sw1OutVolt = kPCA9420_Sw1OutVolt0V600;
    /* Mode 2: VDDCORE off. */
    cfg[2].enableSw1Out = false;

    /* Mode 3: VDDCORE, VDD1V8 and VDDIO off. */
    cfg[3].enableSw1Out  = false;
    cfg[3].enableSw2Out  = false;
    cfg[3].enableLdo2Out = false;
}


void DMIC0_HWVAD_Callback(void)
{
  volatile int i;
  /* reset hwvad internal interrupt */
  DMIC_CtrlClrIntrHwvad(DMIC0, true);
  /* wait for HWVAD to settle */
  if(dsp_started == 0) {    
      dsp_started = 1;        
//      DSP_Start();  
      
      /* VIT Library has I2S ID checking */
      i2s_config_t s_TxConfig;
      CLOCK_AttachClk(kFRO_DIV4_to_FLEXCOMM1);
      I2S_TxGetDefaultConfig(&s_TxConfig);
      I2S_TxInit(I2S1, &s_TxConfig);
      
      DSP_RESUME();
      DMIC_PDM_24M();
      DisableDeepSleepIRQ(HWVAD0_IRQn);
      CTIMER_StartTimer(CTIMER2);
  }

  for (i = 0; i <= 500U; i++)
  {
    
  }
  /*HWVAD Normal operation */
  DMIC_CtrlClrIntrHwvad(DMIC0, false);
}


void DMIC0_Transfer_Callback(DMIC_Type *base, dmic_dma_handle_t *handle, status_t status, void *userData) 
{
  AudioSource_Queue_Element element;
  uint32_t size;
  element.pBufAddress = &audio_capture_dmic_DMABuff[AUDIO_CAPTURE_TRANSFER_SIZE * audio_capture_dmic_buffer_index];
  element.uBufSize = AUDIO_CAPTURE_TRANSFER_SIZE * sizeof(int16_t); /* uBufSize in samples */
  
  audio_capture_dmic_buffer_index ^= 1;
//  PRINTF("%d\r\n",audio_capture_dmic_buffer_index);
  
  if(audio_capture_dmic_started == 1)
  {
      p_dmic_last_address = element.pBufAddress;
      dmic_last_size = element.uBufSize;
#ifdef __XCC__
      ret = xos_msgq_put(p_audio_capture_dmic_queue, (uint32_t *)&element);
      if (ret != XOS_OK) {
        PRINTF("xos_msgq_put p_audio_capture_dmic_queue %d", ret);
      }
#else

      size = dc_ringbuf_write(p_dcringbuf_context, (uint8_t *)element.pBufAddress, element.uBufSize);

#ifdef  CHECK_DATA     
   if((audio_dump_pos < AUDIO_DUMP_MAX) && (audio_dump_pos != AUDIO_DUMP_DONE)) {
               p_audio_dump_addr = (uint8_t *)(AUDIO_DUMP_START_0 + audio_dump_pos);
               memcpy(p_audio_dump_addr, element.pBufAddress, element.uBufSize);
               audio_dump_pos += element.uBufSize;
        } else {
            if(audio_dump_pos != AUDIO_DUMP_DONE)
            {
                PRINTF("Dump Done\n");
            }
            audio_dump_pos = AUDIO_DUMP_DONE;
                                }
#endif

      if (size != element.uBufSize) {
          PRINTF("dc_ringbuf_write overflow\n");
	  }
#endif
  }
  

}


void ctimer_match0_callback(uint32_t flags)
{
  uint32_t hwvad_flag;  
  hwvad_flag = NVIC_GetPendingIRQ(HWVAD0_IRQn);
  if(hwvad_flag)
  {
      PRINTF("Y\r\n");
      DMIC_CtrlClrIntrHwvad(DMIC0, true);
      /* Delay to clear first spurious interrupt and let the filter converge */
      for (int i = 0; i <= 500U; i++)
      {
        
      }
      /*HWVAD Normal operation */
      DMIC_CtrlClrIntrHwvad(DMIC0, false);
      NVIC_ClearPendingIRQ(HWVAD0_IRQn);		   
  }  
  else
  {
     PRINTF("N\r\n");
     dsp_started = 0;
     GPIO_PinWrite(GPIO, 0U, 14U, 0);
     EnableDeepSleepIRQ(HWVAD0_IRQn);  
 //    DSP_Stop();  
     DSP_POWERDOWN();     
     CTIMER_StopTimer(CTIMER2);
     DMIC_PDM_500k();
     dc_ringbuf_setmode(p_dcringbuf_context, RINGBUF_MODE_HALF);
  } 
}


void DMIC_PDM_24M(void)
{
    CLOCK_AttachClk(kFRO_DIV4_to_DMIC);
#if USE_FRO_96M
    CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 30);
#else
    CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 60);
#endif
    DMIC_ResetChannelDecimator(DMIC0, kDMIC_EnableChannel0, true);
    dmic_channel_cfg.osr = 25U;
    dmic_channel_cfg.gainshft = 5U;
    DMIC_ConfigChannel(DMIC0, kDMIC_Channel0, kDMIC_Left, &dmic_channel_cfg);
    DMIC_ResetChannelDecimator(DMIC0, kDMIC_EnableChannel0, false);
}

void DMIC_PDM_500k(void)
{
    CLOCK_AttachClk(kLPOSC_to_DMIC);
    CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 2U);
    DMIC_ResetChannelDecimator(DMIC0, kDMIC_EnableChannel0, true);
    dmic_channel_cfg.osr = 16U;
    dmic_channel_cfg.gainshft = 8U;
    DMIC_ConfigChannel(DMIC0, kDMIC_Channel0, kDMIC_Left, &dmic_channel_cfg);
    DMIC_ResetChannelDecimator(DMIC0, kDMIC_EnableChannel0, false);
}

void Init_DMIC_DMA(void)
{
   /* Configure DMAMUX. */
    RESET_PeripheralReset(kINPUTMUX_RST_SHIFT_RSTn);
    INPUTMUX_Init(INPUTMUX);
    /* Enable DMA request */
    INPUTMUX_EnableSignal(INPUTMUX, kINPUTMUX_Dmic0Ch0ToDmac0Ch16RequestEna, true);
    /* Turnoff clock to inputmux to save power. Clock is only needed to make changes */
    INPUTMUX_Deinit(INPUTMUX);
    
    CLOCK_AttachClk(kLPOSC_to_DMIC);
    CLOCK_SetClkDiv(kCLOCK_DivDmicClk, 2U);
    DMA_Init(DMA0);
    DMA_EnableChannel(DMA0, 16U);
    DMA_CreateHandle(&s_dmicRxDmaHandle, DMA0, 16U);
    
    memset(&dmic_channel_cfg,0U,sizeof(dmic_channel_config_t));
    dmic_channel_cfg.divhfclk			 = kDMIC_PdmDiv1;
    dmic_channel_cfg.osr		         = 16;
    dmic_channel_cfg.gainshft			 = 8U;
    dmic_channel_cfg.preac2coef 		 = kDMIC_CompValueZero;
    dmic_channel_cfg.preac4coef 		 = kDMIC_CompValueZero;
    dmic_channel_cfg.dc_cut_level		 = kDMIC_DcCut155;
    dmic_channel_cfg.post_dc_gain_reduce         = 1U;
    dmic_channel_cfg.saturate16bit		 = 1U;
    dmic_channel_cfg.sample_rate		 = kDMIC_PhyFullSpeed;
#if defined(FSL_FEATURE_DMIC_CHANNEL_HAS_SIGNEXTEND) && (FSL_FEATURE_DMIC_CHANNEL_HAS_SIGNEXTEND)
    dmic_channel_cfg.enableSignExtend = false;
#endif
    
    DMIC_Init(DMIC0);
#if !(defined(FSL_FEATURE_DMIC_HAS_NO_IOCFG) && FSL_FEATURE_DMIC_HAS_NO_IOCFG)
    DMIC_SetIOCFG(DMIC0, kDMIC_PdmStereo);
#endif
    
    DMIC_Use2fs(DMIC0, true);
    DMIC_EnableChannelDma(DMIC0, kDMIC_Channel0, true);
    DMIC_ConfigChannel(DMIC0, kDMIC_Channel0, kDMIC_Left, &dmic_channel_cfg);
    DMIC_FifoChannel(DMIC0, kDMIC_Channel0, FIFO_DEPTH, true, true);
    
    DMIC_EnableChannnel(DMIC0, DMIC_CHANEN_EN_CH0(1));
    
    DMIC_TransferCreateHandleDMA(DMIC0, &s_dmicDmaHandle, DMIC0_Transfer_Callback, NULL, &s_dmicRxDmaHandle);
    DMIC_InstallDMADescriptorMemory(&s_dmicDmaHandle, s_dmaDescriptorPingpong, 2U);       
}




void InitHWVAD(void)
{
      /*Gain of the noise estimator */
    DMIC_SetGainNoiseEstHwvad(DMIC0, 0x02U);

    /*Gain of the signal estimator */
    DMIC_SetGainSignalEstHwvad(DMIC0, 0x01U);

    /* 00 = first filter by-pass, 01 = hpf_shifter=1, 10 = hpf_shifter=4 */
    DMIC_SetFilterCtrlHwvad(DMIC0, 0x01U);

    /*input right-shift of (GAIN x 2 -10) bits (from -10bits (0000) to +14bits (1100)) */
    DMIC_SetInputGainHwvad(DMIC0, 0x04U);

    DisableDeepSleepIRQ(HWVAD0_IRQn);
    DisableIRQ(HWVAD0_IRQn);

    DMIC_FilterResetHwvad(DMIC0, true);
    DMIC_FilterResetHwvad(DMIC0, false);
    /* reset hwvad internal interrupt */
    DMIC_CtrlClrIntrHwvad(DMIC0, true);
    /* Delay to clear first spurious interrupt and let the filter converge */
    SDK_DelayAtLeastUs(200000, SystemCoreClock);
    /*HWVAD Normal operation */
    DMIC_CtrlClrIntrHwvad(DMIC0, false);
    DMIC_HwvadEnableIntCallback(DMIC0, DMIC0_HWVAD_Callback);
}




void StartDMIC_DMA_Transfer(void)
{
  SYSCTL0->HWWAKE |= SYSCTL0_HWWAKE_DMICWAKE_MASK | SYSCTL0_HWWAKE_DMAC0WAKE_MASK;
  DMIC_TransferReceiveDMA(DMIC0, &s_dmicDmaHandle, s_receiveXfer, kDMIC_Channel0);
  EnableDeepSleepIRQ(HWVAD0_IRQn);
  audio_capture_dmic_started = 1;
}

void InitCTimer(void)
{
  ctimer_config_t config;
  /* Use MAIN clock for the Ctimer2 */
  CLOCK_AttachClk(kMAIN_CLK_to_CTIMER2);  
  CTIMER_GetDefaultConfig(&config);
  CTIMER_Init(CTIMER2, &config);
  
  /* Configuration 0 */
  matchConfig0.enableCounterReset = true;
  matchConfig0.enableCounterStop  = false;
  matchConfig0.matchValue         = CLOCK_GetCtimerClkFreq(2) * 3;//3s
  matchConfig0.outControl         = kCTIMER_Output_NoAction;
  matchConfig0.outPinInitState    = false;
  matchConfig0.enableInterrupt    = true;
  
  CTIMER_RegisterCallBack(CTIMER2, &ctimer_callback_table[0], kCTIMER_SingleCallback);
  CTIMER_SetupMatch(CTIMER2, kCTIMER_Match_0, &matchConfig0);
}


void VIT_Launcher(void *Handle)
{
  /* Init board hardware. */
  pca9420_modecfg_t pca9420ModeCfg[4];
  uint32_t exclude_from_pd[4];
  /* BE CAUTIOUS TO SET CORRECT VOLTAGE RANGE ACCORDING TO YOUR BOARD/APPLICATION. PAD SUPPLY BEYOND THE RANGE DO
     HARM TO THE SILICON. */
  power_pad_vrange_t vrange = {.Vdde0Range = kPadVol_171_198,
                               .Vdde1Range = kPadVol_171_198,
                               .Vdde2Range = kPadVol_171_198,
                               .Vdde3Range = kPadVol_300_360,
                               .Vdde4Range = kPadVol_171_198};
  
  exclude_from_pd[0] = (SYSCTL0_PDRUNCFG0_RBBSRAM_PD_MASK | SYSCTL0_PDRUNCFG0_RBB_PD_MASK | SYSCTL0_PDRUNCFG0_LPOSC_PD_MASK);
  exclude_from_pd[1] = (SYSCTL0_PDSLEEPCFG1_FLEXSPI0_SRAM_APD_MASK | SYSCTL0_PDSLEEPCFG1_SRAM_SLEEP_MASK);
  exclude_from_pd[2] = 0xFFFFFFFFU;
  exclude_from_pd[3] = 0;
  
 /* PMIC PCA9420 */
  BOARD_InitPmic();
  for (uint32_t i = 0; i < ARRAY_SIZE(pca9420ModeCfg); i++)
  {
      PCA9420_GetDefaultModeConfig(&pca9420ModeCfg[i]);
  }
  BOARD_ConfigPMICModes(pca9420ModeCfg, ARRAY_SIZE(pca9420ModeCfg));
  PCA9420_WriteModeConfigs(&pca9420Handle, kPCA9420_Mode0, &pca9420ModeCfg[0], ARRAY_SIZE(pca9420ModeCfg));

#if USE_FRO_96M  
  PMIC_LVD_Adjust();
  BOARD_SetPmicVoltage_0V750();
#else
    /* Configure PMIC Vddcore value according to CM33 clock. DSP not used in this demo. */
  BOARD_SetPmicVoltageForFreq(CLOCK_GetFreq(kCLOCK_CoreSysClk), 0U); 
#endif 


  /* Indicate to power library that PMIC is used. */
  POWER_UpdatePmicRecoveryTime(1);

  POWER_SetPadVolRange(&vrange);
  
  /* Determine the power mode before bring up. */
  if ((POWER_GetEventFlags() & PMC_FLAGS_DEEPPDF_MASK) != 0)
  {
      PRINTF("Board wake up from deep or full deep power down mode.\r\n");
      POWER_ClearEventFlags(PMC_FLAGS_DEEPPDF_MASK);
  }
  
  /* Enable MUA clock before run DSP core */
  CLOCK_EnableClock(kCLOCK_Mu);
  RESET_PeripheralReset(kMU_RST_SHIFT_RSTn);


  RESET_PeripheralReset(kSEMA_RST_SHIFT_RSTn);
  SEMA42_Init(SEMA42);
  SEMA42_ResetAllGates(SEMA42);
  memset((void *)BOARD_TX_AUDIOSOURCE_ARM_SHMEM_BASE, 0, BOARD_TX_AUDIOSOURCE_ARM_SHMEM_SIZE);
  p_dcringbuf_context = dc_ringbuf_create((uint8_t*)BOARD_TX_AUDIOSOURCE_ARM_SHMEM_BASE,
                        BOARD_TX_AUDIOSOURCE_ARM_SHMEM_SIZE);
  dc_ringbuf_setmode(p_dcringbuf_context, RINGBUF_MODE_HALF);


  POWER_EnablePD(kPDRUNCFG_PD_SYSPLL_ANA);
  POWER_EnablePD(kPDRUNCFG_PD_SYSPLL_LDO);
  POWER_EnablePD(kPDRUNCFG_APD_FLEXSPI1_SRAM);
  POWER_EnablePD(kPDRUNCFG_PPD_FLEXSPI1_SRAM);
  POWER_EnablePD(kPDRUNCFG_LP_HSPAD_SDIO0_VDET);
  POWER_EnablePD(kPDRUNCFG_PD_HSPAD_SDIO0_REF);
  POWER_EnablePD(kPDRUNCFG_LP_HSPAD_SDIO1_VDET);
  POWER_EnablePD(kPDRUNCFG_PD_HSPAD_SDIO1_REF);
  POWER_EnablePD(kPDRUNCFG_PD_ROM);
  POWER_EnablePD(kPDRUNCFG_PD_OTP);
  POWER_ApplyPD();
 
  InitCTimer();
  Init_DMIC_DMA();
  InitHWVAD();
  StartDMIC_DMA_Transfer();
//  BOARD_DSP_Init();
    
  PRINTF("Deepsleep\r\n");
  while((USART0->STAT & USART_STAT_TXIDLE_MASK) != USART_STAT_TXIDLE_MASK);
  while (1)
  {
    if(dsp_started == 0)
    {
      POWER_EnterDeepSleep(exclude_from_pd);
//      POWER_EnterSleep();
    }
    else
    {
      POWER_EnterSleep();
    }
  
  }
}

