/*
 * Copyright 2022 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

//#define FUSIONF1_XOS_SUPPORTED 1
#include "ringbuf.h"
#include "fsl_debug_console.h"

#define AUDIO_BUFFER_ONE_QUEUE_SIZE      10 * 16 //10ms data,160 samples
#define AUDIO_BUFFER_SIZE                40 * AUDIO_BUFFER_ONE_QUEUE_SIZE//total size 400ms,6400 samples
#define AUDIO_BASE_ADDRESS (void *)      0x00A00000

//#define CHECK_AUDIO_DATA


//#ifdef CHECK_AUDIO_DATA
//    #define AUDIO_DUMP_START_0 0x00A80000
//    #define AUDIO_DUMP_MAX     0x00020000
//    #define AUDIO_DUMP_DONE    0x00CC0000
//static uint32_t audio_dump_pos = 0;
//uint8_t *p_audio_dump_addr = NULL;
//#endif


static uint32_t totalSize = 0;//total size wrote to array(samples)
static uint32_t isRepeat = 0;//if the write data is repeated
static uint32_t firstRead = 1;

typedef struct _AudioBuf {
    AudioSample buf[AUDIO_BUFFER_SIZE];
    uint32_t read_pt;
    uint32_t write_pt;
    uint32_t size;
} AudioBuf;

#if (FUSIONF1_XOS_SUPPORTED == 1)
static XosMutex ring_buf_mutex;
#endif


static AudioBuf audio_buffer;


//#pragma location = 0x20200000
//__root const uint16_t array[AUDIO_BUFFER_SIZE] = {0};

//#pragma location = 0x00A00000
//__root static AudioBuf audio_buffer;



void audiobuf_init(void)
{
    audio_buffer.read_pt = 0;
    audio_buffer.write_pt = 0;
    audio_buffer.size = 0;


#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_create(&ring_buf_mutex, XOS_MUTEX_WAIT_PRIORITY, 0);
#endif
}

void audiobuf_reset(void)
{
    audio_buffer.read_pt = 0;
    audio_buffer.write_pt = 0;
    audio_buffer.size = 0;
}

/* */
int32_t audiobuf_fill(AudioSample *data, uint32_t size_in_sample)
{
    uint32_t temp = 0;
#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_lock(&ring_buf_mutex);
#endif
    if(totalSize + size_in_sample <= AUDIO_BUFFER_SIZE)
    {
      isRepeat = 0;
      memcpy(&audio_buffer.buf[audio_buffer.write_pt],data,size_in_sample * sizeof(AudioSample));
      totalSize += size_in_sample;//calculate by samples
      audio_buffer.write_pt += size_in_sample;//calculate by samples
      audio_buffer.size += size_in_sample;
    }
    else
    {
      isRepeat = 1;
      if(audio_buffer.write_pt + size_in_sample <= AUDIO_BUFFER_SIZE)
      {
         memcpy(&audio_buffer.buf[audio_buffer.write_pt], data, size_in_sample * sizeof(AudioSample));
         audio_buffer.write_pt += size_in_sample;
         totalSize += size_in_sample;
      }
      else
      {
         temp = AUDIO_BUFFER_SIZE - audio_buffer.write_pt;
         memcpy(&audio_buffer.buf[audio_buffer.write_pt], data, temp * sizeof(AudioSample));
         memcpy(&audio_buffer.buf[0], data + temp, (size_in_sample - temp)* sizeof(AudioSample));
         audio_buffer.write_pt = size_in_sample - temp;
         totalSize += size_in_sample;
      }
      audio_buffer.size = ((audio_buffer.size + size_in_sample > AUDIO_BUFFER_SIZE)? AUDIO_BUFFER_SIZE : (audio_buffer.size + size_in_sample));
    }
    if(audio_buffer.write_pt >= AUDIO_BUFFER_SIZE)
      audio_buffer.write_pt -= AUDIO_BUFFER_SIZE;

#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_unlock(&ring_buf_mutex);
#endif
    return 0;
}


int32_t audiobuf_retrive(AudioSample *data, uint32_t size_in_sample)
{
    uint32_t temp = 0;

#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_lock(&ring_buf_mutex);
#endif
    if (size_in_sample > audio_buffer.size)
    {
        return -1;
    }
    if(firstRead)
    {
      firstRead = 0;
      if(audio_buffer.write_pt > AUDIO_BUFFER_SIZE / 2)
      {
        audio_buffer.read_pt = audio_buffer.write_pt - (AUDIO_BUFFER_SIZE / 2);  //Reserve 100ms of audio data
      }
      else
      {
        audio_buffer.read_pt = AUDIO_BUFFER_SIZE - (AUDIO_BUFFER_SIZE/2 - audio_buffer.write_pt);
      }
    }

    if (audio_buffer.read_pt + size_in_sample <= AUDIO_BUFFER_SIZE) {
        memcpy(data, &audio_buffer.buf[audio_buffer.read_pt], size_in_sample * sizeof(AudioSample));
        audio_buffer.read_pt += size_in_sample;
    } else {
        temp = AUDIO_BUFFER_SIZE - audio_buffer.read_pt;
        memcpy(data, &audio_buffer.buf[audio_buffer.read_pt], temp * sizeof(AudioSample));
        memcpy(data + temp, &audio_buffer.buf[0], (size_in_sample - temp)* sizeof(AudioSample));
        audio_buffer.read_pt = size_in_sample - temp;
    }

    audio_buffer.size -= size_in_sample;
#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_unlock(&ring_buf_mutex);
#endif
    return 0;
}

static uint32_t audio_read_pt;



int32_t audiobuf_retrive_dsp(AudioSample *data, uint32_t size_in_sample,uint32_t write_pt,uint32_t size)
{
    uint32_t temp = 0;
    uint32_t audio_write_pt = write_pt;
    uint32_t audio_size = size;
    uint32_t testTemp;

#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_lock(&ring_buf_mutex);
#endif

	if(size_in_sample > audio_size)
	{
		return -1;
	}
	if(firstRead)
	{
		firstRead = 0;
		if(audio_write_pt >= AUDIO_BUFFER_SIZE / 2)
		{
			audio_read_pt = (audio_write_pt - AUDIO_BUFFER_SIZE / 2);//Reserve 100ms of audio data
		}
		else
		{
			audio_read_pt = AUDIO_BUFFER_SIZE - (AUDIO_BUFFER_SIZE/2 - audio_write_pt);

		}

	}
    if (audio_read_pt + size_in_sample <= AUDIO_BUFFER_SIZE) {
    	memcpy(data, AUDIO_BASE_ADDRESS + (audio_read_pt * sizeof(int16_t)),size_in_sample * sizeof(AudioSample));
    	audio_read_pt += size_in_sample;
    } else {
        temp = AUDIO_BUFFER_SIZE - audio_read_pt;
        memcpy(data, AUDIO_BASE_ADDRESS + (audio_read_pt * sizeof(int16_t)), temp * sizeof(AudioSample));
        memcpy(data + temp, AUDIO_BASE_ADDRESS, (size_in_sample - temp)* sizeof(AudioSample));
        audio_read_pt = size_in_sample - temp;
    }

    if(audio_read_pt >= AUDIO_BUFFER_SIZE)
    {
    	audio_read_pt -= AUDIO_BUFFER_SIZE;
    }

#ifdef CHECK_AUDIO_DATA
    testTemp = audio_read_pt;
    if(audio_read_pt >= size_in_sample)
    	audio_read_pt -= size_in_sample;
    else
    	audio_read_pt = AUDIO_BUFFER_SIZE - size_in_sample;

    if((audio_dump_pos < AUDIO_DUMP_MAX) && (audio_dump_pos != AUDIO_DUMP_DONE)) {
                                                  p_audio_dump_addr = (uint8_t *)(AUDIO_DUMP_START_0 + audio_dump_pos);
                memcpy(p_audio_dump_addr, (AUDIO_BASE_ADDRESS + (audio_read_pt * sizeof(int16_t))), size_in_sample * sizeof(int16_t));
                                                  audio_dump_pos += size_in_sample * sizeof(int16_t);
          } else {
              if(audio_dump_pos != AUDIO_DUMP_DONE)
              {
                  PRINTF("Dump Done\n");
              }
              audio_dump_pos = AUDIO_DUMP_DONE;
                                  }
    audio_read_pt = testTemp;
#endif



    audio_size -= size_in_sample;

#if (FUSIONF1_XOS_SUPPORTED == 1)
	xos_mutex_unlock(&ring_buf_mutex);
#endif
    return audio_read_pt;

}


int32_t audiobuf_data_avail(void)
{
    return audio_buffer.size;
}

int32_t audiobuf_maxsize(void)
{
    return AUDIO_BUFFER_SIZE;
}

